@tool
extends EditorPlugin

# importers
const NoopImportPlugin = preload("importers/noop_import_plugin.gd")
const SpriteFramesImportPlugin = preload("importers/sprite_frames_import_plugin.gd")
const SpriteFramesSplitImportPlugin = preload("res://addons/AsepriteWizard/importers/sprite_frames_multiple_import_plugin.gd")
const TilesetTextureImportPlugin = preload("importers/tileset_texture_import_plugin.gd")
const TextureImportPlugin = preload("importers/static_texture_import_plugin.gd")
const TextureSplitImportPlugin = preload("res://addons/AsepriteWizard/importers/static_texture_multiple_import_plugin.gd")
const LayerSpriteFramesImportPlugin = preload("res://addons/AsepriteWizard/importers/split_layer_importers/layer_import_plugin.gd")
const LayerTextureImportPlugin = preload("res://addons/AsepriteWizard/importers/split_layer_importers/layer_texture_import_plugin.gd")
const FileSystemHelper = preload("importers/helpers/file_system_helper.gd")
# export
const ExportPlugin = preload("export/export_plugin.gd")
# interface
const ConfigDialog = preload('config/config_dialog.tscn')
const WizardWindow = preload("interface/docks/wizard/as_wizard_dock_container.tscn")
const AsepriteDockImportsWindow = preload('interface/imports_manager/aseprite_imports_manager.tscn')
const ImportsManagerPanels = preload('interface/imports_manager/import_panels.tscn')

const AnimatedSpriteInspectorPlugin = preload("interface/docks/animated_sprite/inspector_plugin.gd")
const SpriteInspectorPlugin = preload("interface/docks/sprite/inspector_plugin.gd")

const tool_menu_name = "Aseprite Wizard"
const menu_item_name = "Spritesheet Wizard Dock..."
const config_menu_item_name = "Config..."
const import_menu_item_name = "Imports Manager..."

var file_lifecycle_listeners = preload("files/file_lifecycle_listeners.gd").new()
var config = preload("config/config.gd").new()
var window: TabContainer
var config_window: Window
var imports_list_window: Window
var imports_list_panel: MarginContainer
var export_plugin : EditorExportPlugin
var sprite_inspector_plugin: EditorInspectorPlugin
var animated_sprite_inspector_plugin: EditorInspectorPlugin

var file_system_helper

var _importers = []

var _is_import_list_docked = false

func _enter_tree():
	_load_config()
	_setup_menu_entries()
	_setup_importer()
	_setup_exporter()
	_setup_animated_sprite_inspector_plugin()
	_setup_sprite_inspector_plugin()
	_setup_file_listeners()


func _exit_tree():
	_disable_plugin()


func _disable_plugin():
	_remove_menu_entries()
	_remove_importer()
	_remove_exporter()
	_remove_wizard_dock()
	_remove_inspector_plugins()
	_remove_file_listeners()


func _load_config():
	config.initialize_project_settings()


func _setup_menu_entries():
	var submenu = PopupMenu.new()
	add_tool_submenu_item(tool_menu_name, submenu)
	submenu.add_item(menu_item_name)
	submenu.add_item(import_menu_item_name)
	submenu.add_item(config_menu_item_name)
	submenu.index_pressed.connect(_on_tool_menu_pressed)


func _remove_menu_entries():
	remove_tool_menu_item(tool_menu_name)


func _setup_importer():
	file_system_helper = FileSystemHelper.new()
	add_child(file_system_helper)

	_importers = [
		NoopImportPlugin.new(),
		SpriteFramesImportPlugin.new(),
		SpriteFramesSplitImportPlugin.new(file_system_helper),
		TilesetTextureImportPlugin.new(),
		TextureImportPlugin.new(),
		TextureSplitImportPlugin.new(file_system_helper),
		LayerSpriteFramesImportPlugin.new(),
		LayerTextureImportPlugin.new(),
	]

	for i in _importers:
		add_import_plugin(i)


func _remove_importer():
	for i in _importers:
		remove_import_plugin(i)

	if file_system_helper != null:
		file_system_helper.queue_free()
		file_system_helper = null


func _setup_exporter():
	export_plugin = ExportPlugin.new()
	add_export_plugin(export_plugin)


func _remove_exporter():
	remove_export_plugin(export_plugin)


func _setup_sprite_inspector_plugin():
	sprite_inspector_plugin = SpriteInspectorPlugin.new()
	add_inspector_plugin(sprite_inspector_plugin)


func _setup_animated_sprite_inspector_plugin():
	animated_sprite_inspector_plugin = AnimatedSpriteInspectorPlugin.new()
	add_inspector_plugin(animated_sprite_inspector_plugin)


func _remove_inspector_plugins():
	remove_inspector_plugin(sprite_inspector_plugin)
	remove_inspector_plugin(animated_sprite_inspector_plugin)


func _remove_wizard_dock():
	if window:
		remove_control_from_bottom_panel(window)
		window.queue_free()
		window = null


func _open_window():
	if window:
		make_bottom_panel_item_visible(window)
		return

	window = WizardWindow.instantiate()
	window.connect("close_requested",Callable(self,"_on_window_closed"))
	add_control_to_bottom_panel(window, "Aseprite Wizard")
	make_bottom_panel_item_visible(window)


func _open_config_dialog():
	if is_instance_valid(config_window):
		config_window.queue_free()

	config_window = ConfigDialog.instantiate()
	get_editor_interface().get_base_control().add_child(config_window)
	config_window.popup_centered_ratio(0.5)


func _open_import_list_dialog():
	if is_instance_valid(imports_list_window):
		imports_list_window.queue_free()

	if is_instance_valid(imports_list_panel):
		if _is_import_list_docked:
			remove_control_from_bottom_panel(imports_list_panel)
			_is_import_list_docked = false
		imports_list_panel.queue_free()
		imports_list_panel = null

	imports_list_panel = ImportsManagerPanels.instantiate()
	imports_list_panel.dock_requested.connect(_on_import_list_dock_requested)
	_create_imports_manager_window(imports_list_panel)


func _on_window_closed():
	if window:
		remove_control_from_bottom_panel(window)
		window.queue_free()
		window = null


func _on_tool_menu_pressed(index):
	match index:
		0: # wizard dock
			_open_window()
		1: # imports
			_open_import_list_dialog()
		2: # config
			_open_config_dialog()


func _on_import_list_dock_requested():
	if _is_import_list_docked:
		remove_control_from_bottom_panel(imports_list_panel)
		_is_import_list_docked = false
		_create_imports_manager_window(imports_list_panel)
		imports_list_panel.show()
		imports_list_panel.anchors_preset = Control.PRESET_FULL_RECT
		imports_list_panel.size_flags_vertical = Control.SIZE_EXPAND_FILL
		imports_list_panel.size_flags_horizontal = Control.SIZE_EXPAND_FILL
		imports_list_panel.set_as_floating()
		return

	_is_import_list_docked = true
	imports_list_panel.set_as_docked()
	imports_list_window.remove_child(imports_list_panel)
	imports_list_window.queue_free()
	add_control_to_bottom_panel(imports_list_panel, "Aseprite Imports Manager")
	make_bottom_panel_item_visible(imports_list_panel)


func _create_imports_manager_window(panel: MarginContainer):
	imports_list_window = AsepriteDockImportsWindow.instantiate()
	imports_list_window.add_child(panel)
	get_editor_interface().get_base_control().add_child(imports_list_window)
	imports_list_window.popup_centered_ratio(0.5)


func _setup_file_listeners() -> void:
	file_lifecycle_listeners.setup_listeners()


func _remove_file_listeners() -> void:
	file_lifecycle_listeners.remove_listeners()
