@tool
extends RefCounted

const logger = preload("../config/logger.gd")
var _config = preload("../config/config.gd").new()

#
# Output:
# {
#   "data_file": file path to the json file
#   "sprite_sheet": file path to the raw image file
# }
func export_file(file_name: String, output_folder: String, options: Dictionary) -> Dictionary:
	var exception_pattern = options.get('exception_pattern', "")
	var only_visible_layers = options.get('only_visible_layers', false)
	var output_name = file_name if options.get('output_filename') == "" else options.get('output_filename', file_name)
	var first_frame_only = options.get("first_frame_only", false)
	var scale = options.get('scale', '1')
	var basename = _get_file_basename(output_name)
	var output_dir = ProjectSettings.globalize_path(output_folder)
	var data_file = "%s/%s.json" % [output_dir, basename]
	var sprite_sheet = "%s/%s.png" % [output_dir, basename]
	var output = []
	var arguments = _export_command_common_arguments(file_name, data_file, sprite_sheet)

	if scale != "1":
		arguments.push_back("--scale")
		arguments.push_back(scale)

	if not only_visible_layers:
		arguments.push_front("--all-layers")

	if first_frame_only:
		arguments.push_front("'[0, 0]'")
		arguments.push_front("--frame-range")

	_add_sheet_type_arguments(arguments, options)

	_add_ignore_layer_arguments(file_name, arguments, exception_pattern)

	var local_sprite_sheet_path = ProjectSettings.localize_path(sprite_sheet)
	var is_new = not ResourceLoader.exists(local_sprite_sheet_path)

	var exit_code = _execute(arguments, output)
	if exit_code != 0:
		logger.error('Failed to export spritesheet: %s' % output, file_name)
		return {}

	return {
		"data_file": ProjectSettings.localize_path(data_file),
		"sprite_sheet": local_sprite_sheet_path,
		"is_first_import": is_new,
	}


func export_layers(file_name: String, output_folder: String, options: Dictionary) -> Array:
	var exception_pattern = options.get('exception_pattern', "")
	var only_visible_layers = options.get('only_visible_layers', false)
	var basename = _get_file_basename(file_name)
	var layers = list_layers(file_name, only_visible_layers)
	var exception_regex = _compile_regex(exception_pattern)

	var output = []

	for layer in layers:
		if layer != "" and (not exception_regex or exception_regex.search(layer) == null):
			output.push_back(export_file_with_layers(file_name, [layer], output_folder, options))

	return output


func export_file_with_layers(file_name: String, layer_names: Array, output_folder: String, options: Dictionary) -> Dictionary:
	var output_prefix = options.get('output_filename', "").strip_edges()
	var output_dir = output_folder.replace("res://", "./").strip_edges()
	var flat_file_name = layer_names[0].replace("/", "_")
	var base_output_path = "%s/%s%s" % [output_dir, output_prefix, flat_file_name if layer_names.size() == 1 else ""]
	var data_file = "%s.json" % base_output_path
	var sprite_sheet = "%s.png" % base_output_path
	var trim_cels = options.get("trim_cels", false)
	var first_frame_only = options.get("first_frame_only", false)
	var scale = options.get('scale', '1')
	var output = []
	var arguments = _export_command_common_arguments(file_name, data_file, sprite_sheet)

	if scale != "1":
		arguments.push_back("--scale")
		arguments.push_back(scale)

	for layer_name in layer_names:
		arguments.push_front(layer_name)
		arguments.push_front("--layer")

	if trim_cels:
		arguments.push_front("--trim")

	if first_frame_only:
		arguments.push_front("'[0, 0]'")
		arguments.push_front("--frame-range")

	_add_sheet_type_arguments(arguments, options)

	var local_sprite_sheet_path = ProjectSettings.localize_path(sprite_sheet)
	var is_new = not ResourceLoader.exists(local_sprite_sheet_path)

	var exit_code = _execute(arguments, output)
	if exit_code != 0:
		logger.error('Failed to export layer spritesheet: %s' % output, file_name)
		return {}

	return {
		"data_file": ProjectSettings.localize_path(data_file),
		"sprite_sheet": local_sprite_sheet_path,
		"is_first_import": is_new,
	}


func _add_ignore_layer_arguments(file_name: String, arguments: Array, exception_pattern: String):
	var layers = _get_exception_layers(file_name, exception_pattern)
	if not layers.is_empty():
		for l in layers:
			arguments.push_front(l)
			arguments.push_front('--ignore-layer')


func _add_sheet_type_arguments(arguments: Array, options : Dictionary):
	var sheet_type = options.get("sheet_type", "packed")
	var item_count = options.get("sheet_columns", 0)

	if (sheet_type == "columns" or sheet_type == "rows") and item_count == 0:
		sheet_type = "packed"
	elif options.get("sheet_merge_duplicates", true):
		arguments.push_back("--merge-duplicates")

	if sheet_type == "columns":
		arguments.push_back("--sheet-columns")
		arguments.push_back(item_count)
	else:
		arguments.push_back("--sheet-type")
		arguments.push_back(sheet_type)

	var frame_padding = options.get("frame_padding", 0)
	arguments.push_back("--shape-padding")
	arguments.push_back(frame_padding)


func _get_exception_layers(file_name: String, exception_pattern: String) -> Array:
	var layers = list_layers(file_name)
	var regex = _compile_regex(exception_pattern)
	if regex == null:
		return []

	var exception_layers = []
	for layer in layers:
		if regex.search(layer) != null:
			exception_layers.push_back(layer)

	return exception_layers


func list_valid_layers(file_path: String, exception_pattern: String = "", show_only_visible: bool = false, should_merge_duplicates: bool = false, skip_group_layers: bool = false) -> Array:
	var layers = []

	if should_merge_duplicates:
		layers = list_layers_without_duplicates(file_path, show_only_visible)
	else:
		layers = list_layers(file_path, show_only_visible, skip_group_layers)

	var exception_regex = _compile_regex(exception_pattern)

	var output = []

	for layer in layers:
		if layer != "" and (not exception_regex or exception_regex.search(layer) == null):
			output.push_back(layer)

	return output


func list_layers(file_path: String, only_visible = false, skip_group_layers: bool = false) -> Array:
	var output = []
	var arguments = ["-b", "--list-layer-hierarchy", file_path]

	if not only_visible:
		arguments.push_front("--all-layers")

	var exit_code = _execute(arguments, output)

	if exit_code != 0:
		logger.error('Failed listing layers: %s' % output, file_path)
		return []

	if output.is_empty():
		return output

	var lines: PackedStringArray = output[0].strip_edges().split('\n')
	var paths = []
	var stack = []
	
	for line in lines:
		var indent_level = 0
		while line.begins_with('  '):
			indent_level += 1
			line = line.substr(2)
		
		stack = stack.slice(0, indent_level)
		
		if line.ends_with('/'):
			var dir_name = line.rstrip('/').strip_edges()
			stack.append(dir_name)
			if not skip_group_layers:
				paths.append("/".join(stack))
		else:
			var file_name = line.strip_edges()
			var full_path = '/'.join(stack + [file_name])
			paths.append(full_path)
	return paths


func list_layers_without_duplicates(file_path: String, only_visible = false) -> Array:
	var output_dir = OS.get_cache_dir()
	var file_name = file_path.get_file()
	var data_path = "%s/%s.json" % [output_dir, file_name];

	var arguments = [
		"-b",
		"--split-layers",
		"--trim",
		"--merge-duplicates",
		"--format", "json-array",
		"--data", data_path,
		file_path
	]

	if not only_visible:
		arguments.push_front("--all-layers")

	var output = []
	var exit_code = _execute(arguments, output)

	if exit_code != 0:
		logger.error('Failed listing layers: %s' % output, file_path)
		return []

	var file = FileAccess.open(data_path, FileAccess.READ)
	var data = JSON.parse_string(file.get_as_text())

	if data.is_empty():
		return output

	var regex = RegEx.new()
	regex.compile("(?<=\\().*(?=\\))")

	var frames = []
	frames = data.frames;
	var sanitizedFrames = {}
	for frame in frames:
		if(sanitizedFrames.find_key(frame.frame) == null):
			var result = regex.search(frame.filename)
			sanitizedFrames[frame.frame] = result.get_string()

	return sanitizedFrames.values()


func list_slices(file_name: String) -> Array:
	var output = []
	var arguments = ["-b", "--list-slices", file_name]

	var exit_code = _execute(arguments, output)

	if exit_code != 0:
		logger.error('Failed listing slices: %s' % output, file_name)
		return []

	if output.is_empty():
		return output

	var raw = output[0].split('\n')
	var sanitized = []
	for s in raw:
		sanitized.append(s.strip_edges())
	return sanitized


func _export_command_common_arguments(source_name: String, data_path: String, spritesheet_path: String) -> Array:
	return [
		"-b",
		"--list-tags",
		"--list-slices",
		"--data",
		data_path,
		"--format",
		"json-array",
		"--sheet",
		spritesheet_path,
		source_name,
	]


func _execute(arguments, output):
	return OS.execute(_aseprite_command(), arguments, output, true, true)


func _aseprite_command() -> String:
	return _config.get_command()


func _get_file_basename(file_path: String) -> String:
	return file_path.get_file().trim_suffix('.%s' % file_path.get_extension())


func _compile_regex(pattern):
	if pattern == "":
		return

	var rgx = RegEx.new()
	if rgx.compile(pattern) == OK:
		return rgx

	logger.error('Exception regex error')


func test_command() -> bool:
	var exit_code = OS.execute(_aseprite_command(), ['--version'], [], true)
	return exit_code == 0


func is_valid_spritesheet(content):
	return content is Dictionary and content.has("frames") and content.has("meta") and content.meta.has('image')


func get_content_frames(content):
	return content.frames if typeof(content.frames) == TYPE_ARRAY  else content.frames.values()


func get_slice_rect(content: Dictionary, slice_name: String) -> Variant:
	if not content.has("meta") or not content.meta.has("slices"):
		return null
	for slice in content.meta.slices:
		if slice.name == slice_name:
			if slice.keys.size() > 0:
				var p = slice.keys[0].bounds
				return Rect2(p.x, p.y, p.w, p.h)
	return null


##
## Exports tileset layers
##
## Return (dictionary):
##      data_file: path to aseprite generated JSON file
##      sprite_sheet: localized path to spritesheet file
func export_tileset_texture(file_name: String, output_folder: String, options: Dictionary) -> Dictionary:
	var exception_pattern = options.get('exception_pattern', "")
	var only_visible_layers = options.get('only_visible_layers', false)
	var output_name = file_name if options.get('output_filename') == "" else options.get('output_filename', file_name)
	var basename = _get_file_basename(output_name)
	var output_dir = ProjectSettings.globalize_path(output_folder)
	var data_path = "%s/%s.json" % [output_dir, basename]
	var sprite_sheet = "%s/%s.png" % [output_dir, basename]
	var output = []

	var arguments = [
		"-b",
		"--export-tileset",
		"--data",
		data_path,
		"--format",
		"json-array",
		"--sheet",
		sprite_sheet,
		file_name
	]

	if not only_visible_layers:
		arguments.push_front("--all-layers")

	_add_sheet_type_arguments(arguments, options)

	_add_ignore_layer_arguments(file_name, arguments, exception_pattern)

	var exit_code = _execute(arguments, output)
	if exit_code != 0:
		logger.error('Failed to export spritesheet: %s' % output, file_name)
		return {}

	return {
		"data_file": ProjectSettings.localize_path(data_path),
		"sprite_sheet": ProjectSettings.localize_path(sprite_sheet)
	}
