from pathlib import Path
import tempfile
import pytest
import co_op_translator.utils.common.file_utils as file_utils
from co_op_translator.utils.common.file_utils import (
    delete_translated_images_by_language_code,
    delete_translated_markdown_files_by_language_code,
    filter_files,
    generate_translated_filename,
    get_actual_image_path,
    get_filename_and_extension,
    get_unique_id,
    handle_empty_document,
    map_original_to_translated,
    migrate_images_to_webp,
    migrate_translated_image_filenames,
    read_input_file,
    reset_translation_directories,
    update_readme_languages_table,
    write_output_file,
)


@pytest.fixture
def temp_dir(tmp_path):
    """Create a temporary directory for testing."""
    return tmp_path


@pytest.fixture
def test_file(temp_dir):
    """Create a test file with content."""
    file_path = temp_dir / "test.txt"
    with file_path.open("w", encoding="utf-8") as f:
        f.write("Test content\nLine 2")
    return file_path


@pytest.fixture
def empty_file(temp_dir):
    """Create an empty test file."""
    file_path = temp_dir / "empty.txt"
    file_path.touch()
    return file_path


def test_read_input_file(test_file):
    """Test reading content from a file."""
    content = read_input_file(test_file)
    assert content == "Test content\nLine 2"


def test_write_output_file(temp_dir):
    """Test writing content to a file."""
    output_file = temp_dir / "output.txt"
    results = ["Line 1", "Line 2", "Line 3"]
    write_output_file(output_file, results)

    content = output_file.read_text(encoding="utf-8")
    assert content.strip() == "\n".join(results)


def test_handle_empty_document(empty_file, temp_dir):
    """Test handling empty document by copying it."""
    output_file = temp_dir / "output.txt"
    handle_empty_document(empty_file, output_file)
    assert output_file.exists()
    assert output_file.stat().st_size == 0


def test_get_unique_id(temp_dir):
    """Test generating unique ID for a file path."""
    file_path = temp_dir / "dir" / "file.txt"
    file_path.parent.mkdir(parents=True)
    file_path.touch()

    unique_id = get_unique_id(file_path, temp_dir)
    assert isinstance(unique_id, str)
    assert len(unique_id) > 0


def test_cross_platform_path_hash_consistency(temp_dir, monkeypatch):
    """Test that path hashing is consistent across different OS path separators."""
    import os

    # Setup test directory structure
    test_dir = temp_dir / "test_dir"
    test_dir.mkdir()
    file_path = test_dir / "subdir" / "test_file.txt"
    file_path.parent.mkdir(parents=True)
    file_path.touch()

    # Case 1: Get hash with default OS path separator
    original_hash = get_unique_id(file_path, temp_dir)

    # Case 2: Simulate Windows-style paths with backslashes
    windows_style_path = str(file_path).replace("/", "\\")
    windows_hash = get_unique_id(windows_style_path, temp_dir)

    # Case 3: Simulate Unix-style paths with forward slashes
    unix_style_path = str(file_path).replace("\\", "/")
    unix_hash = get_unique_id(unix_style_path, temp_dir)

    # All hashes should be identical regardless of path separator style
    assert (
        original_hash == windows_hash
    ), "Hash should be the same regardless of Windows or default style paths"
    assert (
        original_hash == unix_hash
    ), "Hash should be the same regardless of Unix or default style paths"
    assert (
        windows_hash == unix_hash
    ), "Hash should be the same regardless of Windows or Unix style paths"


def test_get_filename_and_extension():
    """Test extracting filename and extension."""
    test_cases = [
        ("file.txt", ("file", ".txt")),
        ("path/to/file.md", ("file", ".md")),
        ("file", ("file", "")),
        (".gitignore", (".gitignore", "")),
    ]

    for input_path, expected in test_cases:
        filename, ext = get_filename_and_extension(input_path)
        assert (filename, ext) == expected


def test_filter_files(temp_dir):
    """Test filtering files in a directory."""
    # Create test directory structure
    (temp_dir / "dir1").mkdir()
    (temp_dir / "dir2").mkdir()
    (temp_dir / "file1.txt").touch()
    (temp_dir / "file2.txt").touch()
    (temp_dir / "dir1/file3.txt").touch()

    excluded_dirs = {"dir1"}
    files = filter_files(temp_dir, excluded_dirs)

    assert len(files) == 2
    assert all(f.name.endswith(".txt") for f in files)
    assert not any("dir1" in str(f) for f in files)


def test_generate_translated_filename(temp_dir):
    """Test generating translated filename."""
    file_path = temp_dir / "dir" / "file.txt"
    file_path.parent.mkdir(parents=True)
    file_path.touch()

    language_code = "ko"
    full_hash = get_unique_id(file_path, temp_dir)
    filename = generate_translated_filename(file_path, language_code, temp_dir)

    # All translated images are now saved as WebP for optimal compression
    assert filename.endswith(".webp")

    parts = filename.split(".")
    # Expect: basename, hash_prefix, ext
    assert len(parts) >= 3
    basename = parts[0]
    hash_prefix = parts[-2]
    ext = "." + parts[-1]

    assert basename == "file"
    assert ext == ".webp"  # Always WebP for translated images

    # Hash prefix should be exactly 16 hex (truncated from the full 64-hex hash)
    assert len(hash_prefix) == 16
    assert full_hash.startswith(hash_prefix)


def test_migrate_translated_image_filenames(temp_dir):
    """Test migrating legacy filenames to language-subdir + 16-hex hash filenames."""

    image_dir = temp_dir / "translated_images"
    lang = "fr"
    lang_dir = image_dir / lang
    lang_dir.mkdir(parents=True, exist_ok=True)

    # Legacy filename embedding a full 64-hex hash
    full_hash = "a" * 64
    legacy_name = f"logo.{full_hash}.{lang}.png"
    legacy_path = lang_dir / legacy_name
    legacy_path.write_text("legacy image content", encoding="utf-8")

    # File that already looks truncated should not be migrated (but may be moved/renamed to drop lang segment)
    truncated_name = f"icon.abcdef1234567890.{lang}.png"
    (lang_dir / truncated_name).write_text("truncated image content", encoding="utf-8")

    rename_map = migrate_translated_image_filenames(image_dir, [lang])

    # Legacy filename should have been migrated to lang/base.hash.ext
    assert legacy_name in rename_map
    new_rel = rename_map[legacy_name]
    new_path = image_dir / new_rel
    assert new_path.exists()
    assert not (lang_dir / legacy_name).exists()

    new_name = new_path.name
    parts = new_name.split(".")
    assert len(parts) >= 3
    base = parts[0]
    hash_prefix = parts[-2]
    extension = parts[-1]

    assert base == "logo"
    assert extension == "png"
    assert len(hash_prefix) == 16
    assert full_hash.startswith(hash_prefix)

    # The already-truncated file should exist but without language segment if migrated
    # It should either be kept as-is under lang dir or renamed to drop '.lang'
    # Accept either presence under new canonical naming or original truncated name
    truncated_new = lang_dir / ("icon.abcdef1234567890.png")
    assert (truncated_new.exists()) or (lang_dir / truncated_name).exists()


def test_migrate_images_to_webp_scoped_to_language_codes(temp_dir):
    from PIL import Image

    image_dir = temp_dir / "translated_images"
    ko_dir = image_dir / "ko"
    ja_dir = image_dir / "ja"
    ko_dir.mkdir(parents=True, exist_ok=True)
    ja_dir.mkdir(parents=True, exist_ok=True)

    # Create simple PNG images for each language
    img = Image.new("RGB", (10, 10), color="red")
    ko_png = ko_dir / "sample.ko.png"
    ja_png = ja_dir / "sample.ja.png"
    img.save(ko_png)
    img.save(ja_png)

    rename_map = migrate_images_to_webp(image_dir, ["ko"])

    # ko images should be converted to WebP
    ko_webp = ko_dir / "sample.ko.webp"
    assert not ko_png.exists()
    assert ko_webp.exists()
    assert "ko/sample.ko.png" in rename_map
    assert rename_map["ko/sample.ko.png"] == "ko/sample.ko.webp"

    # ja images should remain untouched
    ja_webp = ja_dir / "sample.ja.webp"
    assert ja_png.exists()
    assert not ja_webp.exists()
    assert "ja/sample.ja.png" not in rename_map
    assert "sample.ja.png" not in rename_map


def test_get_actual_image_path(temp_dir):
    """Test resolving actual image path."""
    md_file = temp_dir / "doc/readme.md"
    md_file.parent.mkdir(parents=True)
    md_file.touch()

    image_path = "../images/test.png"
    actual_path = get_actual_image_path(image_path, md_file)

    assert isinstance(actual_path, Path)
    assert str(actual_path).endswith("test.png")


def test_reset_translation_directories(temp_dir):
    """Test resetting translation directories."""
    translations_dir = temp_dir / "translations"
    image_dir = temp_dir / "images"
    language_codes = ["ko", "en"]

    # Create existing directories with content
    translations_dir.mkdir()
    image_dir.mkdir()
    (translations_dir / "old.txt").touch()

    reset_translation_directories(translations_dir, image_dir, language_codes)

    assert translations_dir.exists()
    assert image_dir.exists()
    assert not (translations_dir / "old.txt").exists()
    assert all((translations_dir / code).exists() for code in language_codes)


def test_delete_translated_markdown_files(temp_dir):
    """Test deleting translated markdown files."""
    translations_dir = temp_dir / "translations"
    ko_dir = translations_dir / "ko"
    ko_dir.mkdir(parents=True)
    (ko_dir / "test.md").touch()

    delete_translated_markdown_files_by_language_code("ko", translations_dir)
    assert not ko_dir.exists()


def test_map_original_to_translated_exists_and_missing(temp_dir):
    """map_original_to_translated should return the translated path when it exists, else None."""
    root_dir = temp_dir
    translations_dir = root_dir / "translations"
    lang = "ko"

    # Create original structure
    orig_nb = root_dir / "notebook" / "01" / "foo.ipynb"
    orig_nb.parent.mkdir(parents=True)
    orig_nb.touch()

    # Case 1: missing translated counterpart -> None
    mapped_missing = map_original_to_translated(
        original_abs=orig_nb,
        language_code=lang,
        root_dir=root_dir,
        translations_dir=translations_dir,
    )
    assert mapped_missing is None

    # Case 2: exists translated counterpart -> Path
    translated_nb = translations_dir / lang / "notebook" / "01" / "foo.ipynb"
    translated_nb.parent.mkdir(parents=True)
    translated_nb.touch()

    mapped_exists = map_original_to_translated(
        original_abs=orig_nb,
        language_code=lang,
        root_dir=root_dir,
        translations_dir=translations_dir,
    )
    assert mapped_exists is not None
    assert mapped_exists.exists()
    assert mapped_exists == translated_nb.resolve()


def create_dummy_image(lang_dir, fname):
    f = lang_dir / fname
    f.write_text("dummy image content")
    return f


def test_image_dir_language_subfolders():
    with tempfile.TemporaryDirectory() as tmpdir:
        translations_dir = Path(tmpdir) / "translations"
        image_dir = Path(tmpdir) / "translated_images"
        langs = ["fr", "de"]
        reset_translation_directories(translations_dir, image_dir, langs)
        for lang in langs:
            assert (translations_dir / lang).is_dir()
            assert (image_dir / lang).is_dir()


def test_delete_translated_images_by_language_code():
    with tempfile.TemporaryDirectory() as tmpdir:
        image_dir = Path(tmpdir) / "translated_images"
        langs = ["fr", "de"]
        # Setup dirs and dummy files
        reset_translation_directories(Path(tmpdir) / "translations", image_dir, langs)
        for lang in langs:
            lang_dir = image_dir / lang
            lang_dir.mkdir(parents=True, exist_ok=True)
            create_dummy_image(lang_dir, f"img1.{lang}.png")
            create_dummy_image(lang_dir, f"img2.{lang}.jpg")
            assert any(lang_dir.iterdir())
        # Delete 'fr' images
        delete_translated_images_by_language_code("fr", image_dir)
        assert not (image_dir / "fr").exists()
        assert (image_dir / "de").is_dir() and any((image_dir / "de").iterdir())


def _make_readme_with_markers(tmp_path: Path) -> Path:
    readme = tmp_path / "README.md"
    readme.write_text(
        "before\n"
        f"{file_utils.LANG_TABLE_START}\nold-content\n{file_utils.LANG_TABLE_END}\n"
        "after",
        encoding="utf-8",
    )
    return readme


def test_update_readme_languages_table_with_repo_url(monkeypatch, tmp_path):
    template = (
        f"{file_utils.LANG_TABLE_START}\n"
        "[Lang](./translations/lang/README.md)\n\n"
        "> **Prefer to Clone Locally?**\n\n"
        "> ```bash\n"
        "> git clone <repo_url>\n"
        "> cd <repo_name>\n"
        "> ```\n\n"
        "> ```cmd\n"
        "> git clone https://github.com/*****.git\n"
        "> cd *****\n"
        "> ```\n"
        f"{file_utils.LANG_TABLE_END}"
    )
    monkeypatch.setattr(
        file_utils,
        "load_languages_table_template",
        lambda: template,
    )

    readme = _make_readme_with_markers(tmp_path)
    repo_url = "https://github.com/org/repo.git"

    updated = update_readme_languages_table(readme, repo_url=repo_url)
    assert updated is True

    final_text = readme.read_text(encoding="utf-8")
    assert repo_url in final_text
    assert "<repo_url>" not in final_text
    assert "cd repo" in final_text
    assert "cd *****" not in final_text


def test_update_readme_languages_table_without_repo_url(monkeypatch, tmp_path):
    template = (
        f"{file_utils.LANG_TABLE_START}\n"
        "[Lang](./translations/lang/README.md)\n"
        "> git clone <repo_url>\n"
        "> cd <repo_name>\n"
        f"{file_utils.LANG_TABLE_END}"
    )
    monkeypatch.setattr(
        file_utils,
        "load_languages_table_template",
        lambda: template,
    )

    readme = _make_readme_with_markers(tmp_path)

    updated = update_readme_languages_table(readme)
    assert updated is True

    final_text = readme.read_text(encoding="utf-8")
    assert "<repo_url>" in final_text
    assert "<repo_name>" in final_text
