/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.world.gen;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import javax.annotation.Nullable;
import net.minecraft.crash.CrashReport;
import net.minecraft.crash.CrashReportCategory;
import net.minecraft.entity.EnumCreatureType;
import net.minecraft.util.ReportedException;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.ChunkPos;
import net.minecraft.world.MinecraftException;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraft.world.biome.Biome;
import net.minecraft.world.chunk.Chunk;
import net.minecraft.world.chunk.IChunkProvider;
import net.minecraft.world.chunk.storage.AnvilChunkLoader;
import net.minecraft.world.chunk.storage.IChunkLoader;
import net.minecraft.world.gen.IChunkGenerator;
import net.minecraftforge.common.DimensionManager;
import net.minecraftforge.common.ForgeChunkManager;
import net.minecraftforge.common.chunkio.ChunkIOExecutor;
import net.minecraftforge.fml.common.FMLLog;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class ChunkProviderServer
implements IChunkProvider {
    private static final Logger LOGGER = LogManager.getLogger();
    private final Set<Long> droppedChunks = Sets.newHashSet();
    public final IChunkGenerator chunkGenerator;
    public final IChunkLoader chunkLoader;
    public final Long2ObjectMap<Chunk> loadedChunks = new Long2ObjectOpenHashMap(8192);
    public final WorldServer world;
    private final Set<Long> loadingChunks = Sets.newHashSet();

    public ChunkProviderServer(WorldServer worldObjIn, IChunkLoader chunkLoaderIn, IChunkGenerator chunkGeneratorIn) {
        this.world = worldObjIn;
        this.chunkLoader = chunkLoaderIn;
        this.chunkGenerator = chunkGeneratorIn;
    }

    public Collection<Chunk> getLoadedChunks() {
        return this.loadedChunks.values();
    }

    public void queueUnload(Chunk chunkIn) {
        if (this.world.provider.canDropChunk(chunkIn.x, chunkIn.z)) {
            this.droppedChunks.add(ChunkPos.asLong(chunkIn.x, chunkIn.z));
            chunkIn.unloadQueued = true;
        }
    }

    public void queueUnloadAll() {
        for (Chunk chunk : this.loadedChunks.values()) {
            this.queueUnload(chunk);
        }
    }

    @Override
    @Nullable
    public Chunk getLoadedChunk(int x, int z) {
        long i = ChunkPos.asLong(x, z);
        Chunk chunk = (Chunk)this.loadedChunks.get(i);
        if (chunk != null) {
            chunk.unloadQueued = false;
        }
        return chunk;
    }

    @Nullable
    public Chunk loadChunk(int x, int z) {
        return this.loadChunk(x, z, null);
    }

    @Nullable
    public Chunk loadChunk(int x, int z, @Nullable Runnable runnable) {
        Chunk chunk = this.getLoadedChunk(x, z);
        if (chunk == null) {
            long pos = ChunkPos.asLong(x, z);
            chunk = ForgeChunkManager.fetchDormantChunk(pos, this.world);
            if (chunk != null || !(this.chunkLoader instanceof AnvilChunkLoader)) {
                if (!this.loadingChunks.add(pos)) {
                    FMLLog.bigWarning("There is an attempt to load a chunk ({},{}) in dimension {} that is already being loaded. This will cause weird chunk breakages.", x, z, this.world.provider.getDimension());
                }
                if (chunk == null) {
                    chunk = this.loadChunkFromFile(x, z);
                }
                if (chunk != null) {
                    this.loadedChunks.put(ChunkPos.asLong(x, z), (Object)chunk);
                    chunk.onLoad();
                    chunk.populate(this, this.chunkGenerator);
                }
                this.loadingChunks.remove(pos);
            } else {
                AnvilChunkLoader loader = (AnvilChunkLoader)this.chunkLoader;
                if (runnable == null || !ForgeChunkManager.asyncChunkLoading) {
                    chunk = ChunkIOExecutor.syncChunkLoad(this.world, loader, this, x, z);
                } else if (loader.isChunkGeneratedAt(x, z)) {
                    ChunkIOExecutor.queueChunkLoad(this.world, loader, this, x, z, runnable);
                    return null;
                }
            }
        }
        if (runnable != null) {
            runnable.run();
        }
        return chunk;
    }

    @Override
    public Chunk provideChunk(int x, int z) {
        Chunk chunk = this.loadChunk(x, z);
        if (chunk == null) {
            long i = ChunkPos.asLong(x, z);
            try {
                chunk = this.chunkGenerator.generateChunk(x, z);
            }
            catch (Throwable throwable) {
                CrashReport crashreport = CrashReport.makeCrashReport(throwable, "Exception generating new chunk");
                CrashReportCategory crashreportcategory = crashreport.makeCategory("Chunk to be generated");
                crashreportcategory.addCrashSection("Location", String.format("%d,%d", x, z));
                crashreportcategory.addCrashSection("Position hash", i);
                crashreportcategory.addCrashSection("Generator", this.chunkGenerator);
                throw new ReportedException(crashreport);
            }
            this.loadedChunks.put(i, (Object)chunk);
            chunk.onLoad();
            chunk.populate(this, this.chunkGenerator);
        }
        return chunk;
    }

    @Nullable
    private Chunk loadChunkFromFile(int x, int z) {
        try {
            Chunk chunk = this.chunkLoader.loadChunk(this.world, x, z);
            if (chunk != null) {
                chunk.setLastSaveTime(this.world.getTotalWorldTime());
                this.chunkGenerator.recreateStructures(chunk, x, z);
            }
            return chunk;
        }
        catch (Exception exception) {
            LOGGER.error("Couldn't load chunk", (Throwable)exception);
            return null;
        }
    }

    private void saveChunkExtraData(Chunk chunkIn) {
        try {
            this.chunkLoader.saveExtraChunkData(this.world, chunkIn);
        }
        catch (Exception exception) {
            LOGGER.error("Couldn't save entities", (Throwable)exception);
        }
    }

    private void saveChunkData(Chunk chunkIn) {
        try {
            chunkIn.setLastSaveTime(this.world.getTotalWorldTime());
            this.chunkLoader.saveChunk(this.world, chunkIn);
        }
        catch (IOException ioexception) {
            LOGGER.error("Couldn't save chunk", (Throwable)ioexception);
        }
        catch (MinecraftException minecraftexception) {
            LOGGER.error("Couldn't save chunk; already in use by another instance of Minecraft?", (Throwable)minecraftexception);
        }
    }

    public boolean saveChunks(boolean all) {
        int i = 0;
        ArrayList list = Lists.newArrayList((Iterable)this.loadedChunks.values());
        for (int j = 0; j < list.size(); ++j) {
            Chunk chunk = (Chunk)list.get(j);
            if (all) {
                this.saveChunkExtraData(chunk);
            }
            if (!chunk.needsSaving(all)) continue;
            this.saveChunkData(chunk);
            chunk.setModified(false);
            if (++i != 24 || all) continue;
            return false;
        }
        return true;
    }

    public void flushToDisk() {
        this.chunkLoader.flush();
    }

    @Override
    public boolean tick() {
        if (!this.world.disableLevelSaving) {
            if (!this.droppedChunks.isEmpty()) {
                for (ChunkPos forced : this.world.getPersistentChunks().keySet()) {
                    this.droppedChunks.remove(ChunkPos.asLong(forced.x, forced.z));
                }
                Iterator<Long> iterator = this.droppedChunks.iterator();
                int i = 0;
                while (i < 100 && iterator.hasNext()) {
                    Long olong = iterator.next();
                    Chunk chunk = (Chunk)this.loadedChunks.get((Object)olong);
                    if (chunk != null && chunk.unloadQueued) {
                        chunk.onUnload();
                        ForgeChunkManager.putDormantChunk(ChunkPos.asLong(chunk.x, chunk.z), chunk);
                        this.saveChunkData(chunk);
                        this.saveChunkExtraData(chunk);
                        this.loadedChunks.remove((Object)olong);
                        ++i;
                    }
                    iterator.remove();
                }
            }
            if (this.loadedChunks.isEmpty()) {
                DimensionManager.unloadWorld(this.world.provider.getDimension());
            }
            this.chunkLoader.chunkTick();
        }
        return false;
    }

    public boolean canSave() {
        return !this.world.disableLevelSaving;
    }

    @Override
    public String makeString() {
        return "ServerChunkCache: " + this.loadedChunks.size() + " Drop: " + this.droppedChunks.size();
    }

    public List<Biome.SpawnListEntry> getPossibleCreatures(EnumCreatureType creatureType, BlockPos pos) {
        return this.chunkGenerator.getPossibleCreatures(creatureType, pos);
    }

    @Nullable
    public BlockPos getNearestStructurePos(World worldIn, String structureName, BlockPos position, boolean findUnexplored) {
        return this.chunkGenerator.getNearestStructurePos(worldIn, structureName, position, findUnexplored);
    }

    public boolean isInsideStructure(World worldIn, String structureName, BlockPos pos) {
        return this.chunkGenerator.isInsideStructure(worldIn, structureName, pos);
    }

    public int getLoadedChunkCount() {
        return this.loadedChunks.size();
    }

    public boolean chunkExists(int x, int z) {
        return this.loadedChunks.containsKey(ChunkPos.asLong(x, z));
    }

    @Override
    public boolean isChunkGeneratedAt(int x, int z) {
        return this.loadedChunks.containsKey(ChunkPos.asLong(x, z)) || this.chunkLoader.isChunkGeneratedAt(x, z);
    }
}

