﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.CodeAnalysis.Completion.Log;
using Microsoft.CodeAnalysis.Internal.Log;
using Microsoft.CodeAnalysis.LanguageService;
using Microsoft.CodeAnalysis.Shared.Extensions;
using Microsoft.CodeAnalysis.Shared.Extensions.ContextQuery;
using Roslyn.Utilities;

namespace Microsoft.CodeAnalysis.Completion.Providers;

internal abstract class AbstractExtensionMemberImportCompletionProvider : AbstractImportCompletionProvider
{
    protected abstract bool SupportsStaticExtensionMembers { get; }
    protected abstract string GenericSuffix { get; }

    // Don't provide unimported extension methods if adding import is not supported,
    // since we are current incapable of making a change using its fully qualify form.
    protected override bool ShouldProvideCompletion(CompletionContext completionContext, SyntaxContext syntaxContext)
        => syntaxContext.IsRightOfNameSeparator && IsAddingImportsSupported(completionContext.Document, completionContext.CompletionOptions);

    protected override void LogCommit()
        => CompletionProvidersLogger.LogCommitOfExtensionMethodImportCompletionItem();

    protected override void WarmUpCacheInBackground(Document document)
    {
        _ = ExtensionMemberImportCompletionHelper.WarmUpCacheAsync(document.Project, CancellationToken.None);
    }

    protected override async Task AddCompletionItemsAsync(
        CompletionContext completionContext,
        SyntaxContext syntaxContext,
        HashSet<string> namespaceInScope,
        CancellationToken cancellationToken)
    {
        using (Logger.LogBlock(FunctionId.Completion_ExtensionMethodImportCompletionProvider_GetCompletionItemsAsync, cancellationToken))
        {
            var syntaxFacts = completionContext.Document.GetRequiredLanguageService<ISyntaxFactsService>();
            var (receiverTypeSymbol, isStatic) = TryGetReceiverTypeSymbol(syntaxContext, syntaxFacts, cancellationToken);
            if (receiverTypeSymbol != null)
            {
                var inferredTypes = completionContext.CompletionOptions.TargetTypedCompletionFilter
                    ? syntaxContext.InferredTypes
                    : [];

                var totalTime = SharedStopwatch.StartNew();

                var completionItems = await ExtensionMemberImportCompletionHelper.GetUnimportedExtensionMembersAsync(
                    syntaxContext,
                    receiverTypeSymbol,
                    isStatic,
                    namespaceInScope,
                    inferredTypes,
                    forceCacheCreation: completionContext.CompletionOptions.ForceExpandedCompletionIndexCreation,
                    hideAdvancedMembers: completionContext.CompletionOptions.MemberDisplayOptions.HideAdvancedMembers,
                    cancellationToken).ConfigureAwait(false);

                if (!completionItems.IsDefault)
                {
                    var receiverTypeKey = SymbolKey.CreateString(receiverTypeSymbol, cancellationToken);
                    completionContext.AddItems(completionItems.Select(i => Convert(i, receiverTypeKey)));
                }
            }
        }
    }

    private (ITypeSymbol? receiverTypeSymbol, bool isStatic) TryGetReceiverTypeSymbol(
        SyntaxContext syntaxContext,
        ISyntaxFactsService syntaxFacts,
        CancellationToken cancellationToken)
    {
        var parentNode = syntaxContext.TargetToken.Parent;

        // Even though implicit access to extension method is allowed, we decide not support it for simplicity 
        // e.g. we will not provide completion for unimported extension method in this case
        // New Bar() {.X = .$$ }
        var expressionNode = syntaxFacts.GetLeftSideOfDot(parentNode, allowImplicitTarget: false);
        if (expressionNode is null)
            return default;

        return TryGetReceiverTypeSymbol(syntaxContext.SemanticModel, expressionNode, cancellationToken);
    }

    protected virtual (ITypeSymbol? receiverTypeSymbol, bool isStatic) TryGetReceiverTypeSymbol(
        SemanticModel semanticModel,
        SyntaxNode expressionNode,
        CancellationToken cancellationToken)
    {
        // Check if we are accessing members of a type, no extension methods are exposed off of types.
        if (semanticModel.GetSymbolInfo(expressionNode, cancellationToken).GetAnySymbol() is ITypeSymbol typeSymbol)
        {
            return this.SupportsStaticExtensionMembers ? (typeSymbol, isStatic: true) : default;
        }
        else
        {
            // The expression we're calling off of needs to have an actual instance type.
            // We try to be more tolerant to errors here so completion would still be available in certain case of partially typed code.
            var receiverTypeSymbol = semanticModel.GetTypeInfo(expressionNode, cancellationToken).Type;
            if (receiverTypeSymbol is IErrorTypeSymbol errorTypeSymbol)
                receiverTypeSymbol = errorTypeSymbol.CandidateSymbols.Select(GetSymbolType).FirstOrDefault(s => s != null);

            return (receiverTypeSymbol, isStatic: false);
        }
    }

    private static ITypeSymbol? GetSymbolType(ISymbol symbol)
        => symbol switch
        {
            ILocalSymbol localSymbol => localSymbol.Type,
            IFieldSymbol fieldSymbol => fieldSymbol.Type,
            IPropertySymbol propertySymbol => propertySymbol.Type,
            IParameterSymbol parameterSymbol => parameterSymbol.Type,
            IAliasSymbol aliasSymbol => aliasSymbol.Target as ITypeSymbol,
            _ => symbol as ITypeSymbol,
        };

    private CompletionItem Convert(SerializableImportCompletionItem serializableItem, string receiverTypeSymbolKey)
        => ImportCompletionItem.Create(
            serializableItem.Name,
            serializableItem.Arity,
            serializableItem.ContainingNamespace,
            serializableItem.Glyph,
            GenericSuffix,
            CompletionItemFlags.Expanded,
            (serializableItem.SymbolKeyData, receiverTypeSymbolKey, serializableItem.AdditionalOverloadCount),
            serializableItem.IncludedInTargetTypeCompletion);
}
