import { app } from "/scripts/app.js";
import { $el } from "/scripts/ui.js";

app.registerExtension({
	name: "comfyui.canvas_toolkit.liuguang_highlight",
	setup() {
        console.log("%c ComfyUI 流光高亮 扩展已加载 ", "background: #222; color: #bada55");

        const coerceBool = (value, fallback) => {
            if (value === true || value === "true" || value === 1 || value === "1") return true;
            if (value === false || value === "false" || value === 0 || value === "0") return false;
            return fallback;
        };
        const coerceNumber = (value, fallback) => {
            const n = Number(value);
            return Number.isFinite(n) ? n : fallback;
        };
        const normalizeHex = (value, fallback) => {
            if (typeof value !== "string") return fallback;
            const v = value.trim();
            if (/^#([0-9a-fA-F]{6})$/.test(v)) return v;
            const m = /^#([0-9a-fA-F]{3})$/.exec(v);
            if (m) {
                return `#${m[1].split("").map(ch => ch + ch).join("")}`;
            }
            return fallback;
        };
        const normalizeColor = (value, fallback) => {
            if (typeof value !== "string") return fallback;
            const parts = value.split(",").map(s => s.trim()).filter(Boolean);
            if (parts.length === 0) return fallback;
            const norm = parts.map(p => normalizeHex(p, null));
            if (norm.some(v => !v)) return fallback;
            return norm.join(",");
        };
        const clamp01 = (value, fallback) => {
            const n = coerceNumber(value, fallback);
            if (n < 0) return 0;
            if (n > 1) return 1;
            return n;
        };
        const formatElapsed = (ms) => {
            const totalMs = Math.max(0, Math.floor(ms));
            const totalSeconds = Math.floor(totalMs / 1000);
            const cs = Math.floor((totalMs % 1000) / 10); // centiseconds
            const seconds = totalSeconds % 60;
            const minutes = Math.floor(totalSeconds / 60) % 60;
            const hours = Math.floor(totalSeconds / 3600);
            if (hours > 0) {
                return `${String(hours).padStart(2, "0")}:${String(minutes).padStart(2, "0")}:${String(seconds).padStart(2, "0")}.${String(cs).padStart(2, "0")}`;
            }
            return `${String(minutes).padStart(2, "0")}:${String(seconds).padStart(2, "0")}.${String(cs).padStart(2, "0")}`;
        };

        let highlightEnabled = true;
        let breathingEnabled = false;
        let autoBreathingEnabled = false;
        let breathingPeriodMs = 500;
        let breathingStrength = 100;
        let breathingColor = "#fafafa";
        let breathingSizeScale = 2;
        let breathingBrightness = 2;
        let timeEnabled = true;
        let timeColor = "#ff4d00";
        let timeBgOpacity = 1.0;
        let timeShadowOpacity = 0.5;
        let runningNodeId = null;
        let runningStartTime = 0;
        let lastRunningNodeId = null;
        let lastMouseMoveTime = 0;
        let mouseBreathPeriodMs = 0;
        let mouseBreathPhaseStart = 0;
        let mouseListenerAttached = false;
        let tickHandle = null;
        let lastHighlightTime = 0;
        let lastMouseMoveHandleTime = 0;
        
        // New State Variables
        let missingInputColor = "#FF9F0A";
        let errorColor = "#FF3B30";
        let loadedPresets = [];
        let defaultPresets = [];
        let userPresets = [];
        const USER_PRESET_KEY = "ct_highlight_user_presets_v1";
        const CONFIG_KEY = "ct_highlight_config_v1";
        let lastErrorNodeId = null;
        let pendingSave = 0;
        
        const COLOR_SETTING_ID = "Liuguang.Highlight.Breathing.Color";
        const TIME_COLOR_SETTING_ID = "Liuguang.Highlight.Time.Color";
        let highlightSetting, enabledSetting, autoBreathingSetting, periodSetting, strengthSetting, sizeSetting, brightnessSetting, colorSetting, timeEnabledSetting, timeColorSetting, timeBgOpacitySetting, timeShadowOpacitySetting;

        const applyDefaults = (defaults) => {
            if (!defaults) return;
            if (defaults.highlight_enabled !== undefined) highlightEnabled = coerceBool(defaults.highlight_enabled, true);
            if (defaults.breathing_enabled !== undefined) breathingEnabled = coerceBool(defaults.breathing_enabled, false);
            if (defaults.auto_breathing !== undefined) autoBreathingEnabled = coerceBool(defaults.auto_breathing, false);
            if (defaults.breathing_period_ms !== undefined) breathingPeriodMs = coerceNumber(defaults.breathing_period_ms, 500);
            if (defaults.breathing_strength !== undefined) breathingStrength = coerceNumber(defaults.breathing_strength, 100);
            if (defaults.breathing_size_scale !== undefined) breathingSizeScale = coerceNumber(defaults.breathing_size_scale, 2);
            if (defaults.breathing_brightness !== undefined) breathingBrightness = coerceNumber(defaults.breathing_brightness, 2);
            if (defaults.breathing_color !== undefined) breathingColor = normalizeColor(defaults.breathing_color, "#fafafa");
            if (defaults.time_enabled !== undefined) timeEnabled = coerceBool(defaults.time_enabled, true);
            if (defaults.time_color !== undefined) timeColor = normalizeHex(defaults.time_color, "#ff4d00");
            if (defaults.time_bg_opacity !== undefined) timeBgOpacity = clamp01(defaults.time_bg_opacity, 1.0);
            if (defaults.time_shadow_opacity !== undefined) timeShadowOpacity = clamp01(defaults.time_shadow_opacity, 0.5);
        };

        const getConfig = () => ({
            highlightEnabled,
            breathingEnabled,
            autoBreathingEnabled,
            breathingPeriodMs,
            breathingStrength,
            breathingColor,
            breathingSizeScale,
            breathingBrightness,
            timeEnabled,
            timeColor,
            timeBgOpacity,
            timeShadowOpacity,
            missingInputColor,
            errorColor,
        });

        const loadConfig = () => {
            try{
                const raw = localStorage.getItem(CONFIG_KEY);
                if(!raw) return null;
                const cfg = JSON.parse(raw);
                if(!cfg || typeof cfg !== "object") return null;
                return cfg;
            }catch{
                return null;
            }
        };

        const migrateConfig = (cfg) => {
            if (!cfg || typeof cfg !== "object") return cfg;
            const legacyMissing = new Set(["#E0B0FF", "#e0b0ff"]);
            const legacyError = new Set(["#9932CC", "#9932cc"]);
            if (typeof cfg.missingInputColor === "string" && legacyMissing.has(cfg.missingInputColor.trim())) {
                cfg.missingInputColor = "#FF9F0A";
            }
            if (typeof cfg.errorColor === "string" && legacyError.has(cfg.errorColor.trim())) {
                cfg.errorColor = "#FF3B30";
            }
            return cfg;
        };

        const saveConfig = () => {
            try{
                const cfg = getConfig();
                localStorage.setItem(CONFIG_KEY, JSON.stringify(cfg));
            }catch{}
        };

        const scheduleSave = () => {
            if(pendingSave) return;
            pendingSave = window.setTimeout(() => {
                pendingSave = 0;
                saveConfig();
            }, 200);
        };

        const setConfig = (patch) => {
            if (!patch || typeof patch !== "object") return;
            if (patch.highlightEnabled !== undefined) highlightEnabled = coerceBool(patch.highlightEnabled, highlightEnabled);
            if (patch.breathingEnabled !== undefined) breathingEnabled = coerceBool(patch.breathingEnabled, breathingEnabled);
            if (patch.autoBreathingEnabled !== undefined) autoBreathingEnabled = coerceBool(patch.autoBreathingEnabled, autoBreathingEnabled);
            if (patch.breathingPeriodSec !== undefined) breathingPeriodMs = coerceNumber(patch.breathingPeriodSec, breathingPeriodMs / 1000) * 1000;
            if (patch.breathingPeriodMs !== undefined) breathingPeriodMs = coerceNumber(patch.breathingPeriodMs, breathingPeriodMs);
            if (patch.breathingStrength !== undefined) breathingStrength = coerceNumber(patch.breathingStrength, breathingStrength);
            if (patch.breathingColor !== undefined) {
                breathingColor = normalizeColor(patch.breathingColor, breathingColor);
                if (colorSetting && breathingColor && colorSetting.value !== breathingColor) {
                    try {
                        colorSetting.value = breathingColor;
                        app.ui?.settings?.setSettingValue?.(COLOR_SETTING_ID, breathingColor);
                    } catch {}
                }
            }
            if (patch.breathingSizeScale !== undefined) breathingSizeScale = coerceNumber(patch.breathingSizeScale, breathingSizeScale);
            if (patch.breathingBrightness !== undefined) breathingBrightness = coerceNumber(patch.breathingBrightness, breathingBrightness);
            if (patch.timeEnabled !== undefined) timeEnabled = coerceBool(patch.timeEnabled, timeEnabled);
            if (patch.timeColor !== undefined) timeColor = normalizeHex(patch.timeColor, timeColor);
            if (patch.timeBgOpacity !== undefined) timeBgOpacity = clamp01(patch.timeBgOpacity, timeBgOpacity);
            if (patch.timeShadowOpacity !== undefined) timeShadowOpacity = clamp01(patch.timeShadowOpacity, timeShadowOpacity);
            if (patch.missingInputColor !== undefined) missingInputColor = normalizeColor(patch.missingInputColor, missingInputColor);
            if (patch.errorColor !== undefined) errorColor = normalizeColor(patch.errorColor, errorColor);

            app.graph?.setDirtyCanvas?.(true, true);
            if (tickHandle === null) scheduleTick(0);
            scheduleSave();
        };

        const notifyReady = () => {
            window.__ct_highlight_api = {
                getConfig,
                setConfig,
                getPresets: () => loadedPresets.slice(),
                addPreset: (name, colors) => {
                    const cleanName = (name || "").trim() || `自定义预设${(userPresets?.length || 0) + 1}`;
                    const normalized = normalizeColor(colors, "");
                    if (!normalized) return false;
                    const existing = userPresets.find(p => (p?.colors || []).join(",") === normalized);
                    if (existing) {
                        existing.name = cleanName;
                    } else {
                        userPresets.push({ name: cleanName, colors: normalized.split(","), __user: true });
                    }
                    try {
                        const out = userPresets.map(p => {
                            const { __user, ...rest } = p || {};
                            return rest;
                        });
                        localStorage.setItem(USER_PRESET_KEY, JSON.stringify(out));
                    } catch {}
                    loadedPresets = [...defaultPresets, ...userPresets];
                    return true;
                },
                removePreset: (colors) => {
                    const normalized = normalizeColor(colors, "");
                    if (!normalized) return "invalid";
                    const idx = userPresets.findIndex(p => (p?.colors || []).join(",") === normalized);
                    if (idx >= 0) {
                        userPresets.splice(idx, 1);
                        try {
                            const out = userPresets.map(p => {
                                const { __user, ...rest } = p || {};
                                return rest;
                            });
                            localStorage.setItem(USER_PRESET_KEY, JSON.stringify(out));
                        } catch {}
                        loadedPresets = [...defaultPresets, ...userPresets];
                        return "removed";
                    }
                    const inDefault = defaultPresets.some(p => (p?.colors || []).join(",") === normalized);
                    if (inDefault) return "readonly";
                    return "not-found";
                },
            };
            window.dispatchEvent(new CustomEvent("ct-highlight-ready"));
        };

        const registerSettings = () => {
            if (app.ui?.settings?.addSetting) {
                highlightSetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Enabled",
                    name: "流光高亮：高亮框",
                    type: "boolean",
                    defaultValue: highlightEnabled,
                    onChange(value) {
                        highlightEnabled = coerceBool(value, true);
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                highlightEnabled = coerceBool(highlightSetting?.value, highlightEnabled);

                const enabledSetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Breathing.Enabled",
                    name: "流光高亮：鼠标触发呼吸",
                    type: "boolean",
                    defaultValue: breathingEnabled,
                    onChange(value) {
                        breathingEnabled = coerceBool(value, true);
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                breathingEnabled = coerceBool(enabledSetting?.value, breathingEnabled);

                const autoBreathingSetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Breathing.Auto",
                    name: "流光高亮：自动呼吸",
                    type: "boolean",
                    defaultValue: autoBreathingEnabled,
                    onChange(value) {
                        autoBreathingEnabled = coerceBool(value, true);
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                autoBreathingEnabled = coerceBool(autoBreathingSetting?.value, autoBreathingEnabled);

                const periodSetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Breathing.PeriodMs",
                    name: "流光高亮：呼吸周期 (s)",
                    defaultValue: breathingPeriodMs,
                    type: () => {
                        const id = "Liuguang-Highlight-Breathing-PeriodMs";
                        const input = $el("input", {
                            id,
                            type: "number",
                            min: 0.1,
                            step: 0.1,
                            value: (coerceNumber(periodSetting.value, breathingPeriodMs) / 1000).toFixed(1),
                            style: { 
                                width: "80px",
                                padding: "2px 6px",
                                borderRadius: "4px",
                                border: "1px solid var(--border-color, #333)",
                                backgroundColor: "var(--comfy-input-bg, #222)",
                                color: "var(--input-text, #ddd)"
                            },
                            oninput: (e) => {
                                // Only update local variable for preview, don't trigger setting save/UI refresh
                                const val = parseFloat(e.target.value);
                                if (!isNaN(val) && val > 0) {
                                    breathingPeriodMs = val * 1000;
                                    app.graph?.setDirtyCanvas?.(true, true);
                                }
                            },
                            onchange: (e) => {
                                // Update persistent setting on commit (blur/enter)
                                const val = parseFloat(e.target.value);
                                if (!isNaN(val) && val > 0) {
                                    periodSetting.value = val * 1000;
                                    breathingPeriodMs = val * 1000;
                                }
                            },
                            onkeydown: (e) => {
                                e.stopPropagation();
                            }
                        });
                        return $el("tr", [
                            $el("td", [$el("label", { for: id, textContent: "流光高亮：呼吸周期 (s):" })]),
                            $el("td", [input]),
                        ]);
                    },
                    onChange(value) {
                        breathingPeriodMs = coerceNumber(value, 1885);
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                breathingPeriodMs = coerceNumber(periodSetting?.value, breathingPeriodMs);

                const strengthSetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Breathing.Strength",
                    name: "流光高亮：呼吸强度",
                    defaultValue: breathingStrength,
                    type: () => {
                        const id = "Liuguang-Highlight-Breathing-Strength";
                        const input = $el("input", {
                            id,
                            type: "number",
                            min: 0,
                            max: 100,
                            step: 1,
                            value: coerceNumber(strengthSetting.value, breathingStrength),
                            style: { 
                                width: "80px",
                                padding: "2px 6px",
                                borderRadius: "4px",
                                border: "1px solid var(--border-color, #333)",
                                backgroundColor: "var(--comfy-input-bg, #222)",
                                color: "var(--input-text, #ddd)"
                            },
                            oninput: (e) => {
                                // Only update local variable for preview
                                let val = coerceNumber(e.target.value, 60);
                                if (val < 0) val = 0;
                                if (val > 100) val = 100;
                                breathingStrength = val;
                                app.graph?.setDirtyCanvas?.(true, true);
                            },
                            onchange: (e) => {
                                // Update persistent setting on commit
                                let val = coerceNumber(e.target.value, 60);
                                if (val < 0) val = 0;
                                if (val > 100) val = 100;
                                strengthSetting.value = val;
                                breathingStrength = val;
                            },
                             onkeydown: (e) => {
                                e.stopPropagation();
                            }
                        });
                        const unitLabel = $el("span", {
                            textContent: "%",
                            style: { marginLeft: "4px" },
                        });
                        return $el("tr", [
                            $el("td", [$el("label", { for: id, textContent: "流光高亮：呼吸强度:" })]),
                            $el("td", [input, unitLabel]),
                        ]);
                    },
                    onChange(value) {
                        breathingStrength = coerceNumber(value, 60);
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                breathingStrength = coerceNumber(strengthSetting?.value, breathingStrength);

                const sizeSetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Breathing.Size",
                    name: "流光高亮：呼吸大小",
                    defaultValue: breathingSizeScale,
                    type: () => {
                        const id = "Liuguang-Highlight-Breathing-Size";
                        const input = $el("input", {
                            id,
                            type: "number",
                            min: 0.2,
                            max: 10,
                            step: 0.1,
                            value: coerceNumber(sizeSetting.value, breathingSizeScale),
                            style: { 
                                width: "80px",
                                padding: "2px 6px",
                                borderRadius: "4px",
                                border: "1px solid var(--border-color, #333)",
                                backgroundColor: "var(--comfy-input-bg, #222)",
                                color: "var(--input-text, #ddd)"
                            },
                            oninput: (e) => {
                                let val = coerceNumber(e.target.value, 1);
                                if (val < 0.2) val = 0.2;
                                if (val > 10) val = 10;
                                breathingSizeScale = val;
                                app.graph?.setDirtyCanvas?.(true, true);
                            },
                            onchange: (e) => {
                                let val = coerceNumber(e.target.value, 1);
                                if (val < 0.2) val = 0.2;
                                if (val > 10) val = 10;
                                sizeSetting.value = val;
                                breathingSizeScale = val;
                            },
                             onkeydown: (e) => {
                                e.stopPropagation();
                            }
                        });
                        return $el("tr", [
                            $el("td", [$el("label", { for: id, textContent: "流光高亮：呼吸大小:" })]),
                            $el("td", [input]),
                        ]);
                    },
                    onChange(value) {
                        let val = coerceNumber(value, 1);
                        if (val < 0.2) val = 0.2;
                        if (val > 10) val = 10;
                        breathingSizeScale = val;
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                breathingSizeScale = coerceNumber(sizeSetting?.value, breathingSizeScale);

                const brightnessSetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Breathing.Brightness",
                    name: "流光高亮：呼吸亮度",
                    defaultValue: breathingBrightness,
                    type: () => {
                        const id = "Liuguang-Highlight-Breathing-Brightness";
                        const input = $el("input", {
                            id,
                            type: "number",
                            min: 0,
                            max: 2,
                            step: 0.1,
                            value: coerceNumber(brightnessSetting.value, breathingBrightness),
                            style: { 
                                width: "80px",
                                padding: "2px 6px",
                                borderRadius: "4px",
                                border: "1px solid var(--border-color, #333)",
                                backgroundColor: "var(--comfy-input-bg, #222)",
                                color: "var(--input-text, #ddd)"
                            },
                            oninput: (e) => {
                                let val = coerceNumber(e.target.value, 1);
                                if (val < 0) val = 0;
                                if (val > 2) val = 2;
                                breathingBrightness = val;
                                app.graph?.setDirtyCanvas?.(true, true);
                            },
                            onchange: (e) => {
                                let val = coerceNumber(e.target.value, 1);
                                if (val < 0) val = 0;
                                if (val > 2) val = 2;
                                brightnessSetting.value = val;
                                breathingBrightness = val;
                            },
                             onkeydown: (e) => {
                                e.stopPropagation();
                            }
                        });
                        return $el("tr", [
                            $el("td", [$el("label", { for: id, textContent: "流光高亮：呼吸亮度:" })]),
                            $el("td", [input]),
                        ]);
                    },
                    onChange(value) {
                        let val = coerceNumber(value, 1);
                        if (val < 0) val = 0;
                        if (val > 2) val = 2;
                        breathingBrightness = val;
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                breathingBrightness = coerceNumber(brightnessSetting?.value, breathingBrightness);

                colorSetting = app.ui.settings.addSetting({
                    id: COLOR_SETTING_ID,
                    name: "流光高亮：颜色设置",
                    defaultValue: breathingColor,
                    type: () => {
                        const id = "Liuguang-Highlight-Breathing-Color";
                        
                        // --- Helper to update state ---
                        const parseColors = (val) => {
                            const parts = (typeof val === "string" ? val : "")
                                .split(",")
                                .map(s => s.trim())
                                .filter(Boolean);
                            const first = normalizeHex(parts[0], "#22FF22");
                            const second = normalizeHex(parts[1], first);
                            const third = normalizeHex(parts[2], second);
                            return [first, second, third];
                        };

                        let singlePicker;
                        let startPicker;
                        let midPicker;
                        let endPicker;
                        let gradStart = "#FF0000";
                        let gradMid = "#00FF00";
                        let gradEnd = "#0000FF";

                        const updateColor = (val, save) => {
                            if (input) input.value = val;
                            breathingColor = val;
                            const [s, m, e] = parseColors(val);
                            gradStart = s;
                            gradMid = m;
                            gradEnd = e;
                            if (singlePicker) singlePicker.value = s;
                            if (startPicker) startPicker.value = s;
                            if (midPicker) midPicker.value = m;
                            if (endPicker) endPicker.value = e;
                            app.graph?.setDirtyCanvas?.(true, true);
                            if (save) {
                                colorSetting.value = val;
                                app.ui?.settings?.setSettingValue?.(COLOR_SETTING_ID, val);
                            }
                        };

                        // --- 1. Text Input (Source of Truth) ---
                        const input = $el("input", {
                            id,
                            type: "text",
                            value: colorSetting?.value || breathingColor,
                            placeholder: "#22FF22 或 #FF0000,#0000FF",
                            style: { 
                                width: "100%",
                                padding: "4px",
                                marginBottom: "5px",
                                borderRadius: "4px",
                                border: "1px solid var(--border-color, #333)",
                                backgroundColor: "var(--comfy-input-bg, #222)",
                                color: "var(--input-text, #ddd)",
                                fontFamily: "monospace"
                            },
                            oninput: (e) => updateColor(e.target.value, false),
                            onchange: (e) => updateColor(e.target.value, true),
                            onkeydown: (e) => e.stopPropagation()
                        });

                        // --- 2. Color Pickers ---
                        const createPicker = (initial, onPreview, onCommit, title) => {
                            return $el("input", {
                                type: "color",
                                value: normalizeHex(initial, "#000000"),
                                title: title,
                                style: { 
                                    width: "24px", height: "24px", padding: 0, 
                                    border: "1px solid #666", cursor: "pointer", 
                                    backgroundColor: "transparent", borderRadius: "4px"
                                },
                                oninput: (e) => onPreview(e.target.value),
                                onchange: (e) => onCommit(e.target.value)
                            });
                        };

                        const currVal = colorSetting?.value || breathingColor;
                        const [currStart, currMid, currEnd] = parseColors(currVal);
                        
                        // Single Color Picker
                        singlePicker = createPicker(currStart, (val) => updateColor(val, false), (val) => updateColor(val, true), "单色选择");
                        
                        // Gradient Pickers
                        // Try to parse current gradient or default
                        gradStart = currStart;
                        gradMid = currMid;
                        gradEnd = currEnd;
                        
                        const updateGradientPreview = () => updateColor(`${gradStart},${gradMid},${gradEnd}`, false);
                        const updateGradientCommit = () => updateColor(`${gradStart},${gradMid},${gradEnd}`, true);
                        
                        startPicker = createPicker(
                            gradStart,
                            (val) => { gradStart = val; updateGradientPreview(); },
                            (val) => { gradStart = val; updateGradientCommit(); },
                            "渐变起始色"
                        );
                        midPicker = createPicker(
                            gradMid,
                            (val) => { gradMid = val; updateGradientPreview(); },
                            (val) => { gradMid = val; updateGradientCommit(); },
                            "渐变中间色"
                        );
                        endPicker = createPicker(
                            gradEnd,
                            (val) => { gradEnd = val; updateGradientPreview(); },
                            (val) => { gradEnd = val; updateGradientCommit(); },
                            "渐变结束色"
                        );

                        // --- 3. Presets ---
                        const createPreset = (colors, label) => {
                            const bg = colors.includes(",") ? `linear-gradient(to right, ${colors})` : colors;
                            return $el("div", {
                                title: label,
                                style: {
                                    width: "18px", height: "18px", borderRadius: "50%",
                                    background: bg, cursor: "pointer", border: "1px solid #888"
                                },
                                onclick: () => updateColor(colors, true)
                            });
                        };

                        let presetElements = [];
                        if (loadedPresets && loadedPresets.length > 0) {
                            presetElements = loadedPresets.map(p => {
                                const colorStr = Array.isArray(p.colors) ? p.colors.join(",") : p.colors;
                                return createPreset(colorStr, p.name);
                            });
                        } else {
                            presetElements = [
                                createPreset("#22FF22", "默认绿"),
                                createPreset("#FF0000", "红"),
                                createPreset("#0088FF", "蓝"),
                                createPreset("#FF0000,#FFFF00", "火"),
                                createPreset("#00FFFF,#FF00FF", "赛博"),
                                createPreset("#0000FF,#00FFFF", "海洋"),
                                createPreset("#FF0000,#FFFF00,#00FF00,#00FFFF,#0000FF,#FF00FF", "彩虹")
                            ];
                        }

                        // Layout Container
                        const controls = $el("div", { style: { display: "flex", flexDirection: "column", gap: "6px", width: "100%" } }, [
                            input,
                            $el("div", { style: { display: "flex", gap: "8px", alignItems: "center", fontSize: "12px", color: "var(--input-text, #ccc)" } }, [
                                $el("span", { textContent: "单色:" }), singlePicker,
                                $el("div", { style: { width: "1px", height: "16px", background: "#555", margin: "0 4px" } }), // Divider
                                $el("span", { textContent: "渐变:" }), startPicker, midPicker, endPicker
                            ]),
                            $el("div", { style: { display: "flex", gap: "6px", alignItems: "center", flexWrap: "wrap", marginTop: "2px" } }, [
                                $el("span", { textContent: "预设:", style: { fontSize: "12px", color: "var(--input-text, #ccc)" } }), 
                                ...presetElements
                            ])
                        ]);

                        return $el("tr", [
                            $el("td", { style: { verticalAlign: "top", paddingTop: "8px" } }, [
                            $el("label", { for: id, textContent: "流光高亮颜色:" })
                            ]),
                            $el("td", [controls]),
                        ]);
                    },
                    onChange(value) {
                        breathingColor = value || "#22FF22";
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                // Initialize from saved setting or default
                breathingColor = colorSetting?.value || breathingColor;

                const timeEnabledSetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Time.Enabled",
                    name: "流光高亮：时间显示",
                    type: "boolean",
                    defaultValue: timeEnabled,
                    onChange(value) {
                        timeEnabled = coerceBool(value, true);
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                timeEnabled = coerceBool(timeEnabledSetting?.value, timeEnabled);

                timeColorSetting = app.ui.settings.addSetting({
                    id: TIME_COLOR_SETTING_ID,
                    name: "流光高亮：时间颜色",
                    defaultValue: timeColor,
                    type: () => {
                        const id = "Liuguang-Highlight-Time-Color";
                        let textInput;
                        let colorInput;
                        const updateTimeColor = (val, save) => {
                            const nextColor = normalizeHex(val, timeColor || "#22FF22");
                            timeColor = nextColor;
                            if (textInput) textInput.value = nextColor;
                            if (colorInput) colorInput.value = nextColor;
                            app.graph?.setDirtyCanvas?.(true, true);
                            if (save) {
                                timeColorSetting.value = nextColor;
                                app.ui?.settings?.setSettingValue?.(TIME_COLOR_SETTING_ID, nextColor);
                            }
                        };
                        const current = normalizeHex(timeColorSetting?.value || timeColor || "#22FF22", "#22FF22");
                        textInput = $el("input", {
                            id,
                            type: "text",
                            value: current,
                            placeholder: "#22FF22",
                            style: { 
                                width: "100%",
                                padding: "4px",
                                borderRadius: "4px",
                                border: "1px solid var(--border-color, #333)",
                                backgroundColor: "var(--comfy-input-bg, #222)",
                                color: "var(--input-text, #ddd)",
                                fontFamily: "monospace"
                            },
                            oninput: (e) => updateTimeColor(e.target.value, false),
                            onchange: (e) => updateTimeColor(e.target.value, true),
                            onkeydown: (e) => e.stopPropagation()
                        });
                        colorInput = $el("input", {
                            type: "color",
                            value: current,
                            style: { 
                                width: "24px",
                                height: "24px",
                                padding: 0,
                                border: "1px solid #666",
                                cursor: "pointer",
                                backgroundColor: "transparent",
                                borderRadius: "4px"
                            },
                            oninput: (e) => updateTimeColor(e.target.value, false),
                            onchange: (e) => updateTimeColor(e.target.value, true)
                        });
                        return $el("tr", [
                            $el("td", [$el("label", { for: id, textContent: "流光高亮：时间颜色:" })]),
                            $el("td", [
                                $el("div", { style: { display: "flex", alignItems: "center", gap: "8px" } }, [
                                    colorInput,
                                    textInput
                                ])
                            ]),
                        ]);
                    },
                    onChange(value) {
                        timeColor = normalizeHex(value, timeColor || "#22FF22");
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                timeColor = normalizeHex(timeColorSetting?.value || timeColor, "#22FF22");

                const timeBgOpacitySetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Time.BgOpacity",
                    name: "流光高亮：时间背景透明度",
                    defaultValue: timeBgOpacity,
                    type: () => {
                        const id = "Liuguang-Highlight-Time-BgOpacity";
                        const input = $el("input", {
                            id,
                            type: "number",
                            min: 0,
                            max: 1,
                            step: 0.05,
                            value: clamp01(timeBgOpacitySetting.value, timeBgOpacity).toFixed(2),
                            style: { 
                                width: "80px",
                                padding: "2px 6px",
                                borderRadius: "4px",
                                border: "1px solid var(--border-color, #333)",
                                backgroundColor: "var(--comfy-input-bg, #222)",
                                color: "var(--input-text, #ddd)"
                            },
                            oninput: (e) => {
                                timeBgOpacity = clamp01(e.target.value, 0.55);
                                app.graph?.setDirtyCanvas?.(true, true);
                            },
                            onchange: (e) => {
                                timeBgOpacity = clamp01(e.target.value, 0.55);
                                timeBgOpacitySetting.value = timeBgOpacity;
                            },
                            onkeydown: (e) => {
                                e.stopPropagation();
                            }
                        });
                        return $el("tr", [
                            $el("td", [$el("label", { for: id, textContent: "流光高亮：时间背景透明度:" })]),
                            $el("td", [input]),
                        ]);
                    },
                    onChange(value) {
                        timeBgOpacity = clamp01(value, 0.55);
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                timeBgOpacity = clamp01(timeBgOpacitySetting?.value, timeBgOpacity);

                const timeShadowOpacitySetting = app.ui.settings.addSetting({
                    id: "Liuguang.Highlight.Time.ShadowOpacity",
                    name: "流光高亮：时间阴影透明度",
                    defaultValue: timeShadowOpacity,
                    type: () => {
                        const id = "Liuguang-Highlight-Time-ShadowOpacity";
                        const input = $el("input", {
                            id,
                            type: "number",
                            min: 0,
                            max: 1,
                            step: 0.05,
                            value: clamp01(timeShadowOpacitySetting.value, timeShadowOpacity).toFixed(2),
                            style: { 
                                width: "80px",
                                padding: "2px 6px",
                                borderRadius: "4px",
                                border: "1px solid var(--border-color, #333)",
                                backgroundColor: "var(--comfy-input-bg, #222)",
                                color: "var(--input-text, #ddd)"
                            },
                            oninput: (e) => {
                                timeShadowOpacity = clamp01(e.target.value, 0.98);
                                app.graph?.setDirtyCanvas?.(true, true);
                            },
                            onchange: (e) => {
                                timeShadowOpacity = clamp01(e.target.value, 0.98);
                                timeShadowOpacitySetting.value = timeShadowOpacity;
                            },
                            onkeydown: (e) => {
                                e.stopPropagation();
                            }
                        });
                        return $el("tr", [
                            $el("td", [$el("label", { for: id, textContent: "流光高亮：时间阴影透明度:" })]),
                            $el("td", [input]),
                        ]);
                    },
                    onChange(value) {
                        timeShadowOpacity = clamp01(value, 0.98);
                        app.graph?.setDirtyCanvas?.(true, true);
                    },
                });
                timeShadowOpacity = clamp01(timeShadowOpacitySetting?.value, timeShadowOpacity);
            }
        };

        const scriptUrl = import.meta.url;
        const persistedConfig = migrateConfig(loadConfig());
        const presetsUrl = new URL("./highlight_presets.json", scriptUrl).href;
        try {
            const raw = localStorage.getItem(USER_PRESET_KEY);
            if (raw) {
                const parsed = JSON.parse(raw);
                if (Array.isArray(parsed)) {
                    userPresets = parsed
                        .filter(p => p && Array.isArray(p.colors) && p.colors.length > 0)
                        .map(p => ({ ...p, __user: true }));
                }
            }
        } catch {}

        const enableSettingsUI = false;
        fetch(presetsUrl)
            .then(r => r.json())
            .then(data => {
                if (data.presets) defaultPresets = data.presets;
                loadedPresets = [...defaultPresets, ...userPresets];
                if (data.styles) {
                    if (data.styles.missing_input?.color) missingInputColor = normalizeColor(data.styles.missing_input.color, "#FF0000");
                    if (data.styles.error?.color) errorColor = normalizeColor(data.styles.error.color, "#FF0000");
                }
                if (data.defaults) {
                    applyDefaults(data.defaults);
                }
                if (enableSettingsUI) registerSettings();
                if (persistedConfig) {
                    setConfig(persistedConfig);
                }
                notifyReady();
            })
            .catch(e => {
                console.warn("流光高亮：加载预设失败，使用内置默认值。", e);
                loadedPresets = [...userPresets];
                if (enableSettingsUI) registerSettings();
                if (persistedConfig) {
                    setConfig(persistedConfig);
                }
                notifyReady();
            });

        const original_drawNode = LGraphCanvas.prototype.drawNode;

        LGraphCanvas.prototype.drawNode = function(node, ctx) {
            original_drawNode.call(this, node, ctx);

            const currentRunningId = app.runningNodeId || runningNodeId;
            if (currentRunningId && currentRunningId.toString() !== lastRunningNodeId) {
                lastRunningNodeId = currentRunningId.toString();
                runningStartTime = performance.now();
            }
            const isRunning = (
                (currentRunningId && currentRunningId.toString() === node.id.toString())
            );

            // Determine State & Color
            let targetColor = breathingColor;
            let isError = false;
            let isMissingInput = false;

            // Error Check
            if (node.bgcolor === "#FF0000" || node.bgcolor === "red" || 
                (app.lastNodeErrors && app.lastNodeErrors[node.id]) ||
                (lastErrorNodeId && lastErrorNodeId === node.id.toString())
            ) {
                isError = true;
                targetColor = errorColor;
            }

            // Missing Node Check (Class Definition Missing)
            if (!isError && node.type && typeof LiteGraph !== 'undefined' && LiteGraph.registered_node_types && !LiteGraph.registered_node_types[node.type]) {
                 isError = true;
                 targetColor = errorColor;
            }
            // Missing Input Check
            // Only check if not already error and not running
            if (!isRunning && !isError) {
                 if (node.inputs) {
                     // Try to identify required inputs from ComfyUI node definition
                     const nodeData = node.constructor ? node.constructor.nodeData : null;
                     const requiredInputs = (nodeData && nodeData.input && nodeData.input.required) 
                                            ? Object.keys(nodeData.input.required) 
                                            : null;

                     // Critical types that usually MUST be connected if present
                     const CRITICAL_TYPES = new Set([
                         "MODEL", "VAE", "CLIP", "LATENT", "IMAGE", "CONDITIONING", 
                         "MASK", "STYLE_MODEL", "CLIP_VISION", "CONTROL_NET", "AUDIO"
                     ]);

                     for (const inp of node.inputs) {
                         if (inp.link !== null) continue; // Already connected
                         
                         // Skip implicit optional types
                         if (inp.type === "OPTIONAL" || (typeof inp.type === 'string' && inp.type.toLowerCase() === "optional")) continue;
                         if (inp.name === "is_changed") continue; 
                         if (inp.name === "mask") continue; // 'mask' is often optional even if not marked

                         let isRequired = false;
                         
                         // Special handling for Reroute and PrimitiveNode (always required if unconnected)
                         if (node.type === "Reroute" || node.type === "PrimitiveNode") {
                             isRequired = true;
                         } else if (requiredInputs && requiredInputs.includes(inp.name)) {
                             // It is marked as required. 
                             // We further filter to avoid false positives (e.g. widgets that are not converted to inputs).
                             
                             const typeStr = (typeof inp.type === 'string') ? inp.type.toUpperCase() : "";
                             const isCritical = CRITICAL_TYPES.has(typeStr);
                             
                             // Check if there is a corresponding widget. 
                             // If a widget exists with the same name, we assume the user might provide value via widget.
                             // (Converted widgets are usually removed from node.widgets)
                             const hasWidget = node.widgets && node.widgets.some(w => w.name === inp.name);

                             if (isCritical) {
                                 // Critical types usually don't have widgets and must be connected
                                 isRequired = true;
                             } else if (!hasWidget) {
                                 // If it's not critical but has NO widget, it must be connected
                                 isRequired = true;
                             }
                         }

                         if (isRequired) {
                             isMissingInput = true;
                             targetColor = missingInputColor;
                             break;
                         }
                     }
                 }
            }

            // Keep animation loop alive if there are errors or missing inputs
            if (isError || isMissingInput) {
                lastHighlightTime = performance.now();
                if (tickHandle === null) scheduleTick(0);
            }

            if (!isRunning && !isError && !isMissingInput) return;
            if (!highlightEnabled) return;
            if ((isError || isMissingInput) && !isRunning) {
                return;
            }

            try {
                ctx.save();
                const r = window.devicePixelRatio || 1;
                
                // Reset transform? No, drawNode is in node-local space.
                // So (0,0) is top-left of node.
                
                const shape = node._shape || LiteGraph.BOX_SHAPE;

                // --- Helper Function to create path ---
                // pad: padding amount
                // top_pad: extra top padding
                const createPath = (pad, top_pad) => {
                    ctx.beginPath();
                    const isCollapsed = !!(node.flags && node.flags.collapsed) || !!node.collapsed;
                    const titleHeight = LiteGraph.NODE_TITLE_HEIGHT || 30;
                    const width = isCollapsed
                        ? (node._collapsed_width || LiteGraph.NODE_COLLAPSED_WIDTH || (node.size && node.size[0]) || 140)
                        : ((node.size && node.size[0]) || 140);
                    const height = isCollapsed
                        ? titleHeight
                        : ((node.size && node.size[1]) || 60);
                    const bounds = isCollapsed
                        ? { x: 0, y: -titleHeight, w: width, h: titleHeight }
                        : { x: 0, y: -titleHeight, w: width, h: height + titleHeight };
                    const padOuter = pad;
                    if (shape === LiteGraph.CIRCLE_SHAPE) {
                        const radius = Math.max(2, Math.max(bounds.w, bounds.h) * 0.5 + padOuter);
                        ctx.arc(bounds.x + bounds.w * 0.5, bounds.y + bounds.h * 0.5, radius, 0, Math.PI * 2);
                    } else {
                        const x = bounds.x - padOuter;
                        const y = bounds.y - padOuter;
                        const w = bounds.w + padOuter * 2;
                        const h = bounds.h + padOuter * 2;
                        if (ctx.roundRect) {
                            ctx.roundRect(x, y, w, h, 12);
                        } else {
                            ctx.rect(x, y, w, h);
                        }
                    }
                };
                const addPath = (pad) => {
                    const isCollapsed = !!(node.flags && node.flags.collapsed) || !!node.collapsed;
                    const titleHeight = LiteGraph.NODE_TITLE_HEIGHT || 30;
                    const width = isCollapsed
                        ? (node._collapsed_width || LiteGraph.NODE_COLLAPSED_WIDTH || (node.size && node.size[0]) || 140)
                        : ((node.size && node.size[0]) || 140);
                    const height = isCollapsed
                        ? titleHeight
                        : ((node.size && node.size[1]) || 60);
                    const bounds = isCollapsed
                        ? { x: 0, y: -titleHeight, w: width, h: titleHeight }
                        : { x: 0, y: -titleHeight, w: width, h: height + titleHeight };
                    const padOuter = pad;
                    if (shape === LiteGraph.CIRCLE_SHAPE) {
                        const radius = Math.max(2, Math.max(bounds.w, bounds.h) * 0.5 + padOuter);
                        ctx.moveTo(bounds.x + bounds.w * 0.5 + radius, bounds.y + bounds.h * 0.5);
                        ctx.arc(bounds.x + bounds.w * 0.5, bounds.y + bounds.h * 0.5, radius, 0, Math.PI * 2);
                    } else {
                        const x = bounds.x - padOuter;
                        const y = bounds.y - padOuter;
                        const w = bounds.w + padOuter * 2;
                        const h = bounds.h + padOuter * 2;
                        if (ctx.roundRect) {
                            ctx.roundRect(x, y, w, h, 12);
                        } else {
                            ctx.rect(x, y, w, h);
                        }
                    }
                };

                const official_pad = 4;
                const cover_width = 4;
                const ambient_width = 10;
                const main_width = 6;
                const core_width = 2;
                const top_padding_adjustment = 0;
                const side_inset = 0;
                const outer_expand = 0;

                const cover_pad = official_pad;
                const glow_pad = cover_pad;

                // Animation pulse
                const now = performance.now();
                let breathAlpha = 1.0;
                if (breathingEnabled) {
                    const period = Math.max(50, breathingPeriodMs);
                    const strength = Math.min(100, Math.max(0, breathingStrength)) / 100;
                    const minAlpha = 1 - strength * 0.5;
                    if (autoBreathingEnabled) {
                        // Firefly-like breathing: (exp(sin(t)) - 1/e) / (e - 1/e)
                        // This creates a sharper peak (flash) and wider trough (darkness)
                        // Simplified approximation using pow: pow((sin(t) + 1)/2, 2)
                        
                        const t = (now / period) * Math.PI * 2;
                        // Using a slightly modified sine wave for firefly effect
                        // exp(sin(t)) gives the classic firefly curve
                        const sineVal = Math.sin(t);
                        const expVal = Math.exp(sineVal);
                        // Normalize exp(sin(t)) to 0-1 range. Max is e, min is 1/e
                        const maxVal = Math.E; // approx 2.718
                        const minVal = 1 / Math.E; // approx 0.368
                        const pulse = (expVal - minVal) / (maxVal - minVal);
                        
                        breathAlpha = minAlpha + pulse * (1 - minAlpha);
                    } else if (lastMouseMoveTime) {
                        const recentMouse = now - lastMouseMoveTime < 300;
                        if (recentMouse) {
                            const mousePeriod = Math.max(50, mouseBreathPeriodMs || period);
                            const elapsed = now - (mouseBreathPhaseStart || now);
                            const pulse = (Math.sin((elapsed / mousePeriod) * Math.PI * 2) + 1) / 2;
                            breathAlpha = minAlpha + pulse * (1 - minAlpha);
                        }
                    }
                }

                const parseGradient = (str) => {
                    const colors = (typeof str === "string" ? str : "").split(",").map(s => s.trim()).filter(Boolean);
                    if (colors.length === 0) return ["#22FF22"];
                    return colors;
                };
                
                const gradientColors = parseGradient(targetColor);
                
                // Helper to create gradient fill/stroke
                const setGradientStyle = (colors, alphaMultiplier = 1.0) => {
                    if (colors.length === 1) {
                         // Single color
                         const c = colors[0];
                         // Apply alpha if needed? Canvas colors are usually hex.
                         // Convert to rgba if alpha < 1
                         if (alphaMultiplier < 1) {
                             // Simple hex to rgba
                             const hex = c.replace("#", "");
                             const r = parseInt(hex.substring(0,2), 16);
                             const g = parseInt(hex.substring(2,4), 16);
                             const b = parseInt(hex.substring(4,6), 16);
                             ctx.fillStyle = `rgba(${r},${g},${b},${alphaMultiplier})`;
                             ctx.strokeStyle = `rgba(${r},${g},${b},${alphaMultiplier})`;
                         } else {
                             ctx.fillStyle = c;
                             ctx.strokeStyle = c;
                         }
                    } else {
                        // Linear Gradient
                        // For a rect, we can estimate diagonal or horizontal
                        // Use local coords. Node bounds are around (0,0) to (w,h)
                        const width = node.size ? node.size[0] : 140;
                        const height = node.size ? node.size[1] : 60;
                        const grad = ctx.createLinearGradient(0, 0, width, height);
                        colors.forEach((c, i) => {
                             grad.addColorStop(i / (colors.length - 1), c);
                        });
                        // Apply global alpha for gradient?
                        // ctx.globalAlpha affects everything.
                        ctx.fillStyle = grad;
                        ctx.strokeStyle = grad;
                    }
                };
                
                const currentAlpha = ctx.globalAlpha;
                ctx.globalAlpha = breathingBrightness * breathAlpha;

                // 1. Cover Layer (Background behind highlight) - Optional, maybe not needed if we want transparent
                // createPath(cover_pad, 0);
                // ctx.fillStyle = "rgba(0,0,0,0.0)";
                // ctx.fill();

                // 2. Glow/Ambient Layer
                createPath(glow_pad, 0);
                setGradientStyle(gradientColors, 1.0);
                
                // Draw multiple strokes for glow effect
                // Optimized for performance and soft look
                const isSoft = true;
                
                if (isSoft) {
                    const isAlert = isError || isMissingInput;
                    if (isAlert) {
                        // Alert ring: remove outermost frame, draw only inside border
                        const baseAlpha = breathingBrightness * breathAlpha;
                        const drawInnerStroke = (lw, alpha) => {
                            ctx.save();
                            createPath(0, 0);
                            ctx.clip();
                            setGradientStyle(gradientColors, 1.0);
                            ctx.globalAlpha = alpha * baseAlpha;
                            ctx.lineWidth = lw * breathingSizeScale;
                            ctx.stroke();
                            ctx.restore();
                        };
                        drawInnerStroke(main_width + 3, 0.85);
                        drawInnerStroke(core_width + 1, 1.0);
                    } else {
                        // Outer soft glow (stronger, more noticeable)
                        ctx.lineWidth = (ambient_width + 6) * breathingSizeScale;
                        ctx.globalAlpha = 0.18 * breathingBrightness * breathAlpha;
                        ctx.stroke();

                        // Middle glow
                        ctx.lineWidth = (main_width + 2) * breathingSizeScale;
                        ctx.globalAlpha = 0.45 * breathingBrightness * breathAlpha;
                        ctx.stroke();

                        // Core bright line
                        ctx.lineWidth = (core_width + 1) * breathingSizeScale;
                        ctx.globalAlpha = 0.95 * breathingBrightness * breathAlpha;
                        ctx.stroke();
                    }
                } else {
                    // Legacy style
                    ctx.lineWidth = ambient_width * breathingSizeScale;
                    ctx.globalAlpha = 0.15 * breathingBrightness * breathAlpha;
                    ctx.stroke();
    
                    ctx.lineWidth = main_width * breathingSizeScale;
                    ctx.globalAlpha = 0.4 * breathingBrightness * breathAlpha;
                    ctx.stroke();
    
                    ctx.lineWidth = core_width * breathingSizeScale;
                    ctx.globalAlpha = 0.9 * breathingBrightness * breathAlpha;
                    ctx.stroke();
                }
                
                // Restore Alpha
                ctx.globalAlpha = currentAlpha;

                // 3. Time Display
                if (isRunning && timeEnabled) {
                    try {
                        ctx.save();

                        const now = performance.now();
                        const elapsed = Math.max(0, now - runningStartTime);
                        const elapsedText = formatElapsed(elapsed);
                        const label = elapsedText;

                        const lg = window?.LiteGraph;
                        const titleH = lg?.NODE_TITLE_HEIGHT || 30;
                        const nodeW = (node.size && node.size[0]) || 140;

                        ctx.font = "700 13px ui-monospace, SFMono-Regular, Menlo, Consolas, \"Liberation Mono\", monospace";
                        const tm = ctx.measureText(label);
                        const txtW = tm.width;
                        const padX = 6;
                        const padY = 3;
                        const boxW = txtW + padX * 2;
                        const boxH = 18;

                        const x = Math.max(6, nodeW - boxW - 8);
                        const y = -titleH + 5;

                        ctx.beginPath();
                        ctx.fillStyle = `rgba(6, 10, 18, ${clamp01(timeBgOpacity, 0.72)})`;
                        if (ctx.roundRect) {
                            ctx.roundRect(x, y, boxW, boxH, 6);
                        } else {
                            ctx.rect(x, y, boxW, boxH);
                        }
                        ctx.fill();

                        ctx.strokeStyle = "rgba(255,255,255,0.18)";
                        ctx.lineWidth = 1;
                        ctx.stroke();

                        ctx.fillStyle = "#F5F9FF";
                        ctx.shadowColor = `rgba(0, 0, 0, ${clamp01(timeShadowOpacity, 0.6)})`;
                        ctx.shadowBlur = 8 * r;
                        ctx.shadowOffsetX = 0;
                        ctx.shadowOffsetY = 2 * r;
                        ctx.textAlign = "left";
                        ctx.textBaseline = "middle";
                        ctx.fillText(label, x + padX, y + boxH / 2);
                    } finally {
                        ctx.restore();
                    }
                }

            } catch (e) {
                console.error("流光高亮：绘制异常", e);
            } finally {
                ctx.restore();
            }
        };

        // Force a redraw when execution starts/progresses to ensure the highlight updates immediately
        const api = app.api;
        
        // Listen to the executing event to force canvas dirty
        api.addEventListener("executing", (e) => {
             try {
                 const detail = e?.detail || {};
                 const explicitId = detail.node_id || detail.nodeId || (detail.node && detail.node.id);
                 if (explicitId !== undefined && explicitId !== null) {
                     runningNodeId = explicitId.toString();
                     lastErrorNodeId = null;
                     runningStartTime = performance.now();
                     lastRunningNodeId = runningNodeId;
                 } else {
                     runningNodeId = null;
                     lastRunningNodeId = null;
                     runningStartTime = 0;
                 }
             } catch(_) {
                 runningNodeId = null;
                 lastRunningNodeId = null;
                 runningStartTime = 0;
             }
             if (app.canvas) {
                 app.canvas.setDirty(true, true);
             }
             scheduleTick(0);
        });

        api.addEventListener("executed", (e) => {
            try {
                const detail = e?.detail || {};
                const explicitId = detail.node_id || detail.nodeId || (detail.node && detail.node.id);
                if (explicitId !== undefined && explicitId !== null) {
                    const idStr = explicitId.toString();
                    if (lastErrorNodeId === idStr) {
                        lastErrorNodeId = null;
                    }
                    if (app.lastNodeErrors && app.lastNodeErrors[explicitId]) {
                        delete app.lastNodeErrors[explicitId];
                    }
                    if (app.canvas) {
                        app.canvas.setDirty(true, true);
                    }
                }
            } catch (err) {
                console.error("流光高亮：executed 处理异常", err);
            }
        });

        // Listen for execution errors to highlight the failed node immediately
        api.addEventListener("execution_error", (e) => {
            try {
                const detail = e?.detail || {};
                const explicitId = detail.node_id || detail.nodeId;
                if (explicitId !== undefined && explicitId !== null) {
                    lastErrorNodeId = explicitId.toString();
                    if (app.canvas) {
                        app.canvas.setDirty(true, true);
                    }
                    scheduleTick(0);
                }
            } catch (err) {
                console.error("流光高亮：execution_error 处理异常", err);
            }
        });

        const ensureMouseListener = () => {
            if (mouseListenerAttached) return;
            const canvasEl = app.canvas?.canvas || app.canvas?.el || app.canvas?.canvasEl;
            if (!canvasEl || !canvasEl.addEventListener) return;
            canvasEl.addEventListener("mousemove", () => {
                const now = performance.now();
                // Throttle mouse move handling to save resources (limit to ~5fps for detection)
                // We don't need high precision for "waking up" the breathing effect
                if (now - lastMouseMoveHandleTime < 200) return;
                lastMouseMoveHandleTime = now;

                if (lastMouseMoveTime) {
                    const interval = now - lastMouseMoveTime;
                    const newPeriod = Math.min(3000, Math.max(200, interval));
                    if (mouseBreathPeriodMs && mouseBreathPhaseStart) {
                        const elapsed = now - mouseBreathPhaseStart;
                        const phase = ((elapsed / mouseBreathPeriodMs) * Math.PI * 2) % (Math.PI * 2);
                        mouseBreathPhaseStart = now - (phase / (Math.PI * 2)) * newPeriod;
                    } else {
                        mouseBreathPhaseStart = now;
                    }
                    mouseBreathPeriodMs = newPeriod;
                } else {
                    mouseBreathPhaseStart = now;
                }
                lastMouseMoveTime = now;
                scheduleTick(0);
            });
            mouseListenerAttached = true;
        };
        const scheduleTick = (delay) => {
            if (tickHandle !== null) return;
            tickHandle = setTimeout(() => {
                tickHandle = null;
                ensureMouseListener(); 
                const now = performance.now();
                const hasRunning = app.canvas && (app.runningNodeId || runningNodeId) && highlightEnabled;
                const hasActiveHighlight = (now - lastHighlightTime) < 2000; // Keep alive for 2s after last highlight draw

                if (hasRunning || hasActiveHighlight) {
                    app.canvas.setDirty(true, true);
                }
                if (!hasRunning && !hasActiveHighlight) return;
                
                const recentMouse = now - lastMouseMoveTime < 300;
                const activeBreathing = breathingEnabled && (autoBreathingEnabled || recentMouse);
                // Refresh rate: 
                // - If breathing (auto or mouse), use 100ms (10fps) for low power consumption but visible animation
                // - If static, slow check (500ms)
                const nextDelay = activeBreathing ? 100 : 500;
                scheduleTick(nextDelay);
            }, Math.max(0, delay));
        };
        scheduleTick(0);
	}
});
