# =============================================================================
# FILE: addons/TileMapLayer3D/core/autotile/autotile_engine.gd
# PURPOSE: Main autotiling orchestration - coordinates all operations
# DEPENDENCIES: TileSetTerrainReader, TileSetBitmaskMapper, PlaneCoordinateMapper
# =============================================================================
@tool
class_name AutotileEngine
extends RefCounted

## Main autotiling engine. Coordinates bitmask calculation,
## tile matching, and neighbor updates using Godot's TileSet data.
##
## Usage:
##   var engine := AutotileEngine.new(tileset)
##   var uv := engine.get_autotile_uv(grid_pos, orientation, terrain_id, placement_data)

signal lookup_rebuilt()

var _tileset: TileSet
var _source_id: int
var _terrain_set: int

var _reader: TileSetTerrainReader
var _mapper: TileSetBitmaskMapper

# Bitmask cache for placed tiles: tile_key (int) -> bitmask (int)
var _bitmask_cache: Dictionary = {}


func _init(tileset: TileSet = null, source_id: int = GlobalConstants.AUTOTILE_DEFAULT_SOURCE_ID, terrain_set: int = GlobalConstants.AUTOTILE_DEFAULT_TERRAIN_SET) -> void:
	if tileset:
		setup(tileset, source_id, terrain_set)


## Configure the engine with a TileSet
func setup(tileset: TileSet, source_id: int = GlobalConstants.AUTOTILE_DEFAULT_SOURCE_ID, terrain_set: int = GlobalConstants.AUTOTILE_DEFAULT_TERRAIN_SET) -> void:
	_tileset = tileset
	_source_id = source_id
	_terrain_set = terrain_set

	if tileset:
		_reader = TileSetTerrainReader.new(tileset, source_id, terrain_set)
		_mapper = TileSetBitmaskMapper.new(tileset, source_id, terrain_set)
		rebuild_lookup()
	else:
		_reader = null
		_mapper = null


## Rebuild lookup tables from TileSet
## Call this when TileSet terrain configuration changes
func rebuild_lookup() -> void:
	if _mapper:
		_mapper.build()
		_bitmask_cache.clear()
		lookup_rebuilt.emit()


## Rebuilds bitmask cache from TileMapLayer3D columnar storage
## This ensures neighbor detection works correctly for tiles placed before reload
## @param tile_map_layer: TileMapLayer3D to read tile data from
func rebuild_bitmask_cache(tile_map_layer: TileMapLayer3D) -> void:
	_bitmask_cache.clear()

	if not tile_map_layer:
		return

	var cached_count: int = 0
	var tile_count: int = tile_map_layer.get_tile_count()

	for i in range(tile_count):
		var tile_data: Dictionary = tile_map_layer.get_tile_data_at(i)
		if tile_data.is_empty():
			continue

		var terrain_id: int = tile_data.get("terrain_id", GlobalConstants.AUTOTILE_NO_TERRAIN)

		# Only cache autotiled tiles (terrain_id >= 0)
		if terrain_id < 0:
			continue

		var grid_pos: Vector3 = tile_data.get("grid_position", Vector3.ZERO)
		var orientation: int = tile_data.get("orientation", 0)
		var tile_key: int = GlobalUtil.make_tile_key(grid_pos, orientation)

		var bitmask: int = calculate_bitmask(
			grid_pos,
			orientation,
			terrain_id,
			tile_map_layer
		)
		_bitmask_cache[tile_key] = bitmask
		cached_count += 1

	#if cached_count > 0:
	#	print("AutotileEngine: Rebuilt bitmask cache for ", cached_count, " autotiled tiles")
	pass


## Check if engine is ready for autotiling
func is_ready() -> bool:
	return _reader != null and _reader.is_valid() and _mapper != null and not _mapper.is_empty()


## Get all available terrains from the TileSet
func get_terrains() -> Array[Dictionary]:
	if _reader:
		return _reader.get_terrains()
	return []


## Get count of terrains in the TileSet
func get_terrain_count() -> int:
	if _reader:
		return _reader.get_terrain_count()
	return 0


## Calculate bitmask for a position based on its neighbors
## Uses TileMapLayer3D columnar storage for neighbor lookups
## @param tile_map_layer: TileMapLayer3D to read tile data from
func calculate_bitmask(
	grid_pos: Vector3,
	orientation: int,
	terrain_id: int,
	tile_map_layer: TileMapLayer3D
) -> int:
	var bitmask: int = 0

	for dir_name: String in PlaneCoordinateMapper.NEIGHBOR_OFFSETS_2D.keys():
		var offset_2d: Vector2i = PlaneCoordinateMapper.NEIGHBOR_OFFSETS_2D[dir_name]
		var offset_3d: Vector3 = PlaneCoordinateMapper.offset_to_3d(offset_2d, orientation)
		var neighbor_pos: Vector3 = grid_pos + offset_3d

		if _has_matching_terrain(neighbor_pos, orientation, terrain_id, tile_map_layer):
			bitmask |= PlaneCoordinateMapper.BITMASK_VALUES[dir_name]

	return bitmask


## Check if a position has a tile with matching terrain
## Uses TileMapLayer3D columnar storage for efficient lookup
func _has_matching_terrain(
	grid_pos: Vector3,
	orientation: int,
	terrain_id: int,
	tile_map_layer: TileMapLayer3D
) -> bool:
	var tile_key: int = GlobalUtil.make_tile_key(grid_pos, orientation)

	if not tile_map_layer.has_tile(tile_key):
		return false

	return tile_map_layer.get_tile_terrain_id(tile_key) == terrain_id


## Get UV rect for autotile placement at a position
## Returns the correct UV based on neighboring tiles
## Uses TileMapLayer3D columnar storage for neighbor lookups
func get_autotile_uv(
	grid_pos: Vector3,
	orientation: int,
	terrain_id: int,
	tile_map_layer: TileMapLayer3D
) -> Rect2:
	if not is_ready():
		return Rect2()

	var bitmask: int = calculate_bitmask(
		grid_pos, orientation, terrain_id, tile_map_layer
	)

	# Cache bitmask for this position
	var tile_key: int = GlobalUtil.make_tile_key(grid_pos, orientation)
	_bitmask_cache[tile_key] = bitmask

	return _mapper.get_uv(terrain_id, bitmask)


## Update all neighbors of a position and return UV changes
## Returns: Dictionary of tile_key -> new_uv (Rect2)
## Call this after placing or removing a tile
## Uses TileMapLayer3D columnar storage for neighbor lookups
func update_neighbors(
	grid_pos: Vector3,
	orientation: int,
	tile_map_layer: TileMapLayer3D
) -> Dictionary:
	var updates: Dictionary = {}  # tile_key -> new_uv

	var neighbors: Array[Vector3] = PlaneCoordinateMapper.get_neighbor_positions_3d(
		grid_pos, orientation
	)

	for neighbor_pos: Vector3 in neighbors:
		var tile_key: int = GlobalUtil.make_tile_key(neighbor_pos, orientation)

		# Skip if no tile at this position
		if not tile_map_layer.has_tile(tile_key):
			continue

		# Get terrain_id from columnar storage
		var neighbor_terrain_id: int = tile_map_layer.get_tile_terrain_id(tile_key)

		# Skip if not an autotiled tile
		if neighbor_terrain_id < 0:
			continue

		# Calculate new bitmask
		var new_bitmask: int = calculate_bitmask(
			neighbor_pos, orientation, neighbor_terrain_id, tile_map_layer
		)

		# Check if bitmask changed
		var old_bitmask: int = _bitmask_cache.get(tile_key, -1)

		if new_bitmask != old_bitmask:
			var new_uv: Rect2 = _mapper.get_uv(neighbor_terrain_id, new_bitmask)
			updates[tile_key] = new_uv
			_bitmask_cache[tile_key] = new_bitmask

	return updates


## Invalidate cached bitmask for a tile (call when tile is removed)
func invalidate_tile(tile_key: int) -> void:
	_bitmask_cache.erase(tile_key)


## Get UV rect for a terrain and bitmask value (direct lookup without position)
## Used by area fill to avoid redundant bitmask calculations
func get_uv_for_bitmask(terrain_id: int, bitmask: int) -> Rect2:
	if _mapper:
		return _mapper.get_uv(terrain_id, bitmask)
	return Rect2()


## Clear all cached bitmasks
func clear_cache() -> void:
	_bitmask_cache.clear()


## Get tile size from TileSet
func get_tile_size() -> Vector2i:
	if _reader:
		return _reader.get_tile_size()
	return GlobalConstants.DEFAULT_TILE_SIZE


## Get texture from TileSet
func get_texture() -> Texture2D:
	if _reader:
		return _reader.get_texture()
	return null


## Get terrain name by ID
func get_terrain_name(terrain_id: int) -> String:
	if _reader:
		return _reader.get_terrain_name(terrain_id)
	return ""


## Get terrain color by ID
func get_terrain_color(terrain_id: int) -> Color:
	if _reader:
		return _reader.get_terrain_color(terrain_id)
	return Color.WHITE


## Check if a terrain has configured tiles
func has_terrain_tiles(terrain_id: int) -> bool:
	if _mapper:
		return _mapper.has_terrain(terrain_id)
	return false


## Get count of tiles configured for a terrain
func count_terrain_tiles(terrain_id: int) -> int:
	if _reader:
		return _reader.count_configured_tiles(terrain_id)
	return 0


## Get the TileSet being used
func get_tileset() -> TileSet:
	return _tileset


## Get stats for debugging
func get_stats() -> Dictionary:
	var stats := {
		"ready": is_ready(),
		"cache_size": _bitmask_cache.size(),
		"tileset": _tileset.resource_path if _tileset else "null",
		"source_id": _source_id,
		"terrain_set": _terrain_set,
	}
	if _mapper:
		stats["mapper"] = _mapper.get_stats()
	if _reader:
		stats["terrain_count"] = _reader.get_terrain_count()
	return stats
