//========= Copyright Valve Corporation, All rights reserved. ============//
//
// Purpose: 
//
// $NoKeywords: $
//
//=============================================================================//
#include "cbase.h"
#include "ai_activity.h"
#include "activitylist.h"
#include "stringregistry.h"
#include "isaverestore.h"

#include "filesystem.h"
#include <KeyValues.h>
#include "utldict.h"

// memdbgon must be the last include file in a .cpp file!!!
#include "tier0/memdbgon.h"

extern IFileSystem *filesystem;

// NOTE: If CStringRegistry allowed storing arbitrary data, we could just use that.
// in this case we have the "isPrivate" member and the replacement rules 
// (activityIndex can be reused by private activities), so a custom table is necessary
struct activitylist_t
{
	int					activityIndex;
	unsigned short		stringKey;
	short				isPrivate;
};

CUtlVector<activitylist_t> g_ActivityList;

static CUtlDict< CActivityRemapCache, int > m_ActivityRemapDatabase;

// This stores the actual activity names.  Also, the string ID in the registry is simply an index 
// into the g_ActivityList array.
CStringRegistry	g_ActivityStrings;

// this is just here to accelerate adds
static int g_HighestActivity = 0;

int g_nActivityListVersion = 1;


void ActivityList_Init( void )
{
	g_HighestActivity = 0;
}

void ActivityList_Free( void )
{
	g_ActivityStrings.ClearStrings();
	g_ActivityList.Purge();
	m_ActivityRemapDatabase.Purge();

	// So studiohdrs can reindex activity indices
	++g_nActivityListVersion;
}

// add a new activity to the database
activitylist_t *ActivityList_AddActivityEntry( const char *pName, int iActivityIndex, bool isPrivate )
{
	MEM_ALLOC_CREDIT();
	int index = g_ActivityList.AddToTail();
	activitylist_t *pList = &g_ActivityList[index];
	pList->activityIndex = iActivityIndex;
	pList->stringKey = g_ActivityStrings.AddString( pName, index );
	pList->isPrivate = isPrivate;
	
	// UNDONE: This implies that ALL shared activities are added before ANY custom activities
	// UNDONE: Segment these instead?  It's a 32-bit int, how many activities do we need?
	if ( iActivityIndex > g_HighestActivity )
	{
		g_HighestActivity = iActivityIndex;
	}

	return pList;
}

// get the database entry from a string
static activitylist_t *ListFromString( const char *pString )
{
	// just use the string registry to do this search/map
	int stringID = g_ActivityStrings.GetStringID( pString );
	if ( stringID < 0 )
		return NULL;

	return &g_ActivityList[stringID];
}

// Get the database entry for an index
static activitylist_t *ListFromActivity( int activityIndex )
{
	// ugly linear search
	for ( int i = 0; i < g_ActivityList.Size(); i++ )
	{
		if ( g_ActivityList[i].activityIndex == activityIndex )
		{
			return &g_ActivityList[i];
		}
	}

	return NULL;
}

bool ActivityList_RegisterSharedActivity( const char *pszActivityName, int iActivityIndex )
{
	// UNDONE: Do we want to do these checks when not in developer mode? or maybe DEBUG only?
	// They really only matter when you change the list of code controlled activities.  IDs
	// for content controlled activities never collide because they are generated.

	// technically order isn't dependent, but it's too damn easy to forget to add new ACT_'s to all three lists.
	static int lastActivityIndex = -1;
	Assert( iActivityIndex < LAST_SHARED_ACTIVITY && (iActivityIndex == lastActivityIndex + 1 || iActivityIndex == 0) );
	lastActivityIndex = iActivityIndex;

	// first, check to make sure the slot we're asking for is free. It must be for 
	// a shared activity.
	activitylist_t *pList = ListFromString( pszActivityName );
	if ( !pList )
	{
		pList = ListFromActivity( iActivityIndex );
	}

	if ( pList )
	{
		Warning( "***\nShared activity collision! %s<->%s\n***\n", pszActivityName, g_ActivityStrings.GetStringForKey( pList->stringKey ) );
		Assert(0);
		return false;
	}
	// ----------------------------------------------------------------

	ActivityList_AddActivityEntry( pszActivityName, iActivityIndex, false );
	return true;
}


Activity ActivityList_RegisterPrivateActivity( const char *pszActivityName )
{
	activitylist_t *pList = ListFromString( pszActivityName );
	if ( pList )
	{
		// this activity is already in the list. If the activity we collided with is also private, 
		// then the collision is OK. Otherwise, it's a bug.
		if ( pList->isPrivate )
		{
			return (Activity)pList->activityIndex;
		}
		else
		{
			// this private activity collides with a shared activity. That is not allowed.
			Warning( "***\nShared<->Private Activity collision!\n***\n" );
			Assert(0);
			return ACT_INVALID;
		}
	}

	pList = ActivityList_AddActivityEntry( pszActivityName, g_HighestActivity+1, true );
	return (Activity)pList->activityIndex;
}

// Get the index for a given activity name
// Done at load time for all models
int ActivityList_IndexForName( const char *pszActivityName )
{
	// this is a fast O(lgn) search (actually does 2 O(lgn) searches)
	activitylist_t *pList = ListFromString( pszActivityName );

	if ( pList )
	{
		return pList->activityIndex;
	}

	return kActivityLookup_Missing;
}

// Get the name for a given index
// This should only be used in debug code, it does a linear search
// But at least it only compares integers
const char *ActivityList_NameForIndex( int activityIndex )
{
	activitylist_t *pList = ListFromActivity( activityIndex );
	if ( pList )
	{
		return g_ActivityStrings.GetStringForKey( pList->stringKey );
	}
	return NULL;
}

void ActivityList_RegisterSharedActivities( void )
{
	REGISTER_SHARED_ACTIVITY( ACT_RESET );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_TRANSITION );
	REGISTER_SHARED_ACTIVITY( ACT_COVER );
	REGISTER_SHARED_ACTIVITY( ACT_COVER_MED );
	REGISTER_SHARED_ACTIVITY( ACT_COVER_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_WALK );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_CROUCH );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_CROUCH_AIM );
	REGISTER_SHARED_ACTIVITY( ACT_RUN );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_CROUCH );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_CROUCH_AIM );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_PROTECTED );
	REGISTER_SHARED_ACTIVITY( ACT_SCRIPT_CUSTOM_MOVE );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK1 );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK2 );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK1_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK2_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_DIESIMPLE );
	REGISTER_SHARED_ACTIVITY( ACT_DIEBACKWARD );
	REGISTER_SHARED_ACTIVITY( ACT_DIEFORWARD );
	REGISTER_SHARED_ACTIVITY( ACT_DIEVIOLENT );
	REGISTER_SHARED_ACTIVITY( ACT_DIERAGDOLL );
	REGISTER_SHARED_ACTIVITY( ACT_FLY );
	REGISTER_SHARED_ACTIVITY( ACT_HOVER );
	REGISTER_SHARED_ACTIVITY( ACT_GLIDE );
	REGISTER_SHARED_ACTIVITY( ACT_SWIM );
	REGISTER_SHARED_ACTIVITY( ACT_JUMP );
	REGISTER_SHARED_ACTIVITY( ACT_HOP );
	REGISTER_SHARED_ACTIVITY( ACT_LEAP );
	REGISTER_SHARED_ACTIVITY( ACT_LAND );
	REGISTER_SHARED_ACTIVITY( ACT_CLIMB_UP );
	REGISTER_SHARED_ACTIVITY( ACT_CLIMB_DOWN );
	REGISTER_SHARED_ACTIVITY( ACT_CLIMB_DISMOUNT );
	REGISTER_SHARED_ACTIVITY( ACT_SHIPLADDER_UP );
	REGISTER_SHARED_ACTIVITY( ACT_SHIPLADDER_DOWN );
	REGISTER_SHARED_ACTIVITY( ACT_STRAFE_LEFT );
	REGISTER_SHARED_ACTIVITY( ACT_STRAFE_RIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_ROLL_LEFT );
	REGISTER_SHARED_ACTIVITY( ACT_ROLL_RIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_TURN_LEFT );
	REGISTER_SHARED_ACTIVITY( ACT_TURN_RIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCH );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHIDLE );
	REGISTER_SHARED_ACTIVITY( ACT_STAND );
	REGISTER_SHARED_ACTIVITY( ACT_USE );
	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL1 );
	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL2 );
	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL3 );

	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL_ADVANCE );
	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL_FORWARD );
	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL_GROUP );
	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL_HALT );
	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL_LEFT );
	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL_RIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_SIGNAL_TAKECOVER );

	REGISTER_SHARED_ACTIVITY( ACT_LOOKBACK_RIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_LOOKBACK_LEFT );
	REGISTER_SHARED_ACTIVITY( ACT_COWER );
	REGISTER_SHARED_ACTIVITY( ACT_SMALL_FLINCH );
	REGISTER_SHARED_ACTIVITY( ACT_BIG_FLINCH );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK1 );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK2 );
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD );
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD_START );
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD_FINISH );
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_ARM );
	REGISTER_SHARED_ACTIVITY( ACT_DISARM );
	
	REGISTER_SHARED_ACTIVITY( ACT_DROP_WEAPON );
	REGISTER_SHARED_ACTIVITY( ACT_DROP_WEAPON_SHOTGUN );

	REGISTER_SHARED_ACTIVITY( ACT_PICKUP_GROUND );
	REGISTER_SHARED_ACTIVITY( ACT_PICKUP_RACK );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_ANGRY );

	REGISTER_SHARED_ACTIVITY( ACT_IDLE_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_AGITATED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_STEALTH );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_HURT );

	REGISTER_SHARED_ACTIVITY( ACT_WALK_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_AGITATED );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_STEALTH );

	REGISTER_SHARED_ACTIVITY( ACT_RUN_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AGITATED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_STEALTH );

	REGISTER_SHARED_ACTIVITY( ACT_IDLE_AIM_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_AIM_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_AIM_AGITATED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_AIM_STEALTH );

	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM_AGITATED );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM_STEALTH );

	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM_AGITATED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM_STEALTH );

	REGISTER_SHARED_ACTIVITY( ACT_CROUCHIDLE_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHIDLE_AIM_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHIDLE_AGITATED );

	REGISTER_SHARED_ACTIVITY( ACT_WALK_HURT );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_HURT );
	REGISTER_SHARED_ACTIVITY( ACT_SPECIAL_ATTACK1 );
	REGISTER_SHARED_ACTIVITY( ACT_SPECIAL_ATTACK2 );
	REGISTER_SHARED_ACTIVITY( ACT_COMBAT_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_SCARED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_SCARED );
	REGISTER_SHARED_ACTIVITY( ACT_VICTORY_DANCE );
	
	REGISTER_SHARED_ACTIVITY( ACT_DIE_HEADSHOT );
	REGISTER_SHARED_ACTIVITY( ACT_DIE_CHESTSHOT );
	REGISTER_SHARED_ACTIVITY( ACT_DIE_GUTSHOT );
	REGISTER_SHARED_ACTIVITY( ACT_DIE_BACKSHOT );

	REGISTER_SHARED_ACTIVITY( ACT_FLINCH_HEAD );
	REGISTER_SHARED_ACTIVITY( ACT_FLINCH_CHEST );
	REGISTER_SHARED_ACTIVITY( ACT_FLINCH_STOMACH );
	REGISTER_SHARED_ACTIVITY( ACT_FLINCH_LEFTARM );
	REGISTER_SHARED_ACTIVITY( ACT_FLINCH_RIGHTARM );
	REGISTER_SHARED_ACTIVITY( ACT_FLINCH_LEFTLEG );
	REGISTER_SHARED_ACTIVITY( ACT_FLINCH_RIGHTLEG );
	REGISTER_SHARED_ACTIVITY( ACT_FLINCH_PHYSICS );

	REGISTER_SHARED_ACTIVITY( ACT_IDLE_ON_FIRE );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_ON_FIRE );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_ON_FIRE );

	REGISTER_SHARED_ACTIVITY( ACT_RAPPEL_LOOP );

	REGISTER_SHARED_ACTIVITY( ACT_180_LEFT );
	REGISTER_SHARED_ACTIVITY( ACT_180_RIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_90_LEFT );
	REGISTER_SHARED_ACTIVITY( ACT_90_RIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_STEP_LEFT );			
	REGISTER_SHARED_ACTIVITY( ACT_STEP_RIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_STEP_BACK );
	REGISTER_SHARED_ACTIVITY( ACT_STEP_FORE );

	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK1 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK2 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_MELEE_ATTACK1 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_MELEE_ATTACK2 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK1_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK2_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK_SWING_GESTURE );
	
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_SMALL_FLINCH );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_BIG_FLINCH );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_BLAST );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_BLAST_SHOTGUN );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_BLAST_DAMAGED );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_BLAST_DAMAGED_SHOTGUN );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_HEAD );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_CHEST );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_STOMACH );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_LEFTARM );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_RIGHTARM );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_LEFTLEG );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_FLINCH_RIGHTLEG );

	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_LEFT );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_RIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_LEFT45 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_RIGHT45 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_LEFT90 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_RIGHT90 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_LEFT45_FLAT );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_RIGHT45_FLAT );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_LEFT90_FLAT );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_TURN_RIGHT90_FLAT );

	// registered for Half-Life HGRUNT compatability
	REGISTER_SHARED_ACTIVITY( ACT_BARNACLE_HIT );
	REGISTER_SHARED_ACTIVITY( ACT_BARNACLE_PULL );
	REGISTER_SHARED_ACTIVITY( ACT_BARNACLE_CHOMP );
	REGISTER_SHARED_ACTIVITY( ACT_BARNACLE_CHEW );

	REGISTER_SHARED_ACTIVITY( ACT_DO_NOT_DISTURB );

	// Viewmodel activities may belong elsewhere, but since where is unclear right now,
	// they'll be placed here.
	REGISTER_SHARED_ACTIVITY( ACT_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_VM_FIDGET );
	REGISTER_SHARED_ACTIVITY( ACT_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_VM_PULLBACK_HIGH );
	REGISTER_SHARED_ACTIVITY( ACT_VM_PULLBACK_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_VM_THROW );
	REGISTER_SHARED_ACTIVITY( ACT_VM_PULLPIN );
	REGISTER_SHARED_ACTIVITY( ACT_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_VM_SECONDARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_VM_RELOAD );
	REGISTER_SHARED_ACTIVITY( ACT_VM_RELOAD_START );
	REGISTER_SHARED_ACTIVITY( ACT_VM_RELOAD_FINISH );
	REGISTER_SHARED_ACTIVITY( ACT_VM_DRYFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_VM_HITLEFT );
	REGISTER_SHARED_ACTIVITY( ACT_VM_HITLEFT2 );
	REGISTER_SHARED_ACTIVITY( ACT_VM_HITRIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_VM_HITRIGHT2 );
	REGISTER_SHARED_ACTIVITY( ACT_VM_HITCENTER );
	REGISTER_SHARED_ACTIVITY( ACT_VM_HITCENTER2 );
	REGISTER_SHARED_ACTIVITY( ACT_VM_MISSLEFT );
	REGISTER_SHARED_ACTIVITY( ACT_VM_MISSLEFT2 );
	REGISTER_SHARED_ACTIVITY( ACT_VM_MISSRIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_VM_MISSRIGHT2 );
	REGISTER_SHARED_ACTIVITY( ACT_VM_MISSCENTER );
	REGISTER_SHARED_ACTIVITY( ACT_VM_MISSCENTER2 );
	REGISTER_SHARED_ACTIVITY( ACT_VM_HAULBACK );
	REGISTER_SHARED_ACTIVITY( ACT_VM_SWINGHARD );
	REGISTER_SHARED_ACTIVITY( ACT_VM_SWINGMISS );
	REGISTER_SHARED_ACTIVITY( ACT_VM_SWINGHIT );
	REGISTER_SHARED_ACTIVITY( ACT_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_VM_LOWERED_TO_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_VM_RECOIL1 );
	REGISTER_SHARED_ACTIVITY( ACT_VM_RECOIL2 );
	REGISTER_SHARED_ACTIVITY( ACT_VM_RECOIL3 );
	REGISTER_SHARED_ACTIVITY( ACT_VM_PICKUP );
	REGISTER_SHARED_ACTIVITY( ACT_VM_RELEASE );

	REGISTER_SHARED_ACTIVITY ( ACT_VM_ATTACH_SILENCER );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DETACH_SILENCER );

	// TF2 Scout Pack
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DRAW_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_HOLSTER_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PULLBACK_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_SECONDARYATTACK_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_HITCENTER_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_SWINGHARD_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_TO_LOWERED_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_LOWERED_SPECIAL );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_LOWERED_TO_IDLE_SPECIAL );

	REGISTER_SHARED_ACTIVITY ( ACT_FISTS_VM_HITLEFT );
	REGISTER_SHARED_ACTIVITY ( ACT_FISTS_VM_HITRIGHT );
	REGISTER_SHARED_ACTIVITY ( ACT_FISTS_VM_SWINGHARD );
	REGISTER_SHARED_ACTIVITY ( ACT_FISTS_VM_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_FISTS_VM_DRAW );

//===========================
// HL2 Specific Activities
//===========================
	// SLAM Specialty Activites
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_IDLE);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_ND_IDLE);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_ATTACH);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_ATTACH2);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_ND_ATTACH);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_ND_ATTACH2);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_DETONATE);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_DETONATOR_HOLSTER);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_DRAW); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_ND_DRAW); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_TO_THROW); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_TO_THROW_ND); 
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_STICKWALL_TO_TRIPMINE_ND );
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_IDLE);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_ND_IDLE);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_THROW);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_THROW2);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_THROW_ND);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_THROW_ND2);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_DRAW); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_ND_DRAW); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_TO_STICKWALL); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_TO_STICKWALL_ND); 
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_DETONATE);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_DETONATOR_HOLSTER);
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_THROW_TO_TRIPMINE_ND );
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_TRIPMINE_IDLE);
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_TRIPMINE_DRAW);
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_TRIPMINE_ATTACH); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_TRIPMINE_ATTACH2); 
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_TRIPMINE_TO_STICKWALL_ND );
	REGISTER_SHARED_ACTIVITY( ACT_SLAM_TRIPMINE_TO_THROW_ND );
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_DETONATOR_IDLE); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_DETONATOR_DRAW); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_DETONATOR_DETONATE);
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_DETONATOR_HOLSTER);
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_DETONATOR_STICKWALL_DRAW); 
  	REGISTER_SHARED_ACTIVITY( ACT_SLAM_DETONATOR_THROW_DRAW); 

	// SHOTGUN specialty activities
	REGISTER_SHARED_ACTIVITY( ACT_SHOTGUN_RELOAD_START);
	REGISTER_SHARED_ACTIVITY( ACT_SHOTGUN_RELOAD_FINISH);
	REGISTER_SHARED_ACTIVITY( ACT_SHOTGUN_PUMP);

	// SMG2 special activities
	REGISTER_SHARED_ACTIVITY( ACT_SMG2_IDLE2 );
	REGISTER_SHARED_ACTIVITY( ACT_SMG2_FIRE2 );
	REGISTER_SHARED_ACTIVITY( ACT_SMG2_DRAW2 );
	REGISTER_SHARED_ACTIVITY( ACT_SMG2_RELOAD2 );
	REGISTER_SHARED_ACTIVITY( ACT_SMG2_DRYFIRE2 );
	REGISTER_SHARED_ACTIVITY( ACT_SMG2_TOAUTO );
	REGISTER_SHARED_ACTIVITY( ACT_SMG2_TOBURST );

	// Physcannon special activities
	REGISTER_SHARED_ACTIVITY( ACT_PHYSCANNON_UPGRADE );

	// weapon override activities
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_AR1 );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_AR2 );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_AR2_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_AR2_GRENADE );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_HMG1 );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_ML );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_SMG1 );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_SMG1_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_SMG2 );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_SHOTGUN );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_SHOTGUN_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_PISTOL_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_SLAM );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_TRIPWIRE );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_THROW );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_SNIPER_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_ATTACK_RPG );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK_SWING );

	REGISTER_SHARED_ACTIVITY( ACT_RANGE_AIM_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_AIM_SMG1_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_AIM_PISTOL_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RANGE_AIM_AR2_LOW );

	REGISTER_SHARED_ACTIVITY( ACT_COVER_PISTOL_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_COVER_SMG1_LOW );

	// weapon override activities
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_AR1 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_AR2 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_AR2_GRENADE );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_HMG1 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_ML );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_SMG1 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_SMG1_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_SMG2 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_SHOTGUN );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_PISTOL_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_SLAM );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_TRIPWIRE );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_THROW );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RANGE_ATTACK_SNIPER_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_MELEE_ATTACK_SWING );

	REGISTER_SHARED_ACTIVITY( ACT_IDLE_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_SMG1 );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_ANGRY_SMG1 );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_ANGRY_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_ANGRY_SHOTGUN );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_STEALTH_PISTOL );

	REGISTER_SHARED_ACTIVITY( ACT_IDLE_PACKAGE );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_PACKAGE );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_SUITCASE );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_SUITCASE );

	REGISTER_SHARED_ACTIVITY( ACT_IDLE_SMG1_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_SMG1_STIMULATED );

	REGISTER_SHARED_ACTIVITY( ACT_WALK_RIFLE_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_RIFLE_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_RIFLE_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_RIFLE_STIMULATED );

	REGISTER_SHARED_ACTIVITY( ACT_IDLE_AIM_RIFLE_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM_RIFLE_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM_RIFLE_STIMULATED );

	REGISTER_SHARED_ACTIVITY( ACT_IDLE_SHOTGUN_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_SHOTGUN_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_SHOTGUN_AGITATED );

	// Policing activities
	REGISTER_SHARED_ACTIVITY( ACT_WALK_ANGRY );
	REGISTER_SHARED_ACTIVITY( ACT_POLICE_HARASS1 );
	REGISTER_SHARED_ACTIVITY( ACT_POLICE_HARASS2 );

	// Manned guns
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_MANNEDGUN );

	// Melee weapon activities
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_ANGRY_MELEE );

	// RPG activities
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_RPG_RELAXED );
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_RPG ); 
	REGISTER_SHARED_ACTIVITY( ACT_IDLE_ANGRY_RPG );
	REGISTER_SHARED_ACTIVITY( ACT_COVER_LOW_RPG ); 
	REGISTER_SHARED_ACTIVITY( ACT_WALK_RPG );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_RPG ); 
	REGISTER_SHARED_ACTIVITY( ACT_WALK_CROUCH_RPG ); 
	REGISTER_SHARED_ACTIVITY( ACT_RUN_CROUCH_RPG ); 
	REGISTER_SHARED_ACTIVITY( ACT_WALK_RPG_RELAXED ); 
	REGISTER_SHARED_ACTIVITY( ACT_RUN_RPG_RELAXED );

	REGISTER_SHARED_ACTIVITY( ACT_WALK_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_CROUCH_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_CROUCH_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_CROUCH_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_CROUCH_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_STEALTH_PISTOL );

	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM_SHOTGUN );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM_SHOTGUN );

	REGISTER_SHARED_ACTIVITY( ACT_WALK_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_STEALTH_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_AIM_STEALTH_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_RUN_AIM_STEALTH_PISTOL );

	// Reloads
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD_PISTOL_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD_SMG1 );
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD_SMG1_LOW );
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD_SHOTGUN );
	REGISTER_SHARED_ACTIVITY( ACT_RELOAD_SHOTGUN_LOW );

	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RELOAD );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RELOAD_PISTOL );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RELOAD_SMG1 );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_RELOAD_SHOTGUN );

	// Busy animations
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_LEAN_LEFT );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_LEAN_LEFT_ENTRY );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_LEAN_LEFT_EXIT );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_LEAN_BACK );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_LEAN_BACK_ENTRY );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_LEAN_BACK_EXIT );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_SIT_GROUND );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_SIT_GROUND_ENTRY );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_SIT_GROUND_EXIT );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_SIT_CHAIR );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_SIT_CHAIR_ENTRY );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_SIT_CHAIR_EXIT );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_STAND );
	REGISTER_SHARED_ACTIVITY( ACT_BUSY_QUEUE );
	
	// Dodge animations
	REGISTER_SHARED_ACTIVITY( ACT_DUCK_DODGE );

	REGISTER_SHARED_ACTIVITY( ACT_DIE_BARNACLE_SWALLOW );
	REGISTER_SHARED_ACTIVITY( ACT_GESTURE_BARNACLE_STRANGLE );
	REGISTER_SHARED_ACTIVITY( ACT_PHYSCANNON_DETACH );
	REGISTER_SHARED_ACTIVITY( ACT_PHYSCANNON_ANIMATE );
	REGISTER_SHARED_ACTIVITY( ACT_PHYSCANNON_ANIMATE_PRE );
	REGISTER_SHARED_ACTIVITY( ACT_PHYSCANNON_ANIMATE_POST );

	REGISTER_SHARED_ACTIVITY( ACT_DIE_FRONTSIDE );
	REGISTER_SHARED_ACTIVITY( ACT_DIE_RIGHTSIDE );
	REGISTER_SHARED_ACTIVITY( ACT_DIE_BACKSIDE );
	REGISTER_SHARED_ACTIVITY( ACT_DIE_LEFTSIDE );

	REGISTER_SHARED_ACTIVITY( ACT_OPEN_DOOR );

	// Dynamic interactions
	REGISTER_SHARED_ACTIVITY( ACT_DI_ALYX_ZOMBIE_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_DI_ALYX_ZOMBIE_TORSO_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_DI_ALYX_HEADCRAB_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_DI_ALYX_ANTLION );
	REGISTER_SHARED_ACTIVITY( ACT_DI_ALYX_ZOMBIE_SHOTGUN64 );
	REGISTER_SHARED_ACTIVITY( ACT_DI_ALYX_ZOMBIE_SHOTGUN26 );

	//Readiness transitions
	REGISTER_SHARED_ACTIVITY( ACT_READINESS_RELAXED_TO_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_READINESS_RELAXED_TO_STIMULATED_WALK );
	REGISTER_SHARED_ACTIVITY( ACT_READINESS_AGITATED_TO_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_READINESS_STIMULATED_TO_RELAXED );

	REGISTER_SHARED_ACTIVITY( ACT_READINESS_PISTOL_RELAXED_TO_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_READINESS_PISTOL_RELAXED_TO_STIMULATED_WALK );
	REGISTER_SHARED_ACTIVITY( ACT_READINESS_PISTOL_AGITATED_TO_STIMULATED );
	REGISTER_SHARED_ACTIVITY( ACT_READINESS_PISTOL_STIMULATED_TO_RELAXED );

	REGISTER_SHARED_ACTIVITY( ACT_IDLE_CARRY );
	REGISTER_SHARED_ACTIVITY( ACT_WALK_CARRY );

//===========================
// TF2 Specific Activities
//===========================
	REGISTER_SHARED_ACTIVITY( ACT_STARTDYING );
	REGISTER_SHARED_ACTIVITY( ACT_DYINGLOOP );
	REGISTER_SHARED_ACTIVITY( ACT_DYINGTODEAD );
	REGISTER_SHARED_ACTIVITY( ACT_RIDE_MANNED_GUN );

	// All viewmodels
	REGISTER_SHARED_ACTIVITY( ACT_VM_SPRINT_ENTER );
	REGISTER_SHARED_ACTIVITY( ACT_VM_SPRINT_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_VM_SPRINT_LEAVE );

	// Looping weapon firing
	REGISTER_SHARED_ACTIVITY( ACT_FIRE_START );
	REGISTER_SHARED_ACTIVITY( ACT_FIRE_LOOP );
	REGISTER_SHARED_ACTIVITY( ACT_FIRE_END );

	// TF2
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHING_GRENADEIDLE );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHING_GRENADEREADY );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHING_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_OVERLAY_GRENADEIDLE );
	REGISTER_SHARED_ACTIVITY( ACT_OVERLAY_GRENADEREADY );
	REGISTER_SHARED_ACTIVITY( ACT_OVERLAY_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_OVERLAY_SHIELD_UP );
	REGISTER_SHARED_ACTIVITY( ACT_OVERLAY_SHIELD_DOWN );
	REGISTER_SHARED_ACTIVITY( ACT_OVERLAY_SHIELD_UP_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_OVERLAY_SHIELD_ATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_OVERLAY_SHIELD_KNOCKBACK );
	REGISTER_SHARED_ACTIVITY( ACT_SHIELD_UP );
	REGISTER_SHARED_ACTIVITY( ACT_SHIELD_DOWN );
	REGISTER_SHARED_ACTIVITY( ACT_SHIELD_UP_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_SHIELD_ATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_SHIELD_KNOCKBACK );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHING_SHIELD_UP );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHING_SHIELD_DOWN );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHING_SHIELD_UP_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHING_SHIELD_ATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_CROUCHING_SHIELD_KNOCKBACK );

	REGISTER_SHARED_ACTIVITY( ACT_TURNRIGHT45 );
	REGISTER_SHARED_ACTIVITY( ACT_TURNLEFT45 );
	REGISTER_SHARED_ACTIVITY( ACT_TURN );

	REGISTER_SHARED_ACTIVITY( ACT_OBJ_ASSEMBLING );
	REGISTER_SHARED_ACTIVITY( ACT_OBJ_DISMANTLING );
	REGISTER_SHARED_ACTIVITY( ACT_OBJ_STARTUP );
	REGISTER_SHARED_ACTIVITY( ACT_OBJ_RUNNING );
	REGISTER_SHARED_ACTIVITY( ACT_OBJ_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_OBJ_PLACING );
	REGISTER_SHARED_ACTIVITY( ACT_OBJ_DETERIORATING );
	REGISTER_SHARED_ACTIVITY( ACT_OBJ_UPGRADING );

	// Deploy
	REGISTER_SHARED_ACTIVITY( ACT_DEPLOY );
	REGISTER_SHARED_ACTIVITY( ACT_DEPLOY_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_UNDEPLOY );

//===========================
// HL1 Specific Activities
//===========================

	// Grenades
	REGISTER_SHARED_ACTIVITY( ACT_GRENADE_ROLL );
	REGISTER_SHARED_ACTIVITY( ACT_GRENADE_TOSS );

	// Hand grenade
	REGISTER_SHARED_ACTIVITY( ACT_HANDGRENADE_THROW1 );
	REGISTER_SHARED_ACTIVITY( ACT_HANDGRENADE_THROW2 );
	REGISTER_SHARED_ACTIVITY( ACT_HANDGRENADE_THROW3 );

	// Shotgun
	REGISTER_SHARED_ACTIVITY( ACT_SHOTGUN_IDLE_DEEP );
	REGISTER_SHARED_ACTIVITY( ACT_SHOTGUN_IDLE4 );

	// Glock
	REGISTER_SHARED_ACTIVITY( ACT_GLOCK_SHOOTEMPTY );
	REGISTER_SHARED_ACTIVITY( ACT_GLOCK_SHOOT_RELOAD );

	// RPG
	REGISTER_SHARED_ACTIVITY( ACT_RPG_DRAW_UNLOADED );
	REGISTER_SHARED_ACTIVITY( ACT_RPG_HOLSTER_UNLOADED );
	REGISTER_SHARED_ACTIVITY( ACT_RPG_IDLE_UNLOADED );
	REGISTER_SHARED_ACTIVITY( ACT_RPG_FIDGET_UNLOADED );

	// Crossbow
	REGISTER_SHARED_ACTIVITY( ACT_CROSSBOW_DRAW_UNLOADED );
	REGISTER_SHARED_ACTIVITY( ACT_CROSSBOW_IDLE_UNLOADED );
	REGISTER_SHARED_ACTIVITY( ACT_CROSSBOW_FIDGET_UNLOADED );

	// Gauss
	REGISTER_SHARED_ACTIVITY( ACT_GAUSS_SPINUP );
	REGISTER_SHARED_ACTIVITY( ACT_GAUSS_SPINCYCLE );

	// Tripmine
	REGISTER_SHARED_ACTIVITY( ACT_TRIPMINE_GROUND );
	REGISTER_SHARED_ACTIVITY( ACT_TRIPMINE_WORLD );

//===========================
// CSPort Specific Activities
//===========================

	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_SILENCED );		// fire
	REGISTER_SHARED_ACTIVITY ( ACT_VM_RELOAD_SILENCED );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DRYFIRE_SILENCED );				// fire with no ammo loaded.
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_SILENCED );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DRAW_SILENCED );

	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_EMPTY_LEFT );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DRYFIRE_LEFT );

	REGISTER_SHARED_ACTIVITY ( ACT_PLAYER_IDLE_FIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_PLAYER_CROUCH_FIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_PLAYER_CROUCH_WALK_FIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_PLAYER_WALK_FIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_PLAYER_RUN_FIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_IDLETORUN );
	REGISTER_SHARED_ACTIVITY ( ACT_RUNTOIDLE );

//===========================
// DoD Specific Activities
//===========================

	REGISTER_SHARED_ACTIVITY ( ACT_SPRINT );

	// To get in and out of prone state.
	REGISTER_SHARED_ACTIVITY ( ACT_GET_DOWN_STAND );
	REGISTER_SHARED_ACTIVITY ( ACT_GET_UP_STAND );
	REGISTER_SHARED_ACTIVITY ( ACT_GET_DOWN_CROUCH );
	REGISTER_SHARED_ACTIVITY ( ACT_GET_UP_CROUCH );
	REGISTER_SHARED_ACTIVITY ( ACT_PRONE_FORWARD );
	REGISTER_SHARED_ACTIVITY ( ACT_PRONE_IDLE );

	REGISTER_SHARED_ACTIVITY ( ACT_DEEPIDLE1 );
	REGISTER_SHARED_ACTIVITY ( ACT_DEEPIDLE2 );
	REGISTER_SHARED_ACTIVITY ( ACT_DEEPIDLE3 );
	REGISTER_SHARED_ACTIVITY ( ACT_DEEPIDLE4 );
 
	REGISTER_SHARED_ACTIVITY ( ACT_VM_RELOAD_DEPLOYED ); 
	REGISTER_SHARED_ACTIVITY ( ACT_VM_RELOAD_IDLE );

	REGISTER_SHARED_ACTIVITY ( ACT_VM_DRAW_DEPLOYED );

	//Weapon is empty activities
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DRAW_EMPTY );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_EMPTY );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_RELOAD_EMPTY );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_EMPTY );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED_EMPTY );

	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_8 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_7 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_6 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_5 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_4 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_3 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_1 );

	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED_8 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED_7 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED_6 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED_5 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED_4 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED_3 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_IDLE_DEPLOYED_1 );

	// Animation from prone idle to standing/crouch idle. Number designates bullets left
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY_8 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY_7 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY_6 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY_5 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY_4 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY_3 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY_1 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNDEPLOY_EMPTY );

	// Animation from standing/crouch idle to prone idle. Number designates bullets left
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY_8 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY_7 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY_6 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY_5 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY_4 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY_3 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY_1 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_DEPLOY_EMPTY );

	// Shooting animations for standing/crouch position.  Number designates bullets left at START of animation
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_8 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_7 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_6 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_5 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_4 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_3 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_1 );

	// Shooting animations for prone position. Number designates bullets left at START of animation
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED_8 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED_7 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED_6 ); 
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED_5 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED_4 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED_3 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED_1 );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_PRIMARYATTACK_DEPLOYED_EMPTY );

	// Player anim ACTs

	// Base activities, we translate from these
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_IDLE_ZOOMED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_ZOOMED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_ZOOMED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_ZOOMED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_ZOOMED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_FORWARD_ZOOMED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_PRONE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM );

	// Positions
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_PISTOL	);
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_PISTOL	);
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_PISTOL );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_C96	);
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_C96	);
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_C96 );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_RIFLE );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_BOLT );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_TOMMY );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_MP40 );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_MP44 );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_GREASE );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_MG );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_30CAL );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_AIM_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_AIM_GREN_FRAG );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_GREN_STICK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_GREN_STICK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_GREN_STICK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_GREN_STICK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_GREN_STICK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_GREN_STICK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_AIM_GREN_STICK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_AIM_GREN_STICK );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_AIM_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_AIM_KNIFE );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_SPADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_SPADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_SPADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_SPADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_SPADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_SPADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_AIM_SPADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_AIM_SPADE );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_BAZOOKA );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_PSCHRECK );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_AIM_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_AIM_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_AIM_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_AIM_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_AIM_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_AIM_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_BAR );

	// ZOom
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_ZOOM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_ZOOM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_ZOOM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_ZOOM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_ZOOM_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_ZOOM_RIFLE );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_ZOOM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_ZOOM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_ZOOM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_ZOOM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_ZOOM_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_ZOOM_BOLT );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_ZOOM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_ZOOM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_ZOOM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_ZOOM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_ZOOM_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_ZOOM_BAZOOKA );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_ZOOM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_ZOOM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_ZOOM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_ZOOM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_ZOOM_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_ZOOM_PSCHRECK );

	// Deployed Aim
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_DEPLOY_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_DEPLOY_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_DEPLOY_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_DEPLOY_30CAL );

	// Prone Deployed Aim
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_DEPLOY_RIFLE  );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_DEPLOY_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_DEPLOY_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_DEPLOY_30CAL );

	// Attacks

	// Rifle
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_PRONE_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_DEPLOYED_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_DEPLOYED_RIFLE );

	// Bolt
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_BOLT  );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_PRONE_BOLT  );

	// Tommy
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_PRONE_TOMMY  );

	// MP40
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_MP40  );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_PRONE_MP40  );

	// MP44
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_MP44  );

	// Greasegun
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_GREASE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_GREASE  );

	// Pistols (Colt, Luger)
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_PISTOL  );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_C96 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_C96  );

	// Mgs (mg42, mg34)
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_MG  );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_DEPLOYED_MG  );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_DEPLOYED_MG  );

	// 30cal
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_DEPLOYED_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_DEPLOYED_30CAL  );

	// Grenades
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_GREN_STICK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_GREN_STICK );

	// Knife
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_KNIFE );

	// Spade
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_SPADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_SPADE );

	// Bazooka
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_BAZOOKA );

	// Pschreck
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_PSCHRECK  );

	// Bar
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_PRONE_BAR  );

	// Reloads
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_GARAND );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_K43 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_M1CARBINE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_GREASEGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_FG42 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_RIFLEGRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_C96 );

	// Crouch
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_RIFLEGRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_M1CARBINE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_CROUCH_C96 );

	// Bazookas
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_ZOOMLOAD_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_ZOOMLOAD_PSCHRECK );

	// Deployed
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_DEPLOYED_FG42 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_DEPLOYED_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_DEPLOYED_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_DEPLOYED_MG34 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_DEPLOYED_BAR );

	// Prone
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_GARAND );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_M1CARBINE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_K43 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_MP40 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_GREASEGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_FG42 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_RIFLEGRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_C96 );

	// Prone bazooka
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_ZOOMLOAD_PRONE_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_ZOOMLOAD_PRONE_PSCHRECK );

	// Prone deployed
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_DEPLOYED_BAR );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_DEPLOYED_FG42 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_DEPLOYED_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_DEPLOYED_MG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RELOAD_PRONE_DEPLOYED_MG34 );

	// Prone zoomed aim
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_ZOOM_FORWARD_RIFLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_ZOOM_FORWARD_BOLT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_ZOOM_FORWARD_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONE_ZOOM_FORWARD_PSCHRECK );
	
	// Crouch attack.
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_CROUCH );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_CROUCH_SPADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_CROUCH_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_CROUCH_GREN_FRAG );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRIMARYATTACK_CROUCH_GREN_STICK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_CROUCH );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_CROUCH_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SECONDARYATTACK_CROUCH_MP40 );

	// Hand Signals
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_MG42 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_STICKGRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_IDLE_K98 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_30CAL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_BAZOOKA );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_PSCHRECK );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_KNIFE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_MG42 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_STICKGRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_TOMMY );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_MP44 );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_HS_CROUCH_K98 );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_STAND_IDLE_TNT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCH_IDLE_TNT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_CROUCHWALK_IDLE_TNT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_WALK_IDLE_TNT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_RUN_IDLE_TNT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_SPRINT_IDLE_TNT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PRONEWALK_IDLE_TNT );

	REGISTER_SHARED_ACTIVITY ( ACT_DOD_PLANT_TNT );
	REGISTER_SHARED_ACTIVITY ( ACT_DOD_DEFUSE_TNT );

//HL2MP
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP );
	
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_PISTOL );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_PISTOL );

	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_SMG1 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_SMG1 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_SMG1 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_SMG1 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_SMG1 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_SMG1 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_SMG1 );

  	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_AR2 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_AR2 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_AR2 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_AR2 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_AR2 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_AR2 );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_AR2 );

  	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_SHOTGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_SHOTGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_SHOTGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_SHOTGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_SHOTGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_SHOTGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_SHOTGUN );

  	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_RPG );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_RPG );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_RPG );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_RPG );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_RPG );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_RPG );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_RPG );

  	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_GRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_GRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_GRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_GRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_GRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_GRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_GRENADE );

  	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_PHYSGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_PHYSGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_PHYSGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_PHYSGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_PHYSGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_PHYSGUN );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_PHYSGUN );

  	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROSSBOW );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_CROSSBOW );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_CROSSBOW );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_CROSSBOW );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_CROSSBOW );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_CROSSBOW );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_CROSSBOW );

	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_MELEE );

	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_SLAM );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_RUN_SLAM );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_IDLE_CROUCH_SLAM );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_WALK_CROUCH_SLAM );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RANGE_ATTACK_SLAM );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_GESTURE_RELOAD_SLAM );
	REGISTER_SHARED_ACTIVITY ( ACT_HL2MP_JUMP_SLAM );

	// Portal
	REGISTER_SHARED_ACTIVITY ( ACT_VM_FIZZLE );
	
	// Multiplayer
	REGISTER_SHARED_ACTIVITY ( ACT_MP_STAND_IDLE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_IDLE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_DEPLOYED_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DEPLOYED_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RUN );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_WALK );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_AIRWALK );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SPRINT );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_START );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_FLOAT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LAND );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_VCD );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_PRIMARYFIRE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_PRIMARYFIRE_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_SECONDARYFIRE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_GRENADE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_PRIMARYFIRE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_PRIMARYFIRE_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_SECONDARYFIRE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_GRENADE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_PRIMARYFIRE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_SECONDARYFIRE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_GRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_PRIMARYFIRE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_SECONDARYFIRE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_GRENADE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_PREFIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_POSTFIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_STARTFIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_PREFIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_POSTFIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_PREFIRE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_POSTFIRE );

	// Multiplayer - Primary
	REGISTER_SHARED_ACTIVITY ( ACT_MP_STAND_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RUN_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_WALK_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_AIRWALK_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_START_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_FLOAT_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LAND_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DEPLOYED_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_DEPLOYED_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_DEPLOYED_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_DEPLOYED_IDLE_PRIMARY );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_PRIMARY_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_PRIMARY_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_PRIMARY );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_PRIMARY_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_PRIMARY_END );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_PRIMARY_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_PRIMARY_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_PRIMARY_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_PRIMARY_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_PRIMARY_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_PRIMARY_END );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_PRIMARY_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_PRIMARY_LOOP_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_PRIMARY_END_2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_PRIMARY_2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_PRIMARY_LOOP_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_PRIMARY_END_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_PRIMARY_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_PRIMARY_LOOP_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_PRIMARY_END_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_PRIMARY_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_PRIMARY_LOOP_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_PRIMARY_END_2 );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_PRIMARY_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_PRIMARY_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_PRIMARY_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_PRIMARY_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_PRIMARY_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_PRIMARY_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_PRIMARY_LOOP_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_PRIMARY_LOOP_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_PRIMARY_LOOP_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_PRIMARY_END_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_PRIMARY_END_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_PRIMARY_END_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_PRIMARY_ALT );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_PRIMARY_SUPER );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_PRIMARY_SUPER );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_PRIMARY_SUPER );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_GRENADE_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_GRENADE_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_GRENADE_PRIMARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_GRENADE_PRIMARY );	

	// Secondary
	REGISTER_SHARED_ACTIVITY ( ACT_MP_STAND_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RUN_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_WALK_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_AIRWALK_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_START_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_FLOAT_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LAND_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_SECONDARY );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_SECONDARY );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_SECONDARY_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_SECONDARY_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_SECONDARY_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_SECONDARY_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_SECONDARY_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_SECONDARY_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_SECONDARY_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_SECONDARY_END );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_SECONDARY_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_SECONDARY_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_SECONDARY_2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_SECONDARY_2 );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_GRENADE_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_GRENADE_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_GRENADE_SECONDARY );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_GRENADE_SECONDARY );	

	// Secondary2
	REGISTER_SHARED_ACTIVITY ( ACT_MP_STAND_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RUN_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_WALK_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_AIRWALK_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_START_SECONDARY2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_FLOAT_SECONDARY2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LAND_SECONDARY2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_SECONDARY2 );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_SECONDARY2 );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_SECONDARY2 );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_SECONDARY2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_SECONDARY2_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_STAND_SECONDARY2_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_SECONDARY2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_SECONDARY2_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_CROUCH_SECONDARY2_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_SECONDARY2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_SECONDARY2_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_SWIM_SECONDARY2_END );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_SECONDARY2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_SECONDARY2_LOOP );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RELOAD_AIRWALK_SECONDARY2_END );

	// Melee
	REGISTER_SHARED_ACTIVITY ( ACT_MP_STAND_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RUN_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_WALK_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_AIRWALK_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_START_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_FLOAT_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LAND_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_MELEE );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_MELEE_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_MELEE_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_MELEE );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_GRENADE_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_GRENADE_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_GRENADE_MELEE );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_GRENADE_MELEE );	

	// Item1
	REGISTER_SHARED_ACTIVITY( ACT_MP_STAND_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCH_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RUN_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_WALK_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_AIRWALK_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCHWALK_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_START_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_FLOAT_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_LAND_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_SWIM_ITEM1 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_ITEM1_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_ITEM1_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_SWIM_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_AIRWALK_ITEM1 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_DEPLOYED_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_DEPLOYED_IDLE_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCHWALK_DEPLOYED_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCH_DEPLOYED_IDLE_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_PRIMARY_DEPLOYED_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_PRIMARY_DEPLOYED_ITEM1 );

	// Item2
	REGISTER_SHARED_ACTIVITY( ACT_MP_STAND_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCH_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RUN_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_WALK_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_AIRWALK_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCHWALK_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_START_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_FLOAT_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_LAND_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_SWIM_ITEM2 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_ITEM2_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_ITEM2_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_SWIM_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_AIRWALK_ITEM2 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_HARD_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_HARD_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_SWIM_HARD_ITEM2 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_DEPLOYED_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_DEPLOYED_IDLE_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCHWALK_DEPLOYED_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCH_DEPLOYED_IDLE_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_PRIMARY_DEPLOYED_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_PRIMARY_DEPLOYED_ITEM2 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_STAND_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_STAND_ITEM2_LOOP );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_STAND_ITEM2_END );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_CROUCH_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_CROUCH_ITEM2_LOOP );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_CROUCH_ITEM2_END );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_SWIM_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_SWIM_ITEM2_LOOP );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_SWIM_ITEM2_END );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_AIRWALK_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_AIRWALK_ITEM2_LOOP );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_AIRWALK_ITEM2_END );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_NO_AMMO_ITEM2 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_GRENADE_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_GRENADE_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_SWIM_GRENADE_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_AIRWALK_GRENADE_ITEM2 );

	// Passtime
	REGISTER_SHARED_ACTIVITY( ACT_MP_STAND_PASSTIME );	
	REGISTER_SHARED_ACTIVITY( ACT_MP_RUN_PASSTIME );	
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCHWALK_PASSTIME );	

	// Flinches
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_PRIMARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_ITEM2 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_HEAD );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_CHEST );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_STOMACH );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_LEFTARM );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_RIGHTARM );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_LEFTLEG );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_FLINCH_RIGHTLEG );

	// TF Specific
	REGISTER_SHARED_ACTIVITY ( ACT_MP_GRENADE1_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_GRENADE1_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_GRENADE1_ATTACK );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_GRENADE2_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_GRENADE2_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_GRENADE2_ATTACK );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_PRIMARY_GRENADE1_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_PRIMARY_GRENADE1_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_PRIMARY_GRENADE1_ATTACK );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_PRIMARY_GRENADE2_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_PRIMARY_GRENADE2_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_PRIMARY_GRENADE2_ATTACK );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_SECONDARY_GRENADE1_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SECONDARY_GRENADE1_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SECONDARY_GRENADE1_ATTACK );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SECONDARY_GRENADE2_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SECONDARY_GRENADE2_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SECONDARY_GRENADE2_ATTACK );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_MELEE_GRENADE1_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_MELEE_GRENADE1_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_MELEE_GRENADE1_ATTACK );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_MELEE_GRENADE2_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_MELEE_GRENADE2_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_MELEE_GRENADE2_ATTACK );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM1_GRENADE1_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM1_GRENADE1_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM1_GRENADE1_ATTACK );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM1_GRENADE2_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM1_GRENADE2_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM1_GRENADE2_ATTACK );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM2_GRENADE1_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM2_GRENADE1_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM2_GRENADE1_ATTACK );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM2_GRENADE2_DRAW );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM2_GRENADE2_IDLE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ITEM2_GRENADE2_ATTACK );

	// Building
	REGISTER_SHARED_ACTIVITY ( ACT_MP_STAND_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RUN_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_WALK_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_AIRWALK_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_BUILDING );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_START_BUILDING );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_FLOAT_BUILDING );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LAND_BUILDING );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_BUILDING );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_BUILDING );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_GRENADE_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_GRENADE_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_GRENADE_BUILDING );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_GRENADE_BUILDING );	

	// Building - Deployed
	REGISTER_SHARED_ACTIVITY ( ACT_MP_STAND_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RUN_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_WALK_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_AIRWALK_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_BUILDING_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_START_BUILDING_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_FLOAT_BUILDING_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LAND_BUILDING_DEPLOYED );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_BUILDING_DEPLOYED );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_BUILDING_DEPLOYED );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_GRENADE_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_CROUCH_GRENADE_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_GRENADE_BUILDING_DEPLOYED );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_AIRWALK_GRENADE_BUILDING_DEPLOYED );	

	// PDA
	REGISTER_SHARED_ACTIVITY ( ACT_MP_STAND_PDA );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_PDA );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RUN_PDA );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_WALK_PDA );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_AIRWALK_PDA );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_PDA );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_PDA );	
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_START_PDA );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_FLOAT_PDA );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LAND_PDA );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_PDA );	

	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_STAND_PDA );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_ATTACK_SWIM_PDA );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_STAND_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCH_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_RUN_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_WALK_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_AIRWALK_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_CROUCHWALK_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_START_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_FLOAT_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_JUMP_LAND_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_SWIM_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP_LOSERSTATE );

	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP_CROUCH );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP_CROUCH_PRIMARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP_CROUCH_SECONDARY );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP_CROUCH_MELEE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP_CROUCH_ITEM1 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP_CROUCH_ITEM2 );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP_CROUCH_LOSERSTATE );
	REGISTER_SHARED_ACTIVITY ( ACT_MP_DOUBLEJUMP_CROUCH_PASSTIME );

	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_HANDMOUTH );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FINGERPOINT );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FISTPUMP );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_THUMBSUP );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODYES );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODNO );

	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_HANDMOUTH_PRIMARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FINGERPOINT_PRIMARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FISTPUMP_PRIMARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_THUMBSUP_PRIMARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODYES_PRIMARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODNO_PRIMARY );

	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_HANDMOUTH_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FINGERPOINT_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FISTPUMP_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_THUMBSUP_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODYES_SECONDARY );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODNO_SECONDARY );

	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_HANDMOUTH_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FINGERPOINT_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FISTPUMP_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_THUMBSUP_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODYES_MELEE );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODNO_MELEE );

	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_HANDMOUTH_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FINGERPOINT_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FISTPUMP_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_THUMBSUP_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODYES_ITEM1 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODNO_ITEM1 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_HANDMOUTH_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FINGERPOINT_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FISTPUMP_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_THUMBSUP_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODYES_ITEM2 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODNO_ITEM2 );

	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_HANDMOUTH_BUILDING );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FINGERPOINT_BUILDING );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FISTPUMP_BUILDING );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_THUMBSUP_BUILDING );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODYES_BUILDING );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODNO_BUILDING );

	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_HANDMOUTH_PDA );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FINGERPOINT_PDA );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_FISTPUMP_PDA );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_THUMBSUP_PDA );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODYES_PDA );
	REGISTER_SHARED_ACTIVITY( ACT_MP_GESTURE_VC_NODNO_PDA );

	REGISTER_SHARED_ACTIVITY( ACT_MP_STUN_BEGIN );
	REGISTER_SHARED_ACTIVITY( ACT_MP_STUN_MIDDLE );
	REGISTER_SHARED_ACTIVITY( ACT_MP_STUN_END );

	REGISTER_SHARED_ACTIVITY( ACT_MP_PASSTIME_THROW_BEGIN );
	REGISTER_SHARED_ACTIVITY( ACT_MP_PASSTIME_THROW_MIDDLE );
	REGISTER_SHARED_ACTIVITY( ACT_MP_PASSTIME_THROW_END );
	REGISTER_SHARED_ACTIVITY( ACT_MP_PASSTIME_THROW_CANCEL );

	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNUSABLE );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_UNUSABLE_TO_USABLE );
	REGISTER_SHARED_ACTIVITY ( ACT_VM_USABLE_TO_UNUSABLE );

	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_SECONDARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_RELOAD );			
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_RELOAD_START );			
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_RELOAD_FINISH );			
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_DRYFIRE );		
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_LOWERED_TO_IDLE );

	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_RELOAD_2 );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_RELOAD_START_2 );		
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_RELOAD_FINISH_2 );

	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_RELOAD_3 );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_RELOAD_START_3 );		
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_RELOAD_FINISH_3 );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_PRIMARYATTACK_3 );

	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_SECONDARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_RELOAD );			
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_RELOAD_START );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_RELOAD_FINISH );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_RELOAD2 );			
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_DRYFIRE );		
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_LOWERED_TO_IDLE );

	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_DRAW_2 );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_IDLE_2 );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_PRIMARYATTACK_2 );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_RELOAD_2 );

	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_SECONDARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_RELOAD );			
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_DRYFIRE );		
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_LOWERED_TO_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_STUN );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_HITCENTER );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_SWINGHARD );

	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_SECONDARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_RELOAD );			
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_DRYFIRE );		
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_PDA_VM_LOWERED_TO_IDLE );

	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_PDA1_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_PDA2_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_BLD_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_PDA1_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_PDA2_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_BLD_VM_IDLE );

	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_IDLE_2 );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_SECONDARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_RELOAD );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_DRYFIRE );		
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_LOWERED_TO_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_RELOAD_START );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_RELOAD_FINISH );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_HITCENTER );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_VM_SWINGHARD );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_BACKSTAB_VM_UP );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_BACKSTAB_VM_DOWN );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_BACKSTAB_VM_IDLE );			
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_ITEM1_STUN );	

	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_SECONDARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_RELOAD );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_DRYFIRE );		
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_LOWERED_TO_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_CHARGE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_IDLE_2 );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_IDLE_3 );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_CHARGE_IDLE_3 );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_HITCENTER );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_VM_SWINGHARD );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_BACKSTAB_VM_UP );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_BACKSTAB_VM_DOWN );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_BACKSTAB_VM_IDLE );			
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_ITEM2_STUN );

	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_SECONDARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_RELOAD );			
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_DRYFIRE );		
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_LOWERED_TO_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_CHARGE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_IDLE_2 );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_IDLE_3 );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_CHARGE_IDLE_3 );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_HITCENTER );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM3_VM_SWINGHARD );

	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_SECONDARY2ATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_RELOAD );	
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_RELOAD_START );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_RELOAD_FINISH );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_RELOAD2 );			
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_DRYFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY2_VM_LOWERED_TO_IDLE );

	REGISTER_SHARED_ACTIVITY( ACT_BACKSTAB_VM_UP );
	REGISTER_SHARED_ACTIVITY( ACT_BACKSTAB_VM_DOWN );
	REGISTER_SHARED_ACTIVITY( ACT_BACKSTAB_VM_IDLE );

	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_ATTACK_STAND_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_ATTACK_STAND_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_ATTACK_STAND_STARTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_ATTACK_CROUCH_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_ATTACK_CROUCH_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_ATTACK_SWIM_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_ATTACK_SWIM_POSTFIRE );

	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_ATTACK_STAND_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_ATTACK_STAND_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_ATTACK_STAND_STARTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_ATTACK_CROUCH_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_ATTACK_CROUCH_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_ATTACK_SWIM_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_ATTACK_SWIM_POSTFIRE );

	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK_STAND_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK_STAND_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK_STAND_STARTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK_CROUCH_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK_CROUCH_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK_SWIM_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ATTACK_SWIM_POSTFIRE );

	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_ATTACK_STAND_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_ATTACK_STAND_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_ATTACK_STAND_STARTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_ATTACK_CROUCH_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_ATTACK_CROUCH_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_ATTACK_SWIM_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM1_ATTACK_SWIM_POSTFIRE );

	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_ATTACK_STAND_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_ATTACK_STAND_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_ATTACK_STAND_STARTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_ATTACK_CROUCH_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_ATTACK_CROUCH_POSTFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_ATTACK_SWIM_PREFIRE );
	REGISTER_SHARED_ACTIVITY( ACT_ITEM2_ATTACK_SWIM_POSTFIRE );

	REGISTER_SHARED_ACTIVITY( ACT_MP_STAND_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCH_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RUN_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_WALK_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_AIRWALK_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_CROUCHWALK_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_START_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_FLOAT_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_LAND_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_SWIM_MELEE_ALLCLASS );

	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_STAND_MELEE_SECONDARY_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_CROUCH_MELEE_SECONDARY_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_SWIM_MELEE_ALLCLASS );
	REGISTER_SHARED_ACTIVITY( ACT_MP_ATTACK_AIRWALK_MELEE_ALLCLASS );

	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_HOLSTER );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_PULLBACK );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_SECONDARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_RELOAD );			
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_DRYFIRE );		
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_IDLE_TO_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_IDLE_LOWERED );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_LOWERED_TO_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_STUN );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_HITCENTER );		
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_ALLCLASS_VM_SWINGHARD );
		
	REGISTER_SHARED_ACTIVITY( ACT_MP_STAND_BOMB );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_START_BOMB );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_FLOAT_BOMB );
	REGISTER_SHARED_ACTIVITY( ACT_MP_JUMP_LAND_BOMB );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RUN_BOMB );
	REGISTER_SHARED_ACTIVITY( ACT_MP_SWIM_BOMB );
		
	REGISTER_SHARED_ACTIVITY( ACT_VM_DRAW_QRL );
	REGISTER_SHARED_ACTIVITY( ACT_VM_IDLE_QRL );
	REGISTER_SHARED_ACTIVITY( ACT_VM_PULLBACK_QRL );
	REGISTER_SHARED_ACTIVITY( ACT_VM_PRIMARYATTACK_QRL );
	REGISTER_SHARED_ACTIVITY( ACT_VM_RELOAD_QRL );
	REGISTER_SHARED_ACTIVITY( ACT_VM_RELOAD_START_QRL );			
	REGISTER_SHARED_ACTIVITY( ACT_VM_RELOAD_FINISH_QRL );	

	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_STAND_PRIMARY3 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_CROUCH_PRIMARY3 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_AIRWALK_PRIMARY3 );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_STAND_PRIMARY3_LOOP );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_CROUCH_PRIMARY3_LOOP );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_AIRWALK_PRIMARY3_LOOP );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_STAND_PRIMARY3_END );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_CROUCH_PRIMARY3_END );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_AIRWALK_PRIMARY3_END );
	REGISTER_SHARED_ACTIVITY( ACT_MP_RELOAD_SWIM_PRIMARY3 );
	
	REGISTER_SHARED_ACTIVITY( ACT_MP_THROW );
	REGISTER_SHARED_ACTIVITY( ACT_THROWABLE_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_THROWABLE_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_THROWABLE_VM_FIRE );

	REGISTER_SHARED_ACTIVITY( ACT_SPELL_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_SPELL_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_SPELL_VM_ARM );
	REGISTER_SHARED_ACTIVITY( ACT_SPELL_VM_FIRE );

	REGISTER_SHARED_ACTIVITY( ACT_BREADSAPPER_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_BREADSAPPER_VM_IDLE );

	REGISTER_SHARED_ACTIVITY( ACT_BREADGLOVES_VM_HITLEFT );
	REGISTER_SHARED_ACTIVITY( ACT_BREADGLOVES_VM_HITRIGHT ); 
	REGISTER_SHARED_ACTIVITY( ACT_BREADGLOVES_VM_SWINGHARD );
	REGISTER_SHARED_ACTIVITY( ACT_BREADGLOVES_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_BREADGLOVES_VM_DRAW );

	REGISTER_SHARED_ACTIVITY( ACT_BREADMONSTER_GLOVES_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_BREADMONSTER_GLOVES_HITRIGHT );
	REGISTER_SHARED_ACTIVITY( ACT_BREADMONSTER_GLOVES_HITUP );

	REGISTER_SHARED_ACTIVITY( ACT_BREADMONSTER_VM_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_BREADMONSTER_VM_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_BREADMONSTER_VM_PRIMARYATTACK );

	REGISTER_SHARED_ACTIVITY( ACT_PARACHUTE_DEPLOY );
	REGISTER_SHARED_ACTIVITY( ACT_PARACHUTE_DEPLOY_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_PARACHUTE_RETRACT );
	REGISTER_SHARED_ACTIVITY( ACT_PARACHUTE_RETRACT_IDLE );

	REGISTER_SHARED_ACTIVITY( ACT_BOT_SPAWN );
	REGISTER_SHARED_ACTIVITY( ACT_BOT_PANIC );
	REGISTER_SHARED_ACTIVITY( ACT_BOT_PRIMARY_MOVEMENT );
	REGISTER_SHARED_ACTIVITY( ACT_BOT_GESTURE_FLINCH );
	REGISTER_SHARED_ACTIVITY( ACT_BOT_PANIC_START );
	REGISTER_SHARED_ACTIVITY( ACT_BOT_PANIC_END );

	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_REVOLVER_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_REVOLVER_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_REVOLVER_PRIMARYATTACK );
	REGISTER_SHARED_ACTIVITY( ACT_ENGINEER_REVOLVER_RELOAD );

	REGISTER_SHARED_ACTIVITY( ACT_KART_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_KART_ACTION_SHOOT );
	REGISTER_SHARED_ACTIVITY( ACT_KART_ACTION_DASH );
	REGISTER_SHARED_ACTIVITY( ACT_KART_JUMP_START );
	REGISTER_SHARED_ACTIVITY( ACT_KART_JUMP_FLOAT );
	REGISTER_SHARED_ACTIVITY( ACT_KART_JUMP_LAND );
	REGISTER_SHARED_ACTIVITY( ACT_KART_IMPACT );
	REGISTER_SHARED_ACTIVITY( ACT_KART_IMPACT_BIG );
	REGISTER_SHARED_ACTIVITY( ACT_KART_GESTURE_POSITIVE );
	REGISTER_SHARED_ACTIVITY( ACT_KART_GESTURE_NEGATIVE );

	REGISTER_SHARED_ACTIVITY( ACT_GRAPPLE_DRAW );
	REGISTER_SHARED_ACTIVITY( ACT_GRAPPLE_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_GRAPPLE_FIRE_START );
	REGISTER_SHARED_ACTIVITY( ACT_GRAPPLE_FIRE_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_GRAPPLE_PULL_START );
	REGISTER_SHARED_ACTIVITY( ACT_GRAPPLE_PULL_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_GRAPPLE_PULL_END );

	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_INSPECT_START );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_INSPECT_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_PRIMARY_VM_INSPECT_END );

	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_INSPECT_START );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_INSPECT_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_SECONDARY_VM_INSPECT_END );

	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_INSPECT_START );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_INSPECT_IDLE );
	REGISTER_SHARED_ACTIVITY( ACT_MELEE_VM_INSPECT_END );

	AssertMsg( g_HighestActivity == LAST_SHARED_ACTIVITY - 1, "Not all activities from ai_activity.h registered in activitylist.cpp" ); 
} 

// HACKHACK: Keep backwards compatibility on broken activities temporarily
#define ACTIVITY_FILE_TAG 0x80800000

class CActivityDataOps : public CDefSaveRestoreOps
{
public:
	// save data type interface
	virtual void Save( const SaveRestoreFieldInfo_t &fieldInfo, ISave *pSave ) 
	{
		int activityIndex = *((int *)fieldInfo.pField);
		const char *pActivityName = ActivityList_NameForIndex( activityIndex );
		if ( !pActivityName )
		{
			AssertOnce( activityIndex == -1 ); // FIXME: whatever activity this was, it's now being saved out as ACT_RESET
			pActivityName = ActivityList_NameForIndex( 0 );
		}
		int len = strlen(pActivityName) + 1;
		
		// Use the high 16-bits of this int to signify this file format
		// this makes this backwards compatible.
		// UNDONE: Remove after playtest save files are no longer needed
		len |= ACTIVITY_FILE_TAG;
		pSave->WriteInt( &len );
		pSave->WriteString( pActivityName );
	}

	virtual void Restore( const SaveRestoreFieldInfo_t &fieldInfo, IRestore *pRestore ) 
	{
		char nameBuf[1024];

		int *pActivityIndex = (int *)fieldInfo.pField;

		int nameLen = pRestore->ReadInt();
		if ( (nameLen & 0xFFFF0000) != ACTIVITY_FILE_TAG )
		{
			// old save file, this is an index, not a name
			*pActivityIndex = nameLen;
			return;
		}
		nameLen &= 0xFFFF;
		pRestore->ReadString( nameBuf, sizeof(nameBuf), nameLen );
		*pActivityIndex = ActivityList_IndexForName( nameBuf );
		if ( *pActivityIndex < 0 )
		{
			*pActivityIndex = 0;
		}
	}

	virtual bool IsEmpty( const SaveRestoreFieldInfo_t &fieldInfo ) 
	{ 
		int *pActivityIndex = (int *)fieldInfo.pField;
		return (*pActivityIndex == 0);
	}

	virtual void MakeEmpty( const SaveRestoreFieldInfo_t &fieldInfo ) 
	{
		int *pActivityIndex = (int *)fieldInfo.pField;
		*pActivityIndex = 0;
	}
};

static CActivityDataOps g_ActivityDataOps;

ISaveRestoreOps* ActivityDataOps()
{
	return &g_ActivityDataOps;
}




void UTIL_LoadActivityRemapFile( const char *filename, const char *section, CUtlVector <CActivityRemap> &entries )
{
	int iIndex = m_ActivityRemapDatabase.Find( filename );

	if ( iIndex != m_ActivityRemapDatabase.InvalidIndex() )
	{
		CActivityRemapCache *actRemap = &m_ActivityRemapDatabase[iIndex];
		entries.AddVectorToTail( actRemap->m_cachedActivityRemaps );
		return;
	}

	KeyValues *pkvFile = new KeyValues( section );

	if ( pkvFile->LoadFromFile( filesystem, filename, NULL ) )
	{
		KeyValues *pTestKey = pkvFile->GetFirstSubKey();

		CActivityRemapCache actRemap;

		while ( pTestKey )
		{
			Activity ActBase = (Activity)ActivityList_IndexForName( pTestKey->GetName() );

			if ( ActBase != ACT_INVALID )
			{
				KeyValues *pRemapKey = pTestKey->GetFirstSubKey();

				CActivityRemap actMap;
				actMap.activity = ActBase;

				while ( pRemapKey )
				{
					const char *pKeyName = pRemapKey->GetName();
					const char *pKeyValue = pRemapKey->GetString();

					if ( !stricmp( pKeyName, "remapactivity" ) )
					{
						Activity Act = (Activity)ActivityList_IndexForName( pKeyValue );

						if ( Act == ACT_INVALID )
						{
							actMap.mappedActivity = ActivityList_RegisterPrivateActivity( pKeyValue );
						}
						else
						{
							actMap.mappedActivity = Act;
						}
					}
					else if ( !stricmp( pKeyName, "extra" ) )
					{
						actMap.SetExtraKeyValueBlock( pRemapKey->MakeCopy() );
					}

					pRemapKey = pRemapKey->GetNextKey();
				}

				entries.AddToTail( actMap );
			}

			pTestKey = pTestKey->GetNextKey();
		}

		actRemap.m_cachedActivityRemaps.AddVectorToTail( entries );
		m_ActivityRemapDatabase.Insert( filename, actRemap );
	}
}

int ActivityList_HighestIndex()
{
	return g_HighestActivity;
}
