/******************************************************************************/
// Free implementation of Bullfrog's Dungeon Keeper strategy game.
/******************************************************************************/
/** @file cursor_tag.c
 *     Cursor box functions.
 * @par Purpose:
 *     Draws cursor boxes for various player states.
 * @par Comment:
 *     None.
 * @author   KeeperFX Team
 * @date     10 March 2022
 * @par  Copying and copyrights:
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 */
/******************************************************************************/
#include "pre_inc.h"
#include "globals.h"
#include "bflib_basics.h"
#include "map_data.h"
#include "player_data.h"
#include "dungeon_data.h"
#include "game_legacy.h"
#include "roomspace.h"
#include "engine_render.h"
#include "slab_data.h"
#include "power_hand.h"
#include "frontend.h"
#include "thing_physics.h"
#include "thing_navigate.h"
#include "packets.h"
#include "post_inc.h"

#ifdef __cplusplus
extern "C" {
#endif
/******************************************************************************/

/******************************************************************************/
#ifdef __cplusplus
}
#endif
/******************************************************************************/
unsigned char tag_cursor_blocks_dig(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y, TbBool full_slab)
{
    SYNCDBG(7,"Starting for player %d at subtile (%d,%d)",(int)plyr_idx,(int)stl_x,(int)stl_y);
    struct PlayerInfo* player = get_player(plyr_idx);
    struct Packet* pckt = get_packet_direct(player->packet_num);
    MapSlabCoord slb_x = subtile_slab(stl_x);
    MapSlabCoord slb_y = subtile_slab(stl_y);
    int floor_height_z = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    TbBool allowed = false;
    if (player->render_roomspace.slab_count > 0 && full_slab) // if roomspace is not empty
    {
        allowed = true;
    }
    else if (subtile_is_diggable_for_player(plyr_idx, stl_x, stl_y, false)) // else if not using roomspace, is current slab diggable
    {
        allowed = true;
    }
    else if ((player->one_click_lock_cursor) && ((pckt->control_flags & PCtr_LBtnHeld) != 0))
    {
        allowed = true;
    }
    unsigned char line_color = allowed;
    if (allowed)
    {
        if (player->render_roomspace.untag_mode)
        {
            line_color = SLC_YELLOW;
        }
        else
        {
            struct Dungeon* dungeon = get_players_dungeon(player);
            if ( (player->render_roomspace.drag_mode) && (dungeon->task_count + player->boxsize > MAPTASKS_COUNT) )
            {
                line_color = SLC_REDFLASH;
            }
            else if (dungeon->task_count >= MAPTASKS_COUNT)
            {
                line_color = SLC_REDYELLOW;
            }
        }
    }
    if (is_my_player_number(plyr_idx) && !game_is_busy_doing_gui() && (game.small_map_state != 2) && ((pckt->control_flags & PCtr_MapCoordsValid) != 0))
    {
        map_volume_box.visible = 1;
        map_volume_box.color = line_color;
        map_volume_box.beg_x = (!full_slab ? (subtile_coord(stl_x, 0)) : subtile_coord(((player->render_roomspace.centreX - calc_distance_from_roomspace_centre(player->render_roomspace.width,0)) * STL_PER_SLB), 0));
        map_volume_box.beg_y = (!full_slab ? (subtile_coord(stl_y, 0)) : subtile_coord(((player->render_roomspace.centreY - calc_distance_from_roomspace_centre(player->render_roomspace.height,0)) * STL_PER_SLB), 0));
        map_volume_box.end_x = (!full_slab ? (subtile_coord(stl_x + 1, 0)) : subtile_coord((((player->render_roomspace.centreX + calc_distance_from_roomspace_centre(player->render_roomspace.width,(player->render_roomspace.width % 2 == 0))) + 1) * STL_PER_SLB), 0));
        map_volume_box.end_y = (!full_slab ? (subtile_coord(stl_y + 1, 0)) : subtile_coord((((player->render_roomspace.centreY + calc_distance_from_roomspace_centre(player->render_roomspace.height,(player->render_roomspace.height % 2 == 0))) + 1) * STL_PER_SLB), 0));
        map_volume_box.floor_height_z = floor_height_z;
        player->render_roomspace.is_roomspace_a_single_subtile = !full_slab;
    }
    return line_color;
}

void tag_cursor_blocks_thing_in_hand(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y, TbBool is_special_digger, TbBool full_slab)
{
  SYNCDBG(7,"Starting");
  MapSlabCoord slb_x = subtile_slab(stl_x);
  MapSlabCoord slb_y = subtile_slab(stl_y);  
  if (is_my_player_number(plyr_idx) && !game_is_busy_doing_gui() && (game.small_map_state != 2) )
    {
        map_volume_box.visible = true;
        map_volume_box.color = can_drop_thing_here(stl_x, stl_y, plyr_idx, is_special_digger);
        if (full_slab)
        {
            map_volume_box.beg_x = subtile_coord(slab_subtile(slb_x, 0), 0);
            map_volume_box.beg_y = subtile_coord(slab_subtile(slb_y, 0), 0);
            map_volume_box.end_x = subtile_coord(slab_subtile(slb_x, 0) + STL_PER_SLB, 0);
            map_volume_box.end_y = subtile_coord(slab_subtile(slb_y, 0) + STL_PER_SLB, 0);
        }
        else
        {
            map_volume_box.beg_x = subtile_coord(stl_x, 0);
            map_volume_box.beg_y = subtile_coord(stl_y, 0);
            map_volume_box.end_x = subtile_coord(stl_x + 1, 0);
            map_volume_box.end_y = subtile_coord(stl_y + 1, 0); 
        }
        map_volume_box.floor_height_z = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    }
}

TbBool tag_cursor_blocks_sell_area(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y, TbBool full_slab)
{
    SYNCDBG(7,"Starting");
    struct PlayerInfo* player = get_player(plyr_idx);
    MapSlabCoord slb_x = subtile_slab(stl_x);
    MapSlabCoord slb_y = subtile_slab(stl_y);
    struct SlabMap *slb;
    slb = get_slabmap_block(slb_x, slb_y);
    int floor_height_z = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    unsigned char colour = SLC_RED;
    if (player->render_roomspace.slab_count > 0 && full_slab)
    {
        colour = SLC_GREEN; // roomspace selling support is basic, this makes roomspace selling work over any slabtype
    }
    else if (floor_height_z == 1)
    {
        if ( ( ((subtile_is_sellable_room(plyr_idx, stl_x, stl_y)) || ( (slabmap_owner(slb) == plyr_idx) && ( (slab_has_sellable_door(slb_x, slb_y))
            || ((!full_slab) ? (subtile_has_sellable_trap_on(stl_x, stl_y)) : (slab_has_sellable_trap_on(slb_x, slb_y))) ) ) ) )
            && ( slb->kind != SlbT_ENTRANCE && slb->kind != SlbT_DUNGHEART ) )
        {
            colour = SLC_GREEN;
        }
    }
    if ( is_my_player_number(plyr_idx) && !game_is_busy_doing_gui() && game.small_map_state != 2 )
    {
        map_volume_box.visible = 1;
        map_volume_box.color = colour;
        map_volume_box.beg_x = (!full_slab ? (subtile_coord(stl_x, 0)) : subtile_coord(((player->render_roomspace.centreX - calc_distance_from_roomspace_centre(player->render_roomspace.width,0)) * STL_PER_SLB), 0));
        map_volume_box.beg_y = (!full_slab ? (subtile_coord(stl_y, 0)) : subtile_coord(((player->render_roomspace.centreY - calc_distance_from_roomspace_centre(player->render_roomspace.height,0)) * STL_PER_SLB), 0));
        map_volume_box.end_x = (!full_slab ? (subtile_coord(stl_x + 1, 0)) : subtile_coord((((player->render_roomspace.centreX + calc_distance_from_roomspace_centre(player->render_roomspace.width,(player->render_roomspace.width % 2 == 0))) + 1) * STL_PER_SLB), 0));
        map_volume_box.end_y = (!full_slab ? (subtile_coord(stl_y + 1, 0)) : subtile_coord((((player->render_roomspace.centreY + calc_distance_from_roomspace_centre(player->render_roomspace.height,(player->render_roomspace.height % 2 == 0))) + 1) * STL_PER_SLB), 0));
        map_volume_box.floor_height_z = floor_height_z;
        player->render_roomspace.is_roomspace_a_single_subtile = !full_slab;
    }
    return (colour != SLC_RED);
}

TbBool tag_cursor_blocks_place_door(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y)
{
    SYNCDBG(7,"Starting");
    MapSlabCoord slb_x = subtile_slab(stl_x);
    MapSlabCoord slb_y = subtile_slab(stl_y);
    struct SlabMap *slb;
    slb = get_slabmap_block(slb_x, slb_y);
    TbBool allowed = false;
    char Orientation;
    TbBool Check = false;
    int floor_height_z = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    if (floor_height_z == 1)
    {
        Orientation = find_door_angle(stl_x, stl_y, plyr_idx);
        switch(Orientation)
        {
            case 0:
            {
                Check = (!slab_middle_row_has_trap_on(slb_x, slb_y) );
                break;
            }
            case 1:
            {
                Check = (!slab_middle_column_has_trap_on(slb_x, slb_y) );
                break;
            }
        }
        if ( ( (slabmap_owner(slb) == plyr_idx) && (slb->kind == SlbT_CLAIMED) )
            && (Orientation != -1) && ( Check ) 
            && (!slab_has_door_thing_on(slb_x, slb_y)) )
        {
            allowed = true;
        }
    }
    if ( is_my_player_number(plyr_idx) && !game_is_busy_doing_gui() && game.small_map_state != 2 )
    {
        struct PlayerInfo* player = get_player(plyr_idx);
        map_volume_box.visible = 1;
        map_volume_box.beg_x = subtile_coord(slab_subtile(slb_x, 0), 0);
        map_volume_box.beg_y = subtile_coord(slab_subtile(slb_y, 0), 0);
        map_volume_box.end_x = subtile_coord(slab_subtile(slb_x, STL_PER_SLB), 0);
        map_volume_box.end_y = subtile_coord(slab_subtile(slb_y, STL_PER_SLB), 0);
        map_volume_box.floor_height_z = floor_height_z;
        map_volume_box.color = allowed;
        player->render_roomspace.is_roomspace_a_box = true;
        player->render_roomspace.render_roomspace_as_box = true;
        player->render_roomspace.is_roomspace_a_single_subtile = false;
    }
    return allowed;
}

TbBool tag_cursor_blocks_place_room(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y, TbBool full_slab)
{
    SYNCDBG(7,"Starting");
    struct PlayerInfo* player = get_player(plyr_idx);
    MapSlabCoord slb_x;
    MapSlabCoord slb_y;
    slb_x = subtile_slab(stl_x);
    slb_y = subtile_slab(stl_y);
    int floor_height_z = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    unsigned char colour = SLC_RED;
    if(can_build_roomspace(plyr_idx, player->chosen_room_kind, player->render_roomspace) > 0)
    {
        colour = SLC_GREEN;
    }
    else
    {
        #if (BFDEBUG_LEVEL > 7)
            struct SlabMap* slb = get_slabmap_block(slb_x, slb_y); //inside condition because otherwise it would throw a build warning for not using this variable.
            SYNCDBG(7,"Cannot build %s on %s slabs centered at (%d,%d)", room_code_name(player->chosen_room_kind), slab_code_name(slb->kind), (int)slb_x, (int)slb_y);
        #endif
    }
    if (is_my_player_number(plyr_idx) && !game_is_busy_doing_gui() && (game.small_map_state != 2))
    {
        map_volume_box.visible = 1;
        map_volume_box.color = colour;
        map_volume_box.beg_x = (!full_slab ? (subtile_coord(stl_x, 0)) : subtile_coord(((player->render_roomspace.centreX - calc_distance_from_roomspace_centre(player->render_roomspace.width,0)) * STL_PER_SLB), 0));
        map_volume_box.beg_y = (!full_slab ? (subtile_coord(stl_y, 0)) : subtile_coord(((player->render_roomspace.centreY - calc_distance_from_roomspace_centre(player->render_roomspace.height,0)) * STL_PER_SLB), 0));
        map_volume_box.end_x = (!full_slab ? (subtile_coord(stl_x + 1, 0)) : subtile_coord((((player->render_roomspace.centreX + calc_distance_from_roomspace_centre(player->render_roomspace.width,(player->render_roomspace.width % 2 == 0))) + 1) * STL_PER_SLB), 0));
        map_volume_box.end_y = (!full_slab ? (subtile_coord(stl_y + 1, 0)) : subtile_coord((((player->render_roomspace.centreY + calc_distance_from_roomspace_centre(player->render_roomspace.height,(player->render_roomspace.height % 2 == 0))) + 1) * STL_PER_SLB), 0));
        map_volume_box.floor_height_z = floor_height_z;
        player->render_roomspace.is_roomspace_a_single_subtile = !full_slab;
    }
    return (colour != SLC_RED);
}

void tag_cursor_blocks_place_terrain(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y)
{
    SYNCDBG(7,"Starting");
    MapSlabCoord slb_x = subtile_slab(stl_x);
    MapSlabCoord slb_y = subtile_slab(stl_y);
    int floor_height_z = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    if ( is_my_player_number(plyr_idx) && !game_is_busy_doing_gui() && game.small_map_state != 2 )
    {
        map_volume_box.visible = true;
        map_volume_box.beg_x = subtile_coord(slab_subtile(slb_x, 0), 0);
        map_volume_box.beg_y = subtile_coord(slab_subtile(slb_y, 0), 0);
        map_volume_box.end_x = subtile_coord(slab_subtile(slb_x, STL_PER_SLB), 0);
        map_volume_box.end_y = subtile_coord(slab_subtile(slb_y, STL_PER_SLB), 0);
        map_volume_box.floor_height_z = floor_height_z;
        map_volume_box.color = SLC_GREEN;
    }
}

TbBool tag_cursor_blocks_place_thing(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y)
{
    SYNCDBG(7,"Starting");
    MapSlabCoord slb_x = subtile_slab(stl_x);
    MapSlabCoord slb_y = subtile_slab(stl_y);
    int floor_height_z = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    long height = get_floor_height(stl_x, stl_y);
    unsigned char colour;
    if (map_is_solid_at_height(stl_x, stl_y, height, height))
    {
        colour = SLC_RED;
    }
    else if (map_pos_is_lava(stl_x, stl_y))
    {
        colour = SLC_YELLOW;
    }
    else
    {
        colour = SLC_GREEN;
    }
    if ( is_my_player_number(plyr_idx) && !game_is_busy_doing_gui() && game.small_map_state != 2 )
    {
        struct PlayerInfo* player = get_player(plyr_idx);
        map_volume_box.visible = true;
        map_volume_box.beg_x = subtile_coord(stl_x, 0);
        map_volume_box.beg_y = subtile_coord(stl_y, 0);
        map_volume_box.end_x = subtile_coord(stl_x + 1, 0);
        map_volume_box.end_y = subtile_coord(stl_y + 1, 0);
        map_volume_box.floor_height_z = floor_height_z;
        map_volume_box.color = colour;
        player->render_roomspace.is_roomspace_a_single_subtile = true;
    }
    return (colour != SLC_RED);
}

TbBool tag_cursor_blocks_order_creature(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y, struct Thing* creatng)
{
    SYNCDBG(7,"Starting");
    MapSlabCoord slb_x = subtile_slab(stl_x);
    MapSlabCoord slb_y = subtile_slab(stl_y);
    int floor_height_z = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    unsigned char colour;
    struct Coord3d pos;
    pos.x.val = 0;
    pos.y.val = 0;
    pos.x.stl.num = stl_x;
    pos.y.stl.num = stl_y;
    pos.z.val = get_floor_height(stl_x, stl_y);
    if (!creature_can_navigate_to(creatng, &pos, NavRtF_Default))
    {
        colour = SLC_RED;
    }
    else
    {
        colour = SLC_GREEN;
    }
    if ( is_my_player_number(plyr_idx) && !game_is_busy_doing_gui() && game.small_map_state != 2 )
    {
        struct PlayerInfo* player = get_player(plyr_idx);
        map_volume_box.visible = true;
        map_volume_box.beg_x = subtile_coord(stl_x, 0);
        map_volume_box.beg_y = subtile_coord(stl_y, 0);
        map_volume_box.end_x = subtile_coord(stl_x + 1, 0);
        map_volume_box.end_y = subtile_coord(stl_y + 1, 0);
        map_volume_box.floor_height_z = floor_height_z;
        map_volume_box.color = colour;
        player->render_roomspace.is_roomspace_a_single_subtile = true;
    }
    return (colour != SLC_RED);
}

TbBool tag_cursor_blocks_steal_slab(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y)
{
    SYNCDBG(7,"Starting");
    MapSlabCoord slb_x = subtile_slab(stl_x);
    MapSlabCoord slb_y = subtile_slab(stl_y);
    struct SlabMap* slb = get_slabmap_block(slb_x, slb_y);
    struct SlabConfigStats* slabst = get_slab_stats(slb);
    int floor_height_z = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    unsigned char colour;
    struct PlayerInfo* player = get_player(plyr_idx);
    if ( ( ( ((slabst->category == SlbAtCtg_FortifiedGround) || (slabst->category == SlbAtCtg_FortifiedWall) ) && (slabmap_owner(slb) != player->cheatselection.chosen_player) ) )
        || ( (slabst->category == SlbAtCtg_FriableDirt) || ( (slabst->category == SlbAtCtg_Unclaimed) && (slabst->is_safe_land) && (!slab_is_liquid(slb_x, slb_y) ) ) ) )
    {
        colour = SLC_GREEN;
    }
    else
    {
        colour = SLC_RED;
    }
    if ( is_my_player_number(plyr_idx) && !game_is_busy_doing_gui() && game.small_map_state != 2 )
    {
        map_volume_box.visible = true;
        map_volume_box.beg_x = subtile_coord(slab_subtile(slb_x, 0), 0);
        map_volume_box.beg_y = subtile_coord(slab_subtile(slb_y, 0), 0);
        map_volume_box.end_x = subtile_coord(slab_subtile(slb_x, STL_PER_SLB), 0);
        map_volume_box.end_y = subtile_coord(slab_subtile(slb_y, STL_PER_SLB), 0);
        map_volume_box.floor_height_z = floor_height_z;
        map_volume_box.color = colour;
    }
    return (colour != SLC_RED);
}

TbBool tag_cursor_blocks_place_trap(PlayerNumber plyr_idx, MapSubtlCoord stl_x, MapSubtlCoord stl_y, TbBool full_slab, ThingModel trpkind)
{
    SYNCDBG(7,"Starting");
    MapSlabCoord slb_x = subtile_slab(stl_x);
    MapSlabCoord slb_y = subtile_slab(stl_y);
    TbBool can_place = can_place_trap_on(plyr_idx, stl_x, stl_y, trpkind);
    int floor_height = floor_height_for_volume_box(plyr_idx, slb_x, slb_y);
    if (is_my_player_number(plyr_idx))
    {
        if (!game_is_busy_doing_gui() && (game.small_map_state != 2))
        {
            struct PlayerInfo* player = get_player(plyr_idx);
            player->render_roomspace.is_roomspace_a_box = true;
            player->render_roomspace.render_roomspace_as_box = true;
            if (full_slab)
            {
                // Move to first subtile on a slab
                stl_x = slab_subtile(slb_x,0);
                stl_y = slab_subtile(slb_y,0);
                player->render_roomspace.is_roomspace_a_single_subtile = false;
                draw_map_volume_box(subtile_coord(stl_x,0), subtile_coord(stl_y,0),
                subtile_coord(stl_x+STL_PER_SLB,0), subtile_coord(stl_y+STL_PER_SLB,0), floor_height, can_place);
            }
            else
            {
                player->render_roomspace.is_roomspace_a_single_subtile = true;
                draw_map_volume_box(subtile_coord(stl_x,0), subtile_coord(stl_y,0), subtile_coord(stl_x+1,0), subtile_coord(stl_y+1,0), floor_height, can_place);
            }
        }
    }
    return can_place;
}
/******************************************************************************/
