///////////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2021-2022 Edouard Griffiths, F4EXB <f4exb06@gmail.com>          //
// Copyright (C) 2023 Benjamin Menkuec <benjamin@menkuec.de>                     //
//                                                                               //
// This program is free software; you can redistribute it and/or modify          //
// it under the terms of the GNU General Public License as published by          //
// the Free Software Foundation as version 3 of the License, or                  //
// (at your option) any later version.                                           //
//                                                                               //
// This program is distributed in the hope that it will be useful,               //
// but WITHOUT ANY WARRANTY; without even the implied warranty of                //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the                  //
// GNU General Public License V3 for more details.                               //
//                                                                               //
// You should have received a copy of the GNU General Public License             //
// along with this program. If not, see <http://www.gnu.org/licenses/>.          //
///////////////////////////////////////////////////////////////////////////////////

#include "plutosdrmimosettings.h"

#include <QtGlobal>
#include "util/simpleserializer.h"


PlutoSDRMIMOSettings::PlutoSDRMIMOSettings()
{
	resetToDefaults();
}

void PlutoSDRMIMOSettings::resetToDefaults()
{
    m_title = "PlutoSDRMIMO";
	m_devSampleRate = 2500 * 1000;
	m_LOppmTenths = 0;

	m_rxCenterFrequency = 435000 * 1000;
	m_fcPosRx = FC_POS_CENTER;
	m_log2Decim = 0;
	m_dcBlock = false;
	m_iqCorrection = false;
    m_hwBBDCBlock = true;
    m_hwRFDCBlock = true;
    m_hwIQCorrection = true;
	m_lpfBWRx = 1500000;
	m_lpfRxFIREnable = false;
	m_lpfRxFIRBW = 500000U;
	m_lpfRxFIRlog2Decim = 0;
	m_lpfRxFIRGain = 0;
	m_rxTransverterMode = false;
	m_rxTransverterDeltaFrequency = 0;
    m_iqOrder = true;

	m_rx0Gain = 40;
	m_rx0AntennaPath = RFPATHRX_A_BAL;
	m_rx0GainMode = GAIN_MANUAL;

	m_rx1Gain = 40;
	m_rx1AntennaPath = RFPATHRX_A_BAL;
	m_rx1GainMode = GAIN_MANUAL;

	m_txCenterFrequency = 435000 * 1000;
    m_fcPosTx = FC_POS_CENTER;
	m_log2Interp = 0;
	m_lpfBWTx = 1500000;
	m_lpfTxFIREnable = false;
	m_lpfTxFIRBW = 500000U;
	m_lpfTxFIRlog2Interp = 0;
	m_lpfTxFIRGain = 0;
    m_txTransverterMode = false;
    m_txTransverterDeltaFrequency = 0;

	m_tx0Att = -50;
	m_tx0AntennaPath = RFPATHTX_A;

	m_tx1Att = -50;
	m_tx1AntennaPath = RFPATHTX_A;

    m_useReverseAPI = false;
    m_reverseAPIAddress = "127.0.0.1";
    m_reverseAPIPort = 8888;
    m_reverseAPIDeviceIndex = 0;
}

QByteArray PlutoSDRMIMOSettings::serialize() const
{
	SimpleSerializer s(1);

    // Common
    s.writeU64(1,   m_devSampleRate);
    s.writeS32(2,   m_LOppmTenths);
    s.writeString(3, m_title);

    // Rx
    s.writeU64(10,  m_rxCenterFrequency);
	s.writeS32(11,  m_fcPosRx);
    s.writeU32(12,  m_log2Decim);
	s.writeBool(13, m_dcBlock);
    s.writeBool(14, m_iqCorrection);
    s.writeBool(15, m_hwBBDCBlock);
    s.writeBool(16, m_hwRFDCBlock);
    s.writeBool(17, m_hwIQCorrection);
    s.writeU32(18,  m_lpfBWRx);
    s.writeBool(19, m_lpfRxFIREnable);
    s.writeS32(20,  m_lpfRxFIRGain);
    s.writeU32(21,  m_lpfRxFIRlog2Decim);
    s.writeU32(22,  m_lpfRxFIRBW);
    s.writeBool(23, m_rxTransverterMode);
    s.writeS64(24,  m_rxTransverterDeltaFrequency);
    s.writeBool(25, m_iqOrder);

    // Rx0
    s.writeU32(40, m_rx0Gain);
    s.writeS32(41, (int) m_rx0AntennaPath);
    s.writeS32(42, (int) m_rx0GainMode);

    // Rx1
    s.writeU32(50, m_rx1Gain);
    s.writeS32(51, (int) m_rx1AntennaPath);
    s.writeS32(52, (int) m_rx1GainMode);

    // Tx
    s.writeU64(60,  m_txCenterFrequency);
    s.writeS32(61,  m_fcPosTx);
    s.writeU32(62,  m_log2Interp);
    s.writeU32(63,  m_lpfBWTx);
    s.writeBool(64, m_lpfTxFIREnable);
    s.writeU32(65,  m_lpfTxFIRBW);
    s.writeU32(66,  m_lpfTxFIRlog2Interp);
    s.writeS32(67,  m_lpfTxFIRGain);
    s.writeBool(68, m_txTransverterMode);
    s.writeS64(69,  m_txTransverterDeltaFrequency);

    // Tx0
    s.writeS32(80, m_tx0Att);
    s.writeS32(81, (int) m_tx0AntennaPath);

    // Tx1
    s.writeS32(90, m_tx1Att);
    s.writeS32(91, (int) m_tx1AntennaPath);

    // Reverse API
    s.writeBool(100,   m_useReverseAPI);
    s.writeString(101, m_reverseAPIAddress);
    s.writeU32(102,    m_reverseAPIPort);
    s.writeU32(103,    m_reverseAPIDeviceIndex);

	return s.final();
}

bool PlutoSDRMIMOSettings::deserialize(const QByteArray& data)
{
	SimpleDeserializer d(data);

	if (!d.isValid())
	{
		resetToDefaults();
		return false;
	}

	if (d.getVersion() == 1)
	{
		int intval;
		uint32_t uintval;

        // Common
        d.readU64(1, &m_devSampleRate, 2500 * 1000);
        d.readS32(2, &m_LOppmTenths, 0);
        d.readString(3, &m_title, "PlutoSDRMIMO");

        // Rx
        d.readU64(10, &m_rxCenterFrequency, 435000*1000);
		d.readS32(11, &intval, 0);
		if ((intval < 0) || (intval > 2)) {
		    m_fcPosRx = FC_POS_CENTER;
		} else {
	        m_fcPosRx = (fcPos_t) intval;
		}
        d.readU32(12,  &m_log2Decim, 0);
        d.readBool(13, &m_dcBlock, false);
        d.readBool(14, &m_iqCorrection, false);
        d.readBool(15, &m_hwBBDCBlock, true);
        d.readBool(16, &m_hwRFDCBlock, true);
        d.readBool(17, &m_hwIQCorrection, true);
        d.readU32(18,  &m_lpfBWRx, 1500000);
        d.readBool(19, &m_lpfRxFIREnable, false);
        d.readS32(20,  &m_lpfRxFIRGain, 0);
        d.readU32(21,  &uintval, 0);
        if (uintval > 2) {
            m_lpfRxFIRlog2Decim = 2;
        } else {
            m_lpfRxFIRlog2Decim = uintval;
        }
        d.readU32(22,  &m_lpfRxFIRBW, 500000U);
        d.readBool(23, &m_rxTransverterMode, false);
        d.readS64(24,  &m_rxTransverterDeltaFrequency, 0);
        d.readBool(25, &m_iqOrder, true);

        // Rx0
        d.readU32(40, &m_rx0Gain, 40);
        d.readS32(41, &intval, 0);
        if ((intval >= 0) && (intval < (int) RFPATHRX_END)) {
            m_rx0AntennaPath = (RFPathRx) intval;
        } else {
            m_rx0AntennaPath = RFPATHRX_A_BAL;
        }
        d.readS32(42, &intval, 0);
        if ((intval >= 0) && (intval < (int) GAIN_END)) {
            m_rx0GainMode = (GainMode) intval;
        } else {
            m_rx0GainMode = GAIN_MANUAL;
        }

        // Rx1
        d.readU32(50, &m_rx1Gain, 40);
        d.readS32(51, &intval, 0);
        if ((intval >= 0) && (intval < (int) RFPATHRX_END)) {
            m_rx1AntennaPath = (RFPathRx) intval;
        } else {
            m_rx1AntennaPath = RFPATHRX_A_BAL;
        }
        d.readS32(52, &intval, 0);
        if ((intval >= 0) && (intval < (int) GAIN_END)) {
            m_rx1GainMode = (GainMode) intval;
        } else {
            m_rx1GainMode = GAIN_MANUAL;
        }

        // Tx
        d.readU64(60, &m_txCenterFrequency, 435000*1000);
		d.readS32(61, &intval, 0);
		if ((intval < 0) || (intval > 2)) {
		    m_fcPosTx = FC_POS_CENTER;
		} else {
	        m_fcPosTx = (fcPos_t) intval;
		}
        d.readU32(62, &m_log2Interp, 0);
        d.readU32(63, &m_lpfBWTx, 1500000);
        d.readBool(64, &m_lpfTxFIREnable, false);
        d.readU32(65, &m_lpfTxFIRBW, 500000U);
        d.readU32(66, &uintval, 0);
        if (uintval > 2) {
            m_lpfTxFIRlog2Interp = 2;
        } else {
            m_lpfTxFIRlog2Interp = uintval;
        }
        d.readS32(67, &m_lpfTxFIRGain, 0);
        d.readBool(68, &m_txTransverterMode, false);
        d.readS64(69, &m_txTransverterDeltaFrequency, 0);

        // Tx0
        d.readS32(80, &m_tx0Att, -50);
        d.readS32(81, &intval, 0);
        if ((intval >= 0) && (intval < (int) RFPATHTX_END)) {
            m_tx0AntennaPath = (RFPathTx) intval;
        } else {
            m_tx0AntennaPath = RFPATHTX_A;
        }

        // Tx1
        d.readS32(80, &m_tx1Att, -50);
        d.readS32(81, &intval, 0);
        if ((intval >= 0) && (intval < (int) RFPATHTX_END)) {
            m_tx1AntennaPath = (RFPathTx) intval;
        } else {
            m_tx1AntennaPath = RFPATHTX_A;
        }

        // Reverse API
        d.readBool(100, &m_useReverseAPI, false);
        d.readString(101, &m_reverseAPIAddress, "127.0.0.1");
        d.readU32(102, &uintval, 0);

        if ((uintval > 1023) && (uintval < 65535)) {
            m_reverseAPIPort = uintval;
        } else {
            m_reverseAPIPort = 8888;
        }

        d.readU32(103, &uintval, 0);
        m_reverseAPIDeviceIndex = uintval > 99 ? 99 : uintval;

		return true;
    }
	else
	{
		resetToDefaults();
		return false;
	}
}

void PlutoSDRMIMOSettings::applySettings(const QStringList& settingsKeys, const PlutoSDRMIMOSettings& settings)
{
    if (settingsKeys.contains("title")) {
        m_title = settings.m_title;
    }
    if (settingsKeys.contains("devSampleRate")) {
        m_devSampleRate = settings.m_devSampleRate;
    }
    if (settingsKeys.contains("LOppmTenths")) {
        m_LOppmTenths = settings.m_LOppmTenths;
    }
    if (settingsKeys.contains("rxCenterFrequency")) {
        m_rxCenterFrequency = settings.m_rxCenterFrequency;
    }
    if (settingsKeys.contains("fcPosRx")) {
        m_fcPosRx = settings.m_fcPosRx;
    }
    if (settingsKeys.contains("log2Decim")) {
        m_log2Decim = settings.m_log2Decim;
    }
    if (settingsKeys.contains("dcBlock")) {
        m_dcBlock = settings.m_dcBlock;
    }
    if (settingsKeys.contains("iqCorrection")) {
        m_iqCorrection = settings.m_iqCorrection;
    }
    if (settingsKeys.contains("hwBBDCBlock")) {
        m_hwBBDCBlock = settings.m_hwBBDCBlock;
    }
    if (settingsKeys.contains("hwRFDCBlock")) {
        m_hwRFDCBlock = settings.m_hwRFDCBlock;
    }
    if (settingsKeys.contains("hwIQCorrection")) {
        m_hwIQCorrection = settings.m_hwIQCorrection;
    }
    if (settingsKeys.contains("lpfBWRx")) {
        m_lpfBWRx = settings.m_lpfBWRx;
    }
    if (settingsKeys.contains("lpfRxFIREnable")) {
        m_lpfRxFIREnable = settings.m_lpfRxFIREnable;
    }
    if (settingsKeys.contains("lpfRxFIRBW")) {
        m_lpfRxFIRBW = settings.m_lpfRxFIRBW;
    }
    if (settingsKeys.contains("lpfRxFIRlog2Decim")) {
        m_lpfRxFIRlog2Decim = settings.m_lpfRxFIRlog2Decim;
    }
    if (settingsKeys.contains("lpfRxFIRGain")) {
        m_lpfRxFIRGain = settings.m_lpfRxFIRGain;
    }
    if (settingsKeys.contains("rxTransverterMode")) {
        m_rxTransverterMode = settings.m_rxTransverterMode;
    }
    if (settingsKeys.contains("rxTransverterDeltaFrequency")) {
        m_rxTransverterDeltaFrequency = settings.m_rxTransverterDeltaFrequency;
    }
    if (settingsKeys.contains("iqOrder")) {
        m_iqOrder = settings.m_iqOrder;
    }
    if (settingsKeys.contains("rx0Gain")) {
        m_rx0Gain = settings.m_rx0Gain;
    }
    if (settingsKeys.contains("rx0AntennaPath")) {
        m_rx0AntennaPath = settings.m_rx0AntennaPath;
    }
    if (settingsKeys.contains("rx0GainMode")) {
        m_rx0GainMode = settings.m_rx0GainMode;
    }
    if (settingsKeys.contains("rx1Gain")) {
        m_rx1Gain = settings.m_rx1Gain;
    }
    if (settingsKeys.contains("rx1AntennaPath")) {
        m_rx1AntennaPath = settings.m_rx1AntennaPath;
    }
    if (settingsKeys.contains("rx1GainMode")) {
        m_rx1GainMode = settings.m_rx1GainMode;
    }
    if (settingsKeys.contains("txCenterFrequency")) {
        m_txCenterFrequency = settings.m_txCenterFrequency;
    }
    if (settingsKeys.contains("fcPosTx")) {
        m_fcPosTx = settings.m_fcPosTx;
    }
    if (settingsKeys.contains("log2Interp")) {
        m_log2Interp = settings.m_log2Interp;
    }
    if (settingsKeys.contains("lpfBWTx")) {
        m_lpfBWTx = settings.m_lpfBWTx;
    }
    if (settingsKeys.contains("lpfTxFIREnable")) {
        m_lpfTxFIREnable = settings.m_lpfTxFIREnable;
    }
    if (settingsKeys.contains("lpfTxFIRBW")) {
        m_lpfTxFIRBW = settings.m_lpfTxFIRBW;
    }
    if (settingsKeys.contains("lpfTxFIRlog2Interp")) {
        m_lpfTxFIRlog2Interp = settings.m_lpfTxFIRlog2Interp;
    }
    if (settingsKeys.contains("lpfTxFIRGain")) {
        m_lpfTxFIRGain = settings.m_lpfTxFIRGain;
    }
    if (settingsKeys.contains("txTransverterMode")) {
        m_txTransverterMode = settings.m_txTransverterMode;
    }
    if (settingsKeys.contains("txTransverterDeltaFrequency")) {
        m_txTransverterDeltaFrequency = settings.m_txTransverterDeltaFrequency;
    }
    if (settingsKeys.contains("tx0Att")) {
        m_tx0Att = settings.m_tx0Att;
    }
    if (settingsKeys.contains("tx0AntennaPath")) {
        m_tx0AntennaPath = settings.m_tx0AntennaPath;
    }
    if (settingsKeys.contains("tx1Att")) {
        m_tx1Att = settings.m_tx1Att;
    }
    if (settingsKeys.contains("tx1AntennaPath")) {
        m_tx1AntennaPath = settings.m_tx1AntennaPath;
    }
    if (settingsKeys.contains("useReverseAPI")) {
        m_useReverseAPI = settings.m_useReverseAPI;
    }
    if (settingsKeys.contains("reverseAPIAddress")) {
        m_reverseAPIAddress = settings.m_reverseAPIAddress;
    }
    if (settingsKeys.contains("reverseAPIPort")) {
        m_reverseAPIPort = settings.m_reverseAPIPort;
    }
    if (settingsKeys.contains("reverseAPIDeviceIndex")) {
        m_reverseAPIDeviceIndex = settings.m_reverseAPIDeviceIndex;
    }
}

QString PlutoSDRMIMOSettings::getDebugString(const QStringList& settingsKeys, bool force) const
{
    std::ostringstream ostr;

    if (settingsKeys.contains("title") || force) {
        ostr << " m_title: " << m_title.toStdString();
    }
    if (settingsKeys.contains("devSampleRate") || force) {
        ostr << " m_devSampleRate: " << m_devSampleRate;
    }
    if (settingsKeys.contains("LOppmTenths") || force) {
        ostr << " m_LOppmTenths: " << m_LOppmTenths;
    }
    if (settingsKeys.contains("rxCenterFrequency") || force) {
        ostr << " m_rxCenterFrequency: " << m_rxCenterFrequency;
    }
    if (settingsKeys.contains("fcPosRx") || force) {
        ostr << " m_fcPosRx: " << m_fcPosRx;
    }
    if (settingsKeys.contains("log2Decim") || force) {
        ostr << " m_log2Decim: " << m_log2Decim;
    }
    if (settingsKeys.contains("dcBlock") || force) {
        ostr << " m_dcBlock: " << m_dcBlock;
    }
    if (settingsKeys.contains("iqCorrection") || force) {
        ostr << " m_iqCorrection: " << m_iqCorrection;
    }
    if (settingsKeys.contains("hwBBDCBlock") || force) {
        ostr << " m_hwBBDCBlock: " << m_hwBBDCBlock;
    }
    if (settingsKeys.contains("hwRFDCBlock") || force) {
        ostr << " m_hwRFDCBlock: " << m_hwRFDCBlock;
    }
    if (settingsKeys.contains("hwIQCorrection") || force) {
        ostr << " m_hwIQCorrection: " << m_hwIQCorrection;
    }
    if (settingsKeys.contains("lpfBWRx") || force) {
        ostr << " m_lpfBWRx: " << m_lpfBWRx;
    }
    if (settingsKeys.contains("lpfRxFIREnable") || force) {
        ostr << " m_lpfRxFIREnable: " << m_lpfRxFIREnable;
    }
    if (settingsKeys.contains("lpfRxFIRBW") || force) {
        ostr << " m_lpfRxFIRBW: " << m_lpfRxFIRBW;
    }
    if (settingsKeys.contains("lpfRxFIRlog2Decim") || force) {
        ostr << " m_lpfRxFIRlog2Decim: " << m_lpfRxFIRlog2Decim;
    }
    if (settingsKeys.contains("lpfRxFIRGain") || force) {
        ostr << " m_lpfRxFIRGain: " << m_lpfRxFIRGain;
    }
    if (settingsKeys.contains("rxTransverterMode") || force) {
        ostr << " m_rxTransverterMode: " << m_rxTransverterMode;
    }
    if (settingsKeys.contains("rxTransverterDeltaFrequency") || force) {
        ostr << " m_rxTransverterDeltaFrequency: " << m_rxTransverterDeltaFrequency;
    }
    if (settingsKeys.contains("iqOrder") || force) {
        ostr << " m_iqOrder: " << m_iqOrder;
    }
    if (settingsKeys.contains("rx0Gain") || force) {
        ostr << " m_rx0Gain: " << m_rx0Gain;
    }
    if (settingsKeys.contains("rx0AntennaPath") || force) {
        ostr << " m_rx0AntennaPath: " << m_rx0AntennaPath;
    }
    if (settingsKeys.contains("rx0GainMode") || force) {
        ostr << " m_rx0GainMode: " << m_rx0GainMode;
    }
    if (settingsKeys.contains("rx1Gain") || force) {
        ostr << " m_rx1Gain: " << m_rx1Gain;
    }
    if (settingsKeys.contains("rx1AntennaPath") || force) {
        ostr << " m_rx1AntennaPath: " << m_rx1AntennaPath;
    }
    if (settingsKeys.contains("rx1GainMode") || force) {
        ostr << " m_rx1GainMode: " << m_rx1GainMode;
    }
    if (settingsKeys.contains("txCenterFrequency") || force) {
        ostr << " m_txCenterFrequency: " << m_txCenterFrequency;
    }
    if (settingsKeys.contains("fcPosTx") || force) {
        ostr << " m_fcPosTx: " << m_fcPosTx;
    }
    if (settingsKeys.contains("log2Interp") || force) {
        ostr << " m_log2Interp: " << m_log2Interp;
    }
    if (settingsKeys.contains("lpfBWTx") || force) {
        ostr << " m_lpfBWTx: " << m_lpfBWTx;
    }
    if (settingsKeys.contains("lpfTxFIREnable") || force) {
        ostr << " m_lpfTxFIREnable: " << m_lpfTxFIREnable;
    }
    if (settingsKeys.contains("lpfTxFIRBW") || force) {
        ostr << " m_lpfTxFIRBW: " << m_lpfTxFIRBW;
    }
    if (settingsKeys.contains("lpfTxFIRlog2Interp") || force) {
        ostr << " m_lpfTxFIRlog2Interp: " << m_lpfTxFIRlog2Interp;
    }
    if (settingsKeys.contains("lpfTxFIRGain") || force) {
        ostr << " m_lpfTxFIRGain: " << m_lpfTxFIRGain;
    }
    if (settingsKeys.contains("txTransverterMode") || force) {
        ostr << " m_txTransverterMode: " << m_txTransverterMode;
    }
    if (settingsKeys.contains("txTransverterDeltaFrequency") || force) {
        ostr << " m_txTransverterDeltaFrequency: " << m_txTransverterDeltaFrequency;
    }
    if (settingsKeys.contains("tx0Att") || force) {
        ostr << " m_tx0Att: " << m_tx0Att;
    }
    if (settingsKeys.contains("tx0AntennaPath") || force) {
        ostr << " m_tx0AntennaPath: " << m_tx0AntennaPath;
    }
    if (settingsKeys.contains("tx1Att") || force) {
        ostr << " m_tx1Att: " << m_tx1Att;
    }
    if (settingsKeys.contains("tx1AntennaPath") || force) {
        ostr << " m_tx1AntennaPath: " << m_tx1AntennaPath;
    }
    if (settingsKeys.contains("useReverseAPI") || force) {
        ostr << " m_useReverseAPI: " << m_useReverseAPI;
    }
    if (settingsKeys.contains("reverseAPIAddress") || force) {
        ostr << " m_reverseAPIAddress: " << m_reverseAPIAddress.toStdString();
    }
    if (settingsKeys.contains("reverseAPIPort") || force) {
        ostr << " m_reverseAPIPort: " << m_reverseAPIPort;
    }
    if (settingsKeys.contains("reverseAPIDeviceIndex") || force) {
        ostr << " m_reverseAPIDeviceIndex: " << m_reverseAPIDeviceIndex;
    }

    return QString(ostr.str().c_str());
}

void PlutoSDRMIMOSettings::translateRFPathTx(RFPathTx path, QString& s)
{
    switch(path)
    {
    case RFPATHTX_A:
        s = "A";
        break;
    case RFPATHTX_B:
        s = "B";
        break;
    default:
        s = "A";
        break;
    }
}

void PlutoSDRMIMOSettings::translateRFPathRx(RFPathRx path, QString& s)
{
    switch(path)
    {
    case RFPATHRX_A_BAL:
        s = "A_BALANCED";
        break;
    case RFPATHRX_B_BAL:
        s = "B_BALANCED";
        break;
    case RFPATHRX_C_BAL:
        s = "C_BALANCED";
        break;
    case RFPATHRX_A_NEG:
        s = "A_N";
        break;
    case RFPATHRX_A_POS:
        s = "A_P";
        break;
    case RFPATHRX_B_NEG:
        s = "B_N";
        break;
    case RFPATHRX_B_POS:
        s = "B_P";
        break;
    case RFPATHRX_C_NEG:
        s = "C_N";
        break;
    case RFPATHRX_C_POS:
        s = "C_P";
        break;
    case RFPATHRX_TX1MON:
        s = "TX_MONITOR1";
        break;
    case RFPATHRX_TX2MON:
        s = "TX_MONITOR2";
        break;
    case RFPATHRX_TX3MON:
        s = "TX_MONITOR3";
        break;
    default:
        s = "A_BALANCED";
        break;
    }
}

void PlutoSDRMIMOSettings::translateGainMode(GainMode mode, QString& s)
{
    switch(mode)
    {
    case GAIN_MANUAL:
        s = "manual";
        break;
    case GAIN_AGC_SLOW:
        s = "slow_attack";
        break;
    case GAIN_AGC_FAST:
        s = "fast_attack";
        break;
    case GAIN_HYBRID:
        s = "hybrid";
        break;
    default:
        s = "manual";
        break;
    }
}
