///////////////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2021 Edouard Griffiths, F4EXB <f4exb06@gmail.com>                   //
//                                                                                   //
// This program is free software; you can redistribute it and/or modify              //
// it under the terms of the GNU General Public License as published by              //
// the Free Software Foundation as version 3 of the License, or                      //
// (at your option) any later version.                                               //
//                                                                                   //
// This program is distributed in the hope that it will be useful,                   //
// but WITHOUT ANY WARRANTY; without even the implied warranty of                    //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the                      //
// GNU General Public License V3 for more details.                                   //
//                                                                                   //
// You should have received a copy of the GNU General Public License                 //
// along with this program. If not, see <http://www.gnu.org/licenses/>.              //
///////////////////////////////////////////////////////////////////////////////////////
/*
DVB-S2X LDPC parity bit address tables

Entries below copied from:
http://www.etsi.org/deliver/etsi_en/302300_302399/30230702/01.01.01_20/en_30230702v010101a.pdf

Copyright (C) 2018 Ron Economos <w6rz@comcast.net>

Permission to use, copy, modify, and/or distribute this software for any purpose with or without fee is hereby granted.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

#ifndef DVB_S2X_TABLES_HH
#define DVB_S2X_TABLES_HH

namespace ldpctool {

struct DVB_S2X_TABLE_B1
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 14400;
  static const int LINKS_MIN_CN = 3;
  static const int LINKS_MAX_CN = 4;
  static const int LINKS_TOTAL = 201599;
  static const int DEG_MAX = 11;
  static constexpr int DEG[] = {
    11, 3, 0
  };
  static constexpr int LEN[] = {
    20, 20, 0
  };
  static constexpr int POS[] = {
    5332,  8018,  35444, 13098, 9655,  41945, 44273, 22741, 9371,  8727,  43219,
    41410, 43593, 14611, 46707, 16041, 1459,  29246, 12748, 32996, 676,   46909,
    9340,  35072, 35640, 17537, 10512, 44339, 30965, 25175, 9918,  21079, 29835,
    3332,  12088, 47966, 25168, 50180, 42842, 40914, 46726, 17073, 41812, 34356,
    15159, 2209,  7971,  22590, 20020, 27567, 4853,  10294, 38839, 15314, 49808,
    20936, 14497, 23365, 22630, 38728, 28361, 34659, 956,   8559,  44957, 22222,
    28043, 4641,  25208, 47039, 30612, 25796, 14661, 44139, 27335, 12884, 6980,
    32584, 33453, 1867,  20185, 36106, 30357, 809,   28513, 46045, 27862, 4802,
    43744, 13375, 36066, 23604, 30766, 6233,  45051, 23660, 20815, 19525, 25207,
    27522, 3854,  9311,  21925, 41107, 25773, 26323, 24237, 24344, 46187, 44503,
    10256, 20038, 12177, 26635, 5214,  14191, 34404, 45807, 4938,  4173,  31344,
    32043, 26501, 46725, 4648,  16718, 31060, 26633, 19036, 14222, 13886, 26535,
    18103, 8498,  36814, 34600, 36495, 36712, 29833, 27396, 11877, 42861, 1834,
    36592, 1645,  3649,  30521, 14674, 3630,  890,   13307, 41412, 24682, 9907,
    4401,  44543, 13784, 5828,  32862, 25179, 29736, 39614, 5186,  49749, 38317,
    41460, 39101, 50080, 40137, 32691, 26528, 35332, 44067, 8467,  14286, 10470,
    12211, 34019, 37870, 36918, 36419, 33153, 50070, 41498, 47741, 30538, 12342,
    33751, 23988, 33624, 41882, 34075, 25552, 3106,  17611, 13190, 29336, 312,
    5667,  35483, 35460, 16153, 37267, 28308, 50009, 46345, 34204, 32756, 38243,
    5657,  24157, 36834, 6890,  49576, 46244, 43875, 16738, 47225, 2944,  36882,
    30341, 48485, 3700,
    14451, 20438, 18875,
    13634, 41138, 42962,
    46459, 13369, 27974,
    21493, 14629, 2369,
    11351, 40226, 42457,
    34749, 39000, 3912,
    18128, 46776, 47055,
    2221,  26806, 11345,
    35143, 630,   2229,
    44009, 41295, 34646,
    32163, 16657, 26544,
    31770, 23641, 43623,
    45826, 10902, 39490,
    7514,  20480, 28511,
    11429, 19834, 35430,
    50112, 38163, 5738,
    16191, 16862, 6783,
    6085,  39149, 34988,
    41497, 32023, 28688,
  };
};

struct DVB_S2X_TABLE_B2
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 18720;
  static const int LINKS_MIN_CN = 4;
  static const int LINKS_MAX_CN = 5;
  static const int LINKS_TOTAL = 226079;
  static const int DEG_MAX = 12;
  static constexpr int DEG[] = {
    12, 3, 0
  };
  static constexpr int LEN[] = {
    24, 28, 0
  };
  static constexpr int POS[] = {
    15210, 4519,  18217, 34427, 18474, 16813, 28246, 17687, 44527, 31465, 13004, 43601,
    28576, 13611, 24294, 15041, 503,   11393, 26290, 9278,  19484, 20742, 13226, 28322,
    32651, 27323, 22368, 15522, 37576, 20607, 20152, 19741, 26700, 31696, 21061, 35991,
    44168, 27910, 31104, 34776, 38835, 45450, 40002, 31522, 7807,  26330, 2410,  44983,
    15861, 39215, 14631, 42584, 26502, 41864, 27885, 32276, 29049, 16878, 37480, 42550,
    38795, 13012, 7912,  4058,  23869, 3325,  42889, 19921, 13826, 40323, 18162, 10005,
    35100, 5483,  7629,  35166, 1239,  10772, 5289,  286,   16172, 41843, 42612, 38493,
    11997, 40340, 19047, 16236, 43557, 9104,  24032, 2915,  19265, 36209, 6443,  40947,
    43527, 29675, 4195,  31926, 35392, 20400, 7515,  45806, 36068, 33079, 37325, 6301,
    4580,  20492, 40934, 14478, 8238,  2425,  28901, 43602, 7224,  17640, 28259, 6850,
    41859, 14006, 19132, 5690,  16223, 11575, 30562, 44797, 3759,  9833,  36529, 21084,
    45546, 16044, 26763, 13559, 29092, 41595, 5726,  13733, 9164,  15354, 20145, 10655,
    24076, 40883, 13424, 30325, 40589, 32367, 36270, 9286,  40151, 8501,  3871,  22109,
    26239, 29805, 5358,  44835, 11609, 3899,  9760,  39600, 43422, 13295, 45431, 14515,
    5392,  37010, 12386, 40193, 21492, 45146, 12376, 41952, 43153, 45733, 718,   35726,
    33884, 38006, 16927, 20958, 25413, 44561, 11245, 12984, 35198, 30977, 31916, 10657,
    1412,  1048,  14965, 31879, 29967, 41000, 32087, 22,    34773, 768,   27289, 19898,
    43051, 6964,  31807, 4119,  33509, 15950, 6304,  2813,  35192, 38282, 39710, 26356,
    9889,  18957, 6355,  18770, 40381, 1876,  38889, 17958, 20309, 10744, 1744,  228,
    41543, 36505, 32795, 12454, 8520,  4916,  22313, 1363,  13010, 8770,  17057, 8694,
    22987, 29564, 13804, 3110,  1382,  33844, 15117, 42314, 36045, 25295, 28421, 22044,
    15951, 42952, 17458, 6926,  21257, 41243, 8662,  17046, 15054, 15302, 16964, 40079,
    13359, 45754, 16715, 9586,  10960, 25406, 14675, 8880,  5087,  12303, 28993, 13571,
    24824, 31012, 4121,  808,   30962, 28736, 11013, 20488, 7715,  7637,  6217,  25114,
    23615, 5760,  5554,
    18072, 21605, 39242,
    24190, 6592,  12281,
    44681, 6563,  7001,
    18291, 19605, 33476,
    2884,  30927, 18430,
    23674, 36414, 30649,
    15364, 22089, 19757,
    41162, 14454, 17627,
    16676, 28573, 22163,
    8851,  36803, 27589,
    40049, 476,   1413,
    41013, 34505, 33296,
    29782, 38018, 42124,
    22625, 7485,  11772,
    2052,  37567, 14082,
    30106, 43203, 20858,
    7399,  3796,  22396,
    38745, 792,   44483,
    28268, 33355, 41030,
    30098, 37269, 12871,
    35769, 33119, 16738,
    3307,  43434, 13244,
    17852, 9133,  23190,
    35184, 20115, 24202,
    14760, 43026, 19425,
    26414, 16821, 6625,
    30362, 35769, 42608
  };
};

struct DVB_S2X_TABLE_B3
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 29160;
  static const int LINKS_MIN_CN = 6;
  static const int LINKS_MAX_CN = 7;
  static const int LINKS_TOTAL = 249479;
  static const int DEG_MAX = 12;
  static constexpr int DEG[] = {
    12, 4, 3, 0
  };
  static constexpr int LEN[] = {
    27, 9, 45, 0
  };
  static constexpr int POS[] = {
    30649, 35117, 23181, 15492, 2367,  31230, 9368,  13541, 6608,  23384, 18300, 5905,
    1961,  8950,  20589, 17688, 9641,  1877,  4937,  15293, 24864, 14876, 6516,  10165,
    4229,  26034, 28862, 8265,  27847, 3,     22728, 13946, 27162, 26003, 17696, 13261,
    31719, 25669, 17149, 17377, 33106, 12630, 4814,  16334, 1480,  32952, 11187, 3849,
    30186, 20938, 7946,  23283, 11042, 28080, 26642, 34560, 11302, 4991,  5121,  6879,
    13445, 22794, 18048, 15116, 5657,  9853,  15581, 34960, 13240, 11176, 17937, 25081,
    4868,  28235, 30286, 29706, 7073,  6773,  10390, 27002, 13015, 7388,  14772, 19581,
    11765, 16642, 11431, 19588, 20154, 8027,  29758, 5501,  6398,  4268,  21337, 21136,
    2275,  7899,  25943, 12939, 14478, 20369, 22877, 3591,  12217, 19130, 24252, 32444,
    24599, 21382, 4689,  3524,  11304, 20423, 13677, 19639, 10577, 28279, 22330, 30722,
    21622, 26233, 3921,  17722, 6843,  5999,  8186,  2355,  33632, 34632, 30285, 9616,
    19909, 30417, 19587, 27853, 13896, 3689,  155,   20457, 33362, 21739, 22779, 33862,
    3713,  32975, 9403,  2836,  23109, 11099, 3505,  14562, 17309, 26470, 4843,  12279,
    24216, 26340, 22073, 32570, 12936, 19797, 21801, 8918,  7999,  24408, 5783,  25190,
    8817,  29367, 17017, 6208,  21402, 2280,  2110,  7975,  32039, 34605, 1235,  912,
    23116, 33017, 31405, 638,   4707,  31760, 18043, 3507,  11989, 26632, 32829, 11262,
    9274,  2553,  10697, 13507, 15323, 27080, 3752,  33191, 12363, 24664, 14068, 1416,
    21670, 26696, 18570, 25197, 1517,  7765,  32686, 6572,  30901, 28242, 17802, 24056,
    35388, 26895, 8023,  31249, 29290, 13440, 7156,  17367, 21472, 27219, 14447, 9655,
    11100, 27918, 2900,  33262, 15301, 4664,  15728, 1185,  24818, 32995, 31108, 16368,
    34978, 31690, 30464, 13044, 5492,  10047, 2768,  14336, 30880, 32780, 10993, 24750,
    7022,  19718, 26036, 19145, 21177, 33949, 17135, 5193,  33718, 2539,  13920, 25537,
    918,   18514, 14530, 13699, 11902, 22721, 8335,  35346, 24655, 3332,  14708, 20822,
    11191, 24064, 32825, 12321, 11771, 23299, 31325, 25526, 16785, 22212, 34075, 9066,
    31209, 27819, 5974,  19918, 26831, 33338, 26647, 9480,  28489, 7827,  18562, 2401,
    17395, 23192, 10277, 28458, 23028, 18793, 10463, 10740, 616,   24647, 4153,  10128,
    2873,  22381, 8132,  18239, 31614, 4193, 32313,  7575,  25801, 27591, 19872, 17992,
    4609,  9114,  14764, 13516,
    19192, 9882,  13112, 16075,
    12510, 28902, 8784,  32679,
    4578,  34533, 30609, 25543,
    13739, 3465,  5330,  999,
    33254, 13085, 5001,  29061,
    28369, 79,    17750, 13399,
    24851, 9524,  30966, 10422,
    18251, 34810, 12259, 25103,
    25193, 16945, 1059,
    11266, 13612, 30508,
    24778, 25364, 1322,
    14492, 11111, 13693,
    15125, 8205,  1749,
    8494,  9902,  9395,
    23936, 3981,  22799,
    28448, 28076, 26544,
    19652, 13424, 8915,
    2885,  11356, 3241,
    1609,  10284, 24350,
    2462,  19358, 15717,
    29327, 15960, 14743,
    5388,  32927, 1288,
    19074, 6322,  32214,
    34208, 30535, 35462,
    23415, 20836, 21819,
    17986, 12196, 30030,
    8422,  2647,  5710,
    3200,  23132, 23337,
    22307, 29841, 4813,
    15309, 26942, 29970,
    23288, 7493,  3005,
    20661, 34283, 33192,
    23033, 9541,  6424,
    22003, 24665, 5534,
    4684,  1411,  33340,
    26042, 6426,  3808,
    285,   21942, 14302,
    16023, 6825,  20084,
    34878, 12295, 32028,
    2591,  178,   24107,
    16379, 2912,  9912,
    15375, 16120, 28375,
    20170, 726,   11291,
    8185,  13471, 8448,
    23205, 14239, 17896,
    17950, 19308, 1591,
    3170,  23836, 18879,
    12853, 10678, 18431,
    21157, 31624, 3153,
    27682, 12433, 3458,
    312,   4844,  13138,
    17715, 35138, 15456,
    30507, 33307, 30783
  };
};

struct DVB_S2X_TABLE_B4
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 35640;
  static const int LINKS_MIN_CN = 8;
  static const int LINKS_MAX_CN = 9;
  static const int LINKS_TOTAL = 262439;
  static const int DEG_MAX = 13;
  static constexpr int DEG[] = {
    13, 3, 0
  };
  static constexpr int LEN[] = {
    27, 72, 0
  };
  static constexpr int POS[] = {
    20834, 22335, 21330, 11913, 6036,  15830, 11069, 10539, 4244,  15068, 7113,  2704,  16224,
    2010,  5628,  27960, 11690, 22545, 24432, 4986,  21083, 17529, 4104,  11941, 21239, 9602,
    689,   13248, 1777,  4876,  2537,  20869, 15718, 9575,  18164, 5294,  13914, 21711, 23374,
    9675,  21239, 13600, 24710, 10613, 14804, 19412, 23270, 26741, 10503, 25258, 17816, 25210,
    12518, 8680,  6422,  22715, 25097, 26959, 3913,  26493, 7797,  25977, 4896,  27063, 20781,
    21715, 12850, 7963,  4027,  4295,  14931, 18158, 616,   20570, 8720,  16487, 19050, 23925,
    7939,  21089, 15170, 24325, 6651,  22352, 5633,  27903, 2685,  1310,  5594,  9296,  25670,
    25121, 13906, 8217,  25390, 9112,  13945, 9826,  10844, 11418, 10724, 11518, 9280,  9576,
    25979, 23644, 16073, 27407, 3476,  28057, 4003,  2279,  17490, 7558,  9538,  22115, 20439,
    20708, 22572, 14997, 15780, 5159,  11356, 10931, 8514,  23275, 2560,  912,   15935, 20703,
    26467, 17173, 21964, 15469, 21967, 10380, 16222, 15106, 16786, 19542, 28560, 18387, 27909,
    14897, 6167,  24295, 1266,  16902, 9546,  11628, 12048, 24495, 3706,  22629, 14165, 2333,
    19403, 18738, 28140, 13141, 6151,  22785, 9620,  4290,  2342,  4902,  15856, 19033, 22820,
    15761, 1985,  9160,  4435,  11164, 5442,  23572, 6951,  19077, 15406, 16658, 18324, 19229,
    16997, 10094, 19982, 22821, 7810,  19660, 1182,  21968, 16564, 17453, 10780, 17034, 16405,
    11,    28611, 10411, 15799, 15705, 2773,  28601, 19333, 19447, 16790, 4618,  15841, 23854,
    24686, 4131,  1013,  2141,  6052,  11896, 18719, 16813, 22420, 23406, 21052, 4333,  17754,
    16425, 17614, 26883, 12101, 8224,  13979, 6869,  25215, 25991, 28968, 19337, 25361, 20513,
    1671,  14990, 20692, 24951, 19446, 7163,  4959,  13197, 19201, 3883,  22532, 15468, 11856,
    22758, 23586, 16985, 18396, 7434,  11817, 363,   11824, 285,   20897, 16646, 16095, 17011,
    25144, 14916, 6302,  20972, 25439, 6156,  21776, 19701, 27803, 9695,  12941, 23541, 27425,
    6979,  27910, 7378,  8983,  6280,  4134,  28860, 8079,  20892, 28776, 7899,  23399, 87,
    18045, 23929, 25876, 15560, 23629, 18376, 4053,  14655, 2450,  11907, 19535, 28543, 3513,
    4704,  16512, 16554, 14062, 2596,  10357, 17316, 1011,  22090, 11353, 20300, 15300, 18536,
    14293, 4746,  28831, 20028, 16742, 16835, 28405, 11245, 10802, 20242, 17737, 9590,  20693,
    26547, 22557, 22517, 6285,  5336,  3998,  2351,  6628,  22949, 1517,  4712,  1770,  9207,
    28522, 14116, 5455,  13105, 18709, 3030,  4217,  6306,  27448, 1943,  23866, 20212, 18857,
    14794, 21425, 15659,
    4446,  21140, 13454,
    21115, 3271,  1443,
    2153,  12424, 6159,
    23559, 22473, 26065,
    15914, 22980, 12766,
    3482,  16233, 5719,
    27020, 12322, 24014,
    25438, 26499, 26506,
    21987, 16027, 6832,
    17330, 2620,  20756,
    15985, 10471, 23302,
    593,   6869,  27185,
    22961, 9129,  25646,
    10702, 12334, 23959,
    6375,  23299, 26942,
    8029,  4072,  24051,
    15147, 5113,  14725,
    1451,  27291, 28731,
    18808, 11561, 249,
    28962, 21405, 18944,
    6889,  3314,  23457,
    27708, 14530, 8795,
    6185,  28821, 6550,
    2259,  17627, 701,
    20819, 18831, 20140,
    4991,  11369, 4282,
    13230, 3413,  27092,
    14556, 5068,  16209,
    4337,  24652, 498,
    715,   28883, 2285,
    16524, 25513, 26034,
    21067, 15122, 21667,
    27982, 15280, 3313,
    7563,  22779, 22453,
    4744,  17277, 27210,
    19170, 10806, 18815,
    26424, 26442, 7837,
    26264, 28931, 6020,
    4645,  20678, 13160,
    18111, 28045, 23883,
    5128,  10876, 3087,
    28551, 26276, 3541,
    20152, 10181, 28172,
    26430, 14769, 6809,
    4956,  16130, 11348,
    1691,  10216, 5743,
    7848,  20236, 2661,
    10660, 8321,  6155,
    2757,  6963,  2596,
    27791, 6707,  258,
    12785, 21176, 15450,
    7477,  17274, 25201,
    262,   18996, 15836,
    5287,  11970, 13365,
    3098,  17823, 10786,
    21831, 14476, 11447,
    1893,  3625,  25404,
    20880, 21987, 1228,
    20942, 15045, 21358,
    18237, 28914, 15673,
    24273, 284,   9803,
    13949, 15670, 16693,
    15553, 27782, 22644,
    27980, 24820, 27733,
    7015,  20974, 10016,
    26164, 20314, 25916,
    11489, 13663, 11777,
    18230, 11483, 5655,
    1618,  19977, 26521,
    25639, 13184, 28994,
    3821,  18349, 13846
  };
};

struct DVB_S2X_TABLE_B5
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 37440;
  static const int LINKS_MIN_CN = 9;
  static const int LINKS_MAX_CN = 10;
  static const int LINKS_TOTAL = 273599;
  static const int DEG_MAX = 13;
  static constexpr int DEG[] = {
    13, 12, 3, 0
  };
  static constexpr int LEN[] = {
    8, 24, 72, 0
  };
  static constexpr int POS[] = {
    12918, 15296, 894,   10855, 350,   453,   11966, 1667,  18720, 12943, 24437, 8135,  2834,
    11861, 3827,  15431, 8827,  8253,  23393, 15048, 5554,  16297, 2994,  6727,  19453, 2371,
    26414, 3044,  20240, 18313, 11618, 3145,  10976, 5786,  5609,  16358, 2547,  11557, 14755,
    26434, 2510,  26719, 4420,  6753,  917,   7821,  26765, 11684, 9811,  5420,  6653,  19554,
    11928, 20579, 17439, 19103, 21162, 11235, 19172, 22254, 3420,  10558, 3646,  11858, 24120,
    10189, 8172,  5004,  26082, 4345,  5139,  15135, 26522, 6172,  17492, 8462,  4392,  4546,
    27330, 21498, 13424, 8077,  10165, 9739,  482,   23749, 1515,  12788, 10464, 9085,  20875,
    12009, 22276, 18401, 7541,  5871,  23053, 16979, 16300, 13566, 19424, 5293,  18290, 23917,
    9613,  24175, 11374, 11736, 17676, 13126, 20931, 20290, 20659, 2000,  7969,  9386,
    21507, 24494, 11822, 21771, 26776, 21175, 27354, 15815, 7598,  19809, 611,   10144,
    195,   14244, 7229,  13002, 14328, 17987, 14595, 6985,  7642,  9434,  7079,  5571,
    10013, 3641,  14064, 11716, 4620,  18119, 23365, 26446, 26273, 25164, 11262, 26019,
    15166, 19403, 5606,  20138, 1893,  645,   5414,  12097, 18635, 21648, 12255, 13269,
    1895,  9969,  8372,  17737, 21679, 17061, 20219, 2513,  27199, 11242, 17025, 1261,
    12845, 13086, 16256, 15177, 20822, 10862, 18375, 6751,  17532, 24725, 6966,  18489,
    8373,  25550, 20688, 16686, 7894,  24599, 21578, 12516, 7115,  4836,  23473, 25162,
    14375, 9150,  6606,  21633, 16224, 23708, 20350, 4575,  143,   13356, 10239, 22868,
    10760, 19807, 7079,  16382, 26236, 22606, 16777, 24312, 16941, 26684, 8658,  19279,
    15136, 8603,  332,   2898,  21821, 23778, 3232,  12052, 14336, 7832,  5600,  27015,
    14392, 26564, 21616, 8332,  21750, 10379, 19730, 7553,  27352, 2718,  15202, 25661,
    6891,  13210, 15284, 21940, 8742,  10965, 3176,  25034, 25137, 25161, 13267, 7012,
    4993,  9943,  13260, 20980, 20224, 20129, 2120,  23111, 16640, 23548, 21445, 10794,
    4846,  2858,  22663, 12584, 20448, 4629,  17825, 22269, 11278, 26312, 9463,  21085,
    24282, 18233, 9220,  14979, 24106, 14507, 24838, 19689, 17589, 7926,  7893,  21701,
    12253, 26122, 8035,  20823, 2584,  4703,  25178, 5460,  4190,  7057,  1144,  8426,
    12354, 7216,  19484, 4110,  22105, 1452,  11457, 12539, 27106, 14256, 14113, 20701,
    2547,  26926, 25933, 11919, 12026, 24639, 19741, 15457, 9239,  26713, 22838, 6051,
    8782,  14714, 23363, 450,   19972, 2622,  19473, 24182, 2391,  26205, 10018, 9202,
    15690, 10472, 20263, 469,   18876, 23660, 9005,  12595, 23818, 26430, 926,   6156,
    5440,  5209,  14958, 9882,  18843, 22063, 12749, 18473, 22546, 11768, 4493,  12833,
    18540, 3544,  9471,  15893, 14761, 23479, 22010, 15491, 19608, 25035, 9094,  24836,
    15909, 16594, 23538, 25136, 25063, 24995, 5354,  905,   18580, 15476, 20710, 7774,
    6088,  17133, 11498,
    4721,  17594, 18267,
    1645,  23638, 26645,
    14800, 17920, 22016,
    12927, 350,   19391,
    19447, 19886, 25992,
    26120, 1747,  11234,
    1588,  23170, 27232,
    2230,  15468, 18709,
    17410, 11055, 20645,
    3244,  25815, 14204,
    2858,  7980,  12780,
    3256,  20418, 24355,
    24260, 16245, 20948,
    11122, 1503,  15651,
    19272, 24054, 6075,
    4905,  931,   18884,
    23633, 17244, 6067,
    5568,  26403, 490,
    16113, 16055, 10524,
    23013, 8138,  12876,
    20699, 20123, 15435,
    27272, 27296, 22638,
    7658,  17259, 20553,
    14914, 17891, 12137,
    16323, 1085,  18895,
    21503, 17141, 2915,
    21979, 23246, 1271,
    14409, 11303, 12604,
    25591, 12157, 14704,
    18739, 19265, 8140,
    11244, 5962,  6647,
    3589,  6029,  6489,
    16416, 185,   9426,
    1267,  14086, 22473,
    17159, 22404, 23608,
    7230,  22514, 21605,
    7645,  1239,  10717,
    12028, 13404, 12140,
    14784, 15425, 14895,
    26165, 18980, 15386,
    14399, 7725,  14908,
    8463,  22853, 22095,
    5517,  1854,  8283,
    24381, 260,   12595,
    839,   23743, 22445,
    13473, 8017,  7716,
    8697,  13050, 16975,
    26656, 16911, 11972,
    26173, 2504,  15216,
    7493,  6461,  12840,
    4464,  14912, 3745,
    21461, 9734,  25841,
    4659,  7599,  9984,
    17519, 7389,  75,
    12589, 9862,  8680,
    23053, 21981, 25299,
    19246, 3243,  15916,
    21733, 4467,  26491,
    4959,  10093, 20074,
    9140,  15000, 12783,
    854,   10701, 25850,
    13624, 7755,  10789,
    3977,  15812, 10783,
    5830,  6774,  10151,
    21375, 25110, 5830,
    15985, 18342, 2623,
    4716,  27211, 18500,
    18370, 12487, 7335,
    4362,  21569, 16881,
    10421, 15454, 13015,
    5794,  1239,  9934
  };
};

struct DVB_S2X_TABLE_B6
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 40320;
  static const int LINKS_MIN_CN = 9;
  static const int LINKS_MAX_CN = 10;
  static const int LINKS_TOTAL = 244799;
  static const int DEG_MAX = 11;
  static constexpr int DEG[] = {
    11, 7, 3, 0
  };
  static constexpr int LEN[] = {
    24, 4, 84, 0
  };
  static constexpr int POS[] = {
    24402, 4786,  12678, 6376,  23965, 10003, 15376, 15164, 21366, 24252, 3353,
    8189,  3297,  18493, 17994, 16296, 11970, 16168, 15911, 20683, 11930, 3119,
    22463, 11744, 13833, 8279,  21652, 14679, 23663, 4389,  15110, 17254, 17498,
    13616, 426,   18060, 598,   19615, 9494,  3987,  8014,  13361, 4131,  13185,
    4176,  17725, 14717, 3414,  10033, 17879, 8079,  12107, 10852, 1375,  19459,
    1450,  4123,  2111,  17490, 13209, 8048,  15285, 4422,  11667, 18290, 19621,
    2067,  15982, 304,   8658,  19120, 6746,  13569, 19253, 2227,  22778, 23826,
    11667, 11145, 20469, 17485, 13697, 3712,  4258,  16831, 22634, 18035, 7275,
    23804, 14496, 17938, 15883, 14984, 15944, 2816,  22406, 22111, 2319,  14731,
    8541,  12579, 22121, 8602,  16755, 6704,  23740, 16151, 20297, 9633,  1100,
    19569, 10549, 19086, 21110, 11659, 6901,  21295, 7637,  11756, 8293,  9071,
    9527,  9135,  7181,  19534, 2157,  788,   13347, 17355, 17509, 711,   20116,
    21217, 15801, 12175, 9604,  17521, 2127,  21103, 1346,  8921,  7976,  3363,
    11036, 5152,  19173, 8086,  3571,  1955,  4146,  13309, 15934, 19132, 5510,
    12935, 13966, 15399, 16179, 8206,  19233, 16702, 7127,  12185, 15420, 1383,
    6222,  6384,  20549, 18914, 23658, 11189, 638,   9297,  17741, 9747,  13598,
    17209, 11974, 20776, 2146,  9023,  3192,  19646, 3393,  1727,  15588, 20185,
    5008,  3885,  5035,  15852, 5189,  13877, 15177, 3049,  22164, 16540, 21064,
    24004, 10345, 12255, 36,    24008, 8764,  13276, 13131, 2358,  24010, 16203,
    21121, 21691, 8555,  11918, 129,   8860,  23600, 3042,  3949,  19554, 12319,
    22514, 11709, 11874, 11656, 536,   9142,  3901,  580,   1547,  10749, 5529,
    3324,  6251,  1156,  112,   13086, 5373,  5119,  132,   18069, 10482, 19519,
    17279, 2017,  14846, 21417, 17154, 21735, 18788, 11759, 192,   16027, 6234,
    20417, 3788,  15159, 22188, 21251, 16633, 13579, 8128,  1841,  23554, 15056,
    12104, 9182,  6147,  1553,  12750, 4071,  6495,
    4961,  18460, 23266, 10785, 10973, 4405,  2707,
    7665,  7043,  1968,  3589,  15378, 9642,  21148,
    13073, 13298, 20040, 13582, 17124, 348,   12055,
    378,   7476,  9838,
    15454, 5218,  14834,
    17678, 3445,  18453,
    2767,  388,   12638,
    5688,  56,    6360,
    20009, 872,   16872,
    10206, 5551,  477,
    10662, 23689, 19768,
    8965,  17535, 4421,
    19397, 18734, 5422,
    10043, 22104, 21682,
    508,   1588,  23853,
    1092,  7288,  4358,
    2283,  22298, 10504,
    15022, 8592,  22291,
    11844, 17038, 2983,
    17404, 14541, 6446,
    20724, 7498,  2993,
    14715, 9410,  6844,
    20213, 14674, 263,
    4822,  20951, 635,
    20651, 23174, 5057,
    22237, 9229,  4859,
    17280, 9586,  20334,
    19508, 8068,  11375,
    5776,  21209, 9418,
    6872,  6349,  20397,
    11165, 19619, 13108,
    13550, 10715, 5122,
    5655,  10699, 8415,
    9864,  4985,  7986,
    6436,  3754,  7690,
    4257,  17119, 5328,
    659,   4687,  6006,
    527,   10824, 8234,
    11291, 1735,  22513,
    7254,  2617,  1493,
    3015,  7462,  10953,
    15705, 2181,  11992,
    4628,  19430, 18223,
    9426,  21808, 13549,
    17008, 3470,  22568,
    13643, 24195, 21816,
    936,   14226, 22874,
    6156,  19306, 18215,
    23984, 14714, 12907,
    5139,  18639, 15609,
    11908, 5446,  8958,
    6315,  16864, 15814,
    10686, 22570, 16196,
    203,   4208,  13716,
    494,   14172, 11778,
    15112, 14244, 8417,
    21087, 4602,  15570,
    19758, 4401,  22270,
    8218,  11940, 5009,
    23833, 13785, 12569,
    1698,  7113,  18541,
    18711, 19991, 19673,
    8025,  17107, 14784,
    5954,  6817,  19810,
    24143, 12236, 18063,
    23748, 23956, 10369,
    7805,  13982, 13861,
    5198,  10889, 6787,
    10406, 13918, 3305,
    12219, 6523,  12999,
    9964,  2004,  17361,
    23759, 21507, 11984,
    4188,  19754, 13358,
    8027,  3662,  2411,
    19762, 16017, 9125,
    2393,  4619,  5452,
    24176, 6586,  10895,
    15872, 1795,  15801,
    6911,  15300, 14787,
    2584,  4905,  8833,
    1327,  12862, 9476,
    16768, 12633, 7400,
    11983, 6276,  18370,
    12939, 12793, 20048,
    20284, 12949, 21345,
    19545, 4503,  16017,
    1253,  12068, 18813
  };
};

struct DVB_S2X_TABLE_B7
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 41400;
  static const int LINKS_MIN_CN = 9;
  static const int LINKS_MAX_CN = 10;
  static const int LINKS_TOTAL = 233999;
  static const int DEG_MAX = 11;
  static constexpr int DEG[] = {
    11, 6, 3, 0
  };
  static constexpr int LEN[] = {
    20, 5, 90, 0
  };
  static constexpr int POS[] = {
    2475,  3722,  16456, 6081,  4483,  19474, 20555, 10558, 4351,  4052,  20066,
    1547,  5612,  22269, 11685, 23297, 19891, 18996, 21694, 7927,  19412, 15951,
    288,   15139, 7767,  3059,  1455,  12056, 12721, 7938,  19334, 3233,  5711,
    6664,  7486,  17133, 2931,  20176, 20158, 9634,  20002, 13129, 10015, 13595,
    218,   22642, 9357,  11999, 22898, 4446,  8059,  1913,  22365, 10039, 15203,
    10305, 22970, 7928,  16564, 8402,  9988,  7039,  10195, 22389, 5451,  8731,
    19073, 1005,  18826, 11109, 13748, 11891, 21530, 15924, 21128, 6841,  11064,
    3240,  11632, 18386, 22456, 3963,  14719, 4244,  4599,  8098,  7599,  12862,
    5666,  11543, 9276,  19923, 19171, 19591, 6005,  8623,  22777, 1255,  20078,
    17064, 13244, 323,   11349, 6637,  8611,  6695,  4750,  20985, 18144, 5584,
    20309, 6210,  16745, 10959, 14284, 2893,  20916, 10985, 9664,  9065,  11703,
    17833, 21598, 22375, 12890, 10779, 11241, 13115, 9222,  21139, 1217,  15337,
    15514, 12517, 18953, 11458, 17296, 8751,  7213,  12078, 4994,  4391,  14976,
    3842,  21548, 10955, 11679, 16551, 8514,  17999, 20557, 16497, 12122, 23056,
    10551, 20186, 66,    11038, 22049, 2130,  1089,  22093, 9069,  3470,  8079,
    19208, 22044, 2732,  1325,  22309, 967,   22951, 1366,  11745, 5556,  6926,
    2805,  18271, 10046, 4277,  207,   19518, 17387, 9701,  8515,  6813,  10532,
    19714, 21923, 13493, 1768,  18819, 6093,  14086, 13695, 12781, 9782,  445,
    22160, 15778, 13629, 10312, 19769, 8567,  22096, 15558, 19730, 11861, 18492,
    10729, 16847, 273,   4119,  4392,  11480, 20396, 3505,  7220,  390,   5546,
    17277, 8531,  17390, 22364, 7167,  2217,
    7325,  3832,  19899, 21104, 8400,  3906,
    6218,  20330, 14943, 14477, 5614,  1582,
    21534, 14286, 14624, 14809, 6775,  22838,
    15786, 6527,  15848, 5288,  13523, 9692,
    12696, 15315, 602,
    17081, 6828,  13578,
    3492,  6510,  20337,
    6113,  5090,  7290,
    20122, 15539, 19267,
    10412, 19090, 17863,
    2546,  2295,  19448,
    20296, 2296,  2627,
    6740,  14224, 10460,
    12878, 6055,  15452,
    15152, 15699, 563,
    15414, 21900, 19161,
    11126, 15975, 3733,
    4379,  15742, 6475,
    17203, 5870,  18537,
    4912,  260,   21115,
    23164, 4273,  1694,
    1082,  5287,  11152,
    14537, 2277,  19232,
    13414, 15608, 12926,
    17043, 18241, 18313,
    208,   6118,  20777,
    9140,  19241, 22845,
    18527, 5035,  4161,
    20867, 22650, 5585,
    7875,  10358, 1898,
    3563,  14833, 21329,
    14705, 3359,  13959,
    4507,  11976, 20017,
    22424, 12925, 8308,
    8739,  15561, 8010,
    6408,  20723, 20928,
    12337, 7864,  15777,
    12742, 20430, 17351,
    6259,  1865,  9808,
    8343,  17441, 2551,
    2167,  3025,  23181,
    22718, 13243, 4797,
    4223,  4982,  4395,
    1609,  16748, 17625,
    8463,  15204, 19632,
    6583,  9112,  20284,
    11334, 19370, 4763,
    746,   18560, 15222,
    8796,  12725, 15176,
    10245, 15567, 9991,
    17447, 18373, 21523,
    1473,  5286,  15793,
    17675, 21170, 6699,
    15515, 15942, 8733,
    7047,  11348, 14584,
    20435, 19603, 1961,
    18851, 7069,  11402,
    19180, 6487,  2979,
    2650,  13282, 9040,
    22613, 23266, 4786,
    20832, 3001,  23129,
    3850,  5255,  6601,
    19827, 15438, 13956,
    15798, 4430,  11318,
    4724,  8719,  21209,
    18127, 844,   21379,
    7427,  22987, 10233,
    22949, 8145,  21778,
    7622,  14471, 18874,
    8566,  14340, 3381,
    3373,  419,   11514,
    15127, 917,   13136,
    19375, 18740, 4951,
    960,   2856,  17804,
    662,   8107,  10298,
    10993, 11755, 19142,
    11400, 18818, 521,
    7210,  18658, 8285,
    9496,  20836, 5655,
    14654, 13694, 12705,
    20381, 16473, 7271,
    12796, 3280,  23370,
    13893, 7667,  1736,
    5485,  18321, 7789,
    11242, 18771, 17282,
    817,   21060, 15985,
    666,   20461, 22464,
    7696,  19774, 4324,
    12239, 14014, 4759,
    5011,  10472, 4137,
    3047,  2444,  3818,
    1594,  20382, 538,
    7051,  21874, 1697,
    18539, 26,    21487
  };
};

struct DVB_S2X_TABLE_B8
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 45000;
  static const int LINKS_MIN_CN = 12;
  static const int LINKS_MAX_CN = 13;
  static const int LINKS_TOTAL = 257399;
  static const int DEG_MAX = 11;
  static constexpr int DEG[] = {
    11, 9, 3, 0
  };
  static constexpr int LEN[] = {
    25, 5, 95, 0
  };
  static constexpr int POS[] = {
    11863, 9493,  4143,  12695, 8706,  170,   4967,  798,   9856,  6015,  5125,
    12288, 19567, 18233, 15430, 1671,  3787,  10133, 15709, 7883,  14260, 17039,
    2066,  12269, 14620, 7577,  11525, 19519, 6181,  3850,  8893,  272,   12473,
    8857,  12404, 1136,  19464, 15113, 12598, 12147, 4987,  13843, 12152, 13241,
    1354,  12339, 4308,  23,    12677, 11533, 3187,  11609, 4740,  14630, 19630,
    14508, 10946, 3928,  580,   3526,  17836, 3786,  15739, 13991, 1238,  1071,
    6977,  13222, 13811, 585,   8154,  2579,  8314,  12185, 15876, 7738,  5691,
    12901, 12576, 11597, 4893,  17238, 15556, 8106,  12472, 10455, 14530, 17432,
    8373,  12875, 16582, 14611, 14267, 15093, 2405,  9342,  18326, 12125, 9257,
    5861,  12284, 2441,  13280, 2762,  5076,  17758, 4359,  6156,  18961, 13208,
    4400,  8474,  19629, 19528, 14125, 12780, 12740, 19316, 491,   4761,  1719,
    7270,  6615,  1175,  15848, 6943,  18360, 8905,  13921, 10807, 19688, 18757,
    8312,  12234, 17907, 17254, 7699,  18399, 5508,  12215, 4818,  18107, 2874,
    19496, 13973, 10432, 13445, 15320, 13648, 1501,  10549, 6710,  8897,  1998,
    1575,  12713, 10916, 5316,  13713, 11318, 4055,  5782,  5828,  17981, 3141,
    12177, 10726, 4244,  3138,  15996, 6822,  7495,  5257,  8909,  6180,  10680,
    6650,  1909,  19146, 1038,  17229, 10050, 3051,  9793,  10839, 3532,  14759,
    5337,  8448,  4939,  14792, 7585,  17860, 8612,  2229,  18965, 1519,  2031,
    13845, 9320,  579,   15441, 15050, 752,   8303,  6989,  13360, 12927, 15255,
    17286, 3639,  1733,  16883, 8457,  9475,  2939,  3234,  1993,  8554,  9939,
    6359,  15474, 12100, 6992,  13844, 16988, 7481,  16977, 9052,  9262,  15270,
    7181,  3624,  3814,  16379, 182,   4338,  17627, 3315,  5745,  14093, 15574,
    10709, 18662, 6909,  11248, 5268,  412,   5854,  16782, 16059, 10498, 5061,
    13321, 617,   6734,  3718,  15441, 19241, 17214, 1682,  18641, 18646, 6330,
    7377,  16951, 14477, 6507,  9922,  11464, 2563,  5702,  12691, 10606, 17874,
    7198,  12571, 17617, 4862,  18899, 7100,  8130,  9665,  10779,
    6789,  11459, 17651, 3693,  13332, 3854,  7737,  12589, 15189,
    16260, 14569, 9442,  17890, 18097, 6845,  6960,  1376,  8099,
    12719, 14986, 18999, 14013, 3449,  13618, 14807, 265,   1508,
    11231, 966,   15957, 8315,  3384,  2570,  5700,  10911, 17372,
    153,   8445,  19598,
    7841,  14806, 54,
    2492,  14099, 11718,
    18608, 4278,  333,
    59,    3982,  16986,
    3494,  12496, 2775,
    18320, 10650, 16234,
    9739,  16537, 19706,
    7587,  19072, 18775,
    14133, 12042, 2922,
    229,   17958, 15889,
    5130,  11029, 271,
    5122,  7021,  7067,
    12258, 16611, 9245,
    15493, 15347, 15939,
    741,   12055, 2822,
    12804, 3480,  5690,
    18598, 19273, 16354,
    2569,  16771, 13693,
    15051, 853,   956,
    12256, 2756,  15137,
    15685, 2802,  16479,
    14687, 12470, 3583,
    15473, 17781, 867,
    4843,  6765,  13122,
    11287, 3680,  19101,
    4609,  11385, 13470,
    12353, 6632,  206,
    10984, 3116,  1263,
    9419,  14455, 19438,
    9528,  1808,  435,
    2238,  12870, 10119,
    10868, 8402,  11111,
    11081, 7197,  2667,
    13780, 10759, 19722,
    3768,  3052,  1836,
    446,   1642,  12388,
    16876, 8398,  14485,
    7301,  14815, 13811,
    5678,  10419, 14396,
    1877,  14384, 12817,
    19028, 19589, 6893,
    8725,  6346,  676,
    13611, 12486, 2054,
    11203, 14908, 14692,
    18139, 5334,  1253,
    16233, 9749,  16946,
    18885, 4332,  16306,
    3862,  10395, 13871,
    3747,  8900,  3381,
    13367, 14132, 7220,
    15095, 4219,  15869,
    13519, 18079, 17541,
    19012, 13943, 19471,
    2221,  5710,  13711,
    5185,  3363,  10195,
    9580,  17331, 15360,
    14387, 7596,  9614,
    17336, 6371,  6030,
    14629, 10636, 10159,
    2402,  9170,  4321,
    1040,  5899,  153,
    7710,  7637,  13966,
    10919, 8535,  3791,
    1968,  2567,  4986,
    4166,  8744,  17691,
    540,   10695, 10019,
    17710, 1188,  10821,
    5858,  17012, 17389,
    3083,  17587, 12682,
    5354,  9537,  6807,
    4964,  15942, 9653,
    9000,  17053, 13291,
    11685, 8503,  10777,
    13919, 18155, 9877,
    1625,  15314, 13879,
    18520, 7074,  17061,
    3748,  2752,  7298,
    493,   19163, 14139,
    2260,  18339, 10688,
    8928,  17695, 10276,
    7640,  18547, 3561,
    11275, 5297,  13167,
    19691, 19542, 15725,
    11837, 7273,  11297,
    17873, 7840,  19563,
    8109,  3811,  18417,
    17759, 17623, 13175,
    10041, 4152,  2249,
    18452, 1450,  19309,
    9161,  11651, 4614,
    11547, 14058, 639,
    9384,  3272,  12368,
    5898,  2578,  14635,
    15963, 6733,  11048
  };
};

struct DVB_S2X_TABLE_B9
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 46800;
  static const int LINKS_MIN_CN = 13;
  static const int LINKS_MAX_CN = 14;
  static const int LINKS_TOTAL = 251999;
  static const int DEG_MAX = 10;
  static constexpr int DEG[] = {
    10, 3, 0
  };
  static constexpr int LEN[] = {
    30, 100, 0
  };
  static constexpr int POS[] = {
    2510,  12817, 11890, 13009, 5343,  1775,  10496, 13302, 13348, 17880,
    6766,  16330, 2412,  7944,  2483,  7602,  12482, 6942,  3070,  9231,
    16410, 1766,  1240,  10046, 12091, 14475, 7003,  202,   7733,  11237,
    15562, 4695,  13931, 17100, 11102, 770,   3848,  4216,  7132,  10929,
    16469, 17153, 8177,  8723,  12861, 15948, 2251,  1500,  11526, 8590,
    14813, 3505,  12654, 1079,  11736, 6290,  2299,  17073, 6330,  5997,
    390,   16492, 13989, 1320,  14600, 7061,  6583,  458,   894,   1596,
    8625,  7644,  1322,  16647, 15763, 10439, 8740,  5529,  2969,  13893,
    13425, 13121, 5344,  8739,  4953,  7654,  17848, 9334,  9533,  2731,
    12506, 10992, 8762,  5395,  6424,  11688, 3193,  17601, 14679, 8204,
    5466,  15487, 1642,  6671,  13557, 4074,  7182,  4436,  12398, 12973,
    1958,  13041, 6579,  15984, 3762,  16633, 6113,  11509, 7227,  28,
    17202, 4813,  14024, 15099, 2648,  4476,  2260,  6507,  9930,  9232,
    14186, 14510, 6818,  7665,  12708, 2645,  16687, 13255, 8239,  15884,
    1751,  7847,  17987, 11410, 3345,  17133, 17655, 5027,  1261,  17191,
    8056,  4264,  13915, 8217,  6118,  8072,  6278,  6835,  5038,  15008,
    13625, 2999,  5336,  11687, 13500, 5723,  13903, 766,   6293,  155,
    12316, 14093, 7372,  16846, 15357, 9865,  17869, 1429,  16681, 202,
    15062, 1123,  6454,  17625, 3213,  39,    1669,  1770,  13636, 16555,
    13053, 7597,  11481, 1336,  3343,  11387, 5463,  17830, 13741, 5976,
    1956,  13509, 1664,  16867, 8168,  13421, 17078, 3285,  17138, 1572,
    16711, 1499,  4805,  13584, 14759, 2844,  13110, 7356,  5850,  8330,
    6521,  8528,  14170, 6681,  16992, 12867, 14326, 15227, 4082,  8595,
    16176, 8184,  8572,  1923,  935,   8900,  13020, 6812,  9778,  3391,
    3946,  4711,  15314, 15108, 15634, 4144,  4372,  9207,  10715, 1291,
    16601, 5864,  10968, 4724,  9235,  6988,  3307,  6515,  7004,  16328,
    16217, 4227,  9735,  15857, 5003,  2532,  4451,  8574,  2149,  6908,
    9506,  8949,  12035, 9701,  3124,  14295, 8567,  13614, 5159,  16746,
    2418,  8669,  10921, 5738,  147,   1004,  2692,  9065,  12877, 7559,
    16706, 8511,  10314, 3118,  1219,  7071,  12376, 538,   2389,  3297,
    12492, 10589, 5791,
    13528, 1653,  6618,
    10485, 1307,  4102,
    347,   13580, 4039,
    523,   10311, 10540,
    4183,  6192,  17159,
    11458, 6521,  9632,
    11594, 15791, 10384,
    11654, 126,   11715,
    6265,  34,    5091,
    7271,  13900, 7588,
    3960,  11297, 1612,
    9857,  4695,  16399,
    6423,  2197,  15040,
    4219,  5979,  13959,
    2959,  578,   8404,
    4585,  658,   6474,
    15900, 11357, 5249,
    7414,  8642,  1151,
    4130,  9064,  14537,
    14517, 1356,  3748,
    13865, 12085, 17295,
    9530,  5110,  1570,
    10862, 8458,  15322,
    16355, 1774,  5270,
    1229,  11587, 1632,
    17039, 787,   4703,
    11423, 15388, 6136,
    8413,  9703,  13946,
    4678,  4072,  16702,
    6244,  4690,  7164,
    7238,  14169, 5398,
    8679,  122,   11593,
    10954, 15802, 16427,
    9413,  6717,  16406,
    1027,  17863, 7836,
    655,   8827,  10286,
    4124,  12599, 12482,
    12955, 3121,  15318,
    8343,  16634, 6301,
    13568, 5056,  9920,
    1948,  10,    17395,
    8550,  131,   2151,
    15226, 15994, 13093,
    10966, 15412, 2781,
    13425, 15831, 5346,
    2261,  1067,  6346,
    6625,  1966,  13533,
    10575, 4483,  5761,
    14366, 2019,  14426,
    16746, 1450,  4830,
    13109, 7358,  7942,
    15376, 7284,  14035,
    14341, 12625, 3306,
    9375,  7529,  1537,
    13831, 13447, 4549,
    15658, 15299, 8238,
    4005,  13264, 9766,
    4715,  6285,  15383,
    1262,  12883, 15434,
    11123, 14975, 3434,
    5307,  1112,  16967,
    12163, 12009, 3681,
    9174,  13153, 10344,
    13456, 13197, 9562,
    1785,  7549,  15347,
    663,   9748,  9436,
    4961,  11903, 11574,
    16248, 6238,  666,
    11426, 13748, 14763,
    14431, 1443,  2069,
    2376,  8154,  14978,
    13140, 1289,  9046,
    1159,  300,   3319,
    11510, 7769,  15877,
    6430,  14946, 6856,
    8868,  15622, 12458,
    4867,  6622,  6850,
    14721, 11241, 12760,
    14233, 9874,  17682,
    16677, 13195, 15086,
    11155, 7067,  14160,
    12741, 14379, 8922,
    1930,  17055, 11752,
    12361, 6523,  9568,
    12165, 5636,  16011,
    11389, 4754,  9916,
    15903, 15542, 8301,
    12073, 4918,  9754,
    16544, 17907, 14814,
    10839, 1401,  5107,
    12320, 1095,  8592,
    15088, 6521,  12015,
    14802, 3901,  8920,
    17932, 2990,  1643,
    5102,  3870,  2045,
    540,   2643,  2287,
    5844,  2482,  9471,
    10428, 637,   3629,
    8814,  7277,  2678
  };
};

struct DVB_S2X_TABLE_B10
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 50400;
  static const int LINKS_MIN_CN = 17;
  static const int LINKS_MAX_CN = 18;
  static const int LINKS_TOTAL = 259199;
  static const int DEG_MAX = 12;
  static constexpr int DEG[] = {
    12, 5, 3, 0
  };
  static constexpr int LEN[] = {
    20, 20, 100, 0
  };
  static constexpr int POS[] = {
    13057, 12620, 2789,  3553,  6763,  8329,  3333,  7822,  10490, 13943, 4101,  2556,
    658,   11386, 2242,  7249,  5935,  2148,  5291,  11992, 3222,  2957,  6454,  3343,
    93,    1205,  12706, 11406, 9017,  7834,  5358,  13700, 14295, 4152,  6287,  4249,
    6958,  2768,  8087,  1759,  11889, 4474,  3925,  4004,  14392, 8923,  6962,  4822,
    6719,  5436,  1905,  10228, 5059,  4892,  12448, 26,    12891, 10607, 12210, 10424,
    8368,  10667, 9045,  7694,  13097, 3555,  4831,  411,   8539,  6527,  12753, 11530,
    4960,  6647,  13969, 3556,  9997,  7898,  2134,  9931,  3749,  4305,  11242, 10410,
    9125,  9075,  9916,  12370, 8720,  6056,  8128,  5425,  979,   3421,  5660,  9473,
    4348,  11979, 5985,  395,   11255, 13878, 7797,  4962,  13519, 13323, 7596,  5520,
    2852,  8519,  3022,  9432,  3564,  9467,  8569,  12235, 11837, 5031,  4246,  2,
    4081,  3630,  1619,  2525,  3773,  11491, 14076, 9834,  3618,  2008,  4694,  6948,
    7684,  9642,  5970,  1679,  13207, 12368, 262,   7401,  11471, 2861,  5620,  4754,
    7474,  10418, 1422,  10960, 13852, 988,   13465, 6415,  86,    2432,  7595,  12239,
    8539,  11749, 8794,  6350,  1947,  13325, 13061, 7385,  13017, 2536,  13121, 15,
    7944,  13831, 5126,  9938,  11758, 335,   980,   9736,  12143, 5753,  4533,  10814,
    10706, 12618, 6949,  2684,  4107,  14388, 11372, 6321,  13832, 9190,  2838,  13860,
    10830, 1947,  13803, 3257,  2677,  406,   8400,  10536, 12911, 3629,  251,   9784,
    13343, 13304, 301,   801,   6456,  6351,  6155,  6763,  3812,  11337, 8446,  9306,
    524,   5573,  503,   10544, 8990,  673,   2309,  12376, 466,   11441, 960,   1557,
    4403,  3564,  1732,  13453, 12054, 8941,  1383,  12424, 4347,  9830,  3553,  5158,
    2025,  4282,  4983,  13553, 10776,
    11833, 13099, 5078,  4420,  3527,
    1544,  7474,  2780,  7749,  4153,
    11189, 520,   8463,  12230, 7712,
    10409, 13367, 2604,  2966,  9248,
    1412,  420,   3507,  9818,  7955,
    1122,  12483, 9375,  10232, 9456,
    2799,  7033,  10404, 4495,  12059,
    2569,  5970,  6262,  2199,  8045,
    11724, 511,   12693, 12855, 9597,
    756,   12900, 13391, 13623, 10683,
    2095,  13479, 1488,  9469,  11142,
    13849, 1356,  10776, 3530,  9866,
    13449, 14225, 2072,  12772, 9461,
    6466,  6181,  6502,  401,   7439,
    4631,  1086,  3062,  11789, 11811,
    6788,  14007, 2270,  14132, 2764,
    4643,  10272, 11316, 2608,  8511,
    5221,  9028,  2736,  7223,  1051,
    1974,  2737,  6739,  13904, 6156,
    5,     9082,  3915,
    2400,  7195,  3413,
    606,   221,   8171,
    4548,  1267,  5310,
    12795, 2160,  8305,
    10563, 3507,  12190,
    6325,  2499,  9717,
    9251,  6046,  13308,
    11704, 10834, 11241,
    4777,  3774,  11533,
    12487, 10365, 6852,
    58,    2650,  2027,
    7248,  13704, 5573,
    12777, 7834,  8561,
    7906,  8121,  7774,
    554,   3105,  6000,
    11198, 3586,  10410,
    9002,  4094,  11297,
    12058, 1037,  13638,
    1258,  12917, 11078,
    2430,  51,    10276,
    7841,  9451,  10236,
    11045, 1058,  10352,
    9629,  9428,  86,
    8146,  1255,  3802,
    10820, 6337,  4199,
    9364,  7723,  1139,
    438,   6445,  583,
    2683,  5358,  10730,
    8471,  3061,  13380,
    3005,  2840,  4754,
    8210,  1814,  11502,
    8667,  14258, 5985,
    8407,  13336, 10970,
    6363,  11715, 5053,
    104,   13618, 13817,
    6562,  4087,  294,
    1742,  10528, 4626,
    6607,  2692,  1587,
    11097, 8361,  2788,
    13451, 3541,  823,
    4060,  13604, 9816,
    157,   6106,  1062,
    8853,  5159,  4270,
    9352,  13164, 2919,
    7526,  5174,  12501,
    12634, 13077, 5129,
    5750,  1568,  6281,
    269,   5985,  10973,
    8518,  9415,  1028,
    4722,  13275, 634,
    12113, 7104,  7436,
    12787, 1032,  5936,
    3425,  11526, 10797,
    784,   9208,  15,
    11223, 12849, 4913,
    10635, 3553,  8852,
    11749, 10619, 3532,
    4080,  9831,  9219,
    6560,  6049,  6111,
    1304,  11770, 12585,
    13209, 8589,  11287,
    2887,  10699, 14307,
    4752,  456,   4073,
    1175,  13156, 4894,
    12756, 3237,  6279,
    10125, 7074,  2344,
    7533,  7103,  5226,
    4000,  4425,  12173,
    10056, 5312,  1599,
    7445,  8696,  12533,
    11509, 14050, 2483,
    12405, 2876,  5033,
    4512,  4955,  5627,
    5572,  5099,  10987,
    10665, 404,   3082,
    2075,  1583,  13454,
    5666,  7228,  524,
    13290, 7634,  418,
    9006,  7368,  4181,
    9447,  3674,  8171,
    9355,  10211, 9342,
    12572, 3681,  3322,
    3295,  186,   7491,
    7926,  212,   5241,
    5479,  1654,  8097,
    5078,  423,   4817,
    1357,  12780, 3664,
    11900, 402,   13108,
    299,   7166,  12008,
    5750,  3041,  5618,
    8357,  1229,  8884,
    3713,  8791,  13375,
    4390,  6302,  568,
    1009,  4440,  10003,
    1209,  11978, 11711,
    1803,  9838,  13537,
    11318, 9750,  12421,
    2388,  3021,  7880,
    7220,  1062,  6871
  };
};

struct DVB_S2X_TABLE_B11
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 32400;
  static const int LINKS_MIN_CN = 6;
  static const int LINKS_MAX_CN = 8;
  static const int LINKS_TOTAL = 255599;
  static const int DEG_MAX = 18;
  static constexpr int DEG[] = {
    18, 3, 16, 9, 6, 0
  };
  static constexpr int LEN[] = {
    10, 65, 5, 5, 5, 0
  };
  static constexpr int POS[] = {
    708,   1132,  2120,  3208,  3879,  8320,  11948, 14185, 15214, 16594, 17849, 19766, 23814, 26175, 27579, 28052, 31512, 32029,
    2720,  2753,  3716,  6133,  8020,  8305,  9429,  10337, 15503, 19905, 20127, 21963, 25624, 27221, 27907, 27945, 29833, 30270,
    4011,  7807,  11547, 12782, 13040, 14599, 14836, 15218, 17890, 18922, 19668, 20267, 20714, 22151, 24373, 25261, 26101, 27627,
    136,   5341,  7661,  12779, 13392, 13922, 14151, 15054, 16544, 17232, 17478, 19895, 22814, 23820, 25014, 26346, 27575, 31803,
    3456,  3485,  5839,  8312,  8423,  9796,  10018, 11520, 13336, 15520, 19928, 22019, 23144, 25339, 27406, 28729, 29527, 31406,
    1779,  3634,  3930,  4138,  5449,  5924,  6776,  7658,  8703,  11542, 13133, 15086, 16334, 21876, 23860, 24281, 28854, 29557,
    697,   868,   1345,  6257,  7400,  8220,  9761,  11501, 15828, 16175, 16865, 17251, 19298, 21907, 24033, 24175, 24497, 30965,
    991,   1845,  3173,  5609,  11275, 12666, 12903, 14409, 15359, 17537, 17923, 19821, 20323, 21561, 21663, 23378, 25371, 28487,
    446,   3096,  3604,  3688,  6864,  7296,  8128,  9957,  11568, 13204, 14502, 16280, 17655, 19695, 25953, 28006, 31006, 31160,
    3592,  5443,  5450,  8875,  10529, 10721, 15241, 16485, 16905, 17980, 19685, 21639, 21938, 25038, 25322, 26073, 27072, 32305,
    2539,  11274, 18981,
    8099,  17427, 18613,
    7872,  12547, 14776,
    17272, 31146, 31476,
    12171, 20651, 28060,
    5845,  20532, 24021,
    2102,  9627,  12746,
    4406,  13397, 16767,
    7707,  19199, 20221,
    10278, 11526, 13739,
    8902,  13337, 25524,
    5385,  6939,  15968,
    1686,  2985,  18124,
    21285, 22673, 25722,
    4833,  4895,  7657,
    14073, 19518, 27540,
    2832,  27137, 32072,
    8936,  19641, 24786,
    1696,  4408,  9480,
    3941,  9228,  25043,
    1328,  7460,  25237,
    11251, 21361, 23890,
    10450, 10762, 26795,
    1544,  19244, 22553,
    9564,  24279, 27073,
    12214, 15608, 30892,
    6316,  29252, 30504,
    3667,  7784,  26600,
    11435, 20278, 31840,
    7211,  21620, 23899,
    17193, 18368, 23536,
    3750,  18865, 29121,
    2088,  7648,  20893,
    12231, 28534, 28933,
    6316,  14450, 31885,
    2672,  8770,  26920,
    17337, 18624, 26359,
    3098,  17939, 27134,
    1084,  24695, 31846,
    5110,  9148,  10203,
    3943,  19212, 31745,
    6,     6061,  29453,
    2271,  9151,  27015,
    386,   2747,  26182,
    13129, 15456, 30698,
    126,   10222, 23935,
    11008, 17244, 19711,
    17752, 22103, 31308,
    11293, 20670, 23940,
    11627, 14829, 19929,
    2163,  5918,  23666,
    28627, 28709, 31369,
    3161,  3209,  26227,
    1597,  25322, 30792,
    2083,  15971, 16193,
    4795,  10875, 11668,
    12984, 28077, 28863,
    1851,  9750,  30222,
    2252,  8660,  8984,
    6764,  8375,  15896,
    5074,  16399, 31725,
    11507, 15519, 30828,
    3196,  7975,  17614,
    477,   11889, 17748,
    2420,  2852,  25451,
    3683,  4741,  6944,  8199,  8808,  13142, 14071, 15830, 17216, 18589, 20284, 21652, 22542, 24994, 25719, 26187,
    1534,  4620,  4899,  6461,  6737,  9082,  10679, 11544, 16118, 20173, 20662, 21526, 22646, 24778, 29790, 30044,
    2343,  2547,  5620,  6523,  8188,  9029,  14162, 15517, 24143, 25078, 25163, 26616, 28731, 30201, 30889, 32034,
    1876,  4541,  5147,  9087,  12853, 12967, 13887, 16009, 19722, 20475, 21245, 21908, 22547, 25790, 27330, 27640,
    1706,  3168,  6471,  7382,  10195, 11568, 11770, 17719, 19484, 19572, 20375, 20470, 23332, 24372, 30654, 31230,
    996,   3297,  3587,  4313,  12243, 12456, 17510, 20492, 29071,
    7114,  7312,  7937,  8379,  8580,  11514, 13657, 23774, 24569,
    98,    600,   745,   1223,  4298,  6362,  12544, 21620, 28786,
    2585,  4021,  10785, 11294, 20707, 25033, 25465, 26990, 30713,
    1895,  4346,  10404, 16998, 17073, 24131, 24891, 26056, 26444,
    4265,  8641,  8937,  13462, 23815, 26205,
    1468,  2578,  3070,  6258,  8221,  10414,
    5186,  8832,  11589, 25697, 29629, 32022,
    15971, 17493, 18659, 19082, 22089, 26779,
    1597,  1691,  10499, 13815, 18943, 27396
  };
};

struct DVB_S2X_TABLE_B12
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 34560;
  static const int LINKS_MIN_CN = 7;
  static const int LINKS_MAX_CN = 9;
  static const int LINKS_TOTAL = 266759;
  static const int DEG_MAX = 20;
  static constexpr int DEG[] = {
    20, 14, 12, 3, 4, 3, 0
  };
  static constexpr int LEN[] = {
    10, 5, 5, 55, 15, 6, 0
  };
  static constexpr int POS[] = {
    551,   1039,  1564,  1910,  3126,  4986,  5636,  5661,  7079,  9384,  9971,  10460, 11259, 14150, 14389, 14568, 14681, 21772, 27818, 28671,
    384,   1734,  1993,  3890,  4594,  6655,  7483,  8508,  8573,  8720,  10388, 15541, 17306, 18411, 18606, 19048, 19273, 21492, 21970, 29495,
    1104,  2877,  10668, 11101, 12647, 13994, 14598, 15653, 17265, 18435, 18848, 18949, 19209, 19312, 19414, 19465, 24927, 26613, 28809, 28865,
    1185,  6439,  6519,  7790,  8609,  8826,  9934,  16363, 16596, 18088, 18757, 20318, 20446, 21123, 23938, 24235, 25120, 25469, 26036, 28043,
    53,    3630,  4384,  4619,  7805,  8822,  12208, 13312, 14269, 16435, 17925, 18079, 18689, 19042, 21541, 22729, 26066, 27666, 28759, 30107,
    1926,  2549,  9816,  10544, 10980, 12468, 13026, 15658, 15670, 15975, 17200, 22364, 22775, 23343, 24289, 24956, 26230, 28040, 28348, 29718,
    1243,  1673,  4181,  6080,  7829,  8259,  9294,  10556, 10905, 14071, 18080, 18203, 18537, 19707, 24143, 24442, 25877, 27072, 29158, 29690,
    1834,  2523,  5973,  6006,  8054,  8843,  10141, 11668, 12956, 13202, 18659, 21757, 24258, 24675, 24779, 25924, 26980, 27008, 29229, 29899,
    3790,  5716,  7345,  7381,  9081,  9679,  13629, 18038, 19858, 21248, 21348, 22251, 24410, 26790, 27643, 27955, 27996, 28271, 29638, 30198,
    158,   545,   1178,  5181,  8585,  9927,  10721, 11361, 11640, 12552, 12579, 14641, 14928, 15609, 17388, 20551, 24516, 26834, 29850, 30201,
    1076,  3011,  5636,  6947,  7236,  7511,  10629, 16795, 20770, 22796, 22853, 24219, 28516, 29151,
    678,   2553,  3403,  6515,  7079,  8767,  10228, 10791, 10832, 16113, 18718, 21328, 25762, 26925,
    8536,  8706,  9471,  9854,  11186, 12220, 13261, 14034, 14897, 25068, 26338, 26427, 28784, 29281,
    2634,  3562,  11652, 13206, 15185, 17051, 19666, 21058, 23107, 23869, 24590, 25311, 25498, 28891,
    2440,  4164,  7040,  7591,  9321,  9796,  11026, 12204, 13478, 17852, 22183, 25257, 28756, 28813,
    390,   2209,  3095,  4554,  5755,  12285, 12314, 14372, 14957, 15711, 22946, 27713,
    207,   418,   3856,  8719,  11708, 15353, 20717, 21639, 23269, 26732, 27613, 28334,
    2805,  3795,  7105,  8130,  10330, 13888, 15132, 16415, 17093, 23277, 25191, 27630,
    1381,  1955,  3689,  6290,  6342,  9573,  13751, 14633, 16298, 18206, 24248, 24893,
    5991,  8976,  9763,  12308, 12566, 15265, 17068, 21084, 22475, 24371, 25171, 28008,
    8962,  17060, 22260,
    1335,  6691,  14738,
    4118,  10315, 23113,
    4643,  10970, 28091,
    1195,  3683,  26231,
    7486,  17403, 22471,
    7328,  10110, 19764,
    4630,  13732, 28298,
    6139,  19386, 26009,
    19712, 20670, 27993,
    9317,  18037, 19324,
    4422,  4510,  10290,
    1396,  22324, 28601,
    1404,  5708,  22352,
    14783, 17214, 19359,
    7996,  20133, 20614,
    6219,  17582, 24404,
    2481,  20505, 28124,
    4569,  10863, 28776,
    941,   7516,  11474,
    24878, 27151, 28125,
    9053,  10186, 28126,
    1376,  19327, 26055,
    5965,  14239, 16875,
    17434, 18098, 25044,
    5785,  14385, 22243,
    15144, 16981, 25171,
    13671, 25732, 25961,
    95,    20461, 20558,
    4321,  19572, 26175,
    3637,  17351, 18683,
    18096, 23926, 27359,
    7914,  13217, 23098,
    9822,  11342, 26728,
    7591,  18615, 28044,
    112,   16897, 19127,
    10087, 18516, 27292,
    2762,  3323,  21677,
    4533,  20728, 23071,
    7164,  7180,  15097,
    2061,  6184,  20598,
    6310,  13462, 26714,
    8189,  9345,  15315,
    3319,  17370, 24726,
    5217,  9271,  11984,
    10245, 13623, 16621,
    5537,  22325, 22692,
    1592,  19859, 25578,
    7005,  15625, 22572,
    1474,  14387, 28592,
    153,   4254,  20080,
    8709,  25107, 25135,
    11663, 17264, 25598,
    7135,  17226, 18698,
    109,   2603,  26360,
    1408,  9608,  11927, 14872,
    4187,  10410, 27776, 30125,
    1008,  4409,  14938, 20458,
    3641,  12480, 20141, 22605,
    10453, 22378, 24993, 30002,
    19957, 24800, 25196, 29823,
    2431,  2929,  5824,  12333,
    395,   4393,  15571, 22019,
    15040, 24712, 28275, 28794,
    3735,  11988, 13828, 13949,
    4301,  5829,  24675, 26696,
    6406,  8140,  9438,  17274,
    4272,  17638, 19278, 24961,
    3271,  11030, 11481, 28624,
    3792,  5394,  7566,  17557,
    6505,  11108, 20811,
    2722,  28613, 28991,
    11472, 25911, 30170,
    2355,  13553, 25490,
    3284,  13373, 17330,
    9046,  16513, 22243
  };
};

struct DVB_S2X_TABLE_B13
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 36000;
  static const int LINKS_MIN_CN = 7;
  static const int LINKS_MAX_CN = 9;
  static const int LINKS_TOTAL = 255599;
  static const int DEG_MAX = 16;
  static constexpr int DEG[] = {
    16, 3, 15, 10, 8, 3, 0
  };
  static constexpr int LEN[] = {
    10, 65, 5, 5, 5, 10, 0
  };
  static constexpr int POS[] = {
    690,   1366,  2591,  2859,  4224,  5842,  7310,  8181,  12432, 15667, 15717, 16935, 17583, 19696, 20573, 21269,
    2488,  2890,  6630,  6892,  11563, 12518, 15560, 16798, 18355, 18746, 19165, 19295, 21567, 23505, 23617, 23629,
    321,   2844,  2894,  3986,  4538,  7599,  7816,  9831,  10247, 11556, 16068, 17249, 18194, 23043, 23100, 25938,
    2503,  2827,  4771,  5929,  6400,  7437,  8054,  10897, 11633, 14404, 16133, 17101, 24425, 24973, 25086, 25802,
    1462,  2099,  3910,  5131,  5352,  8832,  9495,  9624,  10796, 12906, 13903, 14724, 14946, 17708, 21034, 26612,
    260,   523,   1427,  3435,  4517,  9494,  12594, 12688, 12726, 14163, 16537, 17424, 18424, 20321, 25101, 28269,
    2131,  2407,  4820,  7167,  11783, 15249, 15982, 18761, 22162, 24593, 24971, 25831, 26351, 27005, 28348, 28793,
    2089,  5829,  6119,  7505,  7758,  8122,  9870,  12107, 16656, 17582, 19115, 23738, 27646, 27692, 27862, 28356,
    2714,  3288,  3337,  5013,  6210,  8080,  9348,  12919, 13458, 13621, 18015, 21024, 24044, 24761, 25610, 26317,
    1305,  3402,  5830,  7095,  8852,  9580,  9793,  11157, 12725, 14355, 20659, 21400, 22289, 23823, 26250, 27006,
    12936, 15702, 23593,
    3616,  17219, 18621,
    1234,  12759, 26749,
    396,   3017,  18360,
    10664, 21597, 26165,
    12986, 14553, 24818,
    18403, 21213, 28302,
    6515,  18331, 19413,
    19649, 26219, 27483,
    2538,  15793, 17528,
    7871,  9374,  20785,
    5494,  8084,  21558,
    6691,  7770,  14306,
    3247,  7417,  18827,
    11615, 15987, 20081,
    1527,  15272, 26042,
    10540, 15548, 23849,
    223,   2601,  25888,
    2395,  21465, 28501,
    19703, 21589, 27252,
    12832, 15726, 25300,
    3750,  10030, 16124,
    401,   6474,  28682,
    4424,  19876, 25563,
    590,   12853, 14779,
    25185, 25539, 25920,
    6857,  23329, 25764,
    3467,  23205, 23751,
    9278,  24364, 25033,
    14956, 19104, 22487,
    21856, 26163, 27130,
    2067,  17357, 22055,
    50,    14414, 19142,
    306,   445,   16437,
    2260,  13892, 17608,
    8893,  12230, 16916,
    5943,  8921,  16380,
    5079,  15385, 21951,
    5108,  6038,  8513,
    2126,  6749,  7330,
    3814,  11941, 22949,
    2301,  15416, 26731,
    3498,  14463, 20417,
    2062,  10434, 10746,
    18128, 18960, 23452,
    13080, 13129, 27193,
    18818, 24995, 27087,
    7198,  11948, 23135,
    17206, 18524, 25811,
    5202,  10030, 10076,
    8497,  23410, 23991,
    1553,  1968,  13135,
    4426,  10786, 23259,
    92,    7941,  23045,
    6356,  14028, 23104,
    18319, 20286, 22984,
    5778,  25335, 26191,
    662,   15922, 27478,
    2920,  9733,  18974,
    15337, 27509, 27519,
    8659,  25028, 27723,
    14865, 24564, 26361,
    1383,  21234, 21770,
    10767, 25752, 25843,
    7717,  14536, 24248,
    278,   2803,  2966,  3547,  4128,  4829,  4981,  6699,  6716,  14183, 14239, 15939, 16996, 19694, 20073,
    3022,  3087,  10039, 10174, 11403, 12146, 13689, 14934, 17765, 18121, 18936, 21818, 27202, 27532, 28192,
    817,   3888,  4102,  9441,  10165, 10941, 18131, 20028, 22305, 23832, 25225, 26228, 27208, 27245, 27390,
    6346,  7992,  9053,  11187, 12124, 16435, 16850, 21269, 21580, 22096, 23700, 24751, 26264, 27318, 27576,
    1440,  3291,  5755,  12247, 12272, 15394, 15659, 15764, 16338, 17373, 18840, 19597, 19812, 22415, 27062,
    937,   3118,  8745,  10933, 12703, 13906, 14113, 21442, 21539, 28140,
    247,   2465,  2918,  3189,  5886,  11451, 16862, 17458, 20908, 26608,
    58,    10104, 11815, 14429, 16531, 19797, 24071, 26021, 28000, 28489,
    4367,  5710,  7855,  14203, 18071, 19336, 19880, 20166, 26774, 28554,
    191,   1085,  4068,  7452,  11739, 15962, 17501, 19172, 24130, 28476,
    4961,  19716, 19964, 23479, 24004, 24340, 25537, 27930,
    1688,  2235,  10464, 15112, 15134, 25143, 25910, 28689,
    765,   11839, 17427, 19754, 21445, 22034, 23493, 25296,
    277,   7947,  9952,  12228, 12595, 16563, 19758, 21721,
    1575,  2652,  5226,  8159,  16624, 25446, 26417, 26722,
    10571, 17389, 22602,
    1331,  7875,  18475,
    11738, 13853, 23914,
    9412,  11361, 26507,
    16877, 23022, 27060,
    2627,  16649, 22369,
    9446,  14752, 28540,
    4496,  7705,  22247,
    2439,  19741, 28550,
    6605,  12623, 26774
  };
};

struct DVB_S2X_TABLE_B14
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 37440;
  static const int LINKS_MIN_CN = 8;
  static const int LINKS_MAX_CN = 10;
  static const int LINKS_TOTAL = 271439;
  static const int DEG_MAX = 18;
  static constexpr int DEG[] = {
    18, 14, 3, 7, 4, 3, 0
  };
  static constexpr int LEN[] = {
    10, 10, 55, 5, 10, 14, 0
  };
  static constexpr int POS[] = {
    2087,  6318,  7314,  8327,  9453,  12989, 13156, 13763, 13819, 16963, 18495, 19352, 20510, 20651, 23379, 23847, 23953, 26469,
    2680,  5652,  6816,  7854,  10673, 11431, 12379, 14570, 17081, 19341, 20749, 21056, 22990, 23012, 24902, 25547, 26718, 27284,
    2142,  3940,  4724,  4791,  6617,  6800,  9349,  9380,  10073, 10147, 11750, 12900, 16044, 16156, 17769, 21600, 21669, 22554,
    1588,  3097,  4277,  6181,  6737,  8974,  9793,  12215, 12814, 17953, 18270, 21808, 22625, 24390, 25429, 25750, 25967, 26391,
    561,   5825,  7106,  7166,  7475,  11844, 12905, 13559, 13978, 14176, 14437, 16070, 16587, 19792, 20187, 23754, 26070, 27232,
    673,   1783,  4046,  4887,  5596,  8390,  9229,  12315, 14252, 14415, 14529, 17837, 20013, 20032, 22201, 22487, 24412, 25792,
    1261,  1910,  3767,  6244,  7050,  7367,  9230,  12972, 13229, 13472, 14287, 14494, 16776, 20523, 20738, 21591, 23622, 25206,
    1618,  2106,  3640,  6304,  7984,  8158,  9072,  9311,  12618, 15746, 16985, 18923, 20959, 21267, 23375, 24052, 24260, 24827,
    6256,  6931,  7276,  7356,  7832,  12284, 12405, 13083, 13602, 14750, 19021, 20026, 22661, 23283, 24427, 25301, 25982, 27279,
    2432,  3076,  3399,  5305,  7370,  8406,  8826,  9237,  10537, 15492, 15606, 15619, 16515, 17562, 19550, 22525, 24389, 25740,
    157,   296,   422,   467,   7125,  9849,  9997,  15376, 15506, 16119, 17153, 17857, 18639, 23136,
    1275,  1439,  6162,  8258,  9031,  10207, 10472, 16004, 16641, 17140, 21342, 22191, 23200, 25753,
    110,   1073,  6460,  9208,  10520, 15833, 15951, 17494, 18614, 19970, 20537, 21512, 21796, 22135,
    3771,  5399,  5885,  7905,  8302,  8614,  10205, 11133, 11459, 16044, 22701, 25170, 26255, 27086,
    1597,  2640,  2741,  3790,  5107,  7470,  9160,  12078, 12350, 14020, 18877, 19507, 22658, 24290,
    4957,  5961,  6263,  8201,  8579,  9392,  10133, 11712, 14757, 15678, 15718, 19528, 25107, 25122,
    870,   4508,  5944,  7360,  11724, 15003, 16387, 19543, 19893, 20189, 21942, 23740, 25686, 25849,
    131,   2044,  6731,  7619,  7787,  9109,  9841,  10006, 10275, 13687, 16522, 18212, 24457, 25197,
    504,   1863,  4246,  5075,  5448,  6296,  6930,  11792, 13736, 14588, 16340, 17102, 17807, 26621,
    1137,  1168,  2366,  3818,  4311,  6806,  8583,  10850, 12198, 12357, 21357, 23243, 23568, 25003,
    2353,  11886, 22548,
    1680,  9112,  12175,
    15126, 16642, 27030,
    5571,  5719,  19190,
    6232,  13413, 19728,
    8197,  12068, 17122,
    3220,  3476,  24534,
    1630,  4345,  23890,
    19815, 20676, 24443,
    12761, 14080, 15937,
    41,    7963,  23895,
    7658,  13020, 27128,
    1017,  1447,  3285,
    2268,  22921, 26255,
    261,   13889, 14175,
    13925, 18777, 18987,
    15136, 24523, 27156,
    12008, 18484, 19299,
    4304,  9857,  15134,
    2966,  9278,  9737,
    5469,  15449, 22259,
    11359, 14186, 20635,
    16453, 21262, 23629,
    5613,  7100,  11104,
    3959,  14714, 18116,
    7465,  13803, 24660,
    3939,  7615,  9891,
    12249, 16491, 22373,
    8734,  14253, 25616,
    5781,  18719, 23894,
    6208,  6703,  14626,
    1284,  4730,  23920,
    3920,  13167, 13366,
    3925,  7147,  27268,
    1926,  12777, 21092,
    675,   8186,  22557,
    487,   9590,  12433,
    7090,  16031, 27037,
    3083,  10445, 22950,
    380,   4663,  7195,
    960,   12754, 20597,
    1790,  12476, 24250,
    11307, 22121, 22266,
    3256,  7625,  12046,
    11034, 11800, 17383,
    6142,  14781, 19944,
    2679,  11106, 22783,
    7769,  11060, 15178,
    7384,  9851,  20205,
    14813, 19271, 22600,
    3085,  11637, 19934,
    6518,  7995,  19382,
    11070, 15498, 26380,
    248,   16291, 23824,
    4989,  19436, 26642,
    5954,  16039, 16042, 20349, 21326, 24656, 25427,
    2558,  6628,  9167,  16825, 19069, 20808, 22617,
    317,   13859, 14069, 16104, 18835, 20040, 26633,
    2866,  4153,  5875,  11698, 15287, 19719, 25808,
    536,   6955,  9735,  16098, 20694, 24675, 26881,
    25,    7316,  9961,  21037,
    7823,  19458, 20404, 25186,
    7142,  11057, 17748, 24788,
    11315, 12358, 21583, 21836,
    8995,  9326,  12826, 25981,
    2281,  10560, 10674, 19801,
    5001,  6655,  26231, 26542,
    800,   15131, 18482, 22621,
    9060,  12257, 24786, 25188,
    3462,  17201, 18960, 24462,
    17631, 26360, 26425,
    12774, 20967, 21391,
    14701, 20696, 26807,
    5931,  13144, 14022,
    128,   16460, 26300,
    801,   9487,  25937,
    6153,  11296, 23054,
    2749,  14434, 20049,
    1732,  7646,  20402,
    3839,  11031, 26022,
    2159,  20918, 21407,
    285,   13785, 24234,
    1977,  3899,  7972,
    4120,  19101, 23719
  };
};

struct DVB_S2X_TABLE_B15
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 41760;
  static const int LINKS_MIN_CN = 10;
  static const int LINKS_MAX_CN = 12;
  static const int LINKS_TOTAL = 270359;
  static const int DEG_MAX = 18;
  static constexpr int DEG[] = {
    18, 12, 10, 3, 10, 4, 3, 0
  };
  static constexpr int LEN[] = {
    10, 5, 5, 55, 5, 10, 26, 0
  };
  static constexpr int POS[] = {
    3880,  4377,  6147,  6219,  7873,  8180,  9157,  10311, 10862, 15393, 16522, 17318, 17609, 18398, 19290, 19293, 20296, 22244,
    1056,  1647,  5119,  5201,  6991,  10038, 10843, 11614, 11901, 12026, 14631, 16749, 16772, 16915, 17331, 19235, 19877, 22763,
    501,   2634,  2812,  3085,  3242,  4952,  5087,  8334,  8838,  8993,  12601, 12849, 13142, 13852, 14416, 14444, 15122, 20692,
    343,   1183,  5708,  6798,  6951,  9154,  9160,  9508,  9884,  11874, 11984, 13737, 14933, 17208, 21253, 21822, 22723, 22898,
    3332,  4384,  5137,  8527,  8749,  10414, 10536, 12759, 14769, 16121, 19255, 19326, 20283, 20352, 20629, 20827, 21226, 22087,
    60,    3866,  3895,  4116,  5631,  6985,  7205,  7681,  10031, 12825, 14266, 14644, 16396, 17010, 20221, 20268, 21729, 21752,
    61,    1112,  1392,  1826,  1910,  4370,  5910,  6660,  6943,  7859,  9628,  10213, 10701, 12615, 14453, 17123, 18667, 20688,
    880,   2397,  2669,  7767,  9683,  9705,  10430, 13995, 15972, 16217, 17187, 18246, 18869, 21077, 21884, 21897, 21927, 22475,
    748,   1029,  1446,  2912,  6784,  6926,  7710,  11674, 12014, 12409, 12438, 14411, 14723, 15953, 16020, 17496, 18230, 19547,
    1151,  2295,  2975,  3082,  6502,  8269,  9130,  9629,  10018, 10235, 14871, 15834, 17396, 17777, 19138, 21871, 22035, 22927,
    650,   789,   4632,  4777,  5004,  8796,  13479, 14917, 16826, 16926, 19144, 20754,
    1693,  4906,  5777,  5907,  6472,  9792,  11824, 16134, 16406, 16440, 18395, 22338,
    5172,  5920,  7987,  9381,  10573, 11382, 11512, 13074, 15755, 16591, 19518, 20968,
    1409,  2508,  6542,  8993,  10660, 13691, 14724, 15597, 19641, 20809, 21160, 22767,
    895,   1446,  3298,  4018,  5250,  6269,  8897,  9049,  12052, 15311, 16199, 20677,
    1,     774,   1248,  2362,  7019,  8430,  14321, 14444, 19664, 21475,
    1714,  1973,  4155,  7536,  7975,  9323,  9997,  10627, 20959, 21824,
    586,   1907,  2153,  5914,  7407,  8311,  8900,  10060, 18502, 18818,
    805,   1897,  3019,  7404,  10055, 11779, 11982, 15319, 21802, 21913,
    5276,  5470,  8725,  11080, 11939, 17057, 17960, 18930, 19814, 22546,
    1227,  10140, 18999,
    849,   17266, 18364,
    4436,  6167,  14942,
    11103, 14219, 19204,
    6738,  10043, 20614,
    1885,  3173,  13934,
    2088,  11344, 20627,
    2668,  6722,  20336,
    11274, 18439, 21280,
    2223,  15960, 21282,
    6555,  7521,  11051,
    9037,  11912, 22911,
    12952, 19885, 21298,
    13696, 16793, 17228,
    1040,  4501,  6170,
    1025,  4522,  21287,
    1213,  3817,  12857,
    1392,  6601,  12468,
    835,   16504, 19633,
    634,   16014, 19619,
    6166,  17343, 21067,
    6583,  16107, 18382,
    5481,  9653,  18543,
    14634, 15406, 16179,
    1952,  7810,  16892,
    2271,  12635, 20456,
    8838,  10469, 20629,
    11400, 16788, 18756,
    230,   11373, 17104,
    17204, 17733, 20707,
    8465,  13092, 22087,
    8684,  8983,  10130,
    11468, 13469, 21366,
    9342,  10115, 19130,
    3184,  9535,  11802,
    13495, 16231, 19609,
    8911,  12617, 15190,
    508,   8674,  19422,
    4059,  6197,  8449,
    9440,  11028, 13468,
    1779,  9358,  13459,
    46,    7370,  15160,
    12118, 17458, 21853,
    320,   4449,  20048,
    12300, 14502, 21803,
    9019,  19417, 22280,
    1320,  6434,  7916,
    6850,  10275, 17099,
    301,   5637,  7309,
    8443,  13673, 16625,
    4943,  15268, 20252,
    13246, 17809, 18271,
    3230,  8329,  12330,
    1398,  7959,  18145,
    274,   10500, 12954,
    1326,  2057,  5453,  6588,  11514, 11920, 13687, 14692, 17684, 22043,
    3921,  7217,  8693,  10943, 11769, 12121, 12618, 19781, 19932, 20083,
    2166,  5206,  5482,  11453, 13986, 16877, 18184, 18822, 21663, 22611,
    858,   11727, 13116, 14705, 15517, 16109, 17086, 18439, 19047, 20321,
    216,   414,   726,   2616,  6948,  7028,  8288,  12164, 21697, 22606,
    7441,  14498, 15308, 17321,
    1455,  6627,  10112, 13652,
    7448,  7945,  17043, 21758,
    2947,  7933,  11624, 14237,
    514,   4014,  20406, 22226,
    4454,  9815,  11696, 13946,
    7787,  11797, 13113, 15796,
    2363,  4379,  21733, 22277,
    8437,  16504, 16527, 21350,
    8932,  14444, 15680, 19635,
    1273,  11365, 15443,
    3533,  11221, 13249,
    687,   1927,  14403,
    3914,  4221,  8791,
    12479, 15622, 17384,
    14928, 20923, 22283,
    7729,  13750, 15716,
    88,    12409, 19522,
    6852,  16166, 21884,
    1204,  12049, 16487,
    11178, 11226, 15971,
    6382,  14361, 16863,
    10195, 10247, 18188,
    1819,  5834,  8434,
    286,   3333,  21431,
    13950, 15188, 17771,
    10198, 14887, 16751,
    13811, 18307, 18337,
    1210,  18076, 21869,
    5717,  8482,  11896,
    6501,  15625, 17792,
    3965,  4494,  20272,
    1589,  9900,  14472,
    288,   9421,  12009,
    2177,  4626,  16605,
    710,   4696,  18127
  };
};

struct DVB_S2X_TABLE_B16
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 44640;
  static const int LINKS_MIN_CN = 12;
  static const int LINKS_MAX_CN = 14;
  static const int LINKS_TOTAL = 273239;
  static const int DEG_MAX = 16;
  static constexpr int DEG[] = {
    16, 3, 13, 12, 3, 0
  };
  static constexpr int LEN[] = {
    10, 65, 10, 5, 34, 0
  };
  static constexpr int POS[] = {
    1083,  2862,  3815,  4075,  5519,  8003,  9308,  10029, 12476, 12949, 13759, 13918, 14303, 15028, 19737, 19953,
    392,   3781,  6086,  8378,  9952,  10531, 11369, 11954, 14808, 14948, 16585, 16682, 18445, 18960, 19085, 19423,
    3023,  3727,  4797,  5104,  5342,  5994,  8138,  9758,  10146, 11758, 14763, 15300, 15640, 17947, 18690, 18864,
    854,   1259,  2147,  3861,  4258,  4949,  5555,  5940,  9454,  14940, 15521, 16393, 17029, 18789, 18810, 19840,
    4404,  6608,  7232,  7534,  7721,  8079,  8558,  9851,  11560, 11968, 12678, 13410, 15908, 16918, 18108, 18437,
    519,   1591,  1600,  1964,  7706,  9481,  10789, 11068, 13056, 13373, 13759, 14323, 14999, 15505, 17366, 18254,
    545,   673,   2257,  4060,  4289,  4897,  5287,  7318,  8511,  13835, 14540, 14948, 15475, 16718, 17907, 18067,
    1949,  3426,  3440,  4679,  5103,  8692,  8987,  10075, 10923, 11162, 11625, 12805, 13749, 17487, 17807, 18802,
    858,   1969,  2178,  2421,  2592,  2764,  3504,  7084,  9227,  9349,  9960,  10592, 18149, 18974, 19010, 19891,
    3282,  5061,  5908,  6929,  7551,  7927,  8116,  8388,  11305, 11379, 12527, 13982, 14343, 15064, 16259, 19602,
    3730,  8198,  8789,
    1515,  6545,  9678,
    12411, 14800, 17119,
    1000,  15382, 18607,
    977,   1525,  5376,
    4464,  7676,  8937,
    3684,  6730,  9836,
    10203, 10305, 18629,
    2214,  4904,  10873,
    690,   7077,  12623,
    3094,  11228, 16285,
    2583,  5278,  16500,
    4253,  13495, 14465,
    3323,  17768, 19548,
    7670,  12892, 18704,
    373,   14868, 16337,
    8886,  17314, 17578,
    10636, 12385, 19530,
    5734,  14030, 18619,
    3298,  4905,  10156,
    332,   19282, 19924,
    15,    8741,  16429,
    11482, 14807, 15426,
    6055,  12144, 14026,
    1095,  5737,  10525,
    813,   965,   4520,
    808,   8546,  14057,
    3195,  3814,  14683,
    1184,  17287, 19477,
    12411, 13207, 18549,
    2639,  12198, 15656,
    3527,  5555,  14387,
    5563,  10402, 19122,
    4538,  13134, 18766,
    731,   3368,  5865,
    1253,  2238,  8820,
    2764,  11942, 16705,
    6375,  18789, 19594,
    3387,  11299, 14192,
    2486,  2729,  8580,
    3096,  5778,  10416,
    2513,  10609, 14018,
    2508,  10361, 15415,
    5368,  6612,  17415,
    1998,  5687,  17627,
    2711,  16277, 17350,
    5213,  5820,  9217,
    5744,  17794, 19180,
    9095,  15302, 19441,
    10031, 12094, 18856,
    739,   6709,  11785,
    1496,  10418, 15753,
    9437,  11589, 19552,
    7484,  9656,  12381,
    2371,  7237,  7794,
    748,   7213,  9835,
    1930,  6418,  8683,
    5482,  15289, 18623,
    10791, 15731, 18781,
    3622,  5730,  14230,
    1169,  9420,  19665,
    10170, 13288, 14142,
    3855,  7239,  18843,
    816,   16956, 19454,
    3179,  5689,  16584,
    4313,  6450,  8761,  11594, 13187, 14029, 14509, 14944, 16947, 17850, 18270, 18390, 19934,
    1680,  2214,  3859,  3994,  4276,  6984,  12261, 13869, 14696, 16303, 16467, 16756, 19754,
    433,   1009,  3169,  6825,  7128,  7294,  7327,  8175,  16653, 16657, 17314, 18598, 19472,
    1473,  2110,  2777,  5217,  5902,  7136,  7797,  8650,  9116,  11267, 14270, 15342, 18291,
    349,   2892,  4148,  7493,  10142, 13920, 14079, 14423, 15229, 16255, 16808, 18248, 18362,
    5879,  7078,  7457,  9493,  10771, 11010, 12068, 12994, 13007, 13109, 17983, 19531, 20087,
    483,   804,   993,   1078,  1822,  4646,  4658,  5458,  8116,  8443,  10056, 13440, 18939,
    490,   865,   1309,  4339,  6434,  8210,  9248,  10588, 13094, 16476, 17620, 19378, 19708,
    163,   899,   2396,  4212,  6157,  9036,  11116, 13295, 13928, 15111, 16312, 18369, 19470,
    985,   1298,  3213,  5145,  6917,  7136,  7183,  10870, 11329, 12295, 13466, 14989, 17909,
    89,    582,   812,   1761,  5157,  6039,  7843,  8418,  8747,  11575, 13169, 14160,
    1871,  2701,  3252,  7399,  7646,  9785,  11274, 17041, 17361, 18899, 19430, 19691,
    1328,  2165,  2722,  4120,  4132,  9855,  10802, 14441, 16771, 17679, 18611, 18718,
    1166,  3128,  8585,  9843,  10411, 12365, 14141, 15156, 16987, 17484, 17702, 19204,
    943,   952,   4108,  4832,  6706,  9245,  14304, 16528, 17055, 17698, 18419, 19526,
    1340,  7429,  17768,
    10358, 12400, 16483,
    1070,  4760,  10051,
    6992,  8645,  9886,
    756,   7962,  17532,
    13063, 17794, 18323,
    630,   9881,  20052,
    5786,  7779,  15441,
    5049,  5860,  16575,
    10021, 13811, 20097,
    2167,  6374,  19993,
    1412,  4441,  11765,
    14750, 17242, 18319,
    507,   1756,  18791,
    2277,  6901,  9690,
    14828, 15959, 16658,
    4687,  6452,  16770,
    465,   11415, 13696,
    13370, 15379, 16190,
    2988,  12683, 16796,
    6382,  14227, 14295,
    17221, 18167, 18379,
    9656,  9841,  10968,
    16917, 19014, 19869,
    15255, 15400, 17505,
    6403,  15345, 16248,
    6794,  15772, 18005,
    3252,  12230, 12246,
    9062,  9082,  10245,
    405,   9373,  19195,
    5987,  6006,  6026,
    2865,  2887,  2896,
    14889, 14898, 14924,
    7791,  7800,  7809
  };
};

struct DVB_S2X_TABLE_B17
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 46080;
  static const int LINKS_MIN_CN = 13;
  static const int LINKS_MAX_CN = 15;
  static const int LINKS_TOTAL = 271079;
  static const int DEG_MAX = 15;
  static constexpr int DEG[] = {
    15, 12, 3, 12, 4, 3, 0
  };
  static constexpr int LEN[] = {
    10, 10, 55, 5, 10, 38, 0
  };
  static constexpr int POS[] = {
    790,   1010,  1064,  2157,  2569,  3499,  4637,  4951,  6789,  8177,  9888,  10800, 13254, 13829, 17946,
    597,   693,   862,   900,   4750,  4897,  5410,  5441,  6491,  8815,  11894, 13411, 13696, 14103, 18413,
    903,   2779,  2996,  6100,  7489,  7560,  8637,  8853,  10078, 11372, 12040, 15911, 16944, 17059, 17771,
    1761,  2084,  2099,  2232,  3114,  3120,  7062,  10527, 10823, 11945, 13918, 16359, 17110, 17654, 18370,
    677,   1080,  2329,  5100,  5106,  6261,  6383,  10953, 11968, 12186, 13266, 14451, 16092, 17760, 17871,
    1069,  3672,  5304,  6102,  6907,  8087,  9477,  9654,  11581, 14650, 14808, 14920, 15397, 16179, 18595,
    327,   1161,  2624,  4494,  4516,  5555,  6928,  7455,  7959,  8734,  8933,  9753,  10614, 16263, 17689,
    1922,  1932,  6481,  7488,  7722,  8836,  10326, 10633, 11184, 12555, 13485, 14155, 16373, 17486, 18331,
    1056,  1624,  1991,  3585,  6052,  7838,  10123, 11470, 14228, 15146, 16166, 16390, 17630, 17679, 17774,
    295,   3429,  3587,  4597,  5017,  5105,  5136,  5827,  7308,  8266,  9632,  11612, 14879, 16167, 18491,
    1523,  1615,  3368,  6253,  8510,  9070,  10020, 10368, 10718, 11882, 12014, 15077,
    421,   3234,  4290,  4808,  4983,  9992,  12569, 13331, 14457, 15853, 15943, 18318,
    583,   2081,  4320,  6268,  6284,  9084,  9638,  10941, 13335, 15062, 17310, 17667,
    573,   5180,  5758,  5813,  9655,  9892,  10763, 11209, 11717, 14760, 14972, 16395,
    151,   1917,  4190,  5573,  5629,  6725,  9653,  9974,  10008, 11971, 15132, 18170,
    132,   1270,  3074,  7215,  7878,  8266,  11875, 12274, 13134, 15084, 17120, 17556,
    845,   2417,  2435,  5875,  7758,  7807,  12521, 13907, 16400, 17778, 18260, 18370,
    2848,  4714,  5924,  6507,  7595,  8422,  9281,  13140, 13276, 14589, 15269, 15943,
    278,   931,   1186,  3588,  4072,  6943,  8429,  9348,  9863,  10056, 14376, 15846,
    3480,  3887,  4932,  5888,  10246, 10281, 11065, 11434, 12290, 12345, 12635, 13917,
    528,   14523, 18426,
    4127,  5613,  9647,
    8777,  15790, 18168,
    3491,  5608,  10216,
    5154,  8811,  16363,
    437,   2834,  3470,
    9675,  12773, 17150,
    2456,  7748,  8623,
    3758,  14333, 18097,
    3969,  17136, 18610,
    6745,  13708, 18656,
    6152,  10273, 13316,
    7822,  14888, 15541,
    15501, 16598, 18531,
    2497,  8828,  15453,
    3443,  6899,  7293,
    3721,  13954, 15822,
    719,   13335, 15342,
    1566,  7588,  8362,
    8644,  13389, 17476,
    1513,  8257,  15942,
    2620,  7224,  15557,
    7696,  12178, 17371,
    5285,  8439,  11367,
    4961,  7657,  17125,
    11382, 11542, 16823,
    2429,  7538,  10992,
    680,   7651,  10178,
    6794,  11231, 18328,
    1195,  12837, 15710,
    1156,  17454, 18260,
    6541,  10062, 17436,
    70,    282,   7519,
    608,   1919,  7299,
    3339,  11187, 15788,
    4771,  12599, 13753,
    1822,  4233,  10561,
    5233,  14135, 15888,
    4109,  14837, 18717,
    3011,  15644, 17342,
    10668, 11462, 15065,
    2486,  6822,  7486,
    3851,  6182,  11215,
    595,   11064, 15525,
    9738,  10045, 14128,
    929,   2222,  11949,
    10950, 12273, 15503,
    3672,  6760,  9589,
    3583,  5887,  8907,
    13351, 15134, 17291,
    7770,  9928,  12542,
    268,   10496, 17937,
    1318,  2938,  6971,
    428,   1791,  9729,
    6895,  8896,  10420,
    2946,  4619,  6209,  7377,  7931,  8740,  9223,  12171, 12985, 13795, 14141, 16233,
    217,   958,   995,   3144,  5905,  6178,  6596,  10427, 15013, 15669, 16343, 18465,
    357,   2579,  4550,  5223,  5890,  7642,  7900,  8441,  13416, 17740, 18131, 18679,
    894,   1776,  1852,  3262,  5830,  6008,  7877,  9570,  15838, 16029, 16176, 16583,
    2190,  2698,  3277,  4748,  5575,  6822,  8356,  9692,  11406, 11697, 12991, 15275,
    9695,  12587, 15112, 17987,
    5221,  5710,  15272, 17606,
    3068,  9034,  11853, 17189,
    2503,  7618,  9336,  15768,
    2069,  2258,  7450,  10219,
    778,   8645,  12173, 12429,
    6960,  9073,  12411, 15065,
    3515,  5848,  12776, 15706,
    4725,  5967,  15682, 17350,
    12416, 14871, 16503, 18679,
    4218,  13329, 17613,
    752,   6184,  9180,
    3188,  3971,  11453,
    2580,  17414, 18001,
    10285, 13728, 15896,
    612,   10652, 12442,
    7637,  7724,  15724,
    1427,  15130, 15355,
    77,    5271,  8934,
    3121,  10373, 11930,
    11913, 12253, 15701,
    6582,  9841,  10243,
    11595, 16319, 16332,
    6402,  11107, 14899,
    4593,  5442,  9630,
    1321,  3877,  17467,
    1335,  10771, 12311,
    24,    16695, 18578,
    11396, 17644, 18618,
    7306,  14777, 15287,
    1809,  5769,  10827,
    137,   3555,  5186,
    201,   3340,  10470,
    8954,  12160, 17867,
    6744,  9179,  14780,
    3280,  9637,  17720,
    1867,  10642, 14613,
    4292,  7451,  14924,
    1621,  13335, 16834,
    8289,  14826, 15302,
    3610,  12151, 12159,
    3968,  3976,  5492,
    3491,  14734, 17314,
    3774,  8427,  10437,
    3128,  4202,  15889,
    3821,  9781,  10862,
    8264,  9191,  12337,
    1476,  8123,  8946
  };
};

struct DVB_S2X_TABLE_B18
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 47520;
  static const int LINKS_MIN_CN = 14;
  static const int LINKS_MAX_CN = 16;
  static const int LINKS_TOTAL = 272519;
  static const int DEG_MAX = 15;
  static constexpr int DEG[] = {
    15, 12, 3, 12, 4, 3, 0
  };
  static constexpr int LEN[] = {
    10, 10, 55, 5, 10, 42, 0
  };
  static constexpr int POS[] = {
    214,   632,   923,   3251,  6036,  6570,  8258,  9462,  10399, 11781, 12778, 14807, 15369, 16105, 17153,
    652,   1565,  3710,  3720,  4603,  7139,  7817,  9076,  11532, 13729, 14362, 15379, 15488, 15541, 15777,
    98,    130,   520,   622,   1806,  2474,  3378,  4773,  6896,  7533,  7744,  11317, 11511, 11574, 15853,
    95,    1168,  2985,  4995,  5032,  5310,  6932,  8571,  9181,  9193,  9896,  10257, 12336, 12811, 14754,
    1178,  1969,  2935,  3432,  3628,  4814,  5218,  5676,  6214,  9953,  10421, 11091, 13574, 13772, 15191,
    2356,  7171,  8062,  8652,  8801,  9917,  10037, 10164, 10671, 10975, 13460, 15594, 15936, 16378, 16711,
    1401,  3622,  4493,  5190,  6908,  7193,  9583,  10283, 11832, 12152, 12609, 13343, 13427, 13839, 15591,
    485,   4930,  7552,  7574,  7629,  8514,  10870, 10888, 11614, 11774, 12414, 13159, 15555, 16874, 16897,
    203,   711,   1373,  5149,  7271,  8191,  8523,  9012,  9645,  11391, 13989, 14402, 15572, 16490, 16985,
    1020,  3606,  4996,  5016,  7632,  9959,  11098, 11792, 12649, 12859, 13381, 14579, 16204, 16899, 17030,
    3653,  4711,  4777,  4779,  5203,  8250,  10671, 12130, 12449, 13340, 14148, 14853,
    3209,  4098,  4415,  4777,  5358,  6681,  8049,  9805,  10139, 15608, 15628, 16532,
    37,    279,   2890,  3692,  5680,  7136,  10862, 11442, 13688, 14572, 14978, 16246,
    150,   2430,  2659,  3909,  8619,  9432,  12372, 12720, 13213, 14635, 15316, 15727,
    759,   7725,  8548,  10772, 10897, 11392, 12273, 13619, 14465, 14488, 16191, 17039,
    499,   2346,  4909,  4998,  6555,  10631, 12374, 13539, 13954, 14728, 14768, 16213,
    286,   458,   1072,  1982,  3593,  4541,  5803,  7260,  7681,  10279, 15178, 15701,
    683,   850,   1430,  4534,  4990,  9870,  10385, 10508, 12633, 13516, 14763, 15297,
    1304,  1620,  2788,  4431,  8333,  10080, 11887, 11994, 12430, 12578, 15816, 16317,
    1020,  2376,  3071,  4752,  7844,  12085, 12338, 12790, 13930, 14874, 16264, 16947,
    2917,  14555, 16711,
    7491,  9627,  11576,
    863,   2959,  15686,
    3115,  3698,  4721,
    1992,  6059,  9232,
    6038,  7185,  14573,
    1340,  3421,  3694,
    4609,  8628,  12390,
    2208,  8716,  8858,
    13808, 15922, 16148,
    2249,  11974, 16896,
    5853,  13225, 13788,
    815,   4711,  6530,
    2209,  2468,  14725,
    4410,  5415,  13854,
    6355,  6825,  15280,
    309,   9537,  16469,
    8068,  13746, 14396,
    9323,  10747, 15016,
    6903,  8218,  11652,
    680,   3121,  8534,
    7311,  10942, 15810,
    877,   965,   6600,
    1742,  5760,  12311,
    3137,  4854,  11102,
    2422,  7181,  7657,
    11818, 13570, 15583,
    6318,  13695, 13717,
    3866,  5279,  6931,
    10864, 15450, 15719,
    4540,  7389,  17179,
    4951,  15064, 16397,
    7605,  10323, 11651,
    4137,  6356,  7204,
    5439,  10310, 14018,
    12843, 13416, 14274,
    2804,  9644,  10370,
    11150, 13275, 14293,
    5134,  5240,  11495,
    864,   2151,  13541,
    736,   13561, 17218,
    8287,  13491, 16780,
    5718,  15660, 16593,
    8455,  13981, 15971,
    9467,  14810, 16276,
    2229,  3946,  8111,
    7217,  7241,  12272,
    67,    3678,  5473,
    6684,  10779, 16599,
    9465,  12372, 16208,
    6794,  14248, 16412,
    2214,  10815, 11926,
    3021,  6374,  12487,
    3950,  6042,  9573,
    7939,  11686, 14299,
    350,   3529,  4079,  4256,  5849,  7190,  8860,  10139, 10232, 10819, 11381, 14147,
    317,   992,   2421,  3962,  4699,  6659,  7506,  10225, 10422, 10631, 12471, 17133,
    1042,  1396,  2353,  2995,  3377,  5431,  5872,  6507,  6958,  8392,  10521, 15036,
    2799,  3188,  3338,  4212,  5257,  6667,  7299,  8647,  9365,  9822,  15393, 16334,
    1095,  1357,  1964,  2027,  3439,  5975,  7077,  10182, 11538, 12085, 14873, 15081,
    5063,  15980, 16044, 16895,
    2675,  3343,  8369,  15958,
    186,   10209, 12824, 14269,
    4306,  6720,  10338, 16589,
    2260,  7944,  10926, 16496,
    821,   2379,  3453,  11530,
    818,   3049,  7651,  16046,
    2127,  3717,  10120, 15916,
    3267,  11412, 13437, 15833,
    1386,  7706,  15875, 16377,
    508,   11392, 13620,
    4097,  14269, 15322,
    9921,  12311, 12914,
    7184,  10571, 15214,
    3917,  8952,  11193,
    1241,  11798, 14788,
    10457, 14430, 14892,
    5603,  14302, 16388,
    427,   2770,  6440,
    9317,  10050, 14671,
    3199,  5089,  5353,
    7239,  7411,  13299,
    306,   1674,  14551,
    816,   7484,  12448,
    706,   13444, 15695,
    554,   4597,  9489,
    2104,  6359,  12468,
    9266,  10617, 11381,
    3277,  3793,  6604,
    1731,  1887,  9707,
    885,   5432,  7884,
    1786,  8137,  13590,
    5024,  6886,  16155,
    2777,  7172,  8568,
    3551,  8533,  13805,
    3299,  8732,  15678,
    633,   9789,  14366,
    11345, 14813, 16179,
    1216,  5414,  13845,
    5832,  7474,  10047,
    1074,  3156,  9228,
    4090,  7614,  10391,
    2624,  5520,  13591,
    3462,  12548, 12556,
    2027,  11569, 14106,
    1821,  3571,  8001,
    3979,  7285,  9173,
    11161, 12334, 16935,
    2642,  8811,  8819,
    5359,  11128, 13310,
    200,   6362,  9809,
    1174,  8836,  13549
  };
};

struct DVB_S2X_TABLE_B19
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 48600;
  static const int LINKS_MIN_CN = 15;
  static const int LINKS_MAX_CN = 17;
  static const int LINKS_TOTAL = 264599;
  static const int DEG_MAX = 14;
  static constexpr int DEG[] = {
    14, 11, 3, 11, 4, 3, 0
  };
  static constexpr int LEN[] = {
    10, 10, 55, 5, 10, 45, 0
  };
  static constexpr int POS[] = {
    15,    865,   1308,  2887,  6202,  6440,  7201,  9014,  10015, 10041, 11780, 13602, 14265, 15506,
    1054,  1416,  2903,  3746,  3753,  7608,  9121,  11097, 11761, 12334, 14304, 15284, 15489, 15860,
    388,   942,   2207,  2627,  3453,  6684,  7105,  8633,  9292,  9413,  11574, 11789, 12990, 13558,
    896,   1802,  2100,  4497,  6650,  7324,  7693,  11232, 11848, 12625, 12982, 13238, 13719, 15260,
    2165,  2313,  3930,  5231,  9263,  10942, 12167, 12938, 13529, 13806, 14792, 15118, 15304, 15970,
    286,   951,   1083,  3401,  5527,  10235, 10417, 10717, 12204, 12522, 12974, 13623, 13645, 13721,
    895,   2500,  3051,  4393,  4686,  5972,  7932,  8025,  8731,  9744,  10323, 10619, 14961, 16054,
    1631,  2304,  3149,  3453,  4133,  4459,  5442,  7051,  8622,  10051, 10791, 11566, 12754, 14016,
    2747,  4371,  5647,  5694,  5899,  8378,  8965,  9917,  10472, 12163, 13349, 14760, 15005, 16196,
    1119,  3605,  4141,  4969,  6694,  7031,  7748,  8800,  9268,  9982,  10605, 11768, 12185, 12254,
    2825,  3983,  3991,  6410,  8249,  8457,  8770,  9183,  12028, 12213, 12448,
    604,   1720,  2373,  2919,  7212,  7867,  7967,  8044,  10466, 13860, 14417,
    301,   1594,  5664,  9711,  9763,  10804, 10816, 11681, 11842, 12204, 15041,
    47,    555,   1796,  2032,  3923,  5175,  5204,  7322,  12008, 15192, 15273,
    2564,  2827,  4053,  4355,  5383,  6611,  7951,  10231, 10605, 12712, 15035,
    2243,  3129,  5665,  9703,  9913,  10101, 10212, 13549, 14952, 15661, 15696,
    204,   775,   3771,  5704,  7007,  7095,  7543,  9656,  12426, 12882, 15545,
    4224,  4480,  4521,  5860,  5993,  6200,  6810,  8966,  13588, 13658, 14863,
    528,   2425,  4261,  6534,  9097,  9746,  10277, 11570, 11722, 12614, 14288,
    3612,  4482,  6901,  8453,  8546,  9588,  10302, 11161, 11365, 14894, 15018,
    3742,  10567, 16199,
    7133,  9058,  11953,
    6322,  6923,  15941,
    8088,  9184,  10475,
    677,   2998,  15174,
    4761,  5594,  9851,
    2307,  13609, 15098,
    4022,  12283, 12286,
    5993,  8346,  11208,
    3910,  15175, 15351,
    6964,  10422, 13372,
    6125,  13835, 14292,
    6234,  7466,  8536,
    4785,  6567,  8576,
    6743,  10561, 14130,
    1735,  7324,  11459,
    3414,  5686,  12861,
    5457,  13085, 14942,
    2789,  9707,  10189,
    3474,  7428,  8655,
    393,   4691,  5694,
    4825,  8033,  15186,
    1458,  4367,  5864,
    5843,  11855, 14660,
    7748,  8189,  15486,
    4810,  13665, 13848,
    5818,  6651,  8375,
    693,   5872,  7945,
    172,   11594, 12687,
    5430,  12616, 15658,
    6934,  7909,  11246,
    3637,  12306, 12362,
    3471,  5213,  9689,
    14049, 14507, 15642,
    2668,  3016,  15757,
    3740,  7546,  12925,
    6446,  14217, 15795,
    6834,  12517, 16183,
    6930,  8193,  10296,
    4279,  5463,  11460,
    197,   1031,  3531,
    9722,  9899,  11678,
    2962,  7375,  15462,
    181,   2986,  13487,
    908,   3631,  15042,
    3273,  8070,  10865,
    4099,  6682,  15571,
    2864,  6393,  12639,
    6486,  7891,  14560,
    10819, 11213, 13405,
    71,    6734,  8450,
    3467,  5016,  15956,
    6267,  10180, 15388,
    1625,  2994,  13339,
    2517,  4489,  7357,
    27,    454,   1440,  1874,  2627,  6126,  8518,  9120,  10144, 13007, 13892,
    439,   991,   5000,  5256,  7215,  8109,  8680,  10694, 12078, 13454, 15221,
    1162,  4754,  5101,  5940,  8304,  10907, 11008, 11528, 13514, 13837, 15230,
    1230,  1618,  2547,  2922,  5619,  7415,  12126, 12406, 14378, 15306, 15408,
    140,   446,   3378,  3908,  6904,  7783,  8587,  10420, 10630, 12602, 13597,
    1004,  3374,  7234,  9291,
    8358,  8550,  8658,  11681,
    3669,  7500,  8610,  10360,
    4892,  9971,  11846, 13233,
    329,   1735,  2397,  13639,
    12658, 12789, 14985, 15091,
    8580,  8769,  9451,  15185,
    2383,  3542,  4270,  8840,
    1379,  2256,  2452,  15454,
    1457,  6459,  7332,  12968,
    5323,  7398,  14302,
    6056,  9938,  10618,
    104,   6041,  12225,
    4895,  14004, 14522,
    1942,  6495,  6826,
    8262,  15584, 16179,
    11604, 12644, 12874,
    3538,  9506,  15206,
    666,   6038,  8853,
    5941,  8753,  12449,
    6500,  8604,  16045,
    7937,  12018, 12966,
    8164,  14164, 14528,
    867,   6180,  10192,
    3403,  5208,  10213,
    1752,  7835,  11867,
    1576,  6993,  11260,
    2245,  8237,  14506,
    1284,  1807,  5480,
    9778,  10034, 13115,
    8398,  13975, 15705,
    6906,  7770,  8242,
    1896,  3277,  10631,
    2168,  6889,  8036,
    1616,  6908,  11754,
    11353, 13863, 14389,
    2514,  7212,  12887,
    5661,  6511,  10622,
    4690,  8892,  10754,
    12200, 12486, 14850,
    4663,  15405, 15949,
    302,   309,   1904,
    5265,  7100,  7105,
    4996,  7928,  11084,
    5425,  10367, 15826,
    6766,  8245,  11914,
    8091,  13882, 13887,
    1308,  1348,  7944,
    4730,  10272, 14249,
    5001,  5838,  11633,
    3687,  4732,  15948,
    285,   5437,  10939,
    7254,  10928, 11235,
    2038,  6236,  14036,
    9407,  12341, 16040
  };
};

struct DVB_S2X_TABLE_B20
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 50400;
  static const int LINKS_MIN_CN = 18;
  static const int LINKS_MAX_CN = 20;
  static const int LINKS_TOTAL = 282599;
  static const int DEG_MAX = 15;
  static constexpr int DEG[] = {
    15, 12, 4, 3, 4, 3, 0
  };
  static constexpr int LEN[] = {
    10, 15, 15, 35, 15, 50, 0
  };
  static constexpr int POS[] = {
    66,    862,   939,   3380,  4920,  5225,  5330,  6218,  7204,  7532,  7689,  9135,  9363,  10504, 10694,
    1993,  2656,  4602,  6079,  7569,  7724,  9038,  9647,  9979,  11845, 12641, 12783, 13451, 13661, 14166,
    1360,  2046,  3315,  3423,  3974,  4747,  6535,  6632,  7261,  8173,  8391,  9517,  9928,  11749, 11761,
    3373,  3910,  3965,  4146,  4171,  6195,  6384,  7642,  9337,  9563,  9716,  11490, 12393, 13068, 14016,
    244,   2500,  3970,  6097,  6208,  6669,  7479,  7667,  8234,  9367,  10437, 11623, 12138, 12212, 12454,
    665,   1162,  1990,  2144,  2356,  4400,  6032,  9336,  9988,  10693, 11378, 12021, 12582, 13874, 13958,
    1129,  1527,  1725,  1902,  2039,  2107,  3241,  5991,  6086,  7866,  10793, 11132, 11318, 13636, 14100,
    611,   2141,  2552,  2602,  3049,  3185,  5339,  6495,  7390,  8033,  9068,  10844, 10977, 11047, 13995,
    2805,  4137,  4523,  4841,  7408,  8551,  8667,  8749,  8979,  9232,  9934,  10345, 10634, 11646, 12771,
    144,   1120,  2295,  3469,  4991,  5613,  7186,  7858,  9116,  9328,  10850, 11492, 11837, 12155, 13103,
    803,   1580,  1797,  4719,  6743,  7061,  7753,  8376,  9046,  11635, 11721, 13350,
    1432,  3534,  4822,  6282,  6412,  7180,  7619,  7936,  11278, 12531, 13074, 13084,
    2013,  2575,  2887,  3930,  4725,  5498,  5625,  6209,  6836,  7268,  9062,  10950,
    515,   1037,  2033,  2624,  3044,  6028,  7163,  8729,  8772,  10358, 10659, 12466,
    464,   1685,  2749,  3321,  3778,  5322,  5386,  6294,  7928,  8871,  10278, 13040,
    408,   829,   1672,  2667,  3077,  3545,  3703,  5213,  5381,  7937,  8474,  13126,
    1617,  2490,  2636,  2723,  5431,  6975,  7159,  7900,  10849, 11572, 11887, 12462,
    1402,  2373,  6408,  6656,  6704,  8040,  8841,  9541,  11818, 13891, 14006, 14239,
    1388,  2078,  2136,  3514,  5090,  8083,  8510,  9200,  9814,  11142, 11625, 12980,
    561,   1659,  2611,  3085,  3367,  3804,  6021,  6209,  6348,  8282,  8475,  11386,
    2457,  3223,  4495,  4869,  5314,  5774,  6532,  6552,  8987,  9196,  9199,  11591,
    627,   1069,  3015,  3048,  4275,  4545,  4617,  5606,  6070,  8237,  8659,  8953,
    1028,  4096,  5253,  6370,  8087,  8382,  8950,  8984,  9618,  12843, 13519, 14356,
    560,   604,   663,   2209,  2709,  4421,  6291,  7322,  10054, 11747, 11997, 14192,
    361,   938,   993,   2884,  3386,  9431,  9798,  10155, 11892, 12184, 13140, 13808,
    1045,  5017,  9862,  13620,
    205,   3913,  9136,  13316,
    2994,  4307,  10330, 13439,
    2437,  6367,  9411,  10505,
    5546,  6653,  7663,  12391,
    2825,  3446,  5803,  11254,
    1459,  5988,  7895,  9221,
    3968,  6470,  7739,  12977,
    3298,  4154,  12918, 14282,
    8890,  9389,  10144, 12801,
    2529,  3668,  10005, 11652,
    4558,  8324,  10112, 12696,
    491,   6153,  11815, 12813,
    1300,  12716, 13107, 13847,
    5542,  6160,  11226, 12846,
    5206,  9994,  11133,
    7113,  12088, 12802,
    950,   1805,  8437,
    4411,  10474, 12911,
    3599,  7495,  8984,
    4751,  10097, 10284,
    67,    5056,  11979,
    10633, 10770, 13585,
    1198,  3963,  9181,
    746,   4895,  11288,
    7724,  8170,  9246,
    6542,  8235,  8589,
    1512,  4603,  11098,
    7082,  13053, 13985,
    3887,  9094,  10355,
    3417,  6588,  12800,
    4151,  5569,  8184,
    5557,  8162,  12572,
    2565,  6091,  6359,
    2792,  4430,  6281,
    7936,  10786, 11229,
    677,   3184,  12460,
    2468,  2884,  11714,
    87,    2318,  9383,
    976,   3614,  10292,
    969,   3180,  14286,
    7818,  12285, 13535,
    3471,  10797, 11502,
    3552,  10568, 12836,
    1386,  6971,  13457,
    987,   7598,  9275,
    5039,  13533, 13739,
    1854,  5210,  11261,
    10603, 11753, 12263,
    722,   1030,  12267,
    2720,  5083,  5350,  9274,
    3377,  8717,  9414,  12039,
    1355,  6452,  10130, 13008,
    5112,  7583,  9971,  10955,
    4633,  8781,  12791, 13607,
    1535,  5803,  8062,  10467,
    2326,  8224,  9176,  12082,
    939,   8480,  11823, 13045,
    380,   425,   4943,  10388,
    4001,  4293,  7887,  9858,
    3734,  3746,  9929,  12310,
    1592,  6062,  6419,  10292,
    101,   2538,  6316,  13640,
    3038,  5921,  6156,  6529,
    3820,  10279, 12229, 12404,
    761,   3735,  8874,
    4985,  9636,  14004,
    1744,  2501,  9257,
    3223,  7816,  10249,
    765,   2768,  5363,
    4911,  5948,  13726,
    6745,  9749,  11658,
    1373,  4860,  13952,
    120,   407,   13132,
    862,   2571,  3681,
    3706,  5914,  8019,
    7465,  10479, 12795,
    441,   1017,  1563,
    6638,  8730,  10379,
    3229,  4169,  11333,
    1181,  7252,  11670,
    1090,  4576,  8680,
    943,   9116,  11566,
    3180,  7882,  12535,
    2944,  4411,  12747,
    3153,  5618,  7782,
    428,   2208,  10359,
    447,   6906,  12192,
    8495,  11164, 12870,
    641,   6397,  11868,
    4165,  4534,  11544,
    4594,  7957,  11969,
    3667,  4604,  7920,
    2253,  4617,  13638,
    1099,  4214,  6076,
    461,   8085,  9875,
    8128,  13331, 13740,
    8527,  9749,  12563,
    4216,  6105,  12391,
    6583,  13362, 14130,
    566,   2898,  4772,
    4048,  7696,  8342,
    539,   5111,  9923,
    931,   3789,  7276,
    5306,  13717, 13901,
    1540,  11240, 11353,
    1845,  2752,  6810,
    8553,  10094, 10228,
    9625,  12761, 13252,
    4518,  4526,  9215,
    5394,  6447,  10864,
    7497,  11962, 12795,
    2679,  3310,  3743,
    2764,  10853, 12702,
    6409,  9498,  10387
  };
};

struct DVB_S2X_TABLE_B21
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 55440;
  static const int LINKS_MIN_CN = 28;
  static const int LINKS_MAX_CN = 30;
  static const int LINKS_TOTAL = 273239;
  static const int DEG_MAX = 13;
  static constexpr int DEG[] = {
    13, 12, 3, 12, 5, 3, 0
  };
  static constexpr int LEN[] = {
    10, 5, 60, 10, 5, 64, 0
  };
  static constexpr int POS[] = {
    726,   794,   1587,  2475,  3114,  3917,  4471,  6207,  7451,  8203,  8218,  8583,  8941,
    418,   480,   1320,  1357,  1481,  2323,  3677,  5112,  7038,  7198,  8066,  9260,  9282,
    1506,  2585,  3336,  4543,  4828,  5571,  5954,  6047,  6081,  7691,  8090,  8824,  9153,
    164,   888,   1867,  2685,  2983,  4071,  4848,  4930,  5882,  7085,  7861,  8545,  8689,
    766,   1004,  1143,  1782,  1996,  2506,  2944,  3094,  4085,  5728,  8634,  8737,  8759,
    199,   341,   2068,  2100,  2708,  2896,  4173,  5846,  6167,  7798,  9101,  9159,  9329,
    617,   752,   1647,  2258,  2597,  4708,  5808,  6726,  7293,  7800,  7988,  8825,  9055,
    315,   408,   620,   1256,  1985,  2560,  3226,  5526,  6463,  6927,  7223,  7505,  7669,
    1361,  1528,  2397,  3246,  3370,  4333,  5167,  5333,  7409,  8075,  8197,  8279,  9138,
    665,   876,   2039,  2703,  2864,  3485,  3767,  4823,  7275,  7599,  8274,  8694,  9334,
    1160,  1717,  1750,  2158,  3040,  3506,  3764,  3828,  4261,  4292,  5134,  6789,
    1857,  2119,  2952,  6145,  6897,  7582,  7639,  8032,  8080,  8181,  8221,  8454,
    421,   794,   1511,  2166,  2489,  3936,  4288,  5440,  5901,  6490,  7266,  8858,
    456,   2565,  4071,  4395,  4451,  4888,  5338,  5351,  6608,  7602,  7835,  9024,
    617,   912,   3362,  4085,  4404,  5050,  5244,  6043,  6444,  6721,  7414,  8353,
    4535,  7597,  7853,
    2632,  4652,  6491,
    877,   1378,  8828,
    434,   3309,  8607,
    1075,  2525,  4103,
    958,   2220,  3471,
    2256,  4350,  7277,
    1731,  4306,  8524,
    470,   6369,  9026,
    2605,  3171,  8760,
    1886,  4900,  7558,
    3825,  4488,  9230,
    228,   3806,  8101,
    3607,  7835,  8035,
    5280,  7413,  8380,
    2606,  5100,  5549,
    2309,  4329,  8570,
    1577,  4746,  7473,
    2939,  4664,  7327,
    2440,  8572,  8912,
    4143,  8221,  8561,
    3982,  5316,  7329,
    387,   745,   5199,
    2563,  4077,  9076,
    1428,  6482,  9178,
    4600,  7640,  8483,
    3918,  5239,  5388,
    2006,  6768,  9041,
    5766,  7058,  7758,
    2741,  3382,  5713,
    116,   1863,  4193,
    2339,  4499,  8437,
    1799,  5408,  6711,
    6937,  7536,  8100,
    8313,  8388,  9277,
    1743,  3100,  7424,
    1959,  2130,  2230,
    5007,  6692,  7748,
    808,   3333,  5951,
    1719,  7648,  8645,
    102,   2996,  6153,
    739,   2765,  6496,
    1107,  1760,  7023,
    1067,  2666,  9235,
    1125,  3760,  8095,
    2047,  3143,  6383,
    2698,  3440,  5405,
    1746,  1854,  1965,
    380,   3171,  5816,
    4155,  5210,  9325,
    2290,  2658,  3766,
    167,   6239,  6635,
    1338,  1541,  5813,
    6148,  6574,  7436,
    3598,  3777,  6367,
    731,   4247,  8887,
    2152,  2600,  8950,
    3774,  4099,  6605,
    2819,  3315,  6492,
    1195,  3774,  7126,
    572,   2723,  3467,  3509,  5379,  6756,  6853,  7335,  7973,  8087,  8202,  9000,
    817,   3136,  3533,  3732,  5001,  5148,  5202,  5402,  6602,  7236,  7605,  8275,
    185,   325,   591,   1559,  1635,  2826,  3352,  3634,  3937,  5814,  8101,  8133,
    758,   1780,  1965,  2932,  4010,  4621,  7103,  7249,  7328,  7878,  8754,  8805,
    528,   1433,  2926,  3557,  3780,  4650,  4671,  5253,  5488,  5517,  5681,  8300,
    1172,  2131,  3702,  4455,  4504,  5216,  5920,  6371,  6660,  7953,  9036,  9185,
    639,   1572,  1714,  1957,  3145,  5123,  5330,  5419,  6418,  7163,  7237,  9235,
    166,   486,   708,   1071,  2179,  3700,  4756,  5606,  5685,  6426,  6467,  6902,
    462,   486,   735,   2065,  2558,  3196,  4006,  5344,  5617,  7451,  8141,  8436,
    435,   3016,  4173,  4235,  4415,  4731,  5396,  7340,  8002,  8155,  8279,  9081,
    560,   2200,  2649,  3690,  8636,
    4156,  5971,  7133,  7480,  8218,
    1398,  2219,  3796,  4877,  6376,
    506,   1284,  6906,  7288,  9131,
    643,   1661,  5057,  8011,  8241,
    859,   3508,  5030,
    575,   3942,  6198,
    3472,  5037,  8710,
    3850,  8080,  9216,
    3203,  8128,  8836,
    3059,  5057,  8120,
    3804,  6339,  8539,
    2355,  6922,  8235,
    2035,  2133,  7090,
    4787,  5994,  6966,
    1484,  4897,  7766,
    3977,  7211,  7682,
    3030,  4150,  7823,
    516,   2443,  7919,
    5120,  5715,  6141,
    1551,  6029,  7124,
    1995,  2156,  6952,
    4686,  4944,  8175,
    2763,  4667,  7284,
    3648,  7312,  7361,
    333,   3231,  4029,
    692,   2273,  9068,
    15,    3757,  7203,
    2870,  4217,  8458,
    1501,  2721,  6548,
    92,    2144,  6570,
    1846,  4578,  7972,
    2999,  3542,  4001,
    1658,  8479,  8763,
    4169,  6305,  7774,
    2357,  2475,  8504,
    1418,  1516,  3587,
    2715,  2754,  7789,
    1765,  2387,  8858,
    5115,  8712,  9029,
    160,   2544,  5818,
    1600,  3668,  7706,
    1589,  3143,  7396,
    3310,  3953,  8862,
    2054,  3075,  4821,
    4061,  4355,  6130,
    2086,  2534,  4831,
    4229,  4981,  9057,
    24,    5398,  6062,
    1370,  7446,  8116,
    409,   1199,  6499,
    1088,  1648,  7267,
    176,   8059,  9351,
    558,   3830,  4748,
    4772,  8116,  8277,
    1253,  2418,  3450,
    5305,  5679,  7537,
    437,   561,   7932,
    3058,  4317,  9184,
    382,   1516,  6576,
    471,   6158,  7469,
    5,     955,   2716,
    964,   5239,  8890,
    727,   738,   4868,
    7443,  7560,  7580,
    2075,  2266,  8918,
    4021,  4267,  6797,
    6103,  6111,  8823,
    6523,  6531,  9063
  };
};

struct DVB_S2X_TABLE_B22
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 38880;
  static const int LINKS_MIN_CN = 10;
  static const int LINKS_MAX_CN = 11;
  static const int LINKS_TOTAL = 284759;
  static const int DEG_MAX = 19;
  static constexpr int DEG[] = {
    19, 4, 3, 0
  };
  static constexpr int LEN[] = {
    19, 19, 70, 0
  };
  static constexpr int POS[] = {
    113,   1557,  3316,  5680,  6241,  10407, 13404, 13947, 14040, 14353, 15522, 15698, 16079, 17363, 19374, 19543, 20530, 22833, 24339,
    271,   1361,  6236,  7006,  7307,  7333,  12768, 15441, 15568, 17923, 18341, 20321, 21502, 22023, 23938, 25351, 25590, 25876, 25910,
    73,    605,   872,   4008,  6279,  7653,  10346, 10799, 12482, 12935, 13604, 15909, 16526, 19782, 20506, 22804, 23629, 24859, 25600,
    1445,  1690,  4304,  4851,  8919,  9176,  9252,  13783, 16076, 16675, 17274, 18806, 18882, 20819, 21958, 22451, 23869, 23999, 24177,
    1290,  2337,  5661,  6371,  8996,  10102, 10941, 11360, 12242, 14918, 16808, 20571, 23374, 24046, 25045, 25060, 25662, 25783, 25913,
    28,    42,    1926,  3421,  3503,  8558,  9453,  10168, 15820, 17473, 19571, 19685, 22790, 23336, 23367, 23890, 24061, 25657, 25680,
    0,     1709,  4041,  4932,  5968,  7123,  8430,  9564,  10596, 11026, 14761, 19484, 20762, 20858, 23803, 24016, 24795, 25853, 25863,
    29,    1625,  6500,  6609,  16831, 18517, 18568, 18738, 19387, 20159, 20544, 21603, 21941, 24137, 24269, 24416, 24803, 25154, 25395,
    55,    66,    871,   3700,  11426, 13221, 15001, 16367, 17601, 18380, 22796, 23488, 23938, 25476, 25635, 25678, 25807, 25857, 25872,
    1,     19,    5958,  8548,  8860,  11489, 16845, 18450, 18469, 19496, 20190, 23173, 25262, 25566, 25668, 25679, 25858, 25888, 25915,
    7520,  7690,  8855,  9183,  14654, 16695, 17121, 17854, 18083, 18428, 19633, 20470, 20736, 21720, 22335, 23273, 25083, 25293, 25403,
    48,    58,    410,   1299,  3786,  10668, 18523, 18963, 20864, 22106, 22308, 23033, 23107, 23128, 23990, 24286, 24409, 24595, 25802,
    12,    51,    3894,  6539,  8276,  10885, 11644, 12777, 13427, 14039, 15954, 17078, 19053, 20537, 22863, 24521, 25087, 25463, 25838,
    3509,  8748,  9581,  11509, 15884, 16230, 17583, 19264, 20900, 21001, 21310, 22547, 22756, 22959, 24768, 24814, 25594, 25626, 25880,
    21,    29,    69,    1448,  2386,  4601,  6626,  6667,  10242, 13141, 13852, 14137, 18640, 19951, 22449, 23454, 24431, 25512, 25814,
    18,    53,    7890,  9934,  10063, 16728, 19040, 19809, 20825, 21522, 21800, 23582, 24556, 25031, 25547, 25562, 25733, 25789, 25906,
    4096,  4582,  5766,  5894,  6517,  10027, 12182, 13247, 15207, 17041, 18958, 20133, 20503, 22228, 24332, 24613, 25689, 25855, 25883,
    0,     25,    819,   5539,  7076,  7536,  7695,  9532,  13668, 15051, 17683, 19665, 20253, 21996, 24136, 24890, 25758, 25784, 25807,
    34,    40,    44,    4215,  6076,  7427,  7965,  8777,  11017, 15593, 19542, 22202, 22973, 23397, 23423, 24418, 24873, 25107, 25644,
    1595,  6216,  22850, 25439,
    1562,  15172, 19517, 22362,
    7508,  12879, 24324, 24496,
    6298,  15819, 16757, 18721,
    11173, 15175, 19966, 21195,
    59,    13505, 16941, 23793,
    2267,  4830,  12023, 20587,
    8827,  9278,  13072, 16664,
    14419, 17463, 23398, 25348,
    6112,  16534, 20423, 22698,
    493,   8914,  21103, 24799,
    6896,  12761, 13206, 25873,
    2,     1380,  12322, 21701,
    11600, 21306, 25753, 25790,
    8421,  13076, 14271, 15401,
    9630,  14112, 19017, 20955,
    212,   13932, 21781, 25824,
    5961,  9110,  16654, 19636,
    58,    5434,  9936,  12770,
    6575,  11433, 19798,
    2731,  7338,  20926,
    14253, 18463, 25404,
    21791, 24805, 25869,
    2,     11646, 15850,
    6075,  8586,  23819,
    18435, 22093, 24852,
    2103,  2368,  11704,
    10925, 17402, 18232,
    9062,  25061, 25674,
    18497, 20853, 23404,
    18606, 19364, 19551,
    7,     1022,  25543,
    6744,  15481, 25868,
    9081,  17305, 25164,
    8,     23701, 25883,
    9680,  19955, 22848,
    56,    4564,  19121,
    5595,  15086, 25892,
    3174,  17127, 23183,
    19397, 19817, 20275,
    12561, 24571, 25825,
    7111,  9889,  25865,
    19104, 20189, 21851,
    549,   9686,  25548,
    6586,  20325, 25906,
    3224,  20710, 21637,
    641,   15215, 25754,
    13484, 23729, 25818,
    2043,  7493,  24246,
    16860, 25230, 25768,
    22047, 24200, 24902,
    9391,  18040, 19499,
    7855,  24336, 25069,
    23834, 25570, 25852,
    1977,  8800,  25756,
    6671,  21772, 25859,
    3279,  6710,  24444,
    24099, 25117, 25820,
    5553,  12306, 25915,
    48,    11107, 23907,
    10832, 11974, 25773,
    2223,  17905, 25484,
    16782, 17135, 20446,
    475,   2861,  3457,
    16218, 22449, 24362,
    11716, 22200, 25897,
    8315,  15009, 22633,
    13,    20480, 25852,
    12352, 18658, 25687,
    3681,  14794, 23703,
    30,    24531, 25846,
    4103,  22077, 24107,
    23837, 25622, 25812,
    3627,  13387, 25839,
    908,   5367,  19388,
    0,     6894,  25795,
    20322, 23546, 25181,
    8178,  25260, 25437,
    2449,  13244, 22565,
    31,    18928, 22741,
    1312,  5134,  14838,
    6085,  13937, 24220,
    66,    14633, 25670,
    47,    22512, 25472,
    8867,  24704, 25279,
    6742,  21623, 22745,
    147,   9948,  24178,
    8522,  24261, 24307,
    19202, 22406, 24609
  };
};

struct DVB_S2X_TABLE_B23
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 43200;
  static const int LINKS_MIN_CN = 12;
  static const int LINKS_MAX_CN = 14;
  static const int LINKS_TOTAL = 281159;
  static const int DEG_MAX = 16;
  static constexpr int DEG[] = {
    16, 4, 3, 0
  };
  static constexpr int LEN[] = {
    23, 2, 95, 0
  };
  static constexpr int POS[] = {
    692,   1779,  1973,  2726,  5151,  6088,  7921,  9618,  11804, 13043, 15975, 16214, 16889, 16980, 18585, 18648,
    13,    4090,  4319,  5288,  8102,  10110, 10481, 10527, 10953, 11185, 12069, 13177, 14217, 15963, 17661, 20959,
    2330,  2516,  2902,  4087,  6338,  8015,  8638,  9436,  10294, 10843, 11802, 12304, 12371, 14095, 18486, 18996,
    125,   586,   5137,  5701,  6432,  6500,  8131,  8327,  10488, 11032, 11334, 11449, 12504, 16000, 20753, 21317,
    30,    480,   2681,  3635,  3898,  4058,  12803, 14734, 20252, 20306, 20680, 21329, 21333, 21466, 21562, 21568,
    20,    44,    738,   4965,  5516,  7659,  8464,  8759,  12216, 14630, 18241, 18711, 19093, 20217, 21316, 21490,
    31,    43,    3554,  5289,  5667,  8687,  14885, 16579, 17883, 18384, 18486, 19142, 20785, 20932, 21131, 21308,
    7054,  9276,  10435, 12324, 12354, 13849, 14285, 16482, 19212, 19217, 19221, 20499, 20831, 20925, 21195, 21247,
    9,     13,    4099,  10353, 10747, 14884, 15492, 17650, 19291, 19394, 20356, 20658, 21068, 21117, 21183, 21586,
    28,    2250,  2980,  8988,  10282, 12503, 13301, 18351, 20546, 20622, 21006, 21293, 21344, 21472, 21530, 21542,
    17,    32,    2521,  4374,  5098,  7525,  13035, 14437, 15283, 18635, 19136, 20240, 21147, 21179, 21300, 21349,
    57,    4735,  5657,  7649,  8807,  12375, 16092, 16178, 16379, 17545, 19461, 19489, 20321, 20530, 21453, 21457,
    35,    55,    5333,  14423, 14670, 15438, 19468, 19667, 20823, 21084, 21241, 21344, 21447, 21520, 21554, 21586,
    13,    20,    2025,  11854, 12516, 14938, 15929, 18081, 19730, 19929, 20408, 21338, 21391, 21425, 21468, 21546,
    54,    7451,  8176,  10136, 15240, 16442, 16482, 19431, 19483, 19762, 20647, 20839, 20966, 21512, 21579, 21592,
    26,    465,   3604,  4233,  9831,  11741, 13692, 18953, 18974, 21021, 21039, 21133, 21282, 21488, 21532, 21558,
    1,     7,     16,    59,    6979,  7675,  7717,  9791,  12370, 13050, 18534, 18729, 19846, 19864, 20127, 20165,
    15,    31,    11089, 12360, 13640, 14237, 17937, 18043, 18410, 19443, 21107, 21444, 21449, 21528, 21576, 21584,
    32,    51,    9768,  17848, 18095, 19326, 19594, 19618, 19765, 20440, 20482, 20582, 21236, 21338, 21563, 21587,
    44,    55,    4864,  10253, 11306, 12117, 13076, 13901, 15610, 17057, 18205, 19794, 20939, 21132, 21267, 21573,
    3436,  11304, 15361, 16511, 16860, 18238, 18639, 19341, 20106, 20123, 20407, 21200, 21280, 21452, 21526, 21569,
    679,   8822,  11045, 14403, 16588, 17838, 19117, 19453, 20265, 20558, 21374, 21396, 21428, 21442, 21529, 21590,
    391,   13002, 13140, 14314, 17169, 17175, 17846, 18122, 19447, 20075, 20212, 20436, 20583, 21330, 21359, 21403,
    7601,  10257, 20060, 21285,
    4419,  9150,  18097, 20315,
    4675,  13376, 21435,
    610,   1238,  16704,
    5732,  7096,  21104,
    5690,  13531, 14545,
    4334,  14839, 17357,
    8,     2814,  17674,
    2392,  8128,  18369,
    502,   7403,  15133,
    343,   13624, 20673,
    13188, 15687, 21593,
    321,   16866, 21347,
    1242,  4261,  17449,
    4691,  8086,  8691,
    8500,  11538, 20278,
    6269,  12905, 18192,
    5984,  15452, 17111,
    11541, 18717, 21534,
    16,    10780, 16107,
    12310, 12959, 20390,
    1365,  18306, 19634,
    6125,  19132, 20242,
    3012,  17233, 21533,
    5816,  13021, 21440,
    13207, 17811, 18798,
    2762,  7586,  12139,
    3949,  5545,  13584,
    11374, 18279, 19241,
    2736,  10989, 21209,
    4095,  20677, 21395,
    8251,  10084, 20498,
    7628,  8875,  21406,
    2743,  8943,  9090,
    1817,  7788,  15767,
    9333,  9838,  21268,
    6203,  9480,  12042,
    5747,  21187, 21468,
    2553,  18281, 21500,
    3179,  9155,  15222,
    12498, 18109, 20326,
    14106, 21209, 21592,
    7454,  17484, 20791,
    20804, 21120, 21574,
    5754,  18178, 20935,
    30,    4322,  21381,
    11905, 20416, 21397,
    12452, 19899, 21497,
    1917,  6028,  16868,
    9891,  18710, 18953,
    912,   21083, 21446,
    370,   14355, 18069,
    16519, 19003, 20902,
    11163, 17558, 18424,
    8427,  14396, 21405,
    8885,  11796, 21361,
    4960,  15431, 20653,
    11944, 16839, 21236,
    9967,  14529, 17208,
    14144, 19354, 19745,
    7986,  12680, 21396,
    6097,  11501, 13028,
    33,    13803, 21038,
    3177,  20124, 20803,
    2692,  6841,  18655,
    971,   5892,  14354,
    3887,  19455, 21271,
    17214, 17315, 21148,
    6539,  13910, 21526,
    3809,  5153,  15793,
    3865,  21438, 21510,
    7129,  17787, 19636,
    5972,  13150, 14182,
    7078,  14906, 16911,
    15705, 21160, 21482,
    5479,  13860, 19763,
    16817, 19722, 20001,
    14649, 16147, 18886,
    15138, 18578, 21502,
    2096,  2534,  17760,
    11920, 13460, 19783,
    19876, 20071, 20583,
    6241,  14230, 20775,
    16138, 16386, 21371,
    8616,  15624, 18453,
    6013,  8015,  21599,
    9184,  10688, 20792,
    18122, 21141, 21469,
    10706, 13177, 20957,
    15148, 15584, 20959,
    9114,  9432,  16467,
    5483,  14687, 14705,
    8325,  21161, 21410,
    2328,  17670, 19834,
    7015,  20802, 21385,
    52,    5451,  20379,
    9689,  15537, 19733
  };
};

struct DVB_S2X_TABLE_B24
{
  static const int M = 360;
  static const int N = 64800;
  static const int K = 47520;
  static const int LINKS_MIN_CN = 15;
  static const int LINKS_MAX_CN = 17;
  static const int LINKS_TOTAL = 278999;
  static const int DEG_MAX = 15;
  static constexpr int DEG[] = {
    15, 4, 3, 0
  };
  static constexpr int LEN[] = {
    23, 7, 102, 0
  };
  static constexpr int POS[] = {
    696,   989,   1238,  3091,  3116,  3738,  4269,  6406,  7033,  8048,  9157,  10254, 12033, 16456, 16912,
    444,   1488,  6541,  8626,  10735, 12447, 13111, 13706, 14135, 15195, 15947, 16453, 16916, 17137, 17268,
    401,   460,   992,   1145,  1576,  1678,  2238,  2320,  4280,  6770,  10027, 12486, 15363, 16714, 17157,
    1161,  3108,  3727,  4508,  5092,  5348,  5582,  7727,  11793, 12515, 12917, 13362, 14247, 16717, 17205,
    542,   1190,  6883,  7911,  8349,  8835,  10489, 11631, 14195, 15009, 15454, 15482, 16632, 17040, 17063,
    17,    487,   776,   880,   5077,  6172,  9771,  11446, 12798, 16016, 16109, 16171, 17087, 17132, 17226,
    1337,  3275,  3462,  4229,  9246,  10180, 10845, 10866, 12250, 13633, 14482, 16024, 16812, 17186, 17241,
    15,    980,   2305,  3674,  5971,  8224,  11499, 11752, 11770, 12897, 14082, 14836, 15311, 16391, 17209,
    0,     3926,  5869,  8696,  9351,  9391,  11371, 14052, 14172, 14636, 14974, 16619, 16961, 17033, 17237,
    3033,  5317,  6501,  8579,  10698, 12168, 12966, 14019, 15392, 15806, 15991, 16493, 16690, 17062, 17090,
    981,   1205,  4400,  6410,  11003, 13319, 13405, 14695, 15846, 16297, 16492, 16563, 16616, 16862, 16953,
    1725,  4276,  8869,  9588,  14062, 14486, 15474, 15548, 16300, 16432, 17042, 17050, 17060, 17175, 17273,
    1807,  5921,  9960,  10011, 14305, 14490, 14872, 15852, 16054, 16061, 16306, 16799, 16833, 17136, 17262,
    2826,  4752,  6017,  6540,  7016,  8201,  14245, 14419, 14716, 15983, 16569, 16652, 17171, 17179, 17247,
    1662,  2516,  3345,  5229,  8086,  9686,  11456, 12210, 14595, 15808, 16011, 16421, 16825, 17112, 17195,
    2890,  4821,  5987,  7226,  8823,  9869,  12468, 14694, 15352, 15805, 16075, 16462, 17102, 17251, 17263,
    3751,  3890,  4382,  5720,  10281, 10411, 11350, 12721, 13121, 14127, 14980, 15202, 15335, 16735, 17123,
    26,    30,    2805,  5457,  6630,  7188,  7477,  7556,  11065, 16608, 16859, 16909, 16943, 17030, 17103,
    40,    4524,  5043,  5566,  9645,  10204, 10282, 11696, 13080, 14837, 15607, 16274, 17034, 17225, 17266,
    904,   3157,  6284,  7151,  7984,  11712, 12887, 13767, 15547, 16099, 16753, 16829, 17044, 17250, 17259,
    7,     311,   4876,  8334,  9249,  11267, 14072, 14559, 15003, 15235, 15686, 16331, 17177, 17238, 17253,
    4410,  8066,  8596,  9631,  10369, 11249, 12610, 15769, 16791, 16960, 17018, 17037, 17062, 17165, 17204,
    24,    8261,  9691,  10138, 11607, 12782, 12786, 13424, 13933, 15262, 15795, 16476, 17084, 17193, 17220,
    88,    11622, 14705, 15890,
    304,   2026,  2638,  6018,
    1163,  4268,  11620, 17232,
    9701,  11785, 14463, 17260,
    4118,  10952, 12224, 17006,
    3647,  10823, 11521, 12060,
    1717,  3753,  9199,  11642,
    2187,  14280, 17220,
    14787, 16903, 17061,
    381,   3534,  4294,
    3149,  6947,  8323,
    12562, 16724, 16881,
    7289,  9997,  15306,
    5615,  13152, 17260,
    5666,  16926, 17027,
    4190,  7798,  16831,
    4778,  10629, 17180,
    10001, 13884, 15453,
    6,     2237,  8203,
    7831,  15144, 15160,
    9186,  17204, 17243,
    9435,  17168, 17237,
    42,    5701,  17159,
    7812,  14259, 15715,
    39,    4513,  6658,
    38,    9368,  11273,
    1119,  4785,  17182,
    5620,  16521, 16729,
    16,    6685,  17242,
    210,   3452,  12383,
    466,   14462, 16250,
    10548, 12633, 13962,
    1452,  6005,  16453,
    22,    4120,  13684,
    5195,  11563, 16522,
    5518,  16705, 17201,
    12233, 14552, 15471,
    6067,  13440, 17248,
    8660,  8967,  17061,
    8673,  12176, 15051,
    5959,  15767, 16541,
    3244,  12109, 12414,
    31,    15913, 16323,
    3270,  15686, 16653,
    24,    7346,  14675,
    12,    1531,  8740,
    6228,  7565,  16667,
    16936, 17122, 17162,
    4868,  8451,  13183,
    3714,  4451,  16919,
    11313, 13801, 17132,
    17070, 17191, 17242,
    1911,  11201, 17186,
    14,    17190, 17254,
    11760, 16008, 16832,
    14543, 17033, 17278,
    16129, 16765, 17155,
    6891,  15561, 17007,
    12741, 14744, 17116,
    8992,  16661, 17277,
    1861,  11130, 16742,
    4822,  13331, 16192,
    13281, 14027, 14989,
    38,    14887, 17141,
    10698, 13452, 15674,
    4,     2539,  16877,
    857,   17170, 17249,
    11449, 11906, 12867,
    285,   14118, 16831,
    15191, 17214, 17242,
    39,    728,   16915,
    2469,  12969, 15579,
    16644, 17151, 17164,
    2592,  8280,  10448,
    9236,  12431, 17173,
    9064,  16892, 17233,
    4526,  16146, 17038,
    31,    2116,  16083,
    15837, 16951, 17031,
    5362,  8382,  16618,
    6137,  13199, 17221,
    2841,  15068, 17068,
    24,    3620,  17003,
    9880,  15718, 16764,
    1784,  10240, 17209,
    2731,  10293, 10846,
    3121,  8723,  16598,
    8563,  15662, 17088,
    13,    1167,  14676,
    29,    13850, 15963,
    3654,  7553,  8114,
    23,    4362,  14865,
    4434,  14741, 16688,
    8362,  13901, 17244,
    13687, 16736, 17232,
    46,    4229,  13394,
    13169, 16383, 16972,
    16031, 16681, 16952,
    3384,  9894,  12580,
    9841,  14414, 16165,
    5013,  17099, 17115,
    2130,  8941,  17266,
    6907,  15428, 17241,
    16,    1860,  17235,
    2151,  16014, 16643,
    14954, 15958, 17222,
    3969,  8419,  15116,
    31,    15593, 16984,
    11514, 16605, 17255,
  };
};

struct DVB_S2X_TABLE_C1
{
  static const int M = 360;
  static const int N = 16200;
  static const int K = 3960;
  static const int LINKS_MIN_CN = 3;
  static const int LINKS_MAX_CN = 4;
  static const int LINKS_TOTAL = 48959;
  static const int DEG_MAX = 10;
  static constexpr int DEG[] = {
    10, 3, 0
  };
  static constexpr int LEN[] = {
    5, 6, 0
  };
  static constexpr int POS[] = {
    9054,  9186,  12155, 1000,  7383,  6459,  2992,  4723,  8135,  11250,
    2624,  9237,  7139,  12238, 11962, 4361,  5292,  10967, 11036, 8105,
    2044,  11996, 5654,  7568,  7002,  3549,  4767,  8767,  2872,  8345,
    6966,  8473,  5180,  8084,  3359,  5051,  9576,  5139,  1893,  902,
    3041,  3801,  8252,  11951, 909,   8535,  1038,  8400,  3200,  4585,
    5291,  10484, 10872,
    442,   7516,  3720,
    11469, 769,   10998,
    10575, 1436,  2935,
    6905,  8610,  11285,
    1873,  5634,  6383,
  };
};

struct DVB_S2X_TABLE_C2
{
  static const int M = 360;
  static const int N = 16200;
  static const int K = 4320;
  static const int LINKS_MIN_CN = 4;
  static const int LINKS_MAX_CN = 5;
  static const int LINKS_TOTAL = 58319;
  static const int DEG_MAX = 21;
  static constexpr int DEG[] = {
    21, 4, 3, 0
  };
  static constexpr int LEN[] = {
    3, 6, 3, 0
  };
  static constexpr int POS[] = {
    1953,  2331,  2545,  2623,  4653,  5012,  5700,  6458,  6875,  7605,  7694,  7881,  8416,  8758,  9181,  9555,  9578,  9932,  10068, 11479, 11699,
    514,   784,   2059,  2129,  2386,  2454,  3396,  5184,  6624,  6825,  7533,  7861,  9116,  9473,  9601,  10432, 11011, 11159, 11378, 11528, 11598,
    483,   1303,  1735,  2291,  3302,  3648,  4222,  4522,  5511,  6626,  6804,  7404,  7752,  7982,  8108,  8930,  9151,  9793,  9876,  10786, 11879,
    1956,  7572,  9020,  9971,
    13,    1578,  7445,  8373,
    6805,  6857,  8615,  11179,
    7983,  8022,  10017, 11748,
    4939,  8861,  10444, 11661,
    2278,  3733,  6265,  10009,
    4494,  7974,  10649,
    8909,  11030, 11696,
    3131,  9964,  10480,
  };
};

struct DVB_S2X_TABLE_C3
{
  static const int M = 360;
  static const int N = 16200;
  static const int K = 5040;
  static const int LINKS_MIN_CN = 4;
  static const int LINKS_MAX_CN = 5;
  static const int LINKS_TOTAL = 55799;
  static const int DEG_MAX = 12;
  static constexpr int DEG[] = {
    12, 9, 3, 0
  };
  static constexpr int LEN[] = {
    5, 1, 8, 0
  };
  static constexpr int POS[] = {
    1606,  3617,  7973,  6737,  9495,  4209,  9209,  4565,  4250,  7823,  9384,  400,
    4105,  991,   923,   3562,  3892,  10993, 5640,  8196,  6652,  4653,  9116,  7677,
    6348,  1341,  5445,  1494,  7799,  831,   4952,  5106,  3011,  9921,  6537,  8476,
    7854,  5274,  8572,  3741,  5674,  11128, 4097,  1398,  5671,  7302,  8155,  2641,
    6548,  2103,  590,   5749,  5722,  10,    2682,  1063,  633,   2949,  207,   6065,
    2828,  6366,  4766,  399,   935,   7611,  84,    150,   3146,
    5363,  7455,  7140,
    9297,  482,   4848,
    8458,  1631,  5344,
    5729,  6767,  4836,
    11019, 4463,  3882,
    4107,  9610,  5454,
    11137, 4328,  6307,
    3260,  7897,  3809,
  };
};

struct DVB_S2X_TABLE_C4
{
  static const int M = 360;
  static const int N = 16200;
  static const int K = 7560;
  static const int LINKS_MIN_CN = 8;
  static const int LINKS_MAX_CN = 9;
  static const int LINKS_TOTAL = 73079;
  static const int DEG_MAX = 24;
  static constexpr int DEG[] = {
    24, 4, 3, 0
  };
  static constexpr int LEN[] = {
    4, 8, 9, 0
  };
  static constexpr int POS[] = {
    3,     137,   314,   327,   983,   1597,  2028,  3043,  3217,  4109,  6020,  6178,  6535,  6560,  7146,  7180,  7408,  7790,  7893,  8123,  8313,  8526,  8616,  8638,
    356,   1197,  1208,  1839,  1903,  2712,  3088,  3537,  4091,  4301,  4919,  5068,  6025,  6195,  6324,  6378,  6686,  6829,  7558,  7745,  8042,  8382,  8587,  8602,
    18,    187,   1115,  1417,  1463,  2300,  2328,  3502,  3805,  4677,  4827,  5551,  5968,  6394,  6412,  6753,  7169,  7524,  7695,  7976,  8069,  8118,  8522,  8582,
    714,   2713,  2726,  2964,  3055,  3220,  3334,  3459,  5557,  5765,  5841,  6290,  6419,  6573,  6856,  7786,  7937,  8156,  8286,  8327,  8384,  8448,  8539,  8559,
    3452,  7935,  8092,  8623,
    56,    1955,  3000,  8242,
    1809,  4094,  7991,  8489,
    2220,  6455,  7849,  8548,
    1006,  2576,  3247,  6976,
    2177,  6048,  7795,  8295,
    1413,  2595,  7446,  8594,
    2101,  3714,  7541,  8531,
    10,    5961,  7484,
    3144,  4636,  5282,
    5708,  5875,  8390,
    3322,  5223,  7975,
    197,   4653,  8283,
    598,   5393,  8624,
    906,   7249,  7542,
    1223,  2148,  8195,
    976,   2001,  5005,
  };
};

struct DVB_S2X_TABLE_C5
{
  static const int M = 360;
  static const int N = 16200;
  static const int K = 8640;
  static const int LINKS_MIN_CN = 9;
  static const int LINKS_MAX_CN = 10;
  static const int LINKS_TOTAL = 75239;
  static const int DEG_MAX = 21;
  static constexpr int DEG[] = {
    21, 4, 3, 0
  };
  static constexpr int LEN[] = {
    5, 5, 14, 0
  };
  static constexpr int POS[] = {
    32,    384,   430,   591,   1296,  1976,  1999,  2137,  2175,  3638,  4214,  4304,  4486,  4662,  4999,  5174,  5700,  6969,  7115,  7138,  7189,
    1788,  1881,  1910,  2724,  4504,  4928,  4973,  5616,  5686,  5718,  5846,  6523,  6893,  6994,  7074,  7100,  7277,  7399,  7476,  7480,  7537,
    2791,  2824,  2927,  4196,  4298,  4800,  4948,  5361,  5401,  5688,  5818,  5862,  5969,  6029,  6244,  6645,  6962,  7203,  7302,  7454,  7534,
    574,   1461,  1826,  2056,  2069,  2387,  2794,  3349,  3366,  4951,  5826,  5834,  5903,  6640,  6762,  6786,  6859,  7043,  7418,  7431,  7554,
    14,    178,   675,   823,   890,   930,   1209,  1311,  2898,  4339,  4600,  5203,  6485,  6549,  6970,  7208,  7218,  7298,  7454,  7457,  7462,
    4075,  4188,  7313,  7553,
    5145,  6018,  7148,  7507,
    3198,  4858,  6983,  7033,
    3170,  5126,  5625,  6901,
    2839,  6093,  7071,  7450,
    11,    3735,  5413,
    2497,  5400,  7238,
    2067,  5172,  5714,
    1889,  7173,  7329,
    1795,  2773,  3499,
    2695,  2944,  6735,
    3221,  4625,  5897,
    1690,  6122,  6816,
    5013,  6839,  7358,
    1601,  6849,  7415,
    2180,  7389,  7543,
    2121,  6838,  7054,
    1948,  3109,  5046,
    272,   1015,  7464,
  };
};

struct DVB_S2X_TABLE_C6
{
  static const int M = 360;
  static const int N = 16200;
  static const int K = 9360;
  static const int LINKS_MIN_CN = 9;
  static const int LINKS_MAX_CN = 10;
  static const int LINKS_TOTAL = 68399;
  static const int DEG_MAX = 13;
  static constexpr int DEG[] = {
    13, 12, 3, 0
  };
  static constexpr int LEN[] = {
    2, 6, 18, 0
  };
  static constexpr int POS[] = {
    6106,  5389,  698,   6749,  6294,  1653,  1984,  2167,  6139,  6095,  3832,  2468,  6115,
    4202,  2362,  1852,  1264,  3564,  6345,  498,   6137,  3908,  3302,  527,   2767,  6667,
    3422,  1242,  1377,  2238,  2899,  1974,  1957,  261,   3463,  4994,  215,   2338,
    3016,  5109,  6533,  2665,  5300,  4908,  4967,  5787,  726,   229,   1970,  2789,
    6146,  5765,  6649,  2871,  884,   1670,  2597,  5058,  3659,  6594,  5042,  304,
    5521,  2811,  0,     4214,  2626,  2211,  1236,  3771,  852,   6356,  6797,  3463,
    1523,  1830,  3938,  5593,  2128,  5791,  3421,  3680,  6692,  1377,  3808,  3475,
    5551,  6035,  2247,  3662,  759,   6783,  116,   6380,  4586,  3367,  1,     5003,
    3518,  6557,  6510,
    1830,  839,   4421,
    5431,  5959,  6152,
    3174,  5113,  4520,
    5399,  1303,  2496,
    2841,  741,   220,
    2731,  1830,  4193,
    1875,  3935,  223,
    9,     4720,  423,
    3107,  2676,  840,
    1950,  6177,  6457,
    4091,  94,    5102,
    1907,  6050,  3455,
    714,   3,     559,
    502,   4268,  4164,
    1019,  5558,  271,
    6127,  854,   3221,
    959,   5337,  2735,
  };
};

struct DVB_S2X_TABLE_C7
{
  static const int M = 360;
  static const int N = 16200;
  static const int K = 11520;
  static const int LINKS_MIN_CN = 12;
  static const int LINKS_MAX_CN = 13;
  static const int LINKS_TOTAL = 60839;
  static const int DEG_MAX = 12;
  static constexpr int DEG[] = {
    12, 5, 3, 0
  };
  static constexpr int LEN[] = {
    5, 1, 26, 0
  };
  static constexpr int POS[] = {
    2686,  655,   2308,  1603,  336,   1743,  2778,  1263,  3555,  185,   4212,  621,
    286,   2994,  2599,  2265,  126,   314,   3992,  4560,  2845,  2764,  2540,  1476,
    2670,  3599,  2900,  2281,  3597,  2768,  4423,  2805,  836,   130,   1204,  4162,
    1884,  4228,  1253,  2578,  3053,  3650,  2587,  4468,  2784,  1644,  1490,  4655,
    4258,  1699,  4363,  4555,  3810,  4046,  3806,  344,   2459,  4067,  3327,  3510,
    1021,  2741,  2528,  2168,  2820,
    254,   1080,  616,
    1465,  4192,  2972,
    2356,  2976,  1534,
    4412,  1937,  2724,
    1430,  3024,  600,
    1952,  2136,  3573,
    3009,  3123,  1288,
    4553,  2299,  806,
    2997,  402,   4330,
    3302,  4567,  698,
    2364,  498,   3146,
    1809,  647,   992,
    3512,  32,    4301,
    1238,  251,   450,
    1657,  737,   641,
    560,   1720,  2893,
    1689,  2206,  902,
    3998,  1784,  2094,
    2090,  3126,  1201,
    1565,  764,   3473,
    891,   903,   2413,
    2286,  2900,  2348,
    3026,  2033,  1502,
    2404,  1243,  556,
    308,   2222,  3825,
    1523,  3311,  389,
  };
};

struct DVB_S2X_TABLE_C8
{
  static const int M = 360;
  static const int N = 32400;
  static const int K = 6480;
  static const int LINKS_MIN_CN = 3;
  static const int LINKS_MAX_CN = 4;
  static const int LINKS_TOTAL = 103679;
  static const int DEG_MAX = 13;
  static constexpr int DEG[] = {
    13, 3, 0
  };
  static constexpr int LEN[] = {
    9, 9, 0
  };
  static constexpr int POS[] = {
    18222, 6715,  4908,  21568, 22821, 11708, 4769,  4495,  22243, 25872, 9051,  19072, 13956,
    2038,  5205,  21215, 21009, 9584,  2403,  23652, 20866, 20130, 677,   9509,  6136,  773,
    19936, 14590, 17829, 473,   4432,  23171, 11386, 17937, 22084, 24450, 267,   8822,  19335,
    16376, 16769, 5111,  9794,  18907, 827,   12385, 12370, 21647, 10938, 23619, 11633, 15865,
    23417, 7631,  12243, 21546, 4192,  22117, 14757, 4118,  9686,  17021, 8531,  15989, 8807,
    15533, 16584, 18529, 19699, 17821, 4252,  1254,  5952,  3163,  20295, 6944,  1022,  19743,
    129,   16579, 23524, 25897, 14690, 11222, 16250, 9925,  4268,  999,   7102,  24528, 152,
    18361, 3708,  3454,  16604, 1551,  5809,  20324, 4775,  22418, 19091, 19674, 10975, 7327,
    24133, 10950, 22779, 11388, 13818, 20668, 7556,  12333, 16446, 19684, 12510, 25118, 8162,
    17026, 6850,  1269,
    21895, 7137,  25270,
    11858, 24153, 13303,
    7885,  16438, 12805,
    10473, 15004, 8052,
    2088,  10379, 10067,
    21438, 13426, 10440,
    17696, 727,   12164,
    22623, 8408,  17849,
  };
};

struct DVB_S2X_TABLE_C9
{
  static const int M = 360;
  static const int N = 32400;
  static const int K = 7920;
  static const int LINKS_MIN_CN = 3;
  static const int LINKS_MAX_CN = 4;
  static const int LINKS_TOTAL = 97919;
  static const int DEG_MAX = 10;
  static constexpr int DEG[] = {
    10, 3, 0
  };
  static constexpr int LEN[] = {
    10, 12, 0
  };
  static constexpr int POS[] = {
    20617, 6867,  14845, 11974, 22563, 190,   17207, 4052,  7406,  16007,
    21448, 14846, 2543,  23380, 16633, 20365, 16869, 13411, 19853, 795,
    5200,  2330,  2775,  23620, 20643, 10745, 14742, 6493,  14222, 20939,
    9445,  9523,  12769, 7332,  21792, 18717, 16397, 14016, 9481,  22162,
    2922,  6427,  4497,  4116,  17658, 2581,  14364, 3781,  18851, 22974,
    10383, 2184,  1433,  3889,  12828, 17424, 17580, 20936, 1390,  21374,
    425,   2063,  22398, 20907, 9445,  14790, 4457,  723,   7048,  4072,
    11771, 9640,  23212, 9613,  12042, 8335,  21386, 20129, 13521, 16301,
    14867, 12501, 1086,  21526, 17701, 17731, 20907, 8790,  19224, 5784,
    7107,  19690, 17616, 5800,  9501,  23320, 16878, 794,   15931, 17539,
    4556,  21783, 1524,
    20100, 11706, 23663,
    2535,  15530, 6116,
    12078, 3867,  2663,
    19629, 20246, 7024,
    11748, 11426, 19802,
    15942, 12333, 5316,
    11521, 3170,  17818,
    2289,  23780, 16575,
    6649,  16991, 13025,
    20050, 10619, 10250,
    3944,  13063, 5656,
  };
};

struct DVB_S2X_TABLE_C10
{
  static const int M = 360;
  static const int N = 32400;
  static const int K = 10800;
  static const int LINKS_MIN_CN = 4;
  static const int LINKS_MAX_CN = 5;
  static const int LINKS_TOTAL = 107999;
  static const int DEG_MAX = 12;
  static constexpr int DEG[] = {
    12, 3, 0
  };
  static constexpr int LEN[] = {
    10, 20, 0
  };
  static constexpr int POS[] = {
    7416,  4093,  16722, 1023,  20586, 12219, 9175,  16284, 1554,  10113, 19849, 17545,
    13140, 3257,  2110,  13888, 3023,  1537,  1598,  15018, 18931, 13905, 10617, 1014,
    339,   14366, 3309,  15360, 18358, 3196,  4412,  6023,  7070,  17380, 2777,  6691,
    12720, 17634, 4141,  1400,  8247,  18201, 16077, 11314, 11928, 3494,  3155,  2865,
    21038, 6928,  3860,  1943,  20292, 6526,  12939, 15182, 3957,  5651,  356,   2673,
    20555, 17905, 5724,  13932, 1218,  17763, 5912,  5164,  6233,  6220,  1277,  19209,
    19190, 4498,  4950,  6645,  5482,  5667,  13701, 16334, 15231, 735,   8589,  12344,
    679,   17849, 17807, 16033, 10181, 3368,  5778,  8275,  2736,  14042, 17506, 6005,
    1576,  10259, 10525, 3007,  16522, 697,   7726,  8641,  14323, 2893,  8168,  11070,
    17270, 10180, 18740, 847,   4969,  14715, 19316, 5530,  17428, 11193, 9861,  13562,
    6156,  18787, 10467,
    2422,  3723,  10769,
    8015,  18716, 13406,
    5969,  15949, 3084,
    6855,  13990, 3764,
    10351, 15779, 10392,
    16078, 19034, 11279,
    11747, 6608,  4188,
    19699, 8928,  8045,
    4598,  7219,  11391,
    19766, 11871, 5692,
    7487,  15905, 17621,
    8554,  7751,  16516,
    4981,  20250, 16146,
    12524, 21364, 10793,
    17083, 2051,  8862,
    1315,  6246,  7721,
    18043, 16652, 5502,
    1432,  5674,  2224,
    11257, 1312,  8453,
  };
};

} // namespace ldpctool

#endif

