#pragma once

#include "Buffer.h"

struct GrassCollision : Feature
{
private:
	static constexpr std::string_view MOD_ID = "87816";

public:
	virtual inline std::string GetName() override { return "Grass Collision"; }
	virtual inline std::string GetShortName() override { return "GrassCollision"; }
	virtual inline std::string GetFeatureModLink() override { return MakeNexusModURL(MOD_ID); }
	virtual inline std::string_view GetShaderDefineName() override { return "GRASS_COLLISION"; }
	virtual std::string_view GetCategory() const override { return "Grass"; }

	virtual std::pair<std::string, std::vector<std::string>> GetFeatureSummary() override
	{
		return {
			"Enables dynamic grass interactions where grass bends and moves in response to actors walking through it, creating more immersive environmental reactions.",
			{ "Real-time grass deformation from actor movement",
				"Collision detection for up to 256 simultaneous interactions",
				"Dynamic tracking of actor positions for grass response",
				"Performance-optimized collision calculation",
				"Seamless integration with existing grass rendering" }
		};
	}

	bool HasShaderDefine(RE::BSShader::Type shaderType) override;

	void UpdateCollisionTexture();

	struct Settings
	{
		bool EnableGrassCollision = 1;
		bool TrackRagdolls = 1;
		bool EnableBlur = 1;
	};

	struct alignas(16) BoundingBoxPacked
	{
		float2 MinExtent = { 0, 0 };
		float2 MaxExtent = { 0, 0 };
		uint IndexStart = 0;
		uint IndexEnd = 0;
		float2 pad0;
	};
	STATIC_ASSERT_ALIGNAS_16(BoundingBoxPacked);

	struct alignas(16) PerFrame
	{
		float2 PosOffset;              // cell origin in camera model space
		DirectX::XMUINT2 ArrayOrigin;  // xy: array origin (clipmap wrapping)

		DirectX::XMINT2 ValidMargin;
		float TimeDelta;
		uint BoundingBoxCount;

		float CameraHeightDelta;
		float3 pad0;
	};
	STATIC_ASSERT_ALIGNAS_16(PerFrame);

	Settings settings;

	ConstantBuffer* perFrame = nullptr;

	eastl::unique_ptr<Buffer> collisionBoundingBoxes = nullptr;
	eastl::unique_ptr<Buffer> collisionInstances = nullptr;

	virtual void ClearShaderCache() override;

	ID3D11ComputeShader* GetCollisionUpdateCS();
	ID3D11ComputeShader* collisionUpdateCS;

	Texture2D* collisionTexture = nullptr;

	virtual void SetupResources() override;

	virtual void DrawSettings() override;
	void UpdateCollisions(PerFrame& perFrame);
	void Update();

	virtual void LoadSettings(json& o_json) override;
	virtual void SaveSettings(json& o_json) override;

	virtual void RestoreDefaultSettings() override;

	virtual void PostPostLoad() override;

	virtual bool SupportsVR() override { return true; };

	struct Hooks
	{
		struct BSGrassShader_SetupGeometry
		{
			static void thunk(RE::BSShader* This, RE::BSRenderPass* Pass, uint32_t RenderFlags);
			static inline REL::Relocation<decltype(thunk)> func;
		};

		static void Install()
		{
			stl::write_vfunc<0x6, BSGrassShader_SetupGeometry>(RE::VTABLE_BSGrassShader[0]);
			logger::info("[GRASS COLLISION] Installed hooks");
		}
	};
};
