# Generated by Django 4.0.5 on 2022-06-11 15:40
import logging
import multiprocessing.pool
import shutil
import tempfile
import time
from pathlib import Path

from django.conf import settings
from django.db import migrations

from documents.parsers import run_convert

logger = logging.getLogger("paperless.migrations")


def _do_convert(work_package):
    existing_thumbnail, converted_thumbnail = work_package
    try:
        logger.info(f"Converting thumbnail: {existing_thumbnail}")

        # Run actual conversion
        run_convert(
            density=300,
            scale="500x5000>",
            alpha="remove",
            strip=True,
            trim=False,
            auto_orient=True,
            input_file=f"{existing_thumbnail}[0]",
            output_file=str(converted_thumbnail),
        )

        # Copy newly created thumbnail to thumbnail directory
        shutil.copy(converted_thumbnail, existing_thumbnail.parent)

        # Remove the PNG version
        existing_thumbnail.unlink()

        logger.info(
            "Conversion to WebP completed, "
            f"replaced {existing_thumbnail.name} with {converted_thumbnail.name}",
        )

    except Exception as e:
        logger.error(f"Error converting thumbnail (existing file unchanged): {e}")


def _convert_thumbnails_to_webp(apps, schema_editor):
    start = time.time()

    with tempfile.TemporaryDirectory() as tempdir:
        work_packages = []

        for file in Path(settings.THUMBNAIL_DIR).glob("*.png"):
            existing_thumbnail = file.resolve()

            # Change the existing filename suffix from png to webp
            converted_thumbnail_name = existing_thumbnail.with_suffix(
                ".webp",
            ).name

            # Create the expected output filename in the tempdir
            converted_thumbnail = (
                Path(tempdir) / Path(converted_thumbnail_name)
            ).resolve()

            # Package up the necessary info
            work_packages.append(
                (existing_thumbnail, converted_thumbnail),
            )

        if work_packages:
            logger.info(
                "\n\n"
                "  This is a one-time only migration to convert thumbnails for all of your\n"
                "  documents into WebP format.  If you have a lot of documents though, \n"
                "  this may take a while, so a coffee break may be in order."
                "\n",
            )

            with multiprocessing.pool.Pool(
                processes=min(multiprocessing.cpu_count(), 4),
                maxtasksperchild=4,
            ) as pool:
                pool.map(_do_convert, work_packages)

                end = time.time()
                duration = end - start

            logger.info(f"Conversion completed in {duration:.3f}s")


class Migration(migrations.Migration):
    dependencies = [
        ("documents", "1020_merge_20220518_1839"),
    ]

    operations = [
        migrations.RunPython(
            code=_convert_thumbnails_to_webp,
            reverse_code=migrations.RunPython.noop,
        ),
    ]
