/*
 * This file is part of Betaflight.
 *
 * Betaflight is free software. You can redistribute this software
 * and/or modify this software under the terms of the GNU General
 * Public License as published by the Free Software Foundation,
 * either version 3 of the License, or (at your option) any later
 * version.
 *
 * Betaflight is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this software.
 *
 * If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include "common/time.h"
#include "drivers/io_types.h"

#define ALL_MOTORS 255
#define MOTOR_OUTPUT_LIMIT_PERCENT_MIN 1
#define MOTOR_OUTPUT_LIMIT_PERCENT_MAX 100

typedef enum {
    MOTOR_PROTOCOL_FAMILY_UNKNOWN = 0,
    MOTOR_PROTOCOL_FAMILY_PWM,
    MOTOR_PROTOCOL_FAMILY_DSHOT,
} motorProtocolFamily_e;

typedef enum {
    MOTOR_PROTOCOL_PWM = 0,
    MOTOR_PROTOCOL_ONESHOT125,
    MOTOR_PROTOCOL_ONESHOT42,
    MOTOR_PROTOCOL_MULTISHOT,
    MOTOR_PROTOCOL_BRUSHED,
    MOTOR_PROTOCOL_DSHOT150,
    MOTOR_PROTOCOL_DSHOT300,
    MOTOR_PROTOCOL_DSHOT600,
/*  MOTOR_PROTOCOL_DSHOT1200, removed */
    MOTOR_PROTOCOL_PROSHOT1000,
    MOTOR_PROTOCOL_DISABLED,
    MOTOR_PROTOCOL_MAX
} motorProtocolTypes_e;

// Legacy aliases kept for backward compatibility with older PWM_TYPE_* names
#define PWM_TYPE_PWM              MOTOR_PROTOCOL_PWM
#define PWM_TYPE_ONESHOT125       MOTOR_PROTOCOL_ONESHOT125
#define PWM_TYPE_ONESHOT42        MOTOR_PROTOCOL_ONESHOT42
#define PWM_TYPE_MULTISHOT        MOTOR_PROTOCOL_MULTISHOT
#define PWM_TYPE_BRUSHED          MOTOR_PROTOCOL_BRUSHED
#define PWM_TYPE_DSHOT150         MOTOR_PROTOCOL_DSHOT150
#define PWM_TYPE_DSHOT300         MOTOR_PROTOCOL_DSHOT300
#define PWM_TYPE_DSHOT600         MOTOR_PROTOCOL_DSHOT600
#define PWM_TYPE_PROSHOT1000      MOTOR_PROTOCOL_PROSHOT1000
#define PWM_TYPE_DISABLED         MOTOR_PROTOCOL_DISABLED

typedef struct motorVTable_s {
    // Common
    void (*postInit)(void);
    float (*convertExternalToMotor)(uint16_t externalValue);
    uint16_t (*convertMotorToExternal)(float motorValue);
    bool (*enable)(void);
    void (*disable)(void);
    bool (*isMotorEnabled)(unsigned index);
    bool (*telemetryWait)(void);
    bool (*decodeTelemetry)(void);
    void (*updateInit)(void);
    void (*write)(uint8_t index, float value);
    void (*writeInt)(uint8_t index, uint16_t value);
    void (*updateComplete)(void);
    void (*shutdown)(void);
    bool (*isMotorIdle)(unsigned index);
    IO_t (*getMotorIO)(unsigned index);

    // Digital commands
    void (*requestTelemetry)(unsigned index);
} motorVTable_t;

typedef struct motorDevice_s {
    const motorVTable_t *vTable;
    uint8_t       count;
    bool          initialized;
    bool          enabled;
    timeMs_t      motorEnableTimeMs;
} motorDevice_t;
