/*
 * This file is part of Cleanflight and Betaflight.
 *
 * Cleanflight and Betaflight are free software. You can redistribute
 * this software and/or modify this software under the terms of the
 * GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * Cleanflight and Betaflight are distributed in the hope that they
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software.
 *
 * If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdbool.h>
#include <stdint.h>

#include "platform.h"
#include "common/utils.h"

#ifdef USE_ADC

#include "build/build_config.h"
#include "build/debug.h"
#include "drivers/io.h"

#include "pg/adc.h"

#include "adc.h"

//#define DEBUG_ADC_CHANNELS

#ifdef USE_ADC_INTERNAL

/**
 * Use a measurement of the fixed internal vref to calculate the external Vref+
 *
 * The ADC full range reading equates to Vref+ on the channel. Vref+ is typically
 * fed from Vcc at 3.3V, but since Vcc isn't a critical value it may be off
 * by a little due to variation in the regulator. Some chips are provided with a
 * known internal voltage reference, typically around 1.2V. By measuring this
 * reference with an internally connected ADC channel we can then calculate a more
 * accurate value for Vref+ instead of assuming that it is 3.3V
 *
 * @param intVRefAdcValue reading from the internal calibration voltage
 *
 * @return the calculated value of Vref+
*/
uint16_t adcInternalCompensateVref(uint16_t intVRefAdcValue)
{
    // This is essentially a tuned version of
    // __HAL_ADC_CALC_VREFANALOG_VOLTAGE(vrefAdcValue, ADC_RESOLUTION_12B);
    return (uint16_t)((uint32_t)(adcVREFINTCAL * VREFINT_CAL_VREF) / intVRefAdcValue);
}

int16_t adcInternalComputeTemperature(uint16_t tempAdcValue, uint16_t vrefValue)
{
    // This is essentially a tuned version of
    // __HAL_ADC_CALC_TEMPERATURE(vrefValue, tempAdcValue, ADC_RESOLUTION_12B);

    return ((((int32_t)((tempAdcValue * vrefValue) / TEMPSENSOR_CAL_VREFANALOG) - adcTSCAL1) * adcTSSlopeK) + 500) / 1000 + TEMPSENSOR_CAL1_TEMP;
}
#endif // USE_ADC_INTERNAL
#endif
