/*
 * Copyright 2018- The Pixie Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#pragma once

#include <map>
#include <string>
#include <vector>

#include <absl/container/flat_hash_map.h>

#include "src/common/base/base.h"
#include "src/table_store/schema/relation.h"

namespace px {
namespace table_store {
namespace schema {

/**
 * Schema tracks a collection of relations as generated by operators and is part of
 * the queries compilation state.
 */
class Schema {
 public:
  // Checks if the relation with id exists.
  bool HasRelation(int64_t id) const;
  // Returns all the IDs of the relations.
  std::vector<int64_t> GetIDs() const;
  // Adds a new relation to the schema. Writing to the same ID again
  // will override the relation at the entry.
  void AddRelation(int64_t id, const Relation& relation);

  // Get's the relation by id, if not found returns a status.
  StatusOr<const Relation> GetRelation(int64_t id) const;

  // Get the debug string of the schema.
  std::string DebugString() const;

  static Status ToProto(schemapb::Schema* schema,
                        const absl::flat_hash_map<std::string, schema::Relation>& relation_map);

 private:
  // We use an ordered map here to make sure lookups and ids always return the
  // same results (easier to test and more deterministic).
  std::map<int64_t, Relation> relations_;
};

}  // namespace schema
}  // namespace table_store
}  // namespace px
