/*
 * Copyright (c) 2024 Yunshan Networks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef TRACE_UTILS_H
#define TRACE_UTILS_H

/* Warning, this file is autogenerated by cbindgen. Don't modify this manually. */

#include <stdint.h>
#include "consts.h"

#define LUA_RUNTIME_DETECT_METHOD_LEN 256

#define LUA_RUNTIME_PATH_LEN 1024

#define LUA_RUNTIME_VERSION_LEN 32

#define UNWIND_ENTRIES_PER_SHARD 65535

#define UNWIND_SHARDS_PER_PROCESS 1024

enum CfaType {
    CFA_TYPE_RBP_OFFSET,  /* CFA = FP + offset (RBP on x86_64, X29 on ARM64) */
    CFA_TYPE_RSP_OFFSET,  /* CFA = SP + offset (RSP on x86_64, X31 on ARM64) */
    CFA_TYPE_EXPRESSION,
    CFA_TYPE_UNSUPPORTED,
    CFA_TYPE_NO_ENTRY,
};
typedef uint8_t CfaType;

enum RegType {
    REG_TYPE_UNDEFINED,
    REG_TYPE_SAME_VALUE,
    REG_TYPE_OFFSET,
    REG_TYPE_UNSUPPORTED,
};
typedef uint8_t RegType;

#if defined(__aarch64__)
/* Return Address recovery type for ARM64
 * On x86_64, RA is always at CFA-8 (implicitly handled)
 * On ARM64, RA may be in LR register or saved on stack
 */
enum RaType {
    RA_TYPE_CFA_OFFSET,   /* RA at CFA + offset */
    RA_TYPE_LR_REGISTER,  /* ARM64: RA is in Link Register (X30) */
    RA_TYPE_UNSUPPORTED,
};
typedef uint8_t RaType;
#endif

typedef struct lua_unwind_table_t lua_unwind_table_t;

typedef struct python_unwind_table_t python_unwind_table_t;

typedef struct unwind_table_t unwind_table_t;

typedef struct {
    uint32_t kind;
    uint8_t version[LUA_RUNTIME_VERSION_LEN];
    uint8_t detection_method[LUA_RUNTIME_DETECT_METHOD_LEN];
    uint8_t path[LUA_RUNTIME_PATH_LEN];
} lua_runtime_info_t;

typedef struct {
    uint32_t id;
    uint16_t entry_start;
    uint16_t entry_end;
    uint64_t offset;
    uint64_t pc_min;
    uint64_t pc_max;
} shard_info_t;

typedef struct {
    uint16_t len;
    shard_info_t entries[UNWIND_SHARDS_PER_PROCESS];
} process_shard_list_t;

typedef struct {
    uint64_t pc;
    CfaType cfa_type;
    RegType rbp_type;   /* FP recovery: how to restore frame pointer */
#if defined(__aarch64__)
    RaType ra_type;     /* RA recovery: how to get return address (ARM64) */
#endif
    int16_t cfa_offset;     /* by factor of 8 */
    int16_t rbp_offset;     /* by factor of 8 (FP offset from CFA) */
#if defined(__aarch64__)
    int16_t ra_offset;      /* by factor of 8 (RA offset from CFA, for ARM64) */
#endif
} unwind_entry_t;

typedef struct {
    uint32_t id;
    uint32_t len;
    unwind_entry_t entries[UNWIND_ENTRIES_PER_SHARD];
} unwind_entry_shard_t;

typedef struct {
    uint64_t thread_state_address;
    uint8_t offsets_id;
} python_unwind_info_t;

typedef struct {
    uint64_t executor_globals_address;
    uint64_t jit_return_address;
    uint64_t execute_ex_start;   // absolute address of execute_ex start
    uint64_t execute_ex_end;     // absolute address of execute_ex end (exclusive)
    uint8_t  offsets_id;
    uint8_t  has_jit;
    uint8_t  _reserved[6];
} php_unwind_info_t;

typedef struct {
    int64_t current_frame;
} py_cframe_t;

typedef struct {
    int64_t co_filename;
    int64_t co_firstlineno;
    int64_t co_name;
    int64_t co_varnames;
} py_code_object_t;

typedef struct {
    int64_t f_back;
    int64_t f_code;
    int64_t f_lineno;
    int64_t f_localsplus;
} py_frame_object_t;

typedef struct {
    int64_t owner;
} py_interpreter_frame_t;

typedef struct {
    int64_t tstate_head;
} py_interpreter_state_t;

typedef struct {
    int64_t ob_type;
} py_object_t;

typedef struct {
    int64_t interp_main;
} py_runtime_state_t;

typedef struct {
    int64_t data;
    int64_t size;
} py_string_t;

typedef struct {
    int64_t cframe;
    int64_t frame;
    int64_t interp;
    int64_t native_thread_id;
    int64_t next;
    int64_t thread_id;
} py_thread_state_t;

typedef struct {
    int64_t ob_item;
} py_tuple_object_t;

typedef struct {
    int64_t tp_name;
} py_type_object_t;

typedef struct {
    py_cframe_t cframe;
    py_code_object_t code_object;
    py_frame_object_t frame_object;
    py_interpreter_frame_t interpreter_frame;
    py_interpreter_state_t interpreter_state;
    py_object_t object;
    py_runtime_state_t runtime_state;
    py_string_t string;
    py_thread_state_t thread_state;
    py_tuple_object_t tuple_object;
    py_type_object_t type_object;
} python_offsets_t;

typedef struct {
    uint8_t offsets_id;
    uint8_t reserved[7];
    uint64_t state_address;
} lua_unwind_info_t;

typedef struct {
    uint32_t features;
    uint32_t off_l_ci;
    uint32_t off_l_base_ci;
    uint32_t off_l_end_ci;
    uint32_t off_ci_func;
    uint32_t off_ci_top;
    uint32_t off_ci_savedpc;
    uint32_t off_ci_prev;
    uint32_t off_tvalue_tt;
    uint32_t off_tvalue_val;
    uint32_t off_closure_isc;
    uint32_t off_lclosure_p;
    uint32_t off_cclosure_f;
    uint32_t off_proto_source;
    uint32_t off_proto_linedefined;
    uint32_t off_proto_code;
    uint32_t off_proto_sizecode;
    uint32_t off_proto_lineinfo;
    uint32_t off_proto_abslineinfo;
    uint32_t off_tstring_len;
    uint32_t sizeof_tstring;
    uint32_t sizeof_callinfo;
    uint32_t sizeof_tvalue;
} lua_ofs;

typedef struct {
    uint8_t fr2;
    uint8_t gc64;
    uint16_t pad;
    uint32_t tv_sz;
    uint32_t off_l_base;
    uint32_t off_l_stack;
    uint32_t off_gcproto_firstline;
    uint32_t off_gcproto_chunkname;
    uint32_t off_gcstr_data;
    uint32_t off_gcfunc_cfunc;
    uint32_t off_gcfunc_ffid;
    uint32_t off_gcfunc_pc;
    uint32_t off_gcproto_bc;
    uint32_t off_gcstr_len;
    uint32_t off_l_glref;
    uint32_t off_global_state_dispatchmode;
} lj_ofs;

typedef struct {
    uint16_t current_execute_data;
} php_executor_globals_t;

typedef struct {
    uint8_t opline;
    uint8_t function;
    uint8_t this_type_info;
    uint8_t prev_execute_data;
} php_execute_data_t;

typedef struct {
    uint8_t common_type;
    uint8_t common_funcname;
    uint8_t common_scope;
    uint32_t op_array_filename;
    uint32_t op_array_linestart;
    uint32_t sizeof_struct;
} php_function_t;

typedef struct {
    uint64_t val;
} php_string_t;

typedef struct {
    uint8_t lineno;
} php_op_t;

typedef struct {
    uint64_t name;
} php_class_entry_t;

typedef struct {
    php_executor_globals_t executor_globals;
    php_execute_data_t execute_data;
    php_function_t function;
    php_string_t string;
    php_op_t op;
    php_class_entry_t class_entry;
} php_offsets_t;

typedef struct {
    uint64_t isolate_address;
    uint8_t offsets_id;
    uint32_t version;
} v8_unwind_info_t;

typedef struct {
    int16_t marker;
    int16_t function;
    int16_t bytecode_offset;
} v8_frame_pointers_t;

typedef struct {
    uint16_t shared;
    uint16_t code;
} v8_js_function_t;

typedef struct {
    uint16_t name_or_scope_info;
    uint16_t function_data;
    uint16_t script_or_debug_info;
} v8_shared_function_info_t;

typedef struct {
    uint16_t instruction_start;
    uint16_t instruction_size;
    uint16_t flags;
} v8_code_t;

typedef struct {
    uint16_t name;
    uint16_t source;
} v8_script_t;

typedef struct {
    uint16_t source_position_table;
} v8_bytecode_array_t;

typedef struct {
    v8_frame_pointers_t frame_pointers;
    v8_js_function_t js_function;
    v8_shared_function_info_t shared_function_info;
    v8_code_t code;
    v8_script_t script;
    v8_bytecode_array_t bytecode_array;
} v8_offsets_t;

bool frame_pointer_heuristic_check(uint32_t pid);

int32_t is_lua_process(uint32_t pid);

bool is_python_process(uint32_t pid);

bool is_php_process(uint32_t pid);

bool is_v8_process(uint32_t pid);

int32_t lua_detect(uint32_t pid, lua_runtime_info_t *out);

char *lua_format_folded_stack_trace(void *tracer,
                                    uint32_t pid,
                                    const uint64_t *frames,
                                    uint32_t frame_count,
                                    bool new_cache,
                                    void *info_p,
                                    const char *err_tag);

void lua_set_map_fds(int32_t lang_flags_fd,
                     int32_t unwind_info_fd,
                     int32_t lua_offsets_fd,
                     int32_t luajit_offsets_fd);

lua_unwind_table_t *lua_unwind_table_create(int32_t lang_flags_fd,
                                            int32_t unwind_info_fd,
                                            int32_t lua_offsets_fd,
                                            int32_t luajit_offsets_fd);

void lua_unwind_table_destroy(lua_unwind_table_t *table);

void lua_unwind_table_load(lua_unwind_table_t *table, uint32_t pid);

void lua_unwind_table_unload(lua_unwind_table_t *table, uint32_t pid);

size_t merge_lua_stacks(void *trace_str, size_t len, const void *u_trace, const void *i_trace);

size_t merge_python_stacks(void *trace_str, size_t len, const void *i_trace, const void *u_trace);

size_t merge_php_stacks(void *trace_str, size_t len, const void *i_trace, const void *u_trace);

size_t merge_v8_stacks(void *trace_str, size_t len, const void *i_trace, const void *u_trace);

char *resolve_php_frame(uint32_t pid, uint64_t zend_function_ptr, uint64_t lineno, uint64_t is_jit);

char *resolve_v8_frame(uint32_t pid, uint64_t pointer_and_type, uint64_t delta_or_marker, uint64_t sfi_fallback);

int32_t protect_cpu_affinity_c(void);

python_unwind_table_t *python_unwind_table_create(int32_t unwind_info_map_fd,
                                                  int32_t offsets_map_fd);

void python_unwind_table_destroy(python_unwind_table_t *table);

void python_unwind_table_load(python_unwind_table_t *table, uint32_t pid);

void python_unwind_table_unload(python_unwind_table_t *table, uint32_t pid);

typedef struct php_unwind_table_t php_unwind_table_t;

php_unwind_table_t *php_unwind_table_create(int32_t unwind_info_map_fd,
                                            int32_t offsets_map_fd);

void php_unwind_table_destroy(php_unwind_table_t *table);

void php_unwind_table_load(php_unwind_table_t *table, uint32_t pid);

void php_unwind_table_unload(php_unwind_table_t *table, uint32_t pid);

typedef struct v8_unwind_table_t v8_unwind_table_t;

v8_unwind_table_t *v8_unwind_table_create(int32_t unwind_info_map_fd);

void v8_unwind_table_destroy(v8_unwind_table_t *table);

void v8_unwind_table_load(v8_unwind_table_t *table, uint32_t pid);

void v8_unwind_table_unload(v8_unwind_table_t *table, uint32_t pid);

int32_t read_offset_of_stack_in_task_struct(void);

int rustc_demangle(const char *mangled, char *out, size_t out_size);

unwind_table_t *unwind_table_create(int32_t process_shard_list_map_fd,
                                    int32_t unwind_entry_shard_map_fd);

void unwind_table_destroy(unwind_table_t *table);

void unwind_table_load(unwind_table_t *table, uint32_t pid);

void unwind_table_unload(unwind_table_t *table, uint32_t pid);

void unwind_table_unload_all(unwind_table_t *table);

#endif  /* TRACE_UTILS_H */
