@icon("file.svg")
@tool
class_name StoryFile extends Resource

enum Part {
	NOTE,
	TALK,
}

const DOT_EXTENSION := "."+EXTENSION
const EXTENSION := "tqstory"
const HEADER := "story.quadrimus.org"
const KEY := "#"
const TIME := "@"

const FileIcon := preload("file.svg")
const NoteIcon := preload("note.svg")
const NoteTranslationPartialIcon := preload("note_translation_partial.svg")
const NoteTranslationMissingIcon := preload("note_translation_missing.svg")
const TalkIcon := preload("talk.svg")
const TalkTranslationPartialIcon := preload("talk_translation_partial.svg")
const TalkTranslationMissingIcon := preload("talk_translation_missing.svg")

var _note_source := ProxyItemSource.new()
var _talk_source := ProxyItemSource.new()
var _original: Array


func _init() -> void:
	_note_source.name = "StoryFile.notes"
	_note_source.source = self
	_note_source.method_get_item_count = "get_note_count"
	_note_source.method_get_item_icon = "_get_note_icon"
	_note_source.method_get_item_metadata = "_get_note_index"
	_note_source.method_get_item_text = "get_note_name"
	_talk_source.name = "StoryFile.talks"
	_talk_source.source = self
	_talk_source.method_get_item_count = "get_talk_count"
	_talk_source.method_get_item_icon = "_get_talk_icon"
	_talk_source.method_get_item_metadata = "_get_talk_index"
	_talk_source.method_get_item_text = "get_talk_name"
	_original = [HEADER, {"uid": _uid_to_text(ResourceUID.create_id()), "notes": [], "talks": []}]


func find_part_index(part: Part, part_name: String) -> int:
	match part:
		Part.NOTE:
			return find_note_index(part_name)

		Part.TALK:
			return find_talk_index(part_name)

		_:
			push_error("StoryFile.find_part_index: unknown part %s at %s" % [part, resource_path])
			return -1


func find_note_index(note_name: String) -> int:
	var notes = _get_notes()
	for i in notes.size():
		if note_name == notes[i].get("name"):
			return i
	return -1


func find_talk_index(talk_name: String) -> int:
	var talks = _get_talks()
	for i in talks.size():
		if talk_name == talks[i].get("name"):
			return i
	return -1


func get_part_source(part: Part) -> Object:
	match part:
		Part.NOTE:
			return get_note_source()

		Part.TALK:
			return get_talk_source()

		_:
			push_error("StoryFile.get_part_source: unknown part %s at %s" % [part, resource_path])
			return null


func get_note_source() -> Object:
	return _note_source


func get_talk_source() -> Object:
	return _talk_source


func get_part_name(part: StoryFile.Part, part_index: int) -> String:
	match part:
		Part.NOTE:
			return get_note_name(part_index)

		Part.TALK:
			return get_talk_name(part_index)

		_:
			push_error("StoryFile.get_part_name: unknown part %s at %s" % [part, resource_path])
			return ""


func get_note_name(note_index: int) -> String:
	return _get_notes()[note_index].get("name")


func get_talk_name(talk_index: int) -> String:
	return _get_talks()[talk_index].get("name")


func set_note_name(note_index: int, note_name: String) -> void:
	return _get_notes()[note_index].set("name", note_name)


func set_talk_name(talk_index: int, talk_name: String) -> void:
	return _get_talks()[talk_index].set("name", talk_name)


func get_note_count() -> int:
	return _get_notes().size()


func get_talk_count() -> int:
	return _get_talks().size()


func _open(file_path: String = "") -> Error:
	var json := FileAccess.get_file_as_string(resource_path if file_path == "" else file_path)
	var error := FileAccess.get_open_error()
	if error != Error.OK:
		return error
	_original = _parse(json)
	return Error.OK


func _save(file_path: String = "") -> Error:
	if file_path != "":
		resource_path = file_path
	if resource_path != "":
		return _save_json(resource_path, _original)
	return Error.OK


func save_as(file_path: String) -> Error:
	return _save_json(file_path, _original)


func create_note(name: String) -> void:
	_get_notes().append({"name": name, "lines": [{}, {}]})


func create_talk(name: String) -> void:
	_get_talks().append({"name": name, "lines": []})


func delete_note(note_index: int) -> void:
	_get_notes().remove_at(note_index)


func delete_talk(talk_index: int) -> void:
	_get_talks().remove_at(talk_index)


func get_note_line(note_index: int, line_index: int) -> String:
	return get_note_line_translation(note_index, line_index, TranslationServer.get_locale())


func get_talk_line(talk_index: int, line_index: int) -> String:
	return get_talk_line_translation(talk_index, line_index, TranslationServer.get_locale())


func get_note_line_translation(note_index: int, line_index: int, locale: String) -> String:
	return _get_note_lines(note_index)[line_index].get(locale, "")


func get_talk_line_translation(talk_index: int, line_index: int, locale: String) -> String:
	return _get_talk_lines(talk_index)[line_index].get(locale, "")


func is_talk_line_key_unique(talk_index: int, key: String) -> bool:
	var lines = _get_talk_lines(talk_index)
	for line in lines:
		if line.get(KEY, "") == key:
			return false
	return true


func get_talk_line_key(talk_index: int, line_index: int) -> String:
	return _get_talk_lines(talk_index)[line_index].get(KEY, "")


func get_talk_line_time(talk_index: int, line_index: int) -> float:
	var time = _get_talk_lines(talk_index)[line_index].get(TIME)
	if time is float:
		return time
	if time is String:
		return float(time)
	return NAN


func get_note_line_count(note_index: int) -> int:
	return _get_note_lines(note_index).size()


func get_talk_line_count(talk_index: int) -> int:
	return _get_talk_lines(talk_index).size()


func get_note_lines_translation(note_index: int, locale: String, fallback_locale: String = "") -> Array:
	var lines = []
	for line in _get_note_lines(note_index):
		lines.append(_get_translation(func() -> String: return _get_note_debug_info(note_index), line, locale, fallback_locale))
	return lines


func get_talk_lines_translation(talk_index: int, locale: String, fallback_locale: String = "") -> Array:
	var lines = []
	for line in _get_talk_lines(talk_index):
		lines.append(_get_translation(func() -> String: return _get_talk_debug_info(talk_index), line, locale, fallback_locale))
	return lines


func has_note_translation(note_index: int, locale: String) -> bool:
	var note = _get_note_lines(note_index)
	for i in note.size():
		if !note[i].has(locale):
			return false
	return true


func has_talk_translation(talk_index: int, locale: String) -> bool:
	var talk = _get_talk_lines(talk_index)
	for i in talk.size():
		if !talk[i].has(locale):
			return false
	return true


func has_note_line_translation(note_index: int, line_index: int, locale: String) -> bool:
	return _get_note_lines(note_index)[line_index].has(locale)


func has_talk_line_translation(talk_index: int, line_index: int, locale: String) -> bool:
	return _get_talk_lines(talk_index)[line_index].has(locale)


func set_note_line_translation(note_index: int, line_index: int, locale: String, text: String) -> void:
	if text == "":
		_get_note_lines(note_index)[line_index].erase(locale)
	else:
		_get_note_lines(note_index)[line_index].set(locale, text)


func set_talk_line_translation(talk_index: int, line_index: int, locale: String, text: String) -> void:
	if text == "":
		_get_talk_lines(talk_index)[line_index].erase(locale)
	else:
		_get_talk_lines(talk_index)[line_index].set(locale, text)


func set_talk_line_key(talk_index: int, line_index: int, key: String) -> void:
	if key == "":
		_get_talk_lines(talk_index)[line_index].erase(KEY)
	else:
		_get_talk_lines(talk_index)[line_index].set(KEY, key)


func set_talk_line_time(talk_index: int, line_index: int, time: float) -> void:
	if is_nan(time):
		_get_talk_lines(talk_index)[line_index].erase(TIME)
	else:
		_get_talk_lines(talk_index)[line_index].set(TIME, time)


func create_talk_line(talk_index: int, line_index: int) -> void:
	var lines := _get_talk_lines(talk_index)
	if line_index < 0 || line_index >= lines.size():
		lines.append({})
		return
	lines.insert(line_index, {})


func move_talk_line(talk_index: int, line_index: int, delta: int) -> int:
	var lines := _get_talk_lines(talk_index)
	if line_index < 0 || line_index >= lines.size() || delta == 0:
		return -1
	var line := lines.get(line_index)
	lines.remove_at(line_index)
	line_index = clampi(line_index+delta, 0, lines.size())
	lines.insert(line_index, line)
	return line_index


func delete_talk_line(talk_index: int, line_index: int) -> void:
	_get_talk_lines(talk_index).remove_at(line_index)


func missing_note_translation(note_index: int, locales: PackedStringArray) -> bool:
	var note = _get_note_lines(note_index)
	for i in note.size():
		var line = note[i]
		for locale in locales:
			if !line.has(locale):
				return true
	return false


func missing_talk_translation(talk_index: int, locales: PackedStringArray) -> bool:
	var talk = _get_talk_lines(talk_index)
	for i in talk.size():
		var line = talk[i]
		for locale in locales:
			if !line.has(locale):
				return true
	return false


func missing_note_line_translation(note_index: int, line_index: int, locales: PackedStringArray) -> bool:
	var line = _get_note_lines(note_index)[line_index]
	for locale in locales:
		if !line.has(locale):
			return true
	return false


func missing_talk_line_translation(talk_index: int, line_index: int, locales: PackedStringArray) -> bool:
	var line = _get_talk_lines(talk_index)[line_index]
	for locale in locales:
		if !line.has(locale):
			return true
	return false


func _get_notes() -> Array:
	return _original[1].get_or_add("notes", [])


func _get_talks() -> Array:
	return _original[1].get_or_add("talks", [])


func _get_note_icon(_note_index: int) -> Texture2D:
	return NoteIcon


func _get_talk_icon(_talk_index: int) -> Texture2D:
	return TalkIcon


func _get_note_index(note_index: int) -> Variant:
	return note_index


func _get_talk_index(talk_index: int) -> Variant:
	return talk_index


func _get_note_lines(note_index: int) -> Array:
	return _get_notes()[note_index].get("lines")


func _get_talk_lines(talk_index: int) -> Array:
	return _get_talks()[talk_index].get("lines")


func _get_note_debug_info(note_index: int) -> String:
	return "note %s" % get_note_name(note_index)


func _get_talk_debug_info(talk_index: int) -> String:
	return "talk %s" % get_talk_name(talk_index)


func _get_translation(debug_part_info: Callable, from: Dictionary, locale: String, fallback_locale: String = "") -> String:
	var text = from.get(locale, "")
	if text == "":
		print_debug("StoryFile._get_translation: missing %s translation of %s at %s" % [locale, debug_part_info.call(), resource_path])
		if fallback_locale != "":
			return from.get(fallback_locale, "")
	return text


func _to_string() -> String:
	return "<StoryFile:[%s]>" % resource_path


static func is_path(file_path: String) -> bool:
	return file_path.ends_with(StoryFile.DOT_EXTENSION)


static func _get_uid(root: Array) -> int:
	return ResourceUID.text_to_id("uid://"+root[1].get("uid"))


static func _set_uid(root: Array, uid: int) -> void:
	return root[1].set("uid", _uid_to_text(uid))


static func _uid_to_text(uid: int) -> String:
	var text := ResourceUID.id_to_text(uid)
	return text.substr("uid://".length())


# returns Array if OK or String containing error message
static func _parse(json_string: String) -> Variant:
	var json := JSON.new()
	var error = json.parse(json_string)
	if error != Error.OK:
		return "error at line %d: %s" % [json.get_error_line(), json.get_error_message()]
	var root = json.data
	if typeof(root) != Variant.Type.TYPE_ARRAY:
		return "story file root must be array"
	if root.size() < 2:
		return "story file root must contains at least two elements"
	if root[0] != HEADER:
		return "story file root first element must be header"
	var content = root[1]
	if typeof(content) != Variant.Type.TYPE_DICTIONARY:
		return "story file root second element must be object"
	if typeof(content.get("uid", "")) != Variant.Type.TYPE_STRING:
		return "story file content uid must be string"
	var notes = content.get("notes", [])
	if typeof(notes) != Variant.Type.TYPE_ARRAY:
		return "story file content notes must be array"
	for i in notes.size():
		var note = notes[i]
		if typeof(note) != Variant.Type.TYPE_DICTIONARY:
			return "story file content talk %d must be dictionary" % i
		if typeof(note.get("name", "")) != Variant.Type.TYPE_STRING:
			return "story file content note %d name must be string" % i
		var lines = note.get("lines", [])
		if typeof(lines) != Variant.Type.TYPE_ARRAY:
			return "story file content note %d lines must be array" % i
		for j in lines.size():
			if typeof(lines[j]) != Variant.Type.TYPE_DICTIONARY:
				return "story file content note %d line %d must be dictionary" % [i, j]
	var talks = content.get("talks", [])
	if typeof(talks) != Variant.Type.TYPE_ARRAY:
		return "story file content talks must be array"
	for i in talks.size():
		var talk = talks[i]
		if typeof(talk) != Variant.Type.TYPE_DICTIONARY:
			return "story file content talk %d must be dictionary" % i
		if typeof(talk.get("name", "")) != Variant.Type.TYPE_STRING:
			return "story file content talk %d name must be string" % i
		var lines = talk.get("lines", [])
		if typeof(lines) != Variant.Type.TYPE_ARRAY:
			return "story file content talk %d lines must be array" % i
		for j in lines.size():
			if typeof(lines[j]) != Variant.Type.TYPE_DICTIONARY:
				return "story file content talk %d line %d must be dictionary" % [i, j]
	return root


static func _save_json(path: String, root: Array) -> Error:
	var json := _stringify(root)
	var file := FileAccess.open(path, FileAccess.WRITE)
	var error := FileAccess.get_open_error()
	if error != Error.OK:
		return error
	file.store_string(json)
	file.flush()
	file.close()
	return Error.OK


static func _stringify(root: Array) -> String:
	var json := JSON.stringify(root, "\t", false)
	var i := json.find("\"")
	if i > 0:
		json = "["+json.substr(i)
	return json+"\n"
