@tool
class_name NoteEdit extends VSplitContainer

signal on_note_changed()

@onready var note := $Note
@onready var current_label := $Line/CurrentLabel
@onready var current_text := $Line/CurrentText
@onready var translate_label := $Line/TranslateLabel
@onready var translate_text := $Line/TranslateText

var _icon_translation_missing := _get_editor_icon("StatusError")
var _icon_translation_partial := _get_editor_icon("StatusWarning")
var _icon_translation_done := _get_editor_icon("StatusSuccess")
var _locale_current: String = ""
var _locale_translate: String = ""
var _locales: PackedStringArray
var _story_file: StoryFile = null
var _note_index: int = -1
var _note_selected_index: int = -1


func get_current_locale() -> String:
	return _locale_current


func set_current_locale(locale: String) -> void:
	_locale_current = locale
	current_label.text = TranslationServer.get_locale_name(locale)+" ("+locale+")"
	_refresh_locales()
	_refresh_note(_note_selected_index)


func get_translate_locale() -> String:
	return _locale_translate


func set_translate_locale(locale: String) -> void:
	_locale_translate = locale
	translate_label.text = TranslationServer.get_locale_name(locale)+" ("+locale+")"
	_refresh_locales()
	_refresh_note(_note_selected_index)


func get_story_file() -> StoryFile:
	return _story_file


func set_story_file(file: StoryFile) -> void:
	_story_file = file
	_note_index = -1
	_refresh_note(-1)


func get_note_index() -> int:
	return _note_index


func set_note_index(note_index: int) -> void:
	_note_index = note_index
	_refresh_note(-1)


func _set_locales(locales: PackedStringArray) -> void:
	_locales = locales
	for i in note.get_item_count():
		_refresh_line_translation_status(i)


func _refresh_locales() -> void:
	var has_translate := _locale_current != _locale_translate
	translate_label.visible = has_translate
	translate_text.visible = has_translate


func _refresh_note(select_after_refresh: int) -> void:
	if _is_empty():
		for i in note.get_item_count():
			_note_set_item_text(i, "")
		_set_lines_selected_index(-1)
		return
	for i in note.get_item_count():
		_note_set_item_text(i, _story_file.get_note_line_translation(_note_index, i, _locale_current))
		_refresh_line_translation_status(i)
	if select_after_refresh < 0 || select_after_refresh >= note.get_item_count():
		note.deselect_all()
		_set_lines_selected_index(-1)
	else:
		note.select(select_after_refresh)
		_set_lines_selected_index(select_after_refresh)


func _refresh_line() -> void:
	if _story_file == null || _note_selected_index < 0:
		current_text.editable = false
		translate_text.editable = false
		current_text.text = ""
		translate_text.text = ""
		return
	current_text.set_text(_story_file.get_note_line_translation(_note_index, _note_selected_index, _locale_current))
	translate_text.set_text(_story_file.get_note_line_translation(_note_index, _note_selected_index, _locale_translate))
	current_text.editable = true
	translate_text.editable = true


func _refresh_line_translation_status(index: int) -> void:
	if _locale_current == _locale_translate:
		note.set_item_icon(index, null)
	else:
		var icon = _icon_translation_missing
		if _story_file.has_note_line_translation(_note_index, index, _locale_translate):
			icon = _icon_translation_partial if _story_file.missing_note_line_translation(_note_index, index, _locales) else _icon_translation_done
		note.set_item_icon(index, icon)


func _on_lines_selected(index: int) -> void:
	_set_lines_selected_index(index)


func _on_current_text_focus_exited() -> void:
	if _story_file == null || _note_selected_index < 0:
		return
	var text: String = current_text.get_text()
	_story_file.set_note_line_translation(_note_index, _note_selected_index, _locale_current, text)
	_note_set_item_text(_note_selected_index, text)
	on_note_changed.emit()


func _on_translate_text_focus_exited() -> void:
	if _story_file == null || _note_selected_index < 0:
		return
	var text: String = translate_text.get_text()
	_story_file.set_note_line_translation(_note_index, _note_selected_index, _locale_translate, text)
	_refresh_line_translation_status(_note_selected_index)
	on_note_changed.emit()


func _set_lines_selected_index(index: int) -> void:
	_note_selected_index = index
	_refresh_line()


func _is_empty() -> bool:
	return _story_file == null || _note_index < 0


func _note_set_item_text(idx: int, text: String) -> void:
	note.set_item_text(idx, (tr("Title") if idx == 0 else tr("Content"))+" — "+("∅" if text == "" else text))


static func _get_editor_icon(name: StringName) -> Texture2D:
	return EditorInterface.get_editor_theme().get_icon(name, "EditorIcons")
