@tool
class_name TeamQuadrimusStoryEditor extends Control

@onready var name_dialog := $NameDialog
@onready var tools_reload_file_list := $Columns/Left/Tools/ReloadFileList
@onready var tools_save_all_files := $Columns/Left/Tools/SaveAllFiles
@onready var tools_locales_switch := $Columns/Left/Tools/LocalesSwitch
@onready var tools_locales := $Columns/Left/Tools/Locales
@onready var tools_current_locale := $Columns/Left/Tools/Locales/Current
@onready var tools_arrow := $Columns/Left/Tools/Locales/Arrow
@onready var tools_translate_locale := $Columns/Left/Tools/Locales/Translate
@onready var files := $Columns/Left/Selector/Files
@onready var notes := $Columns/Left/Selector/Parts/Notes/Notes
@onready var notes_control_new := $Columns/Left/Selector/Parts/Notes/Controls/New
@onready var notes_control_rename := $Columns/Left/Selector/Parts/Notes/Controls/Rename
@onready var notes_control_delete := $Columns/Left/Selector/Parts/Notes/Controls/Delete
@onready var note_edit := $Columns/Right/NoteEdit
@onready var talks := $Columns/Left/Selector/Parts/Talks/Talks
@onready var talks_control_new := $Columns/Left/Selector/Parts/Talks/Controls/New
@onready var talks_control_rename := $Columns/Left/Selector/Parts/Talks/Controls/Rename
@onready var talks_control_delete := $Columns/Left/Selector/Parts/Talks/Controls/Delete
@onready var talk_edit := $Columns/Right/TalkEdit

var _locales: PackedStringArray = []
var _locale_current: String = ""
var _locale_translate: String = ""
var _selected_file_note_source_translation := ProxyItemSource.new()
var _selected_file_talk_source_translation := ProxyItemSource.new()


func _init() -> void:
	_selected_file_note_source_translation.name = "StoryFile.notes_translation"
	_selected_file_note_source_translation.source = self
	_selected_file_note_source_translation.method_get_item_count = "_selected_file_note_get_item_count"
	_selected_file_note_source_translation.method_get_item_icon = "_selected_file_note_get_item_icon"
	_selected_file_note_source_translation.method_get_item_metadata = "_selected_file_note_get_item_metadata"
	_selected_file_note_source_translation.method_get_item_text = "_selected_file_note_get_item_text"
	_selected_file_talk_source_translation.name = "StoryFile.talks_translation"
	_selected_file_talk_source_translation.source = self
	_selected_file_talk_source_translation.method_get_item_count = "_selected_file_talk_get_item_count"
	_selected_file_talk_source_translation.method_get_item_icon = "_selected_file_talk_get_item_icon"
	_selected_file_talk_source_translation.method_get_item_metadata = "_selected_file_talk_get_item_metadata"
	_selected_file_talk_source_translation.method_get_item_text = "_selected_file_talk_get_item_text"


func _ready() -> void:
	var icon_add := _get_editor_icon("Add")
	var icon_edit := _get_editor_icon("Edit")
	var icon_remove := _get_editor_icon("Remove")
	tools_reload_file_list.icon = _get_editor_icon("Reload")
	tools_save_all_files.icon = _get_editor_icon("Save")
	tools_locales_switch.icon = _get_editor_icon("Translation")
	tools_arrow.texture = _get_editor_icon("ArrowRight")
	notes_control_new.icon = icon_add
	notes_control_rename.icon = icon_edit
	notes_control_delete.icon = icon_remove
	talks_control_new.icon = icon_add
	talks_control_rename.icon = icon_edit
	talks_control_delete.icon = icon_remove
	files.set_item_source(StoryFileSource.singleton())
	notes.set_item_source(_selected_file_note_source_translation)
	talks.set_item_source(_selected_file_talk_source_translation)
	_reload_locales()
	var file_system_dock := EditorInterface.get_file_system_dock()
	file_system_dock.file_removed.connect(_on_editor_file_system_dock_file_removed)
	file_system_dock.files_moved.connect(_on_editor_file_system_dock_files_moved)


func _on_tools_reload_file_list_pressed() -> void:
	StoryFileSource.reload_file_list()
	_reload_files()


func _on_tools_save_all_files_pressed() -> void:
	StoryFileSource.save_all_files()


func _on_tools_locales_switch() -> void:
	tools_locales.visible = tools_locales_switch.button_pressed


func _on_files_item_selected(_text: String, metadata: Variant) -> void:
	var file: StoryFile = metadata
	note_edit.set_story_file(file)
	_reload_notes()
	talk_edit.set_story_file(file)
	_reload_talks()


func _on_parts_tab_changed(tab: int) -> void:
	if tab == 0:
		talk_edit.hide()
		note_edit.show()
	else:
		note_edit.hide()
		talk_edit.show()


func _on_notes_item_selected(_text: String, metadata: Variant) -> void:
	var note_index: int = metadata
	note_edit.set_note_index(note_index)
	_refresh_note_controls()


func _on_talks_item_selected(_text: String, metadata: Variant) -> void:
	var talk_index: int = metadata
	talk_edit.set_talk_index(talk_index)
	_refresh_talk_controls()


func _on_line_changed() -> void:
	_refresh_talk_translation_status()


func _reload_files(select: StoryFile = null) -> void:
	# keep selected file if any
	if select == null:
		select = files.get_selected_item_metadata()

	if select == null:
		files.refresh_items()
	else:
		files.refresh_items({"metadata": select})

	note_edit.set_story_file(select)
	_reload_notes()
	talk_edit.set_story_file(select)
	_reload_talks()


func _reload_locales() -> void:
	_locales = _get_locales()
	talk_edit._set_locales(_locales)
	tools_current_locale.clear()
	tools_translate_locale.clear()
	for locale in _locales:
		var locale_name := TranslationServer.get_locale_name(locale)+" ("+locale+")"
		var id: int = tools_current_locale.item_count
		tools_current_locale.add_item(locale_name, id)
		tools_current_locale.set_item_metadata(id, locale)
		tools_translate_locale.add_item(locale_name, id)
		tools_translate_locale.set_item_metadata(id, locale)
	tools_current_locale.select(0)
	tools_current_locale.item_selected.emit(0)
	var has_translate: bool = tools_current_locale.get_item_count() > 1
	tools_arrow.visible = has_translate
	tools_translate_locale.visible = has_translate
	tools_translate_locale.select(0)
	tools_translate_locale.item_selected.emit(0)
	_refresh_talk_translation_status()


func _reload_notes() -> void:
	notes.refresh_items()

	if notes.is_anything_selected():
		var note_index: int = notes.get_selected_item_metadata()
		note_edit.set_note_index(note_index)
	else:
		note_edit.set_note_index(-1)

	_refresh_note_controls()


func _reload_talks() -> void:
	talks.refresh_items()

	if talks.is_anything_selected():
		var talk_index: int = talks.get_selected_item_metadata()
		talk_edit.set_talk_index(talk_index)
	else:
		talk_edit.set_talk_index(-1)

	_refresh_talk_controls()


func _refresh_current_locale(selected: int) -> void:
	if selected >= 0:
		_locale_current = tools_current_locale.get_item_metadata(selected)
		note_edit.set_current_locale(_locale_current)
		_refresh_note_translation_status()
		talk_edit.set_current_locale(_locale_current)
		_refresh_talk_translation_status()


func _refresh_translate_locale(selected: int) -> void:
	if selected >= 0:
		_locale_translate = tools_current_locale.get_item_metadata(selected)
		note_edit.set_translate_locale(_locale_translate)
		_refresh_note_translation_status()
		talk_edit.set_translate_locale(_locale_translate)
		_refresh_talk_translation_status()


func _refresh_note_translation_status() -> void:
	notes.refresh_items()


func _refresh_talk_translation_status() -> void:
	talks.refresh_items()


func _refresh_note_controls() -> void:
	notes_control_new.disabled = note_edit.get_story_file() == null
	var no_note_selected: bool = !notes.is_anything_selected()
	notes_control_rename.disabled = no_note_selected
	notes_control_delete.disabled = no_note_selected


func _refresh_talk_controls() -> void:
	talks_control_new.disabled = talk_edit.get_story_file() == null
	var no_talk_selected: bool = !talks.is_anything_selected()
	talks_control_rename.disabled = no_talk_selected
	talks_control_delete.disabled = no_talk_selected


func _on_note_new_pressed() -> void:
	var file: StoryFile = note_edit.get_story_file()
	if file == null:
		return
	var callback := func(value: String, ok: bool):
		file.create_note(value)
		_reload_notes()
	name_dialog.popup_create(tr("Note"), callback)


func _on_talk_new_pressed() -> void:
	var file: StoryFile = talk_edit.get_story_file()
	if file == null:
		return
	var callback := func(value: String, ok: bool):
		file.create_talk(value)
		_reload_talks()
	name_dialog.popup_create(tr("Talk"), callback)


func _on_note_rename_pressed() -> void:
	var note_index: int = notes.get_selected_item_metadata()
	if note_index < 0:
		return
	var file: StoryFile = note_edit.get_story_file()
	var callback := func(value: String, ok: bool):
		if ok:
			file.set_note_name(note_index, value)
			notes.refresh_items()
	name_dialog.popup_rename(tr("Talk"), file.get_note_name(note_index), callback)


func _on_talk_rename_pressed() -> void:
	var talk_index: int = talks.get_selected_item_metadata()
	if talk_index < 0:
		return
	var file: StoryFile = talk_edit.get_story_file()
	var callback := func(value: String, ok: bool):
		if ok:
			file.set_talk_name(talk_index, value)
			talks.refresh_items()
	name_dialog.popup_rename(tr("Talk"), file.get_talk_name(talk_index), callback)


func _on_note_delete_pressed() -> void:
	var note_index: int = notes.get_selected_item_metadata()
	if note_index < 0:
		return
	var file: StoryFile = note_edit.get_story_file()
	var callback := func(value: String, ok: bool):
		if ok:
			file.delete_note(note_index)
			_reload_notes()
	name_dialog.popup_delete(tr("Note"), file.get_note_name(note_index), callback)


func _on_talk_delete_pressed() -> void:
	var talk_index: int = talks.get_selected_item_metadata()
	if talk_index < 0:
		return
	var file: StoryFile = talk_edit.get_story_file()
	var callback := func(value: String, ok: bool):
		if ok:
			file.delete_talk(talk_index)
			_reload_talks()
	name_dialog.popup_delete(tr("Talk"), file.get_talk_name(talk_index), callback)


func _on_editor_file_system_dock_file_removed(_file: String) -> void:
	StoryFileSource.reload_file_list()
	_reload_files()


func _on_editor_file_system_dock_files_moved(_old_file: String, _new_file: String) -> void:
	StoryFileSource.reload_file_list()
	_reload_files()


func _selected_file() -> StoryFile:
	return files.get_selected_item_metadata()


func _selected_file_note_source() -> ProxyItemSource:
	var file := _selected_file()
	if file != null:
		return file.get_note_source()
	return null


func _selected_file_note_get_item_count() -> int:
	var source := _selected_file_note_source()
	if source != null:
		return source.get_item_count()
	return 0


func _selected_file_note_get_item_icon(index: int) -> Texture2D:
	if _locale_current != _locale_translate:
		var file := _selected_file()
		if file != null:
			var icon = StoryFile.NoteTranslationMissingIcon
			if file.has_note_translation(index, _locale_translate):
				icon = StoryFile.NoteTranslationPartialIcon if file.missing_note_translation(index, _locales) else StoryFile.NoteIcon
			return icon

	return StoryFile.NoteIcon


func _selected_file_note_get_item_metadata(index: int) -> Variant:
	var source := _selected_file_note_source()
	if source != null:
		return source.get_item_metadata(index)

	return null


func _selected_file_note_get_item_text(index: int) -> String:
	var source := _selected_file_note_source()
	if source != null:
		return source.get_item_text(index)

	return ""


func _selected_file_note_get_item_tooltip(index: int) -> String:
	var source := _selected_file_note_source()
	if source != null:
		return source.get_item_tooltip(index)

	return ""


func _selected_file_talk_source() -> ProxyItemSource:
	var file := _selected_file()
	if file != null:
		return file.get_talk_source()
	return null


func _selected_file_talk_get_item_count() -> int:
	var source := _selected_file_talk_source()
	if source != null:
		return source.get_item_count()
	return 0


func _selected_file_talk_get_item_icon(index: int) -> Texture2D:
	if _locale_current != _locale_translate:
		var file := _selected_file()
		if file != null:
			var icon = StoryFile.TalkTranslationMissingIcon
			if file.has_talk_translation(index, _locale_translate):
				icon = StoryFile.TalkTranslationPartialIcon if file.missing_talk_translation(index, _locales) else StoryFile.TalkIcon
			return icon

	return StoryFile.TalkIcon


func _selected_file_talk_get_item_metadata(index: int) -> Variant:
	var source := _selected_file_talk_source()
	if source != null:
		return source.get_item_metadata(index)

	return null


func _selected_file_talk_get_item_text(index: int) -> String:
	var source := _selected_file_talk_source()
	if source != null:
		return source.get_item_text(index)

	return ""


func _selected_file_talk_get_item_tooltip(index: int) -> String:
	var source := _selected_file_talk_source()
	if source != null:
		return source.get_item_tooltip(index)

	return ""


static func _get_editor_icon(name: StringName) -> Texture2D:
	return EditorInterface.get_editor_theme().get_icon(name, "EditorIcons")


static func _get_locales() -> PackedStringArray:
	var locales := TranslationServer.get_loaded_locales()
	var current := TranslationServer.get_locale()
	if !locales.has(current):
		locales.insert(0, current)
	return locales
