import logging
import time

import click
from celery import shared_task

from core.db.session_factory import session_factory
from core.rag.index_processor.index_processor_factory import IndexProcessorFactory
from extensions.ext_redis import redis_client
from models.dataset import DocumentSegment

logger = logging.getLogger(__name__)


@shared_task(queue="dataset")
def disable_segment_from_index_task(segment_id: str):
    """
    Async disable segment from index
    :param segment_id:

    Usage: disable_segment_from_index_task.delay(segment_id)
    """
    logger.info(click.style(f"Start disable segment from index: {segment_id}", fg="green"))
    start_at = time.perf_counter()

    with session_factory.create_session() as session:
        segment = session.query(DocumentSegment).where(DocumentSegment.id == segment_id).first()
        if not segment:
            logger.info(click.style(f"Segment not found: {segment_id}", fg="red"))
            return

        if segment.status != "completed":
            logger.info(click.style(f"Segment is not completed, disable is not allowed: {segment_id}", fg="red"))
            return

        indexing_cache_key = f"segment_{segment.id}_indexing"

        try:
            dataset = segment.dataset

            if not dataset:
                logger.info(click.style(f"Segment {segment.id} has no dataset, pass.", fg="cyan"))
                return

            dataset_document = segment.document

            if not dataset_document:
                logger.info(click.style(f"Segment {segment.id} has no document, pass.", fg="cyan"))
                return

            if (
                not dataset_document.enabled
                or dataset_document.archived
                or dataset_document.indexing_status != "completed"
            ):
                logger.info(click.style(f"Segment {segment.id} document status is invalid, pass.", fg="cyan"))
                return

            index_type = dataset_document.doc_form
            index_processor = IndexProcessorFactory(index_type).init_index_processor()
            index_processor.clean(dataset, [segment.index_node_id])

            # Disable summary index for this segment
            from services.summary_index_service import SummaryIndexService

            try:
                SummaryIndexService.disable_summaries_for_segments(
                    dataset=dataset,
                    segment_ids=[segment.id],
                    disabled_by=segment.disabled_by,
                )
            except Exception as e:
                logger.warning("Failed to disable summary for segment %s: %s", segment.id, str(e))

            end_at = time.perf_counter()
            logger.info(
                click.style(
                    f"Segment removed from index: {segment.id} latency: {end_at - start_at}",
                    fg="green",
                )
            )
        except Exception:
            logger.exception("remove segment from index failed")
            segment.enabled = True
            session.commit()
        finally:
            redis_client.delete(indexing_cache_key)
