package mill.meta

import scala.jdk.CollectionConverters.CollectionHasAsScala
import mill.constants.CodeGenConstants as CGConst
import mill.api.Result
import mill.api.internal.ModuleDepsResolver.{ModuleDepsEntry, ModuleDepsConfig}
import mill.internal.Util.backtickWrap
import mill.api.internal.*
import pprint.Util.literalize
import mill.api.daemon.internal.MillScalaParser
import mill.api.internal.HeaderData

import scala.util.control.Breaks.*

object CodeGen {

  val generatedFileHeader =
    s"// Generated by Mill ${mill.meta.BuildInfo.millRunnerMetaVersion}"

  def generateWrappedAndSupportSources(
      projectRoot: os.Path,
      allScriptCode: Map[os.Path, String],
      wrappedDest: os.Path,
      supportDest: os.Path,
      resourceDest: os.Path,
      millTopLevelProjectRoot: os.Path,
      output: os.Path,
      parser: MillScalaParser
  ): Unit = {
    val scriptSources = allScriptCode.keys.toSeq.sorted
    val allowNestedBuildMillFiles = mill.internal.Util.readBooleanFromBuildHeader(
      projectRoot,
      mill.constants.ConfigConstants.millAllowNestedBuildMill,
      CGConst.rootBuildFileNames.asScala.toSeq
    )

    // Collect moduleDeps configuration from all YAML files to write to a classpath resource
    val moduleDepsConfig = collection.mutable.Map.empty[String, ModuleDepsConfig]

    // Find all directories that contain build.mill files (root build files only)
    // This is used to determine the enclosing build context for nested builds
    val rootBuildFileNamesSet = CGConst.rootBuildFileNames.asScala.toSet
    val nestedBuildFileDirs = scriptSources
      .filter(p => rootBuildFileNamesSet.contains(p.last))
      .map(_ / os.up)
      .toSet

    // All build file names (including package.mill) for child module detection
    val allBuildFileNames =
      (CGConst.nestedBuildFileNames.asScala ++ CGConst.rootBuildFileNames.asScala).toSet

    for (scriptPath <- scriptSources) {
      val scriptFolderPath = scriptPath / os.up
      val packageSegments = DiscoveredBuildFiles.fileImportToSegments(projectRoot, scriptPath)
      val wrappedDestFile = wrappedDest / packageSegments
      val pkgSegments = packageSegments.drop(1).dropRight(1)
      def pkgSelector0(pre: Option[String], s: Option[String]) =
        (pre ++ pkgSegments ++ s).map(backtickWrap).mkString(".")

      val pkg = pkgSelector0(Some(CGConst.globalPackagePrefix), None)

      val segments = calcSegments(scriptFolderPath, projectRoot)
      val supportDestDir = supportDest / packageSegments / os.up

      // Find the nearest enclosing build.mill file's segments by walking up from
      // the current script's folder until we find a directory containing a build.mill file.
      // Only considers build.mill files, not package.mill files, since package.mill files
      // don't create a new build context - they use the enclosing build.mill's context.
      val enclosingBuildSegments = {
        var dir = scriptFolderPath
        while (dir != projectRoot && !nestedBuildFileDirs.contains(dir)) dir = dir / os.up
        calcSegments(dir, projectRoot)
      }

      // Provide `build` as an alias to the enclosing `build_.package_`, since from
      // the user's perspective it looks like they're writing things that live in
      // `package build`, but at compile-time we rename things, so we provide an alias
      // to preserve the fiction. For nested builds, we alias to the nested package
      // so that project-relative imports continue working.
      val aliasImports = {
        val nestedPath = (Seq("build_") ++ enclosingBuildSegments.map(backtickWrap)).mkString(".")
        s"import $nestedPath.{package_ => build}"
      }
      val childNames = scriptSources
        .collect {
          case path
              if path != scriptPath
                && allBuildFileNames.contains(path.last)
                && path / os.up / os.up == scriptFolderPath => (path / os.up).last
        }
        .distinct

      def pkgSelector2(s: Option[String]) =
        s"_root_.${pkgSelector0(Some(CGConst.globalPackagePrefix), s)}"

      val childAliases = childNames
        .map { c =>
          // Dummy references to sub-modules. Just used as metadata for the discover and
          // resolve logic to traverse, cannot actually be evaluated and used
          val lhs = backtickWrap(c)
          val rhs = s"${pkgSelector2(Some(c))}.package_"
          s"final lazy val $lhs: $rhs.type = $rhs // subfolder module reference"
        }
        .mkString("\n  ")

      if (scriptFolderPath == projectRoot) {
        val buildFileImplCode = generateBuildFileImpl(pkg)
        os.write.over(
          supportDestDir / "BuildFileImpl.scala",
          buildFileImplCode,
          createFolders = true
        )
      }

      val miscInfo = generateMillMiscInfo(
        pkg = pkg,
        scriptFolderPath = scriptFolderPath,
        segments = segments,
        millTopLevelProjectRoot = millTopLevelProjectRoot,
        output = output
      )

      if (scriptPath.last.endsWith(".yaml")) {
        val newParent =
          if (segments.isEmpty) "_root_.mill.util.MainRootModule"
          else "_root_.mill.api.internal.SubfolderModule(_root_.build_.package_.millDiscover)"
        val parsedHeaderData = mill.internal.Util.parseHeaderData(scriptPath).get

        val prelude =
          s"""|import MillMiscInfo.*
              |import _root_.mill.util.TokenReaders.given
              |import _root_.mill.runner.autooverride.AutoOverride
              |""".stripMargin

        def processDataRest[T](data: HeaderData)(
            onProperty: (String, upickle.core.BufferedValue) => T,
            onNestedObject: (String, HeaderData) => T
        ): Seq[T] = {
          for ((locatedKeyString, v) <- data.rest.toSeq)
            yield locatedKeyString.value.split(" +") match {
              case Array(k) => onProperty(k, v)
              case Array("object", k) => onNestedObject(
                  k,
                  upickle.core.BufferedValue.transform(
                    v,
                    HeaderData.headerDataReader(scriptPath)
                  )
                )
              case _ => throw new Result.Exception(
                  "",
                  Some(Result.Failure(
                    "Invalid key: " + locatedKeyString.value,
                    scriptPath.toNIO,
                    locatedKeyString.index
                  ))
                )
            }
        }

        val miscInfoWithResource = {
          val header = if (pkg.isBlank()) "" else s"package $pkg"
          val miscInfoBody = if (segments.isEmpty) {
            rootMiscInfo(scriptFolderPath, millTopLevelProjectRoot, output)
          } else {
            subfolderMiscInfo(scriptFolderPath, segments)
          }
          s"""|$generatedFileHeader
              |$header
              |
              |$miscInfoBody
              |""".stripMargin
        }
        os.write.over(
          supportDestDir / "MillMiscInfo.scala",
          miscInfoWithResource,
          createFolders = true
        )

        def renderTemplate(prefix: String, data: HeaderData, path: Seq[String]): String = {
          val extendsConfig = data.`extends`.value.value.map(_.value)
          val definitions = processDataRest(data)(
            onProperty = (_, _) => "", // Properties will be auto-implemented by AutoOverride
            onNestedObject = (k, nestedData) =>
              renderTemplate(s"object $k", nestedData, path :+ k)
          ).filter(_.nonEmpty)

          // Helper to extract ModuleDepsEntry from HeaderData field
          // Each Located[String] contains (path, index, value) - we extract (value, index)
          def extractEntry(deps: Located[Appendable[Seq[Located[String]]]]): ModuleDepsEntry = {
            val appendable = deps.value
            ModuleDepsEntry(appendable.value.map(loc => (loc.value, loc.index)), appendable.append)
          }

          // Collect moduleDeps config for this module path and store in the map
          val modulePathKey = path.mkString(".")
          val moduleDepsEntry = extractEntry(data.moduleDeps)
          val compileModuleDepsEntry = extractEntry(data.compileModuleDeps)
          val runModuleDepsEntry = extractEntry(data.runModuleDeps)
          val bomModuleDepsEntry = extractEntry(data.bomModuleDeps)

          val config = ModuleDepsConfig(
            yamlPath = scriptPath.toString,
            moduleDeps = moduleDepsEntry,
            compileModuleDeps = compileModuleDepsEntry,
            runModuleDeps = runModuleDepsEntry,
            bomModuleDeps = bomModuleDepsEntry
          )

          moduleDepsConfig(modulePathKey) = config

          // Always generate defs without override - use macro to get super value if it exists
          val pathLiteral = literalize(modulePathKey)
          def moduleDepsSnippet(name: String) =
            s"def $name = _root_.mill.api.internal.ModuleDepsResolver.resolveModuleDeps(build, $pathLiteral, ${literalize(name)}, _root_.mill.api.internal.ModuleDepsResolver.superMethod(${literalize(name)}))"

          val moduleDepsSnippets = Seq(
            moduleDepsSnippet("moduleDeps"),
            moduleDepsSnippet("compileModuleDeps"),
            moduleDepsSnippet("runModuleDeps"),
            moduleDepsSnippet("bomModuleDeps")
          )

          val extendsSnippet =
            if (extendsConfig.nonEmpty)
              s" extends ${extendsConfig.mkString(", ")}, AutoOverride[_root_.mill.T[?]]"
            else " extends AutoOverride[_root_.mill.T[?]]"

          val allSnippets = moduleDepsSnippets ++ Seq(
            "inline def autoOverrideImpl[T](): T = ${ mill.api.Task.notImplementedImpl[T] }"
          ) ++ definitions

          s"""$prefix$extendsSnippet {
             |  ${allSnippets.mkString("\n  ")}
             |}
             |""".stripMargin
        }

        os.write.over(
          (wrappedDestFile / os.up) / wrappedDestFile.baseName,
          s"""package $pkg
             |import mill.*, scalalib.*, javalib.*, kotlinlib.*
             |$aliasImports
             |import build.*
             |$prelude
             |//SOURCECODE_ORIGINAL_FILE_PATH=$scriptPath
             |object package_ extends $newParent, package_ {
             |  ${if (segments.isEmpty) millDiscover(segments.nonEmpty) else ""}
             |  $childAliases
             |}
             |${renderTemplate("trait package_", parsedHeaderData, segments)}
             |""".stripMargin,
          createFolders = true
        )

      } else {
        breakable {
          val specialNames =
            (CGConst.nestedBuildFileNames.asScala ++ CGConst.rootBuildFileNames.asScala).toSet

          val isBuildScript = specialNames(scriptPath.last)

          val scriptName = scriptPath.last

          if (
            scriptFolderPath == projectRoot
            && CGConst.nestedBuildFileNames.contains(scriptName)
          ) break()

          if (
            scriptFolderPath != projectRoot
            && CGConst.rootBuildFileNames.contains(scriptName)
            && !allowNestedBuildMillFiles
          ) break()

          val scriptCode = allScriptCode(scriptPath)

          val markerComment =
            s"""//SOURCECODE_ORIGINAL_FILE_PATH=$scriptPath
               |//SOURCECODE_ORIGINAL_CODE_START_MARKER""".stripMargin

          val siblingScripts = scriptSources
            .filter(_ != scriptPath)
            .filter(p => (p / os.up) == (scriptPath / os.up))
            .map(_.last.split('.').head + "_")

          val importSiblingScripts = siblingScripts
            .filter(s => s != "build_" && s != "package_")
            .map(s => s"import $pkg.${backtickWrap(s)}.*").mkString("\n")

          if (isBuildScript) {
            os.write.over(supportDestDir / "MillMiscInfo.scala", miscInfo, createFolders = true)
          }

          val parts =
            if (!isBuildScript) {
              val wrapperName = backtickWrap(scriptPath.last.split('.').head + "_")
              s"""|$generatedFileHeader
                  |package $pkg
                  |
                  |$aliasImports
                  |$importSiblingScripts
                  |
                  |object $wrapperName {
                  |$markerComment
                  |$scriptCode
                  |}
                  |
                  |export $wrapperName._
                  |""".stripMargin
            } else {
              generateBuildScript(
                projectRoot = projectRoot,
                millTopLevelProjectRoot = millTopLevelProjectRoot,
                scriptPath = scriptPath,
                scriptFolderPath = scriptFolderPath,
                childAliases = childAliases,
                pkg = pkg,
                aliasImports = aliasImports,
                scriptCode = scriptCode,
                markerComment = markerComment,
                parser = parser,
                siblingScripts = siblingScripts,
                importSiblingScripts = importSiblingScripts
              )
            }

          os.write(wrappedDestFile, parts, createFolders = true)
        }
      }
    }

    val resourceFile = resourceDest / "mill" / "module-deps-config.json"
    os.write.over(
      resourceFile,
      upickle.default.write(moduleDepsConfig.toMap, indent = 2),
      createFolders = true
    )
  }

  private def calcSegments(scriptFolderPath: os.Path, projectRoot: os.Path) =
    scriptFolderPath.relativeTo(projectRoot).segments

  private def generateMillMiscInfo(
      pkg: String,
      scriptFolderPath: os.Path,
      segments: Seq[String],
      millTopLevelProjectRoot: os.Path,
      output: os.Path
  ): String = {
    val header = if (pkg.isBlank()) "" else s"package $pkg"
    val body =
      if (segments.nonEmpty) subfolderMiscInfo(scriptFolderPath, segments)
      else rootMiscInfo(
        scriptFolderPath,
        millTopLevelProjectRoot,
        output
      )

    s"""|$generatedFileHeader
        |$header
        |
        |$body
        |""".stripMargin
  }

  def generateBuildFileImpl(pkg: String) = {
    s"""|$generatedFileHeader
        |package $pkg
        |
        |object BuildFileImpl extends mill.api.internal.BuildFileCls(${CGConst.wrapperObjectName})
        |""".stripMargin
  }

  private def generateBuildScript(
      projectRoot: os.Path,
      millTopLevelProjectRoot: os.Path,
      scriptPath: os.Path,
      scriptFolderPath: os.Path,
      childAliases: String,
      pkg: String,
      aliasImports: String,
      scriptCode: String,
      markerComment: String,
      parser: MillScalaParser,
      siblingScripts: Seq[String],
      importSiblingScripts: String
  ): String = {
    val segments = calcSegments(scriptFolderPath, projectRoot)

    val exportSiblingScripts =
      siblingScripts.map(s => s"export $pkg.${backtickWrap(s)}.*").mkString("\n")

    val prelude =
      s"""|import MillMiscInfo.*
          |import _root_.mill.util.TokenReaders.given
          |import _root_.mill.api.JsonFormatters.given
          |""".stripMargin

    val objectData = parser.parseObjectData(scriptCode)

    val expectedModuleMsg =
      if (projectRoot != millTopLevelProjectRoot) "MillBuildRootModule" else "mill.Module"

    val headerCode =
      s"""|$generatedFileHeader
          |package $pkg
          |
          |$aliasImports
          |$importSiblingScripts
          |$prelude
          |
          |object ${CGConst.wrapperObjectName} extends ${CGConst.wrapperObjectName} {
          |  ${childAliases.linesWithSeparators.mkString("  ")}
          |  ${exportSiblingScripts.linesWithSeparators.mkString("  ")}
          |  ${millDiscover(segments.nonEmpty)}
          |}
          |""".stripMargin

    val newParent =
      if (segments.isEmpty) "_root_.mill.util.MainRootModule"
      else "_root_.mill.api.internal.SubfolderModule(_root_.build_.package_.millDiscover)"

    objectData.find(o => o.name.text == "`package`") match {
      case Some(objectData) =>

        var newScriptCode = scriptCode
        objectData.endMarker match {
          case Some(endMarker) =>
            newScriptCode = endMarker.applyTo(newScriptCode, CGConst.wrapperObjectName)
          case None =>
            ()
        }
        objectData.finalStat match {
          case Some((_, finalStat)) =>
            val statLines = finalStat.text.linesWithSeparators.toSeq
            val fenced = Seq(
              "",
              if statLines.sizeIs > 1 then statLines.tail.mkString else finalStat.text
            ).mkString(System.lineSeparator())
            newScriptCode = finalStat.applyTo(newScriptCode, fenced)
          case None => ()
        }

        newScriptCode = objectData.parent.applyTo(
          newScriptCode,
          if (objectData.parent.text == null) {
            throw new Result.Exception(
              s"object `package` in ${scriptPath.relativeTo(millTopLevelProjectRoot)} " +
                s"must extend a subclass of `$expectedModuleMsg`"
            )
          } else {
            // `extends` clauses can have the parent followed by either `with` or `,`
            // separators, but it needs to be consistent. So we need to try and see if
            // any separators are already present and if so follow suite. Not 100%
            // precise, but probably works well enough people will rarely hit issues
            val postParent = newScriptCode.drop(objectData.parent.end).trim
            val sep = {
              if (postParent.startsWith(",")) ", "
              else if (postParent.startsWith("with")) " with "
              else ", " // no separator found, just use `,` by default
            }

            newParent + sep + objectData.parent.text
          }
        )

        newScriptCode = objectData.name.applyTo(newScriptCode, CGConst.wrapperObjectName)
        newScriptCode = objectData.obj.applyTo(newScriptCode, "abstract class")

        s"""$headerCode
           |$markerComment
           |$newScriptCode
           |""".stripMargin

      case None =>
        s"""$headerCode
           |abstract class ${CGConst.wrapperObjectName}
           |    extends $newParent { this: ${CGConst.wrapperObjectName}.type =>
           |$markerComment
           |$scriptCode
           |}""".stripMargin

    }
  }

  def subfolderMiscInfo(
      scriptFolderPath: os.Path,
      segments: Seq[String]
  ): String = {
    s"""|object MillMiscInfo
        |    extends mill.api.internal.SubfolderModule.Info(
        |  millSourcePath0 = os.Path(${literalize(scriptFolderPath.toString)}),
        |  segments = _root_.scala.Seq(${segments.map(literalize(_)).mkString(", ")})
        |)
        |""".stripMargin
  }

  def millDiscover(segmentsNonEmpty: Boolean): String = {
    if (segmentsNonEmpty) ""
    else {
      val rhs = "_root_.mill.api.Discover[this.type]"
      s"override lazy val millDiscover: _root_.mill.api.Discover = $rhs"
    }
  }

  def rootMiscInfo(
      scriptFolderPath: os.Path,
      millTopLevelProjectRoot: os.Path,
      output: os.Path
  ): String = {
    s"""|@_root_.scala.annotation.nowarn
        |object MillMiscInfo
        |    extends mill.api.internal.RootModule.Info(
        |  projectRoot0 = ${literalize(scriptFolderPath.toString)},
        |  output0 = ${literalize(output.toString)},
        |  topLevelProjectRoot0 = ${literalize(millTopLevelProjectRoot.toString)}
        |)
        |""".stripMargin
  }

}
