package mill.scalajslib

import mill.api.Discover
import mill.testkit.UnitTester
import mill.testkit.TestRootModule
import utest.*
import mill.scalajslib.api.*
import mill.T

object WasmTests extends TestSuite {
  val remapTo = "https://cdn.jsdelivr.net/gh/stdlib-js/array-base-linspace@esm/index.mjs"

  object Wasm extends TestRootModule with ScalaJSModule {
    override def scalaVersion = sys.props.getOrElse("TEST_SCALA_2_13_VERSION", ???)

    override def scalaJSVersion = "1.20.2"

    override def moduleKind = ModuleKind.ESModule

    override def moduleSplitStyle = ModuleSplitStyle.FewestModules

    override def scalaJSExperimentalUseWebAssembly: T[Boolean] = true

    override lazy val millDiscover = {
      import mill.util.TokenReaders.given
      Discover[this.type]
    }
  }

  val millSourcePath = os.Path(sys.env("MILL_TEST_RESOURCE_DIR")) / "wasm"

  val tests: Tests = Tests {
    test("should emit wasm") {
      UnitTester(Wasm, millSourcePath).scoped { evaluator =>
        val Right(result) =
          evaluator(Wasm.fastLinkJS).runtimeChecked
        val publicModules = result.value.publicModules.toSeq
        val path = result.value.dest.path
        val main = publicModules.head
        assert(main.jsFileName == "main.js")
        val mainPath = path / "main.js"
        assert(os.exists(mainPath))
        val wasmPath = path / "main.wasm"
        assert(os.exists(wasmPath))
        val wasmMapPath = path / "main.wasm.map"
        assert(os.exists(wasmMapPath))
      }
    }

    test("wasm is runnable") {
      UnitTester(Wasm, millSourcePath).scoped { evaluator =>
        val Right(result) = evaluator(Wasm.fastLinkJS).runtimeChecked
        val path = result.value.dest.path
        os.proc("node", "--experimental-wasm-exnref", "main.js").call(
          cwd = path,
          check = true,
          stdin = os.Inherit,
          stdout = os.Inherit,
          stderr = os.Inherit
        )
      }
    }
  }
}
