package mill.javalib

import mill.{T, Task}
import mill.api.{PathRef, ExecResult}
import mill.api.{Discover, Evaluator}
import mill.javalib.publish.{
  Developer,
  License,
  PackagingType,
  PomSettings,
  VersionControl,
  VersionScheme
}
import mill.testkit.UnitTester
import mill.testkit.TestRootModule
import utest.*
import mill.util.TokenReaders.*
import scala.jdk.CollectionConverters.*

object PublishModuleTests extends TestSuite {

  object HelloJavaWithPublish extends TestRootModule {
    object core extends JavaModule with PublishModule {
      override def artifactName = "hello-world"
      override def publishVersion = "0.0.1"
      override def pomSettings = PomSettings(
        organization = "com.lihaoyi",
        description = "hello world ready for real world publishing",
        url = "https://github.com/lihaoyi/hello-world-publish",
        licenses = Seq(License.Common.Apache2),
        versionControl = VersionControl.github("lihaoyi", "hello-world-publish"),
        developers =
          Seq(Developer("lihaoyi", "Li Haoyi", "https://github.com/lihaoyi"))
      )
      override def versionScheme = Some(VersionScheme.EarlySemVer)

      def checkSonatypeCreds(sonatypeCreds: String) = Task.Command {
        internal.PublishModule.checkSonatypeCreds(sonatypeCreds)()
      }
    }

    lazy val millDiscover = Discover[this.type]
  }

  object PomOnly extends TestRootModule {
    object core extends JavaModule with PublishModule {
      override def pomPackagingType: String = PackagingType.Pom
      override def artifactName = "pom-only"
      override def publishVersion = "0.0.1"
      override def pomSettings = PomSettings(
        organization = "com.lihaoyi",
        description = "pom-only artifacts ready for real world publishing",
        url = "https://github.com/lihaoyi/hello-world-publish",
        licenses = Seq(License.Common.Apache2),
        versionControl = VersionControl.github("lihaoyi", "hello-world-publish"),
        developers =
          Seq(Developer("lefou", "Tobias Roeser", "https://github.com/lefou"))
      )
      override def versionScheme = Some(VersionScheme.EarlySemVer)
      override def mvnDeps = Seq(
        mvn"org.slf4j:slf4j-api:2.0.7"
      )
      // ensure, these tasks won't be called
      override def jar: T[PathRef] = Task { ???.asInstanceOf[PathRef] }
      override def docJar: T[PathRef] = Task { ???.asInstanceOf[PathRef] }
      override def sourceJar: T[PathRef] = Task { ???.asInstanceOf[PathRef] }
    }

    lazy val millDiscover = Discover[this.type]
  }

  object compileAndRuntimeStuff extends TestRootModule {
    def organization = "com.lihaoyi.pubmodtests"
    def version = "0.1.0-SNAPSHOT"
    trait TestPublishModule extends PublishModule {
      def publishVersion = version
      def pomSettings = PomSettings(
        organization = organization,
        description = "test thing",
        url = "https://github.com/com-lihaoyi/mill",
        licenses = Seq(License.Common.Apache2),
        versionControl = VersionControl.github("com-lihaoyi", "mill"),
        developers = Nil
      )
    }
    object main extends JavaModule with TestPublishModule {
      def mvnDeps = Seq(
        mvn"org.slf4j:slf4j-api:2.0.15"
      )
      def runMvnDeps = Seq(
        mvn"ch.qos.logback:logback-classic:1.5.12"
      )
    }

    object transitive extends JavaModule with TestPublishModule {
      def moduleDeps = Seq(main)
    }

    object runtimeTransitive extends JavaModule with TestPublishModule {
      def runModuleDeps = Seq(main)
    }

    lazy val millDiscover = Discover[this.type]
  }

  def tests: Tests = Tests {

    test("publish") {
      test(
        "should retrieve credentials from environment variables if direct argument is empty"
      ) {
        UnitTester(
          HelloJavaWithPublish,
          sourceRoot = null,
          env = Evaluator.defaultEnv ++ Seq(
            "SONATYPE_USERNAME" -> "user",
            "SONATYPE_PASSWORD" -> "password"
          )
        ).scoped { eval =>
          val Right(result) =
            eval.apply(HelloJavaWithPublish.core.checkSonatypeCreds("")).runtimeChecked

          assert(
            result.value == "user:password",
            result.evalCount > 0
          )
        }
      }
      test(
        "should prefer direct argument as credentials over environment variables"
      ) {
        UnitTester(
          HelloJavaWithPublish,
          sourceRoot = null,
          env = Evaluator.defaultEnv ++ Seq(
            "SONATYPE_USERNAME" -> "user",
            "SONATYPE_PASSWORD" -> "password"
          )
        ).scoped { eval =>
          val directValue = "direct:value"
          val Right(result) =
            eval.apply(HelloJavaWithPublish.core.checkSonatypeCreds(directValue)).runtimeChecked

          assert(
            result.value == directValue,
            result.evalCount > 0
          )
        }
      }
      test(
        "should throw exception if neither environment variables or direct argument were not passed"
      ) - UnitTester(HelloJavaWithPublish, null).scoped { eval =>
        val Left(ExecResult.Failure(msg = msg)) =
          eval.apply(HelloJavaWithPublish.core.checkSonatypeCreds("")).runtimeChecked

        assert(
          msg.contains(
            "Consider using MILL_SONATYPE_USERNAME/MILL_SONATYPE_PASSWORD environment variables"
          )
        )
      }
    }

    test("pom-packaging-type") - {
      test("pom") - UnitTester(PomOnly, null).scoped { eval =>
        val Right(result) = eval.apply(PomOnly.core.pom).runtimeChecked

        assert(
          os.exists(result.value.path),
          result.evalCount > 0
        )

        val pomXml = scala.xml.XML.loadFile(result.value.path.toString)
        val dependency = pomXml \ "dependencies" \ "dependency"
        assert(
          (pomXml \ "packaging").text == PackagingType.Pom,
          (dependency \ "artifactId").text == "slf4j-api",
          (dependency \ "groupId").text == "org.slf4j"
        )
      }
    }

    test("scopes") - UnitTester(compileAndRuntimeStuff, null).scoped { eval =>
      def assertClassPathContains(cp: Seq[os.Path], fileName: String) =
        assert(cp.map(_.last).contains(fileName))
      def assertClassPathDoesntContain(cp: Seq[os.Path], prefix: String) =
        assert(cp.map(_.last).forall(!_.startsWith(prefix)))

      def nothingClassPathCheck(cp: Seq[os.Path]): Unit = {
        assertClassPathDoesntContain(cp, "slf4j")
        assertClassPathDoesntContain(cp, "logback")
      }
      def compileClassPathCheck(cp: Seq[os.Path]): Unit = {
        assertClassPathContains(cp, "slf4j-api-2.0.15.jar")
        assertClassPathDoesntContain(cp, "logback")
      }
      def runtimeClassPathCheck(cp: Seq[os.Path]): Unit = {
        assertClassPathContains(cp, "slf4j-api-2.0.15.jar")
        assertClassPathContains(cp, "logback-classic-1.5.12.jar")
      }

      val compileCp =
        eval(compileAndRuntimeStuff.main.compileClasspath).right.get.value.toSeq.map(_.path)
      val runtimeCp =
        eval(compileAndRuntimeStuff.main.runClasspath).right.get.value.toSeq.map(_.path)

      compileClassPathCheck(compileCp)
      runtimeClassPathCheck(runtimeCp)

      val ivy2Repo = eval.evaluator.workspace / "ivy2Local"
      val m2Repo = eval.evaluator.workspace / "m2Local"

      eval(compileAndRuntimeStuff.main.publishLocal(ivy2Repo.toString)).right.get
      eval(compileAndRuntimeStuff.transitive.publishLocal(ivy2Repo.toString)).right.get
      eval(compileAndRuntimeStuff.runtimeTransitive.publishLocal(ivy2Repo.toString)).right.get
      eval(compileAndRuntimeStuff.main.publishM2Local(m2Repo.toString)).right.get
      eval(compileAndRuntimeStuff.transitive.publishM2Local(m2Repo.toString)).right.get
      eval(compileAndRuntimeStuff.runtimeTransitive.publishM2Local(m2Repo.toString)).right.get

      def localRepoCp(localRepo: coursierapi.Repository, moduleName: String, config: String) = {
        val dep = coursierapi.Dependency.of("com.lihaoyi.pubmodtests", moduleName, "0.1.0-SNAPSHOT")
        coursierapi.Fetch.create()
          .addDependencies(dep)
          .addRepositories(localRepo)
          .withResolutionParams(
            coursierapi.ResolutionParams.create()
              .withDefaultConfiguration(if (config.isEmpty) null else config)
          )
          .fetch()
          .asScala
          .map(os.Path(_))
          .toSeq
      }
      def ivy2Cp(moduleName: String, config: String) =
        localRepoCp(
          coursierapi.IvyRepository.of(ivy2Repo.toURI.toASCIIString + "[defaultPattern]"),
          moduleName,
          config
        )
      def m2Cp(moduleName: String, config: String) =
        localRepoCp(
          coursierapi.MavenRepository.of(m2Repo.toURI.toASCIIString),
          moduleName,
          config
        )

      val ivy2CompileCp = ivy2Cp("main", "compile")
      val ivy2RunCp = ivy2Cp("main", "runtime")
      val m2CompileCp = m2Cp("main", "compile")
      val m2RunCp = m2Cp("main", "runtime")

      compileClassPathCheck(ivy2CompileCp)
      compileClassPathCheck(m2CompileCp)
      runtimeClassPathCheck(ivy2RunCp)
      runtimeClassPathCheck(m2RunCp)

      val ivy2TransitiveCompileCp = ivy2Cp("transitive", "compile")
      val ivy2TransitiveRunCp = ivy2Cp("transitive", "runtime")
      val m2TransitiveCompileCp = m2Cp("transitive", "compile")
      val m2TransitiveRunCp = m2Cp("transitive", "runtime")

      compileClassPathCheck(ivy2TransitiveCompileCp)
      compileClassPathCheck(m2TransitiveCompileCp)
      runtimeClassPathCheck(ivy2TransitiveRunCp)
      runtimeClassPathCheck(m2TransitiveRunCp)

      val ivy2RuntimeTransitiveCompileCp = ivy2Cp("runtimeTransitive", "compile")
      val ivy2RuntimeTransitiveRunCp = ivy2Cp("runtimeTransitive", "runtime")
      val m2RuntimeTransitiveCompileCp = m2Cp("runtimeTransitive", "compile")
      val m2RuntimeTransitiveRunCp = m2Cp("runtimeTransitive", "runtime")

      // runtime dependency on the main module - doesn't pull anything from it
      // at compile time, hence the nothingClassPathCheck-s
      nothingClassPathCheck(ivy2RuntimeTransitiveCompileCp)
      nothingClassPathCheck(m2RuntimeTransitiveCompileCp)
      runtimeClassPathCheck(ivy2RuntimeTransitiveRunCp)
      runtimeClassPathCheck(m2RuntimeTransitiveRunCp)
    }

    test("docSourcesArgs") - UnitTester(compileAndRuntimeStuff, null).scoped { eval =>
      val ivy2Repo = eval.evaluator.workspace / "ivy2Local"
      val moduleName = "main"
      val subDir =
        os.sub / compileAndRuntimeStuff.organization / moduleName / compileAndRuntimeStuff.version
      def repoHasIvyXml(): Boolean =
        os.isFile(ivy2Repo / subDir / "ivys/ivy.xml")
      def repoHasJar(): Boolean =
        os.isFile(ivy2Repo / subDir / "jars" / s"$moduleName.jar")
      def repoHasSourcesJar(): Boolean =
        os.isFile(ivy2Repo / subDir / "srcs" / s"$moduleName-sources.jar")
      def repoHasDocJar(): Boolean =
        os.isFile(ivy2Repo / subDir / "docs" / s"$moduleName-javadoc.jar")
      def clearRepo(): Unit =
        os.remove.all(ivy2Repo)

      eval(compileAndRuntimeStuff.main.publishLocal(ivy2Repo.toString)).get
      assert(repoHasIvyXml())
      assert(repoHasJar())
      assert(repoHasSourcesJar())
      assert(repoHasDocJar())

      clearRepo()

      eval(compileAndRuntimeStuff.main.publishLocal(ivy2Repo.toString, doc = false)).get
      assert(repoHasIvyXml())
      assert(repoHasJar())
      assert(repoHasSourcesJar())
      assert(!repoHasDocJar())

      clearRepo()

      eval(compileAndRuntimeStuff.main.publishLocal(
        ivy2Repo.toString,
        doc = false,
        sources = false
      )).right.get
      assert(repoHasIvyXml())
      assert(repoHasJar())
      assert(!repoHasSourcesJar())
      assert(!repoHasDocJar())
    }

    test("transitive") - UnitTester(compileAndRuntimeStuff, null).scoped { eval =>
      val ivy2Repo = eval.evaluator.workspace / "ivy2Local"
      val mainModuleName = "main"
      val transitiveModuleName = "transitive"
      val runtimeTransitiveModuleName = "runtimeTransitive"
      def subDir(moduleName: String) =
        os.sub / compileAndRuntimeStuff.organization / moduleName / compileAndRuntimeStuff.version
      def repoHasIvyXml(moduleName: String): Boolean =
        os.isFile(ivy2Repo / subDir(moduleName) / "ivys/ivy.xml")
      def repoHasJar(moduleName: String): Boolean =
        os.isFile(ivy2Repo / subDir(moduleName) / "jars" / s"$moduleName.jar")
      def repoHasSourcesJar(moduleName: String): Boolean =
        os.isFile(ivy2Repo / subDir(moduleName) / "srcs" / s"$moduleName-sources.jar")
      def repoHasDocJar(moduleName: String): Boolean =
        os.isFile(ivy2Repo / subDir(moduleName) / "docs" / s"$moduleName-javadoc.jar")
      def clearRepo(): Unit =
        os.remove.all(ivy2Repo)

      eval(compileAndRuntimeStuff.transitive.publishLocal(ivy2Repo.toString)).get
      assert(!repoHasIvyXml(mainModuleName))
      assert(!repoHasJar(mainModuleName))
      assert(!repoHasSourcesJar(mainModuleName))
      assert(!repoHasDocJar(mainModuleName))
      assert(repoHasIvyXml(transitiveModuleName))
      assert(repoHasJar(transitiveModuleName))
      assert(repoHasSourcesJar(transitiveModuleName))
      assert(repoHasDocJar(transitiveModuleName))

      clearRepo()

      eval(compileAndRuntimeStuff.transitive.publishLocal(
        ivy2Repo.toString,
        transitive = true
      )).get
      assert(repoHasIvyXml(mainModuleName))
      assert(repoHasJar(mainModuleName))
      assert(repoHasSourcesJar(mainModuleName))
      assert(repoHasDocJar(mainModuleName))
      assert(repoHasIvyXml(transitiveModuleName))
      assert(repoHasJar(transitiveModuleName))
      assert(repoHasSourcesJar(transitiveModuleName))
      assert(repoHasDocJar(transitiveModuleName))

      clearRepo()

      eval(compileAndRuntimeStuff.transitive.publishLocal(ivy2Repo.toString, doc = false)).get
      assert(!repoHasIvyXml(mainModuleName))
      assert(!repoHasJar(mainModuleName))
      assert(!repoHasSourcesJar(mainModuleName))
      assert(!repoHasDocJar(mainModuleName))
      assert(repoHasIvyXml(transitiveModuleName))
      assert(repoHasJar(transitiveModuleName))
      assert(repoHasSourcesJar(transitiveModuleName))
      assert(!repoHasDocJar(transitiveModuleName))

      clearRepo()

      eval(compileAndRuntimeStuff.transitive.publishLocal(
        ivy2Repo.toString,
        doc = false,
        transitive = true
      )).right.get
      assert(repoHasIvyXml(mainModuleName))
      assert(repoHasJar(mainModuleName))
      assert(repoHasSourcesJar(mainModuleName))
      assert(!repoHasDocJar(mainModuleName))
      assert(repoHasIvyXml(transitiveModuleName))
      assert(repoHasJar(transitiveModuleName))
      assert(repoHasSourcesJar(transitiveModuleName))
      assert(!repoHasDocJar(transitiveModuleName))

      clearRepo()

      eval(compileAndRuntimeStuff.runtimeTransitive.publishLocal(ivy2Repo.toString)).get
      assert(!repoHasIvyXml(mainModuleName))
      assert(!repoHasJar(mainModuleName))
      assert(!repoHasSourcesJar(mainModuleName))
      assert(!repoHasDocJar(mainModuleName))
      assert(repoHasIvyXml(runtimeTransitiveModuleName))
      assert(repoHasJar(runtimeTransitiveModuleName))
      assert(repoHasSourcesJar(runtimeTransitiveModuleName))
      assert(repoHasDocJar(runtimeTransitiveModuleName))

      clearRepo()

      eval(compileAndRuntimeStuff.runtimeTransitive.publishLocal(
        ivy2Repo.toString,
        transitive = true
      )).get
      assert(repoHasIvyXml(mainModuleName))
      assert(repoHasJar(mainModuleName))
      assert(repoHasSourcesJar(mainModuleName))
      assert(repoHasDocJar(mainModuleName))
      assert(repoHasIvyXml(runtimeTransitiveModuleName))
      assert(repoHasJar(runtimeTransitiveModuleName))
      assert(repoHasSourcesJar(runtimeTransitiveModuleName))
      assert(repoHasDocJar(runtimeTransitiveModuleName))
    }
  }

}
