package mill.exec

import mill.api.ExecResult.{OuterStack, Success}
import mill.api.*
import mill.api.internal.{Appendable, Cached, Located}
import mill.internal.{CodeSigUtils, FileLogger, MultiLogger}

import java.lang.reflect.Method
import java.util.concurrent.ThreadPoolExecutor
import scala.collection.mutable
import scala.util.control.NonFatal
import scala.util.hashing.MurmurHash3
import mill.api.daemon.internal.{
  BaseModuleApi,
  CompileProblemReporter,
  EvaluatorApi,
  TaskApi,
  TestReporter
}
import mill.internal.SpanningForest
import upickle.core.BufferedValue

import java.io.ByteArrayOutputStream

/**
 * Logic around evaluating a single group, which is a collection of [[Task]]s
 * with a single [[Terminal]].
 */
trait GroupExecution {
  def workspace: os.Path
  def outPath: os.Path
  def externalOutPath: os.Path
  def rootModule: BaseModuleApi
  def classLoaderSigHash: Int
  def classLoaderIdentityHash: Int

  /**
   * Tracks tasks invalidated due to version/classloader mismatch, with the reason string.
   * Populated by loadCachedJson, used by ExecutionLogs.logInvalidationTree.
   */
  def versionMismatchReasons: java.util.concurrent.ConcurrentHashMap[Task[?], String]

  /**
   * `String` is the worker name, `Int` is the worker hash, `Val` is the worker instance,
   * `TaskApi[?]` is the worker's Task (for traversing dependencies during ordered closure).
   */
  def workerCache: mutable.Map[String, (Int, Val, TaskApi[?])]

  /**
   * Lazily computed worker dependency graph and its reverse, for efficient worker closure ordering.
   */
  def workerDeps: Seq[(TaskApi[?], Seq[TaskApi[?]])]
  def reverseDeps: Map[TaskApi[?], Seq[TaskApi[?]]]
  def workerTopoIndex: Map[TaskApi[?], Int]

  def env: Map[String, String]
  def failFast: Boolean
  def ec: Option[ThreadPoolExecutor]
  def codeSignatures: Map[String, Int]
  def systemExit: ( /* reason */ String, /* exitCode */ Int) => Nothing
  def exclusiveSystemStreams: SystemStreams
  def getEvaluator: () => EvaluatorApi
  def staticBuildOverrideFiles: Map[java.nio.file.Path, String]

  /** Evaluate a build override YAML value and deserialize it */
  private def evaluateBuildOverride(
      located: Located[Appendable[BufferedValue]],
      labelled: Task.Named[_]
  ): Either[upickle.core.TraceVisitor.TraceException, Any] = {
    try {
      val reader = labelled.readWriterOpt.getOrElse {
        throw new IllegalStateException(
          s"No ReadWriter registered for task '${labelled.ctx.segments.render}' " +
            s"in module '${labelled.ctx.enclosingModule.getClass.getName}'. " +
            s"This task cannot be overridden from YAML configuration."
        )
      }
      Right(PathRef.currentOverrideModulePath.withValue(
        labelled.ctx.enclosingModule.moduleCtx.millSourcePath
      ) {
        upickle.read[Any](interpolateEnvVarsInJson(located.value.value))(
          using reader.asInstanceOf[upickle.Reader[Any]]
        )
      })
    } catch {
      case e: upickle.core.TraceVisitor.TraceException => Left(e)
    }
  }

  /** Create an ExecResult.Failure from a deserialization exception */
  private def buildOverrideDeserializationError(
      e: upickle.core.TraceVisitor.TraceException,
      located: Located[Appendable[BufferedValue]]
  ) = {
    val errorIndex = e.getCause match {
      case abort: upickle.core.AbortException => abort.index
      case _ => located.value.value.index
    }
    val msg = s"Failed de-serializing config override: ${e.getCause.getMessage}"
    (
      ExecResult.Failure(
        msg,
        Some(Result.Failure(msg, path = located.path.toNIO, index = errorIndex))
      ),
      Nil
    )
  }

  val staticBuildOverrides: Map[String, Located[Appendable[BufferedValue]]] = {
    staticBuildOverrideFiles
      .flatMap { case (path0, rawText) =>
        val path = os.Path(path0)
        val headerDataReader = mill.api.internal.HeaderData.headerDataReader(path)

        def rec(
            segments: Seq[String],
            bufValue: upickle.core.BufferedValue
        ): Seq[(String, Located[Appendable[BufferedValue]])] = {
          val upickle.core.BufferedValue.Obj(kvs, _, _) = bufValue
          val (rawKvs, nested) = kvs.partitionMap {
            case (upickle.core.BufferedValue.Str(k, i), v) =>
              k.toString.split(" +") match {
                case Array(k) => Left((k, i, v))
                case Array("object", k) => Right(rec(segments ++ Seq(k), v))
              }
          }

          val currentResults: Seq[(String, Located[Appendable[BufferedValue]])] =
            BufferedValue.transform(
              BufferedValue.Obj(
                rawKvs.map { case (k, i, v) => (BufferedValue.Str(k, i), v) }.to(
                  mutable.ArrayBuffer
                ),
                true,
                -1
              ),
              headerDataReader
            )
              .rest
              .map { case (k, v) =>
                val (actualValue, append) = Appendable.unwrapAppendMarker(v)
                (segments ++ Seq(k.value)).mkString(".") -> Located(
                  path,
                  k.index,
                  Appendable(actualValue, append)
                )
              }
              .toSeq

          val nestedResults: Seq[(String, Located[Appendable[BufferedValue]])] =
            nested.flatten.toSeq

          currentResults ++ nestedResults
        }

        val parsed0 = BufferedValue.Obj(
          mill.internal.Util.parseYaml0(
            path0.toString,
            rawText.replace("\r", ""),
            headerDataReader
          ).get
            .rest
            .map { case (k, v) => (BufferedValue.Str(k.value, k.index), v) }
            .to(mutable.ArrayBuffer),
          true,
          -1
        )
        if ((path / "..").startsWith(workspace)) {
          rec(
            (path / "..").subRelativeTo(workspace).segments,
            if (path == os.Path(rootModule.moduleDirJava) / "../build.mill.yaml") {
              parsed0
                .value0
                .collectFirst { case (BufferedValue.Str("mill-build", _), v) => v }
                .getOrElse(BufferedValue.Obj(mutable.ArrayBuffer.empty, true, 0))
            } else parsed0
          )
        } else Nil
      }
      .toMap
  }

  def offline: Boolean
  def useFileLocks: Boolean

  lazy val constructorHashSignatures: Map[String, Seq[(String, Int)]] =
    CodeSigUtils.constructorHashSignatures(codeSignatures)

  val effectiveThreadCount: Int =
    ec.map(_.getMaximumPoolSize).getOrElse(1)

  private val envVarsForInterpolation = Seq(
    "PWD" -> workspace.toString,
    "PWD_URI" -> workspace.toURI.toString,
    "MILL_VERSION" -> mill.constants.BuildInfo.millVersion,
    "MILL_BIN_PLATFORM" -> mill.constants.BuildInfo.millBinPlatform
  )

  /** Recursively examine all `ujson.Str` values and replace '${VAR}' patterns. */
  private def interpolateEnvVarsInJson(json: upickle.core.BufferedValue): ujson.Value = {
    import scala.jdk.CollectionConverters.*
    val envWithPwd = (env ++ envVarsForInterpolation).asJava

    // recursively convert java data structure to ujson.Value
    def rec(json: ujson.Value): ujson.Value = json match {
      case ujson.Str(s) => mill.constants.Util.interpolateEnvVars(s, envWithPwd)
      case ujson.Arr(xs) => ujson.Arr(xs.map(rec))
      case ujson.Obj(kvs) => ujson.Obj.from(kvs.map((k, v) => (k, rec(v))))
      case v => v
    }

    rec(upickle.core.BufferedValue.transform(json, ujson.Value))
  }

  // the JVM running this code currently
  val javaHomeHash = sys.props("java.home").hashCode

  val invalidateAllHashes = classLoaderSigHash + javaHomeHash

  // those result which are inputs but not contained in this terminal group
  def executeGroupCached(
      terminal: Task[?],
      group: Seq[Task[?]],
      results: Map[Task[?], ExecResult[(Val, Int)]],
      countMsg: String,
      zincProblemReporter: Int => Option[CompileProblemReporter],
      testReporter: TestReporter,
      logger: Logger,
      deps: Seq[Task[?]],
      classToTransitiveClasses: Map[Class[?], IndexedSeq[Class[?]]],
      allTransitiveClassMethods: Map[Class[?], Map[String, Method]],
      executionContext: mill.api.TaskCtx.Fork.Api,
      exclusive: Boolean,
      upstreamPathRefs: Seq[PathRef]
  ): GroupExecution.Results = {

    val inputsHash = {
      val externalInputsHash = MurmurHash3.orderedHash(
        group.flatMap(_.inputs).filter(!group.contains(_))
          .flatMap(results(_).asSuccess.map(_.value._2))
      )
      val sideHashes = MurmurHash3.orderedHash(group.iterator.map(_.sideHash))
      val scriptsHash = MurmurHash3.orderedHash(
        group
          .iterator
          .collect { case namedTask: Task.Named[_] =>
            CodeSigUtils.codeSigForTask(
              namedTask = namedTask,
              classToTransitiveClasses = classToTransitiveClasses,
              allTransitiveClassMethods = allTransitiveClassMethods,
              codeSignatures = codeSignatures,
              constructorHashSignatures = constructorHashSignatures
            )
          }
          .flatten
      )

      externalInputsHash + sideHashes + scriptsHash + invalidateAllHashes
    }

    terminal match {
      case labelled: Task.Named[_] =>
        val out = if (!labelled.ctx.external) outPath else externalOutPath
        val paths = ExecutionPaths.resolve(out, labelled.ctx.segments)
        val dynamicBuildOverride = labelled.ctx.enclosingModule.moduleDynamicBuildOverrides
        val buildOverrideOpt = staticBuildOverrides.get(labelled.ctx.segments.render)
          .orElse(dynamicBuildOverride.get(labelled.ctx.segments.render))

        // Helper to create a cached result (no evaluation occurred)
        def cachedResult(
            execRes: ExecResult[(Val, Int)],
            serializedPaths: Seq[PathRef]
        ): GroupExecution.Results = GroupExecution.Results(
          newResults = Map(labelled -> execRes),
          newEvaluated = Nil,
          cached = true,
          inputsHash = inputsHash,
          previousInputsHash = -1,
          valueHashChanged = false,
          serializedPaths = serializedPaths
        )

        // Helper to evaluate the task with full caching support
        def evaluateTaskWithCaching(): GroupExecution.Results = {
          val cached = loadCachedJson(logger, inputsHash, labelled, paths)

          // `cached.isEmpty` means worker metadata file removed by user so recompute the worker
          val (multiLogger, fileLoggerOpt) = resolveLogger(Some(paths).map(_.log), logger)
          val upToDateWorker = loadUpToDateWorker(
            logger = logger,
            inputsHash = inputsHash,
            labelled = labelled,
            forceDiscard = cached.isEmpty,
            deps = deps,
            paths = Some(paths),
            upstreamPathRefs = upstreamPathRefs,
            exclusive = exclusive,
            multiLogger = multiLogger,
            counterMsg = countMsg,
            destCreator = new GroupExecution.DestCreator(Some(paths)),
            terminal = terminal
          )

          val cachedValueAndHash =
            upToDateWorker.map(w => (w -> Nil, inputsHash))
              .orElse(cached.flatMap { case (_, valOpt, valueHash) =>
                valOpt.map((_, valueHash))
              })

          cachedValueAndHash match {
            case Some(((v, serializedPaths), hashCode)) =>
              cachedResult(ExecResult.Success((v, hashCode)), serializedPaths)

            case _ =>
              // uncached
              if (!labelled.persistent && os.exists(paths.dest)) {
                logger.debug(s"Deleting task dest dir ${paths.dest.relativeTo(workspace)}")
                os.remove.all(paths.dest)
              }

              val (newResults, newEvaluated) =
                executeGroup(
                  group = group,
                  results = results,
                  inputsHash = inputsHash,
                  paths = Some(paths),
                  taskLabelOpt = Some(terminal.toString),
                  counterMsg = countMsg,
                  reporter = zincProblemReporter,
                  testReporter = testReporter,
                  logger = logger,
                  executionContext = executionContext,
                  exclusive = exclusive,
                  deps = deps,
                  upstreamPathRefs = upstreamPathRefs,
                  terminal = labelled
                )

              val (valueHash, serializedPaths) = newResults(labelled) match {
                case ExecResult.Success((v, _)) =>
                  val valueHash = getValueHash(v, terminal, inputsHash)
                  val serializedPaths =
                    handleTaskResult(v, valueHash, paths.meta, inputsHash, labelled)
                  (valueHash, serializedPaths)

                case _ =>
                  // Wipe out any cached meta.json file that exists, so
                  // a following run won't look at the cached metadata file and
                  // assume it's associated with the possibly-borked state of the
                  // destPath after an evaluation failure.
                  os.remove.all(paths.meta)
                  (0, Nil)
              }

              GroupExecution.Results(
                newResults = newResults,
                newEvaluated = newEvaluated.toSeq,
                cached =
                  if (
                    labelled.isInstanceOf[Task.Input[?]] ||
                    labelled.isInstanceOf[Task.Uncached[?]]
                  ) null
                  else false,
                inputsHash = inputsHash,
                previousInputsHash = cached.map(_._1).getOrElse(-1),
                valueHashChanged = !cached.map(_._3).contains(valueHash),
                serializedPaths = serializedPaths
              )
          }
        }

        // Helper to evaluate build override only (no task evaluation)
        def evaluateBuildOverrideOnly(located: Located[Appendable[BufferedValue]])
            : GroupExecution.Results = {

          val (execRes, serializedPaths) =
            if (os.Path(labelled.ctx.fileName).endsWith("mill-build/build.mill")) {
              val msg =
                s"Build header config conflicts with task defined in ${os.Path(labelled.ctx.fileName).relativeTo(workspace)}:${labelled.ctx.lineNum}"
              (
                ExecResult.Failure(
                  msg,
                  Some(Result.Failure(msg, path = located.path.toNIO, index = located.index))
                ),
                Nil
              )
            } else {
              evaluateBuildOverride(located, labelled) match {
                case Right(yamlValue) =>
                  val (data, serializedPaths) = PathRef.withSerializedPaths { yamlValue }
                  // Write build header override JSON to meta `.json` file to support `show`
                  writeCacheJson(
                    paths.meta,
                    upickle.core.BufferedValue.transform(located.value.value, ujson.Value),
                    data.##,
                    inputsHash + located.value.value.##
                  )
                  (ExecResult.Success(Val(data), data.##), serializedPaths)
                case Left(e) => buildOverrideDeserializationError(e, located)
              }
            }

          cachedResult(execRes, serializedPaths)
        }

        // Three-way conditional:
        // 1. Build override only (!append): just evaluate YAML
        // 2. Both (append): evaluate task with caching, evaluate YAML, merge
        // 3. Task only (no override): evaluate task with caching
        buildOverrideOpt match {
          case Some(appendLocated) if appendLocated.value.append =>
            val taskResults = evaluateTaskWithCaching()

            // Check if task evaluation failed - if so, propagate the failure
            taskResults.newResults.get(labelled) match {
              case Some(ExecResult.Success((v, _))) =>
                val taskValue = v.value.asInstanceOf[Seq[Any]]
                // Evaluate the YAML override and merge
                evaluateBuildOverride(appendLocated, labelled) match {
                  case Right(yamlValue) =>
                    val (mergedData, serializedPaths) = PathRef.withSerializedPaths {
                      (taskValue ++ yamlValue.asInstanceOf[Seq[Any]]).asInstanceOf[Any]
                    }
                    // Don't write merged result to paths.meta - that would overwrite the task
                    // cache and cause the task to re-execute every time. The task cache stays
                    // valid with inputsHash, and show uses the in-memory result from newResults.
                    taskResults.copy(
                      newResults =
                        Map(labelled -> ExecResult.Success(Val(mergedData), mergedData.##)),
                      serializedPaths = serializedPaths
                    )
                  case Left(e) =>
                    val (failure, _) = buildOverrideDeserializationError(e, appendLocated)
                    taskResults.copy(
                      newResults = Map(labelled -> failure),
                      serializedPaths = Nil
                    )
                }
              // Task evaluation failed - propagate the failure
              case _ => taskResults
            }

          // Build override only (no append)
          case Some(appendLocated) => evaluateBuildOverrideOnly(appendLocated)

          // Task only (no build override)
          case None => evaluateTaskWithCaching()
        }
      case _ =>
        val (newResults, newEvaluated) = executeGroup(
          group = group,
          results = results,
          inputsHash = inputsHash,
          paths = None,
          taskLabelOpt = None,
          counterMsg = countMsg,
          reporter = zincProblemReporter,
          testReporter = testReporter,
          logger = logger,
          executionContext = executionContext,
          exclusive = exclusive,
          deps = deps,
          upstreamPathRefs = upstreamPathRefs,
          terminal = terminal
        )
        GroupExecution.Results(
          newResults = newResults,
          newEvaluated = newEvaluated.toSeq,
          cached = null,
          inputsHash = inputsHash,
          previousInputsHash = -1,
          valueHashChanged = false,
          serializedPaths = Nil
        )

    }
  }

  private def executeGroup(
      group: Seq[Task[?]],
      results: Map[Task[?], ExecResult[(Val, Int)]],
      inputsHash: Int,
      paths: Option[ExecutionPaths],
      taskLabelOpt: Option[String],
      counterMsg: String,
      reporter: Int => Option[CompileProblemReporter],
      testReporter: TestReporter,
      logger: mill.api.Logger,
      executionContext: mill.api.TaskCtx.Fork.Api,
      exclusive: Boolean,
      deps: Seq[Task[?]],
      upstreamPathRefs: Seq[PathRef],
      terminal: Task[?]
  ): (Map[Task[?], ExecResult[(Val, Int)]], mutable.Buffer[Task[?]]) = {
    val newEvaluated = mutable.Buffer.empty[Task[?]]
    val newResults = mutable.Map.empty[Task[?], ExecResult[(Val, Int)]]

    val nonEvaluatedTasks = group.toIndexedSeq.filterNot(results.contains)
    val (multiLogger, fileLoggerOpt) = resolveLogger(paths.map(_.log), logger)

    val destCreator = new GroupExecution.DestCreator(paths)

    for (task <- nonEvaluatedTasks) {
      newEvaluated.append(task)
      val taskInputValues = task.inputs
        .map { x => newResults.getOrElse(x, results(x)) }
        .collect { case ExecResult.Success((v, _)) => v }

      val res = {
        if (taskInputValues.length != task.inputs.length) ExecResult.Skipped
        else {
          val args = new mill.api.TaskCtx.Impl(
            args = taskInputValues.map(_.value).toIndexedSeq,
            dest0 = () => destCreator.makeDest(),
            log = multiLogger,
            env = env,
            reporter = reporter,
            testReporter = testReporter,
            workspace = workspace,
            _systemExitWithReason = systemExit,
            fork = executionContext,
            jobs = effectiveThreadCount,
            offline = offline,
            useFileLocks = useFileLocks
          )

          GroupExecution.wrap(
            workspace = workspace,
            deps = deps,
            outPath = outPath,
            paths = paths,
            upstreamPathRefs = upstreamPathRefs,
            exclusive = exclusive,
            multiLogger = multiLogger,
            logger = logger,
            exclusiveSystemStreams = exclusiveSystemStreams,
            counterMsg = counterMsg,
            destCreator = destCreator,
            evaluator = getEvaluator().asInstanceOf[Evaluator],
            terminal = terminal,
            classLoader = rootModule.getClass.getClassLoader
          ) {
            try {
              task.evaluate(args) match {
                case Result.Success(v) => ExecResult.Success(Val(v))
                case f: Result.Failure => ExecResult.Failure(f.error, Some(f))
              }
            } catch {
              case ex: Result.Exception => ExecResult.Failure(ex.error, ex.failure)
              case NonFatal(e) =>
                ExecResult.Exception(
                  e,
                  new OuterStack(new Exception().getStackTrace.toIndexedSeq)
                )
              case e: Throwable => throw e
            }
          }
        }
      }

      newResults(task) = for (v <- res) yield (v, getValueHash(v, task, inputsHash))
    }

    fileLoggerOpt.foreach(_.close())

    if (!failFast) taskLabelOpt.foreach { taskLabel =>
      val taskFailed = newResults.exists(task => task._2.isInstanceOf[ExecResult.Failing[?]])
      if (taskFailed) logger.error(s"$taskLabel task failed")
    }

    (newResults.toMap, newEvaluated)
  }

  // Include the classloader identity hash as part of the worker hash. This is
  // because unlike other tasks, workers are long-lived in memory objects,
  // and are not re-instantiated every run. Thus, we need to make sure we
  // invalidate workers in the scenario where a worker classloader is
  // re-created - so the worker *class* changes - but the *value* inputs to the
  // worker does not change. This typically happens when the worker class is
  // brought in via `//| mvnDeps`, since the class then comes from the
  // non-bootstrap classloader which can be re-created when the `build.mill` file
  // changes.
  //
  // We do not want to do this for normal tasks, because those are always
  // read from disk and re-instantiated every time, so whether the
  // classloader/class is the same or different doesn't matter.
  def workerCacheHash(inputHash: Int): Int = inputHash + classLoaderIdentityHash

  private def handleTaskResult(
      v: Val,
      hashCode: Int,
      metaPath: os.Path,
      inputsHash: Int,
      labelled: Task.Named[?]
  ): Seq[PathRef] = {
    for (w <- labelled.asWorker) workerCache.synchronized {
      workerCache.update(w.ctx.segments.render, (workerCacheHash(inputsHash), v, labelled))
    }

    def normalJson(w: upickle.Writer[?]) = PathRef.withSerializedPaths {
      upickle.writeJs(v.value)(using w.asInstanceOf[upickle.Writer[Any]])
    }
    lazy val workerJson = labelled.asWorker.map { _ =>
      Task.workerJson(labelled.toString, v.value, inputsHash) -> Nil
    }

    val terminalResult: Option[(ujson.Value, Seq[PathRef])] = labelled
      .writerOpt
      .map(normalJson)
      .orElse(workerJson)

    terminalResult match {
      case Some((json, serializedPaths)) =>
        writeCacheJson(metaPath, json, hashCode, inputsHash)
        serializedPaths
      case _ =>
        Nil
    }
  }

  def writeCacheJson(metaPath: os.Path, json: ujson.Value, hashCode: Int, inputsHash: Int) = {
    os.write.over(
      metaPath,
      upickle.stream(
        Cached(
          json,
          hashCode,
          inputsHash,
          millVersion = mill.constants.BuildInfo.millVersion,
          millJvmVersion = sys.props("java.version"),
          classLoaderSigHash = classLoaderSigHash
        ),
        indent = 4
      ),
      createFolders = true
    )
  }

  def resolveLogger(
      logPath: Option[os.Path],
      logger: mill.api.Logger
  ): (mill.api.Logger, Option[AutoCloseable]) =
    logPath match {
      case None => (logger, None)
      case Some(path) =>
        val fileLogger = new FileLogger(path)
        val multiLogger = new MultiLogger(
          logger,
          fileLogger,
          logger.streams.in
        )
        (multiLogger, Some(fileLogger))
    }

  private def loadCachedJson(
      logger: Logger,
      inputsHash: Int,
      labelled: Task.Named[?],
      paths: ExecutionPaths
  ): Option[(Int, Option[(Val, Seq[PathRef])], Int)] = {
    for {
      cached <-
        try Some(upickle.read[Cached](paths.meta.toIO, trace = false))
        catch {
          case NonFatal(_) => None
        }
    } yield {
      // Check for version/classloader mismatch - treat as cache miss if they differ
      def checkMatch[T](cachedValue: T, currentValue: T, reasonName: String): Boolean = {
        val matches = cachedValue == currentValue
        if (!matches) {
          versionMismatchReasons.putIfAbsent(labelled, s"$reasonName:$cachedValue->$currentValue")
        }
        matches
      }

      val currentMillVersion = mill.constants.BuildInfo.millVersion
      val currentJvmVersion = sys.props("java.version")
      val millVersionMatches =
        checkMatch(cached.millVersion, currentMillVersion, "mill-version-changed")
      val jvmVersionMatches =
        checkMatch(cached.millJvmVersion, currentJvmVersion, "mill-jvm-version-changed")
      val classLoaderMatches =
        checkMatch(cached.classLoaderSigHash, classLoaderSigHash, "classpath-changed")

      (
        cached.inputsHash,
        for {
          _ <- Option.when(
            cached.inputsHash == inputsHash && millVersionMatches && jvmVersionMatches && classLoaderMatches
          )(())
          reader <- labelled.readWriterOpt
          (parsed, serializedPaths) <-
            try Some(PathRef.withSerializedPaths(upickle.read(cached.value, trace = false)(using
                reader
              )))
            catch {
              case e: PathRef.PathRefValidationException =>
                logger.debug(
                  s"$labelled: re-evaluating; ${e.getMessage}"
                )
                None
              case NonFatal(_) => None
            }
        } yield (Val(parsed), serializedPaths),
        cached.valueHash
      )
    }
  }

  def getValueHash(v: Val, task: Task[?], inputsHash: Int): Int = {
    if (task.isInstanceOf[Task.Worker[?]]) inputsHash else v.## + invalidateAllHashes
  }

  private def loadUpToDateWorker(
      logger: Logger,
      inputsHash: Int,
      labelled: Task.Named[?],
      forceDiscard: Boolean,
      deps: Seq[Task[?]],
      paths: Option[ExecutionPaths],
      upstreamPathRefs: Seq[PathRef],
      exclusive: Boolean,
      multiLogger: Logger,
      counterMsg: String,
      destCreator: GroupExecution.DestCreator,
      terminal: Task[?]
  ): Option[Val] = {
    labelled.asWorker
      .flatMap { w =>
        workerCache.synchronized {
          workerCache.get(w.ctx.segments.render)
        }
      }
      .flatMap {
        case (cachedHash, upToDate, _)
            if cachedHash == workerCacheHash(inputsHash) && !forceDiscard =>
          Some(upToDate)

        case (_, Val(_: AutoCloseable), _) =>
          // Close this worker and all workers that depend on it
          val allToClose =
            SpanningForest.breadthFirst(Seq(labelled: TaskApi[?]))(n =>
              reverseDeps.getOrElse(n, Nil)
            )
          GroupExecution.closeWorkersInReverseTopologicalOrder(
            allToClose,
            workerCache,
            workerTopoIndex,
            closeable =>
              try GroupExecution.wrap(
                  workspace,
                  deps,
                  outPath,
                  paths,
                  upstreamPathRefs,
                  exclusive,
                  multiLogger,
                  logger,
                  exclusiveSystemStreams,
                  counterMsg,
                  destCreator,
                  getEvaluator().asInstanceOf[Evaluator],
                  terminal,
                  rootModule.getClass.getClassLoader
                )(closeable.close())
              catch { case NonFatal(e) => logger.error(s"Error closing worker: ${e.getMessage}") }
          )
          None

        case _ => None
      }
  }
}

object GroupExecution {

  class DestCreator(paths: Option[ExecutionPaths]) {
    var usedDest = Option.empty[os.Path]

    def makeDest() = this.synchronized {
      paths match {
        case Some(dest) =>
          if (usedDest.isEmpty) os.makeDir.all(dest.dest)
          usedDest = Some(dest.dest)
          dest.dest

        case None => throw new Exception("No `dest` folder available here")
      }
    }
  }

  class ExecutionChecker(
      workspace: os.Path,
      isCommand: Boolean,
      isInput: Boolean,
      terminal: Task[?],
      validReadDests: Seq[os.Path],
      validWriteDests: Seq[os.Path]
  ) extends os.Checker {
    def onRead(path: os.ReadablePath): Unit = path match {
      case path: os.Path =>
        if (!isCommand && !isInput && mill.api.FilesystemCheckerEnabled.value) {
          if (path.startsWith(workspace) && !validReadDests.exists(path.startsWith)) {
            sys.error(
              s"Reading from ${path.relativeTo(workspace)} not allowed during execution of `$terminal`.\n" +
                "You can only read files referenced by `Task.Source` or `Task.Sources`, or within a `Task.Input"
            )
          }
        }
      case _ =>
    }

    def onWrite(path: os.Path): Unit = {
      if (!isCommand && mill.api.FilesystemCheckerEnabled.value) {
        if (path.startsWith(workspace) && !validWriteDests.exists(path.startsWith)) {
          sys.error(
            s"Writing to ${path.relativeTo(workspace)} not allowed during execution of `$terminal`.\n" +
              "Normal `Task`s can only write to files within their `Task.dest` folder, only `Task.Command`s can write to other arbitrary files."
          )
        }
      }
    }
  }

  def wrap[T](
      workspace: os.Path,
      deps: Seq[Task[?]],
      outPath: os.Path,
      paths: Option[ExecutionPaths],
      upstreamPathRefs: Seq[PathRef],
      exclusive: Boolean,
      multiLogger: Logger,
      logger: Logger,
      exclusiveSystemStreams: SystemStreams,
      counterMsg: String,
      destCreator: DestCreator,
      evaluator: Evaluator,
      terminal: Task[?],
      classLoader: ClassLoader
  )(t: => T): T = {
    // Tasks must be allowed to write to upstream worker's dest folders, because
    // the point of workers is to manualy manage long-lived state which includes
    // state on disk.
    val validWriteDests =
      deps.collect { case n: Task.Worker[?] =>
        ExecutionPaths.resolve(outPath, n.ctx.segments).dest
      } ++
        paths.map(_.dest)

    val validReadDests = validWriteDests ++ upstreamPathRefs.map(_.path)

    val isCommand = terminal.isInstanceOf[Task.Command[?]]
    val isInput = terminal.isInstanceOf[Task.Input[?]]
    val executionChecker =
      new ExecutionChecker(workspace, isCommand, isInput, terminal, validReadDests, validWriteDests)
    val (streams, destFunc) =
      if (exclusive) (exclusiveSystemStreams, () => workspace)
      else (multiLogger.streams, () => destCreator.makeDest())

    os.dynamicPwdFunction.withValue(destFunc) {
      os.checker.withValue(executionChecker) {
        mill.api.SystemStreamsUtils.withStreams(streams) {
          val exposedEvaluator =
            if (exclusive) evaluator.asInstanceOf[Evaluator]
            else new EvaluatorProxy(() =>
              sys.error(
                "No evaluator available here; Evaluator is only available in exclusive commands"
              )
            )

          Evaluator.withCurrentEvaluator(exposedEvaluator) {
            // Ensure the class loader used to load user code
            // is set as context class loader when running user code.
            // This is useful if users rely on libraries that look
            // for resources added by other libraries, by using
            // using java.util.ServiceLoader for example.
            mill.api.ClassLoader.withContextClassLoader(classLoader) {
              if (!exclusive) t
              else {
                // For exclusive tasks, we print the task name once and then we disable the
                // prompt/ticker so the output of the exclusive task can "clean" while still
                // being identifiable
                logger.prompt.logPrefixedLine(Seq(counterMsg), new ByteArrayOutputStream(), false)
                logger.prompt.withPromptPaused {
                  t
                }
              }
            }
          }
        }
      }
    }
  }
  case class Results(
      newResults: Map[Task[?], ExecResult[(Val, Int)]],
      newEvaluated: Seq[Task[?]],
      cached: java.lang.Boolean,
      inputsHash: Int,
      previousInputsHash: Int,
      valueHashChanged: Boolean,
      serializedPaths: Seq[PathRef]
  )

  def workerDependencies(
      workerCache: Map[String, (Int, Val, TaskApi[?])]
  ): Seq[(TaskApi[?], Seq[TaskApi[?]])] = {
    // Build worker-to-worker edges only (direct worker inputs)
    val workers = workerCache.values.map(_._3).toSeq
    workers.map { worker =>
      val directWorkerInputs = worker.inputsApi.filter(_.workerNameApi.isDefined)
      (worker, directWorkerInputs)
    }
  }

  def closeWorkersInReverseTopologicalOrder(
      workersToClose: Iterable[TaskApi[?]],
      workerCache: mutable.Map[String, (Int, Val, TaskApi[?])],
      topoIndex: Map[TaskApi[?], Int],
      closeAction: AutoCloseable => Unit
  ): Unit = {
    for (worker <- workersToClose.toSeq.sortBy(w => -topoIndex(w))) {
      val name = worker.workerNameApi.get
      workerCache.synchronized(workerCache.remove(name)).foreach {
        case (_, Val(closeable: AutoCloseable), _) => closeAction(closeable)
        case _ =>
      }
    }
  }
}
