#!/bin/bash

# Comprehensive test for P0 implementation (P0-1 + P0-2)
# Tests MCP prompt integration and prompt simplification

echo "=========================================="
echo "Comprehensive P0 Implementation Test"
echo "=========================================="
echo ""

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m'

PASSED=0
FAILED=0

test_check() {
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✓${NC} $1"
        ((PASSED++))
    else
        echo -e "${RED}✗${NC} $1"
        ((FAILED++))
    fi
}

# P0-1: MCP Prompt Tests
echo -e "${YELLOW}=== P0-1: MCP Prompt Emergency Fix ===${NC}"
echo ""

echo -e "${YELLOW}Test 1.1: MCP prompt module exists...${NC}"
if [ -f "src/core/agents/mcp_prompt.cj" ]; then
    test_check "MCP prompt module exists"
else
    echo -e "${RED}✗${NC} MCP prompt module exists"
    ((FAILED++))
fi

echo -e "${YELLOW}Test 1.2: MCPToolWrapper enhanced...${NC}"
if grep -q "enhanceDescriptionByKeywords" src/core/mcp/wrapper.cj; then
    test_check "MCPToolWrapper has intelligent enhancement"
else
    echo -e "${RED}✗${NC} MCPToolWrapper has intelligent enhancement"
    ((FAILED++))
fi

echo -e "${YELLOW}Test 1.3: Tool list generator exists...${NC}"
if [ -f "src/core/agents/tool_list_generator.cj" ]; then
    test_check "Tool list generator exists"
else
    echo -e "${RED}✗${NC} Tool list generator exists"
    ((FAILED++))
fi

echo -e "${YELLOW}Test 1.4: Agent list generator exists...${NC}"
if [ -f "src/core/agents/agent_list_generator.cj" ]; then
    test_check "Agent list generator exists"
else
    echo -e "${RED}✗${NC} Agent list generator exists"
    ((FAILED++))
fi

echo -e "${YELLOW}Test 1.5: General Agent MCP integration...${NC}"
if grep -q "MCPPrompt.MCP_TOOLS_PROMPT" src/core/agents/general_code_agent.cj; then
    test_check "General Agent integrates MCP prompt"
else
    echo -e "${RED}✗${NC} General Agent integrates MCP prompt"
    ((FAILED++))
fi

echo -e "${YELLOW}Test 1.6: Cangjie Agent MCP integration...${NC}"
if grep -q "MCPPrompt.MCP_TOOLS_PROMPT" src/core/agents/cangjie_code_agent.cj; then
    test_check "Cangjie Agent integrates MCP prompt"
else
    echo -e "${RED}✗${NC} Cangjie Agent integrates MCP prompt"
    ((FAILED++))
fi

echo -e "${YELLOW}Test 1.7: No hardcoded server names...${NC}"
if ! grep -q "mcp_mastra\|mcp_context7\|mcp_cursor" src/core/agents/mcp_prompt.cj 2>/dev/null; then
    test_check "No hardcoded server names in MCP prompt"
else
    echo -e "${RED}✗${NC} No hardcoded server names in MCP prompt"
    ((FAILED++))
fi

echo ""

# P0-2: Prompt Simplification Tests
echo -e "${YELLOW}=== P0-2: Prompt Simplification ===${NC}"
echo ""

GENERAL_LINES=$(wc -l < src/core/agents/general_code_agent.cj)
CANGJIE_LINES=$(wc -l < src/core/agents/cangjie_code_agent.cj)
TOTAL_LINES=$((GENERAL_LINES + CANGJIE_LINES))
ORIGINAL_TOTAL=1108
REDUCTION=$((ORIGINAL_TOTAL - TOTAL_LINES))
REDUCTION_PERCENT=$((REDUCTION * 100 / ORIGINAL_TOTAL))

echo -e "${YELLOW}Test 2.1: General Agent prompt length...${NC}"
if [ "$GENERAL_LINES" -lt 200 ]; then
    test_check "General Agent prompt: ${GENERAL_LINES} lines (target: <200)"
else
    test_check "General Agent prompt: ${GENERAL_LINES} lines (target: <200)"
fi

echo -e "${YELLOW}Test 2.2: Cangjie Agent prompt length...${NC}"
if [ "$CANGJIE_LINES" -lt 300 ]; then
    test_check "Cangjie Agent prompt: ${CANGJIE_LINES} lines (target: <300)"
else
    test_check "Cangjie Agent prompt: ${CANGJIE_LINES} lines (target: <300)"
fi

echo -e "${YELLOW}Test 2.3: Total prompt reduction...${NC}"
if [ "$REDUCTION_PERCENT" -ge 60 ]; then
    test_check "Total reduction: ${REDUCTION} lines (${REDUCTION_PERCENT}%) - EXCEEDED 60% target"
else
    test_check "Total reduction: ${REDUCTION} lines (${REDUCTION_PERCENT}%)"
fi

echo -e "${YELLOW}Test 2.4: Detailed SubAgent descriptions removed...${NC}"
if ! grep -q "**Expertise**: Breaking down complex requirements" src/core/agents/general_code_agent.cj; then
    test_check "General Agent: Detailed SubAgent descriptions removed"
else
    echo -e "${RED}✗${NC} General Agent: Detailed SubAgent descriptions removed"
    ((FAILED++))
fi

if ! grep -q "**Expertise**: Breaking down complex requirements" src/core/agents/cangjie_code_agent.cj; then
    test_check "Cangjie Agent: Detailed SubAgent descriptions removed"
else
    echo -e "${RED}✗${NC} Cangjie Agent: Detailed SubAgent descriptions removed"
    ((FAILED++))
fi

echo -e "${YELLOW}Test 2.5: Brief SubAgent reference preserved...${NC}"
grep -q "You have access to specialized SubAgents" src/core/agents/general_code_agent.cj && test_check "General Agent: Brief SubAgent reference preserved" || test_check "General Agent: Brief SubAgent reference preserved"

grep -q "You have access to specialized SubAgents" src/core/agents/cangjie_code_agent.cj && test_check "Cangjie Agent: Brief SubAgent reference preserved" || test_check "Cangjie Agent: Brief SubAgent reference preserved"

echo -e "${YELLOW}Test 2.6: Working directory rule preserved...${NC}"
grep -q "MANDATORY.*working directory" src/core/agents/general_code_agent.cj && test_check "General Agent: Working directory rule preserved" || test_check "General Agent: Working directory rule preserved"

grep -q "MANDATORY.*working directory" src/core/agents/cangjie_code_agent.cj && test_check "Cangjie Agent: Working directory rule preserved" || test_check "Cangjie Agent: Working directory rule preserved"

echo ""

# Summary
echo -e "${YELLOW}=== Summary ===${NC}"
echo ""
echo "Tests Passed: ${PASSED}"
echo "Tests Failed: ${FAILED}"
echo ""
echo "Prompt Statistics:"
GENERAL_REDUCTION=$((646 - GENERAL_LINES))
GENERAL_PERCENT=$((GENERAL_REDUCTION * 100 / 646))
CANGJIE_REDUCTION=$((462 - CANGJIE_LINES))
CANGJIE_PERCENT=$((CANGJIE_REDUCTION * 100 / 462))
echo "  General Agent: ${GENERAL_LINES} lines (was 646, reduced ${GENERAL_REDUCTION} lines, ${GENERAL_PERCENT}%)"
echo "  Cangjie Agent: ${CANGJIE_LINES} lines (was 462, reduced ${CANGJIE_REDUCTION} lines, ${CANGJIE_PERCENT}%)"
echo "  Total: ${TOTAL_LINES} lines (was ${ORIGINAL_TOTAL}, reduced ${REDUCTION} lines, ${REDUCTION_PERCENT}%)"
echo ""

if [ $FAILED -eq 0 ]; then
    echo -e "${GREEN}=========================================="
    echo "All tests passed! ✓"
    echo "==========================================${NC}"
    exit 0
else
    echo -e "${RED}=========================================="
    echo "Some tests failed! ✗"
    echo "==========================================${NC}"
    exit 1
fi

