#!/bin/bash
###############################################################################
# Codelin 完整安装脚本（生成 codebin）
# 版本: 2.1
# 更新: 
#   - 适配最新 codelin 项目结构，使用 build-codelin.sh 构建
#   - 统一命名：cli -> codelin
#   - 支持多种二进制文件位置和名称的自动检测
###############################################################################

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
NC='\033[0m'

info() { echo -e "${CYAN}ℹ️  $1${NC}"; }
success() { echo -e "${GREEN}✅ $1${NC}"; }
warning() { echo -e "${YELLOW}⚠️  $1${NC}"; }
error() { echo -e "${RED}❌ $1${NC}"; exit 1; }

# Get directories
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"

# Check environment
[ -z "${MAGIC_PATH}" ] && error "MAGIC_PATH not set"
[ -z "${CANGJIE_HOME}" ] && error "CANGJIE_HOME not set"

echo ""
echo -e "${PURPLE}╔══════════════════════════════════════╗${NC}"
echo -e "${PURPLE}  Codelin 完整安装脚本 v2.1${NC}"
echo -e "${PURPLE}╚══════════════════════════════════════╝${NC}"
echo ""

# Build project
echo -e "${BLUE}🚀 编译项目...${NC}"
cd "$PROJECT_ROOT"

# 检查构建脚本是否存在
if [ ! -f "$SCRIPT_DIR/build-codelin.sh" ]; then
    error "构建脚本不存在: $SCRIPT_DIR/build-codelin.sh"
fi

# 记录编译前的二进制文件时间戳（如果存在）
OLD_BINARY_TIME=0
if [ -f "$PROJECT_ROOT/binary/codelin-cli" ]; then
    OLD_BINARY_TIME=$(stat -f "%m" "$PROJECT_ROOT/binary/codelin-cli" 2>/dev/null || echo "0")
fi
if [ -f "$PROJECT_ROOT/binary/codelin" ]; then
    NEW_TIME=$(stat -f "%m" "$PROJECT_ROOT/binary/codelin" 2>/dev/null || echo "0")
    if [ "$NEW_TIME" -gt "$OLD_BINARY_TIME" ]; then
        OLD_BINARY_TIME=$NEW_TIME
    fi
fi

# 尝试编译，如果失败则使用已有的二进制文件
BUILD_LOG="/tmp/codelin-install-build-$(date +%s).log"
BUILD_SUCCESS=false

info "开始编译项目..."
info "构建日志: $BUILD_LOG"

if bash "$SCRIPT_DIR/build-codelin.sh" 2>&1 | tee "$BUILD_LOG"; then
    success "编译完成"
    BUILD_SUCCESS=true
else
    BUILD_EXIT_CODE=$?
    warning "构建脚本返回非零退出码: $BUILD_EXIT_CODE"
    
    # 检查是否有新的二进制文件生成（即使构建脚本返回失败）
    NEW_BINARY_TIME=0
    if [ -f "$PROJECT_ROOT/binary/codelin" ]; then
        NEW_BINARY_TIME=$(stat -f "%m" "$PROJECT_ROOT/binary/codelin" 2>/dev/null || echo "0")
    fi
    if [ -f "$PROJECT_ROOT/binary/codelin-cli" ]; then
        NEW_TIME=$(stat -f "%m" "$PROJECT_ROOT/binary/codelin-cli" 2>/dev/null || echo "0")
        if [ "$NEW_TIME" -gt "$NEW_BINARY_TIME" ]; then
            NEW_BINARY_TIME=$NEW_TIME
        fi
    fi
    
    # 如果生成了新的二进制文件，认为编译成功
    if [ "$NEW_BINARY_TIME" -gt "$OLD_BINARY_TIME" ] && [ "$NEW_BINARY_TIME" -gt 0 ]; then
        success "编译完成（检测到新的二进制文件，时间戳: $(stat -f "%Sm" -t "%Y-%m-%d %H:%M:%S" "$PROJECT_ROOT/binary/codelin" 2>/dev/null || echo "未知")）"
        BUILD_SUCCESS=true
    else
        # 检查是否有任何可用的二进制文件（包括旧版本）
        HAS_BINARY=false
        if [ -f "$PROJECT_ROOT/binary/codelin" ]; then
            HAS_BINARY=true
            BINARY_AGE=$(stat -f "%Sm" -t "%Y-%m-%d %H:%M:%S" "$PROJECT_ROOT/binary/codelin" 2>/dev/null || echo "未知")
            warning "发现二进制文件: binary/codelin (时间: $BINARY_AGE)"
        elif [ -f "$PROJECT_ROOT/binary/codelin-cli" ]; then
            HAS_BINARY=true
            BINARY_AGE=$(stat -f "%Sm" -t "%Y-%m-%d %H:%M:%S" "$PROJECT_ROOT/binary/codelin-cli" 2>/dev/null || echo "未知")
            warning "发现二进制文件: binary/codelin-cli (时间: $BINARY_AGE)"
        elif [ -f "$PROJECT_ROOT/target/release/bin/codelin" ]; then
            HAS_BINARY=true
            BINARY_AGE=$(stat -f "%Sm" -t "%Y-%m-%d %H:%M:%S" "$PROJECT_ROOT/target/release/bin/codelin" 2>/dev/null || echo "未知")
            warning "发现二进制文件: target/release/bin/codelin (时间: $BINARY_AGE)"
        elif [ -f "$PROJECT_ROOT/target/release/bin/cli" ]; then
            HAS_BINARY=true
            BINARY_AGE=$(stat -f "%Sm" -t "%Y-%m-%d %H:%M:%S" "$PROJECT_ROOT/target/release/bin/cli" 2>/dev/null || echo "未知")
            warning "发现二进制文件: target/release/bin/cli (时间: $BINARY_AGE)"
        fi
        
        if [ "$HAS_BINARY" = true ]; then
            warning "⚠️  编译可能失败，但发现已有的二进制文件，将尝试使用现有文件"
            warning "⚠️  建议检查构建日志并重新编译以确保使用最新版本"
            info "构建日志: $BUILD_LOG"
            info "查看构建错误: tail -50 $BUILD_LOG"
    else
        error "编译失败且没有可用的二进制文件"
            echo ""
            error "构建日志: $BUILD_LOG"
            info "查看构建错误:"
            info "  tail -50 $BUILD_LOG"
            echo ""
            info "常见问题排查:"
            info "  1. 检查环境变量:"
            info "     echo \$MAGIC_PATH"
            info "     echo \$CANGJIE_HOME"
            info "  2. 检查依赖:"
            info "     ls -la \$MAGIC_PATH/libs/"
            info "  3. 手动编译测试:"
            info "     cd $PROJECT_ROOT"
            info "     cjpm build"
            echo ""
            error "请修复编译错误后重试。"
            exit 1
        fi
    fi
fi

# Create codebin
INSTALL_DIR="$PROJECT_ROOT/codebin"
rm -rf "$INSTALL_DIR"
mkdir -p "$INSTALL_DIR"/{bin,lib,.codelin}
success "目录创建完成"

# Copy binaries - 支持多种可能的源文件位置和名称，优先使用最新版本
BINARY_COPIED=false
BINARY_SOURCE=""
BINARY_TIME=0

# 函数：检查并记录二进制文件（如果比当前记录更新）
check_binary() {
    local file_path="$1"
    local description="$2"
    if [ -f "$file_path" ]; then
        local file_time=$(stat -f "%m" "$file_path" 2>/dev/null || echo "0")
        if [ "$file_time" -gt "$BINARY_TIME" ]; then
            BINARY_SOURCE="$file_path"
            BINARY_TIME=$file_time
            info "发现二进制文件: $description (时间戳: $(stat -f "%Sm" -t "%Y-%m-%d %H:%M:%S" "$file_path" 2>/dev/null || echo "未知"))"
        fi
    fi
}

# 检查所有可能的二进制文件位置，选择最新的
info "检查可用的二进制文件..."

# 优先级：1. target/release/bin/codelin (直接编译输出，最新，优先使用)
check_binary "$PROJECT_ROOT/target/release/bin/codelin" "target/release/bin/codelin (直接编译，最新)"

# 优先级：2. binary/codelin (build-codelin.sh 的新输出)
check_binary "$PROJECT_ROOT/binary/codelin" "binary/codelin (最新编译)"

# 优先级：3. target/release/bin/cli (历史遗留名称，但可能是最新编译的)
check_binary "$PROJECT_ROOT/target/release/bin/cli" "target/release/bin/cli (历史遗留名称)"

# 优先级：4. binary/codelin-cli (旧版本，仅作为后备，会给出警告)
check_binary "$PROJECT_ROOT/binary/codelin-cli" "binary/codelin-cli (旧版本，后备)"

# 复制选中的二进制文件
if [ -n "$BINARY_SOURCE" ] && [ "$BINARY_TIME" -gt 0 ]; then
    cp "$BINARY_SOURCE" "$INSTALL_DIR/bin/codelin"
    chmod +x "$INSTALL_DIR/bin/codelin"
    BINARY_COPIED=true
    
    # 显示文件信息（不使用 local，因为不在函数内）
    file_size=$(du -h "$INSTALL_DIR/bin/codelin" 2>/dev/null | cut -f1)
    file_date=$(stat -f "%Sm" -t "%Y-%m-%d %H:%M:%S" "$BINARY_SOURCE" 2>/dev/null || echo "未知")
    success "二进制文件已复制到 bin/codelin"
    info "  源文件: $BINARY_SOURCE"
    info "  文件大小: $file_size"
    info "  编译时间: $file_date"
    
    # 如果使用的是旧版本，给出警告
    if [[ "$BINARY_SOURCE" == *"binary/codelin-cli"* ]]; then
        warning "⚠️  使用的是旧版本的二进制文件 (binary/codelin-cli)"
        warning "⚠️  该文件可能是之前编译的旧版本，建议重新编译"
        info "  建议执行: bash scripts/build-codelin.sh"
        info "  然后重新运行安装脚本以确保使用最新版本"
    elif [[ "$BINARY_SOURCE" == *"target/release/bin/cli"* ]]; then
        info "ℹ️  使用的是历史遗留名称的二进制文件 (cli)，但可能是最新编译的"
    fi
else
    error "未找到可用的二进制文件。"
    echo ""
    info "已检查的位置："
    info "  ❌ $PROJECT_ROOT/target/release/bin/codelin"
    info "  ❌ $PROJECT_ROOT/binary/codelin"
    info "  ❌ $PROJECT_ROOT/target/release/bin/cli"
    info "  ❌ $PROJECT_ROOT/binary/codelin-cli"
    echo ""
    info "解决方案："
    info "  1. 运行编译脚本："
    info "     bash scripts/build-codelin.sh"
    echo ""
    info "  2. 检查编译是否成功："
    info "     ls -lah target/release/bin/"
    info "     ls -lah binary/"
    echo ""
    info "  3. 如果编译失败，检查日志："
    info "     tail -50 /tmp/codelin-build-*.log"
    echo ""
    error "请先编译项目，然后重新运行安装脚本。"
    exit 1
fi

# Copy libraries
info "复制依赖库..."

LIB_COUNT=0

# 检测操作系统和架构
OS_TYPE=$(uname -s)
ARCH_TYPE=$(uname -m)

# 1. Codelin 库（如果存在，动态链接时）
# 检查 target/release/codelin 或 target/release/cli（历史遗留）
CODELIN_LIB_DIR=""
if [ -d "$PROJECT_ROOT/target/release/codelin" ]; then
    CODELIN_LIB_DIR="$PROJECT_ROOT/target/release/codelin"
elif [ -d "$PROJECT_ROOT/target/release/cli" ]; then
    CODELIN_LIB_DIR="$PROJECT_ROOT/target/release/cli"
fi

if [ -n "$CODELIN_LIB_DIR" ]; then
    FOUND=$(find "$CODELIN_LIB_DIR" -name "*.dylib" -o -name "*.so" 2>/dev/null | wc -l | tr -d ' ')
    if [ "$FOUND" -gt 0 ]; then
        cp "$CODELIN_LIB_DIR"/*.{dylib,so} "$INSTALL_DIR/lib/" 2>/dev/null || true
        LIB_COUNT=$((LIB_COUNT + FOUND))
        info "Codelin 库已复制 ($FOUND 个)"
    fi
fi

# 2. FFI 库（如果存在）
if [ -d "$PROJECT_ROOT/ffi" ]; then
    FOUND=$(find "$PROJECT_ROOT/ffi" -maxdepth 1 -name "*.dylib" -o -name "*.so" 2>/dev/null | wc -l | tr -d ' ')
    if [ "$FOUND" -gt 0 ]; then
    cp "$PROJECT_ROOT/ffi"/*.{dylib,so} "$INSTALL_DIR/lib/" 2>/dev/null || true
        LIB_COUNT=$((LIB_COUNT + FOUND))
        info "FFI 库已复制 ($FOUND 个)"
    fi
fi

# 3. Magic 库（动态链接时）
if [ -d "$MAGIC_PATH/target/release/magic" ]; then
    FOUND=$(find "$MAGIC_PATH/target/release/magic" -name "*.dylib" -o -name "*.so" 2>/dev/null | wc -l | tr -d ' ')
    if [ "$FOUND" -gt 0 ]; then
    cp "$MAGIC_PATH/target/release/magic"/*.{dylib,so} "$INSTALL_DIR/lib/" 2>/dev/null || true
        LIB_COUNT=$((LIB_COUNT + FOUND))
        info "Magic 库已复制 ($FOUND 个)"
    fi
fi

# 4. Cangjie runtime 库（动态链接时）
RUNTIME_LIB=$(find "$CANGJIE_HOME" -name "libcangjie-runtime.dylib" -o -name "libcangjie-runtime.so" 2>/dev/null | head -1)
if [ -n "$RUNTIME_LIB" ]; then
    RUNTIME_DIR=$(dirname "$RUNTIME_LIB")
    FOUND=$(find "$RUNTIME_DIR" -maxdepth 1 -name "*.dylib" -o -name "*.so" 2>/dev/null | wc -l | tr -d ' ')
    if [ "$FOUND" -gt 0 ]; then
        cp "$RUNTIME_DIR"/*.{dylib,so} "$INSTALL_DIR/lib/" 2>/dev/null || true
        LIB_COUNT=$((LIB_COUNT + FOUND))
        info "Cangjie runtime 库已复制 ($FOUND 个)"
    fi
fi

# 5. Cangjie stdx 库（根据静态/动态链接选择）
# 优先查找静态链接库（从 MAGIC_PATH/libs/）
STDX_STATIC_DIR=""
STDX_DYNAMIC_DIR=""

# 检测平台类型
if [[ "$OS_TYPE" == "Darwin" ]]; then
    if [[ "$ARCH_TYPE" == "arm64" ]]; then
        STDX_STATIC_DIR=$(find "$MAGIC_PATH/libs" -type d -path "*/cangjie-stdx-mac-aarch64-*/darwin_aarch64_llvm/static/stdx" 2>/dev/null | head -1)
        STDX_DYNAMIC_DIR=$(find "$MAGIC_PATH" -type d -path "*/darwin_aarch64_llvm/dynamic/stdx" 2>/dev/null | head -1)
    else
        STDX_STATIC_DIR=$(find "$MAGIC_PATH/libs" -type d -path "*/cangjie-stdx-mac-x64-*/darwin_x86_64_llvm/static/stdx" 2>/dev/null | head -1)
        STDX_DYNAMIC_DIR=$(find "$MAGIC_PATH" -type d -path "*/darwin_x86_64_llvm/dynamic/stdx" 2>/dev/null | head -1)
fi
elif [[ "$OS_TYPE" == "Linux" ]]; then
    if [[ "$ARCH_TYPE" == "aarch64" ]]; then
        STDX_STATIC_DIR=$(find "$MAGIC_PATH/libs" -type d -path "*/cangjie-stdx-linux-aarch64-*/linux_aarch64_llvm/static/stdx" 2>/dev/null | head -1)
        STDX_DYNAMIC_DIR=$(find "$MAGIC_PATH" -type d -path "*/linux_aarch64_llvm/dynamic/stdx" 2>/dev/null | head -1)
    else
        STDX_STATIC_DIR=$(find "$MAGIC_PATH/libs" -type d -path "*/cangjie-stdx-linux-x64-*/linux_x86_64_llvm/static/stdx" 2>/dev/null | head -1)
        STDX_DYNAMIC_DIR=$(find "$MAGIC_PATH" -type d -path "*/linux_x86_64_llvm/dynamic/stdx" 2>/dev/null | head -1)
    fi
fi

# 优先使用静态库（如果存在），否则使用动态库
STDX_DIR=""
if [ -n "$STDX_STATIC_DIR" ] && [ -d "$STDX_STATIC_DIR" ]; then
    STDX_DIR="$STDX_STATIC_DIR"
    info "使用静态链接的 Cangjie stdx 库"
elif [ -n "$STDX_DYNAMIC_DIR" ] && [ -d "$STDX_DYNAMIC_DIR" ]; then
    STDX_DIR="$STDX_DYNAMIC_DIR"
    info "使用动态链接的 Cangjie stdx 库"
fi

if [ -n "$STDX_DIR" ] && [ -d "$STDX_DIR" ]; then
    FOUND=$(find "$STDX_DIR" -maxdepth 1 -name "*.dylib" -o -name "*.so" 2>/dev/null | wc -l | tr -d ' ')
    if [ "$FOUND" -gt 0 ]; then
        cp "$STDX_DIR"/*.{dylib,so} "$INSTALL_DIR/lib/" 2>/dev/null || true
        LIB_COUNT=$((LIB_COUNT + FOUND))
        info "Cangjie stdx 库已复制 ($FOUND 个)"
    else
        warning "Cangjie stdx 库目录存在但未找到库文件: $STDX_DIR"
    fi
else
    warning "未找到 Cangjie stdx 库目录（静态或动态）"
fi

if [ "$LIB_COUNT" -gt 0 ]; then
    success "所有依赖库已复制 (共 $LIB_COUNT 个)"
else
    warning "未复制任何库文件（可能是静态链接）"
fi

# Create configuration files
echo -e "${BLUE}📝 创建配置文件...${NC}"

cat > "$INSTALL_DIR/.codelin/codelin.json" << 'EOF'
{
  "language": "cangjie",
  "model": "zhipuai:glm-4.6",
  "fastModel": "zhipuai:glm-4-flash",
  "fallbackModels": ["maas:deepseek-v3.2-exp", "zhipuai:glm-4"],
  "temperature": 1
}
EOF
success "配置文件已创建"

# Create .env.example
cat > "$INSTALL_DIR/.codelin/.env.example" << 'EOF'
# 智谱 AI (推荐)
ZHIPU_API_KEY=your_zhipu_api_key_here

# 华为 MaaS (备用)
MAAS_BASE_URL=https://api.modelarts-maas.com/v2
MAAS_API_KEY=your_maas_api_key_here

# 其他
MOONSHOT_API_KEY=your_moonshot_api_key_here
DEEPSEEK_API_KEY=your_deepseek_api_key_here
CONTEXT7_API_KEY=your_context7_api_key_here
TAVILY_API_KEY=your_tavily_api_key_here
EOF
success "环境变量模板已创建"

# Copy .env if exists in project
if [ -f "$PROJECT_ROOT/.codelin/.env" ]; then
    cp "$PROJECT_ROOT/.codelin/.env" "$INSTALL_DIR/.codelin/.env"
    info "已复制项目的 .env 文件"
fi

# Create startup script
echo -e "${BLUE}🚀 创建启动脚本...${NC}"

cat > "$INSTALL_DIR/bin/codelinx" << 'EOF'
#!/bin/bash
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
INSTALL_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"

# Load .env
ENV_FILE="$INSTALL_DIR/.codelin/.env"
[ -f "$ENV_FILE" ] && while IFS= read -r line; do
    [[ "$line" =~ ^[[:space:]]*# ]] || [[ -z "${line// }" ]] && continue
    [[ "$line" =~ ^[[:space:]]*([A-Za-z_][A-Za-z0-9_]*)= ]] && export "$line"
done < "$ENV_FILE"

# Set library path
[[ "$OSTYPE" == "darwin"* ]] && export DYLD_LIBRARY_PATH="$INSTALL_DIR/lib:$DYLD_LIBRARY_PATH"
[[ "$OSTYPE" == "linux-gnu"* ]] && export LD_LIBRARY_PATH="$INSTALL_DIR/lib:$LD_LIBRARY_PATH"

exec "$INSTALL_DIR/bin/codelin" "$@"
EOF
chmod +x "$INSTALL_DIR/bin/codelinx"
success "启动脚本已创建"

# Create README
echo -e "${BLUE}📚 创建文档...${NC}"

cat > "$INSTALL_DIR/README.txt" << 'EOF'
# Codelin 安装包

## 快速开始
1. cp .codelin/.env.example .codelin/.env
2. 编辑 .env，填入 API Keys
3. ./bin/codelin

## 配置
- 默认模型: zhipuai:glm-4.6
- 备用模型: maas:deepseek-v3.2-exp
- 配置文件: .codelin/codelin.json

## API Keys
- 智谱 AI: https://open.bigmodel.cn/
- 华为 MaaS: https://console.huaweicloud.com/modelarts-maas/
EOF
success "README 已创建"

echo ""
echo -e "${GREEN}╔══════════════════════════════════════╗${NC}"
echo -e "${GREEN}  ✅ 安装包生成完成${NC}"
echo -e "${GREEN}╚══════════════════════════════════════╝${NC}"
echo ""

# 统计信息
info "安装目录: $INSTALL_DIR"
BINARY_SIZE=$(du -h "$INSTALL_DIR/bin/codelin" 2>/dev/null | cut -f1 || echo "未知")
info "二进制文件大小: $BINARY_SIZE"

# 统计库文件
DYLIB_COUNT=$(ls -1 "$INSTALL_DIR/lib"/*.dylib 2>/dev/null | wc -l | tr -d ' ' || echo "0")
SO_COUNT=$(ls -1 "$INSTALL_DIR/lib"/*.so 2>/dev/null | wc -l | tr -d ' ' || echo "0")
TOTAL_LIBS=$((DYLIB_COUNT + SO_COUNT))

if [ "$TOTAL_LIBS" -gt 0 ]; then
    info "库文件数: $TOTAL_LIBS 个 (dylib: $DYLIB_COUNT, so: $SO_COUNT)"
    LIB_SIZE=$(du -sh "$INSTALL_DIR/lib" 2>/dev/null | cut -f1 || echo "未知")
    info "库文件总大小: $LIB_SIZE"
else
    info "库文件: 无（可能是静态链接）"
fi

INSTALL_SIZE=$(du -sh "$INSTALL_DIR" 2>/dev/null | cut -f1 || echo "未知")
info "安装包总大小: $INSTALL_SIZE"

echo ""
echo -e "${CYAN}📝 下一步：${NC}"
echo "1. cd $INSTALL_DIR"
echo "2. 配置 API Keys (可选):"
echo "   cp .codelin/.env.example .codelin/.env"
echo "   vim .codelin/.env"
echo "3. 运行程序:"
echo "   ./bin/codelin --help"
echo ""
echo -e "${CYAN}💡 提示：${NC}"
echo "- 如果使用静态链接，可能不需要库文件"
echo "- 确保设置了正确的环境变量（MAGIC_PATH, CANGJIE_HOME）"
echo "- 查看 README.txt 了解更多信息"
echo ""
success "安装完成！"
