#!/bin/bash
# build-windows.sh - 在 Mac 上构建 Windows 版本的 Codelin
# 使用方法: bash scripts/build-windows.sh [target]
# target 选项: x86_64-unknown-windows-gnu 或 x86_64-w64-mingw32 (默认)

set -e

# 颜色输出
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

error() {
    echo -e "${RED}❌ $1${NC}" >&2
}

success() {
    echo -e "${GREEN}✅ $1${NC}"
}

warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

step() {
    echo -e "\n${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${BLUE}📦 $1${NC}"
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}\n"
}

complete() {
    echo -e "\n${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${GREEN}🎉 $1${NC}"
    echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}\n"
}

# 获取脚本目录
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"

cd "$PROJECT_ROOT"

# 解析参数
TARGET="${1:-x86_64-w64-mingw32}"

# 验证目标平台
if [[ "$TARGET" != "x86_64-unknown-windows-gnu" && "$TARGET" != "x86_64-w64-mingw32" ]]; then
    error "无效的目标平台: $TARGET"
    info "支持的目标平台:"
    info "  - x86_64-unknown-windows-gnu (GNU ABI)"
    info "  - x86_64-w64-mingw32 (MinGW-w64)"
    exit 1
fi

step "开始构建 Windows 版本 (目标: $TARGET)"

# 检查环境变量
if [ -z "$MAGIC_PATH" ]; then
    error "MAGIC_PATH 环境变量未设置"
    info "请设置: export MAGIC_PATH=/path/to/magic"
    exit 1
fi

# 检查本地 Windows SDK
WINDOWS_SDK_PATH="$PROJECT_ROOT/sdk/cangjie"
if [ -d "$WINDOWS_SDK_PATH" ]; then
    success "找到本地 Windows 仓颉 SDK: $WINDOWS_SDK_PATH"
    # 保存原始的 CANGJIE_HOME（如果存在）
    ORIGINAL_CANGJIE_HOME="$CANGJIE_HOME"
    # 不替换 CANGJIE_HOME，而是将 Windows SDK 的工具链添加到 PATH
    # 这样编译器可以使用 Windows 工具链，但仍然能找到 macOS 模块
    info "将使用 Windows SDK 的工具链进行交叉编译"
elif [ -n "$CANGJIE_HOME" ]; then
    info "使用系统 CANGJIE_HOME: $CANGJIE_HOME"
    ORIGINAL_CANGJIE_HOME="$CANGJIE_HOME"
else
    warning "CANGJIE_HOME 未设置，且未找到本地 Windows SDK"
    info "尝试使用系统默认的 Cangjie SDK"
    ORIGINAL_CANGJIE_HOME=""
fi

info "项目根目录: $PROJECT_ROOT"
info "Magic 路径: $MAGIC_PATH"
info "目标平台: $TARGET"

# 检查 Windows 标准库依赖
WINDOWS_STDX_PATH="${MAGIC_PATH}/libs/cangjie-stdx-windows-x64-1.0.0.1/windows_x86_64_llvm/static/stdx"
WINDOWS_SDK_STD_PATH="$WINDOWS_SDK_PATH/modules/windows_x86_64_llvm/std"

if [ -d "$WINDOWS_STDX_PATH" ]; then
    success "找到 Windows 标准库: $WINDOWS_STDX_PATH"
elif [ -d "$WINDOWS_SDK_STD_PATH" ]; then
    success "找到 Windows SDK 标准库: $WINDOWS_SDK_STD_PATH"
    info "将使用 Windows SDK 的标准库（源代码形式）"
else
    warning "Windows 标准库未找到: $WINDOWS_STDX_PATH"
    if [ -d "$WINDOWS_SDK_PATH" ]; then
        info "Windows SDK 存在，将使用 SDK 的标准库模块"
    else
        warning "未找到任何 Windows 标准库"
        info "如果使用 Cangjie SDK 自带的工具链，可能会自动处理标准库"
        info "继续构建，如果失败请安装 Windows 版本的 cangjie-stdx"
    fi
fi

# 检查交叉编译工具链
step "检查交叉编译工具链"

# 检查是否有 MinGW-w64 工具链（用于 x86_64-w64-mingw32）
if [[ "$TARGET" == "x86_64-w64-mingw32" ]]; then
    if ! command -v x86_64-w64-mingw32-gcc &> /dev/null; then
        warning "未找到 MinGW-w64 交叉编译工具链"
        info "在 Mac 上安装 MinGW-w64:"
        info "  brew install mingw-w64"
        info ""
        info "或者使用 Cangjie SDK 自带的工具链（如果已配置）"
    else
        success "找到 MinGW-w64 工具链: $(which x86_64-w64-mingw32-gcc)"
    fi
fi

# 备份配置文件
step "备份和准备配置文件"

ORIGINAL_CJPM="$PROJECT_ROOT/cjpm.toml"
BACKUP_CJPM="$PROJECT_ROOT/cjpm.toml.backup"
STATIC_CJPM="$SCRIPT_DIR/cli-static.toml"

if [ ! -f "$STATIC_CJPM" ]; then
    error "静态配置文件未找到: $STATIC_CJPM"
    exit 1
fi

if [ -f "$ORIGINAL_CJPM" ]; then
    cp "$ORIGINAL_CJPM" "$BACKUP_CJPM"
    success "已备份原始 cjpm.toml"
fi

# 使用静态配置
cp "$STATIC_CJPM" "$ORIGINAL_CJPM"

# 处理 Magic 项目的配置
MAGIC_CJPM="${MAGIC_PATH}/cjpm.toml"
MAGIC_CJPM_BACKUP="${MAGIC_PATH}/cjpm.toml.backup.windows-build"

if [ -f "$MAGIC_CJPM" ]; then
    cp "$MAGIC_CJPM" "$MAGIC_CJPM_BACKUP"
    info "已备份 Magic 项目的 cjpm.toml"
    
    # 注释掉 Magic 项目中 Windows 相关的 bin-dependencies
    if [[ "$TARGET" == "x86_64-w64-mingw32" ]]; then
        sed -i.bak 's/^\[target\.x86_64-w64-mingw32\.bin-dependencies\]/# [target.x86_64-w64-mingw32.bin-dependencies]/' "$MAGIC_CJPM"
        sed -i.bak 's|^  path-option = \["\.\/libs\/cangjie-stdx-windows-x64-1\.0\.0\.1\/windows_x86_64_llvm\/dynamic\/stdx"\]|#   path-option = ["./libs/cangjie-stdx-windows-x64-1.0.0.1/windows_x86_64_llvm/dynamic/stdx"]|' "$MAGIC_CJPM"
    elif [[ "$TARGET" == "x86_64-unknown-windows-gnu" ]]; then
        sed -i.bak 's/^\[target\.x86_64-unknown-windows-gnu\.bin-dependencies\]/# [target.x86_64-unknown-windows-gnu.bin-dependencies]/' "$MAGIC_CJPM"
        sed -i.bak 's|^  path-option = \["\.\/libs\/cangjie-stdx-windows-x64-1\.0\.0\.1\/windows_x86_64_llvm\/dynamic\/stdx"\]|#   path-option = ["./libs/cangjie-stdx-windows-x64-1.0.0.1/windows_x86_64_llvm/dynamic/stdx"]|' "$MAGIC_CJPM"
    fi
    info "已注释掉 Magic 项目中 Windows bin-dependencies"
fi

# 如果 Windows 标准库不存在，注释掉 bin-dependencies 配置
# 如果使用 Windows SDK，标准库会通过 CANGJIE_HOME 自动找到
if [ ! -d "$WINDOWS_STDX_PATH" ] && [ ! -d "$WINDOWS_SDK_STD_PATH" ]; then
    warning "Windows 标准库不存在，将注释掉 bin-dependencies 配置"
    # 注释掉 Windows 相关的 bin-dependencies
    if [[ "$TARGET" == "x86_64-w64-mingw32" ]]; then
        sed -i.bak 's/^\[target\.x86_64-w64-mingw32\.bin-dependencies\]/# [target.x86_64-w64-mingw32.bin-dependencies]/' "$ORIGINAL_CJPM"
        sed -i.bak 's/^  path-option = \["\${MAGIC_PATH}\/libs\/cangjie-stdx-windows-x64-1\.0\.0\.1\/windows_x86_64_llvm\/static\/stdx"\]/#   path-option = ["${MAGIC_PATH}\/libs\/cangjie-stdx-windows-x64-1.0.0.1\/windows_x86_64_llvm\/static\/stdx"]/' "$ORIGINAL_CJPM"
    elif [[ "$TARGET" == "x86_64-unknown-windows-gnu" ]]; then
        sed -i.bak 's/^\[target\.x86_64-unknown-windows-gnu\.bin-dependencies\]/# [target.x86_64-unknown-windows-gnu.bin-dependencies]/' "$ORIGINAL_CJPM"
        sed -i.bak 's/^  path-option = \["\${MAGIC_PATH}\/libs\/cangjie-stdx-windows-x64-1\.0\.0\.1\/windows_x86_64_llvm\/static\/stdx"\]/#   path-option = ["${MAGIC_PATH}\/libs\/cangjie-stdx-windows-x64-1.0.0.1\/windows_x86_64_llvm\/static\/stdx"]/' "$ORIGINAL_CJPM"
    fi
    info "已注释掉 Windows bin-dependencies，将使用 Cangjie SDK 自带的工具链"
elif [ -d "$WINDOWS_SDK_STD_PATH" ]; then
    info "使用 Windows SDK 的标准库，无需配置 bin-dependencies"
    # 注释掉 bin-dependencies，让编译器使用 CANGJIE_HOME 的标准库
    if [[ "$TARGET" == "x86_64-w64-mingw32" ]]; then
        sed -i.bak 's/^\[target\.x86_64-w64-mingw32\.bin-dependencies\]/# [target.x86_64-w64-mingw32.bin-dependencies]/' "$ORIGINAL_CJPM"
        sed -i.bak 's/^  path-option = \["\${MAGIC_PATH}\/libs\/cangjie-stdx-windows-x64-1\.0\.0\.1\/windows_x86_64_llvm\/static\/stdx"\]/#   path-option = ["${MAGIC_PATH}\/libs\/cangjie-stdx-windows-x64-1.0.0.1\/windows_x86_64_llvm\/static\/stdx"]/' "$ORIGINAL_CJPM"
    elif [[ "$TARGET" == "x86_64-unknown-windows-gnu" ]]; then
        sed -i.bak 's/^\[target\.x86_64-unknown-windows-gnu\.bin-dependencies\]/# [target.x86_64-unknown-windows-gnu.bin-dependencies]/' "$ORIGINAL_CJPM"
        sed -i.bak 's/^  path-option = \["\${MAGIC_PATH}\/libs\/cangjie-stdx-windows-x64-1\.0\.0\.1\/windows_x86_64_llvm\/static\/stdx"\]/#   path-option = ["${MAGIC_PATH}\/libs\/cangjie-stdx-windows-x64-1.0.0.1\/windows_x86_64_llvm\/static\/stdx"]/' "$ORIGINAL_CJPM"
    fi
fi

success "已应用静态配置"

# 清理函数
restore_files() {
    if [ -f "$BACKUP_CJPM" ]; then
        mv "$BACKUP_CJPM" "$ORIGINAL_CJPM"
        success "已恢复原始 cjpm.toml"
    fi
    
    # 恢复 Magic 项目的配置
    MAGIC_CJPM="${MAGIC_PATH}/cjpm.toml"
    MAGIC_CJPM_BACKUP="${MAGIC_PATH}/cjpm.toml.backup.windows-build"
    if [ -f "$MAGIC_CJPM_BACKUP" ]; then
        mv "$MAGIC_CJPM_BACKUP" "$MAGIC_CJPM"
        info "已恢复 Magic 项目的 cjpm.toml"
    fi
}

# 设置退出时恢复文件
trap restore_files EXIT

# 清理 FFI 目录
step "清理 FFI 目录"
if [ -d "./ffi" ]; then
    find ./ffi -name "*.a" -type f -delete 2>/dev/null || true
    find ./ffi -name "*.so" -type f -delete 2>/dev/null || true
    find ./ffi -name "*.dylib" -type f -delete 2>/dev/null || true
    success "FFI 目录清理完成"
else
    warning "FFI 目录不存在，跳过清理"
fi

# 清理构建目录
step "清理构建目录"
cjpm clean >/dev/null 2>&1 || true
success "构建目录清理完成"

# 构建 Windows 版本
step "构建 Windows 版本 (目标: $TARGET)"

BUILD_LOG="/tmp/codelin-windows-build-$(date +%s).log"
info "构建日志: $BUILD_LOG"

# 设置环境变量以使用 Windows SDK 工具链（仅用于交叉编译）
# 注意：不替换 CANGJIE_HOME，因为需要 macOS 模块来编译依赖
if [ -d "$WINDOWS_SDK_PATH" ]; then
    # 将 Windows SDK 的工具链添加到 PATH（优先使用）
    # 注意：LLVM bin 目录包含链接器
    export PATH="$WINDOWS_SDK_PATH/third_party/llvm/bin:$WINDOWS_SDK_PATH/third_party/mingw/bin:$WINDOWS_SDK_PATH/tools/bin:$WINDOWS_SDK_PATH/bin:$PATH"
    info "已设置 PATH 以使用 Windows SDK 工具链进行交叉编译"
    info "CANGJIE_HOME 保持为: ${ORIGINAL_CANGJIE_HOME:-未设置}（用于 macOS 模块）"
    
    # 验证链接器是否存在
    if command -v x86_64-w64-mingw32-ld &> /dev/null || [ -f "$WINDOWS_SDK_PATH/third_party/llvm/bin/x86_64-w64-mingw32-ld" ] || [ -f "$WINDOWS_SDK_PATH/third_party/mingw/bin/x86_64-w64-mingw32-ld" ]; then
        success "找到 Windows 链接器"
    else
        warning "未找到 x86_64-w64-mingw32-ld，可能使用 LLVM 的 lld"
        # 检查是否有 lld
        if [ -f "$WINDOWS_SDK_PATH/third_party/llvm/bin/lld" ] || command -v lld &> /dev/null; then
            info "找到 lld，可能使用 lld 作为链接器"
        fi
    fi
fi

# 执行交叉编译
if cjpm build --target="$TARGET" -V 2>&1 | tee "$BUILD_LOG"; then
    success "编译成功"
else
    BUILD_EXIT_CODE=$?
    error "编译失败，退出码: $BUILD_EXIT_CODE"
    error "构建日志: $BUILD_LOG"
    info "最后 50 行构建输出:"
    tail -50 "$BUILD_LOG" | sed 's/^/  /'
    exit 1
fi

# 查找生成的二进制文件
step "查找生成的二进制文件"

# Windows 二进制文件可能的位置
BINARY_DIRS=(
    "./target/${TARGET}/release/bin"
    "./target/${TARGET}/release"
    "./target/${TARGET}/debug/bin"
    "./target/${TARGET}/debug"
    "./target/release/bin"
    "./target/release"
    "./target/debug/bin"
    "./target/debug"
)

BINARY_NAME="codelin.exe"
BINARY_SOURCE=""
BINARY_FOUND=false

for dir in "${BINARY_DIRS[@]}"; do
    if [ -f "$dir/$BINARY_NAME" ]; then
        BINARY_SOURCE="$dir/$BINARY_NAME"
        BINARY_FOUND=true
        success "找到二进制文件: $BINARY_SOURCE"
        break
    elif [ -f "$dir/codelin" ] && [ -x "$dir/codelin" ]; then
        BINARY_SOURCE="$dir/codelin"
        BINARY_FOUND=true
        success "找到二进制文件: $BINARY_SOURCE (无 .exe 扩展名)"
        break
    fi
done

if [ "$BINARY_FOUND" = false ]; then
    error "未找到生成的二进制文件"
    info "已检查的位置:"
    for dir in "${BINARY_DIRS[@]}"; do
        info "  ❌ $dir/$BINARY_NAME"
        info "  ❌ $dir/codelin"
    done
    info ""
    info "查找所有可能的二进制文件:"
    find ./target -type f -name "*.exe" -o -name "codelin" 2>/dev/null | head -10 | sed 's/^/    /'
    exit 1
fi

# 复制到输出目录
step "复制二进制文件到输出目录"

OUTPUT_DIR="./binary/windows"
mkdir -p "$OUTPUT_DIR"

OUTPUT_BINARY="$OUTPUT_DIR/codelin-${TARGET}.exe"
cp "$BINARY_SOURCE" "$OUTPUT_BINARY"
chmod +x "$OUTPUT_BINARY" 2>/dev/null || true

# 获取文件信息
BINARY_SIZE=$(stat -f "%z" "$OUTPUT_BINARY" 2>/dev/null || stat -c "%s" "$OUTPUT_BINARY" 2>/dev/null || echo "未知")
BINARY_SIZE_MB=$(echo "scale=2; $BINARY_SIZE / 1024 / 1024" | bc 2>/dev/null || echo "未知")

success "二进制文件已复制到: $OUTPUT_BINARY"
info "文件大小: ${BINARY_SIZE_MB} MB (${BINARY_SIZE} 字节)"

# 显示构建信息
complete "Windows 版本构建完成！"

echo ""
info "构建信息:"
info "  目标平台: $TARGET"
info "  输出文件: $OUTPUT_BINARY"
info "  文件大小: ${BINARY_SIZE_MB} MB"
info "  构建日志: $BUILD_LOG"
echo ""
info "下一步:"
info "  1. 将 $OUTPUT_BINARY 复制到 Windows 系统"
info "  2. 确保 Windows 系统有所需的运行时库"
info "  3. 在 Windows 上运行: codelin-${TARGET}.exe"
echo ""

