# CodeQualityAnalyzer 真实实现设计

## 🎯 目标

创建一个真正有用的代码质量分析工具集，提供实质性的代码分析能力，而非简单的包装。

## 📋 核心分析能力

### 1. 函数复杂度分析
**功能**：分析函数的圈复杂度和认知复杂度

**实现方法**：
- 解析函数体，统计控制流语句
- 计算圈复杂度：`M = E - N + 2P`（简化为统计分支点）
- 识别：`if`, `else`, `match`, `for`, `while`, `catch`, `&&`, `||`

**输出**：
```json
{
  "function": "processOrder",
  "complexity": 15,
  "level": "high",
  "recommendations": ["Consider breaking down into smaller functions"]
}
```

### 2. 函数长度分析
**功能**：检查函数是否过长

**标准**：
- 优秀：< 20 行
- 良好：20-50 行
- 一般：50-100 行
- 需要重构：> 100 行

### 3. 嵌套深度分析
**功能**：检查代码嵌套层级

**实现**：
- 解析缩进层级
- 识别代码块：`{...}`
- 统计最大嵌套深度

**标准**：
- 优秀：≤ 2 层
- 良好：3 层
- 一般：4 层
- 需要重构：> 4 层

### 4. 命名规范检查
**功能**：检查标识符命名是否符合规范

**规则**：
- 类名：PascalCase
- 函数/方法：camelCase
- 常量：UPPER_SNAKE_CASE
- 变量：camelCase 或 snake_case

### 5. 代码统计分析
**功能**：提供代码度量指标

**指标**：
- 总行数
- 代码行数（非空非注释）
- 注释行数
- 注释覆盖率
- 函数平均长度
- 最大函数长度

### 6. 代码味道检测
**功能**：识别常见的代码坏味道

**检测项**：
- 魔法数字（未命名的常量）
- 长参数列表（> 5 个参数）
- 重复代码片段
- 过长的函数
- 过深的嵌套

## 🛠️ 技术实现

### 架构设计

```
code_quality_toolset.cj
├── analyzeFunction() - 分析单个函数
├── analyzeFile() - 分析整个文件
├── analyzeFunctions() - 批量分析函数
├── getCodeMetrics() - 获取代码度量
└── detectCodeSmells() - 检测代码味道
```

### 核心算法

#### 1. 圈复杂度计算
```cangjie
func calculateComplexity(functionBody: String): Int64 {
    var complexity: Int64 = 1  // 基础复杂度
    
    // 统计控制流语句
    complexity += countOccurrences(functionBody, "if ")
    complexity += countOccurrences(functionBody, "else ")
    complexity += countOccurrences(functionBody, "match ")
    complexity += countOccurrences(functionBody, "case ")
    complexity += countOccurrences(functionBody, "for ")
    complexity += countOccurrences(functionBody, "while ")
    complexity += countOccurrences(functionBody, "catch ")
    complexity += countOccurrences(functionBody, "&&")
    complexity += countOccurrences(functionBody, "||")
    
    return complexity
}
```

#### 2. 嵌套深度计算
```cangjie
func calculateMaxNesting(code: String): Int64 {
    var maxDepth: Int64 = 0
    var currentDepth: Int64 = 0
    
    for (ch in code.toRuneArray()) {
        if (ch == r'{') {
            currentDepth += 1
            if (currentDepth > maxDepth) {
                maxDepth = currentDepth
            }
        } else if (ch == r'}') {
            currentDepth -= 1
        }
    }
    
    return maxDepth
}
```

#### 3. 命名检查
```cangjie
func checkNaming(name: String, type: SymbolType): Bool {
    match (type) {
        case SymbolType.Class =>
            return isPascalCase(name)
        case SymbolType.Function =>
            return isCamelCase(name)
        case SymbolType.Constant =>
            return isUpperSnakeCase(name)
        case _ =>
            return true
    }
}
```

## 📊 工具接口设计

### Tool 1: analyzeFunction

```cangjie
@tool[
    description: """
Analyze the complexity and quality of a specific function in a Cangjie file.
Provides detailed metrics including cyclomatic complexity, lines of code, nesting depth, and recommendations.

Returns:
- Function name and location
- Cyclomatic complexity score
- Lines of code
- Maximum nesting depth
- Quality level (excellent/good/fair/poor)
- Specific recommendations for improvement
"""
    parameters: {
        filePath: "Absolute path to the Cangjie file",
        functionName: "Name of the function to analyze"
    }
]
public func analyzeFunction(filePath: String, functionName: String): String
```

### Tool 2: analyzeFile

```cangjie
@tool[
    description: """
Analyze the overall code quality of a Cangjie file.
Provides comprehensive metrics and identifies potential issues.

Returns:
- File statistics (total lines, code lines, comment lines)
- Function complexity distribution
- Code smells detected
- Overall quality score
- Top issues to address
"""
    parameters: {
        filePath: "Absolute path to the Cangjie file"
    }
]
public func analyzeFile(filePath: String): String
```

### Tool 3: getCodeMetrics

```cangjie
@tool[
    description: """
Get detailed code metrics for a Cangjie file.
Useful for understanding code characteristics and trends.

Returns:
- Total lines, code lines, comment lines, blank lines
- Number of functions, classes, structs
- Average function length
- Maximum function length
- Comment coverage percentage
- Overall complexity metrics
"""
    parameters: {
        filePath: "Absolute path to the Cangjie file"
    }
]
public func getCodeMetrics(filePath: String): String
```

### Tool 4: detectCodeSmells

```cangjie
@tool[
    description: """
Detect common code smells and anti-patterns in a Cangjie file.
Helps identify areas that may need refactoring.

Detects:
- Long functions (> 50 lines)
- Deep nesting (> 3 levels)
- Long parameter lists (> 5 parameters)
- Magic numbers (unnamed constants)
- Duplicate code patterns
- Poor naming conventions

Returns a list of detected issues with locations and severity.
"""
    parameters: {
        filePath: "Absolute path to the Cangjie file"
    }
]
public func detectCodeSmells(filePath: String): String
```

## 🎯 实施计划

### Phase 1: 核心分析引擎（2-3小时）
1. 创建 `code_quality_toolset.cj`
2. 实现基础的代码解析和统计
3. 实现圈复杂度计算
4. 实现嵌套深度计算

### Phase 2: 工具接口（1-2小时）
1. 实现 `analyzeFunction`
2. 实现 `analyzeFile`
3. 实现 `getCodeMetrics`
4. 实现 `detectCodeSmells`

### Phase 3: 测试和验证（1小时）
1. 编写单元测试
2. 测试各种复杂度的函数
3. 验证输出格式

### Phase 4: 集成和文档（0.5小时）
1. 集成到主工具集
2. 更新 plan1.md
3. 编写使用示例

## ✅ 验收标准

1. ✅ 能够准确计算函数复杂度
2. ✅ 能够识别常见代码味道
3. ✅ 提供可操作的改进建议
4. ✅ 输出格式清晰易读
5. ✅ 编译通过，无错误
6. ✅ 测试覆盖核心功能

## 🚀 预期价值

- Agent 可以自动评估代码质量
- 提供具体的改进建议
- 帮助用户编写更好的代码
- 识别需要重构的代码

---

**实施者**：AI Assistant  
**设计日期**：2024-10-24  
**版本**：v1.0 真实实现

