# UTF-8 字节序列错误修复报告

## 问题描述

**错误信息**:
```
Failed to read file: Invalid utf8 byte sequence.
```

**出现位置**:
- `FSToolset.readFile` 方法
- `FSToolset.batchReadFiles` 方法
- `FSToolset.editFileContent` 方法

**日志示例**:
```
2025-12-08T14:10:25.631201+08:00 ERROR [FSToolset] Failed to read file: Invalid utf8 byte sequence.
2025-12-08T14:10:25.631281+08:00 INFO [DirectoryAnalyzerAgent] Observation: Failed to read file: Invalid utf8 byte sequence.
```

## 问题分析

### 根本原因

1. **文件包含无效 UTF-8 字节序列**:
   - 二进制文件（如图片、可执行文件）
   - 使用非 UTF-8 编码的文本文件（如 GBK、Big5）
   - 损坏的文件

2. **错误处理不够友好**:
   - 错误消息过于简单，用户不清楚原因
   - 没有提供解决方案或建议
   - 错误消息没有说明文件路径

3. **错误传播**:
   - `catRange` 函数已经处理了 UTF-8 错误，但返回的错误消息被 `readFile` 直接使用
   - `readFile` 的 catch 块没有进一步处理 UTF-8 特定的错误

### 代码流程

1. **readFile 方法**:
   ```cangjie
   let content = catRange(resolution.absolute, startLine ?? 1, endLine: endLine ?? Int64.Max)
   ```
   - 调用 `catRange` 读取文件
   - `catRange` 内部调用 `magic.utils.readFile`，可能抛出 UTF-8 异常
   - `catRange` 已经捕获异常并返回友好错误消息
   - 但 `readFile` 的 catch 块直接使用 `ex.message`，没有进一步处理

2. **batchReadFiles 方法**:
   ```cangjie
   match (Utf8Utils.decode(bytes, label: Some(filePath))) {
       case Some(content) => ...
       case None => results[fileIndex] = Some("Error reading ${filePath}: Invalid UTF-8 sequence")
   }
   ```
   - 使用 `Utf8Utils.decode` 安全解码
   - 但错误消息不够友好

3. **editFileContent 方法**:
   ```cangjie
   match (Utf8Utils.decode(File.readFrom(resolution.absolute), label: Some("FSToolset.editFile"))) {
       case Some(content) => ...
       case None => return "Invalid UTF-8 byte sequence in file: ${resolution.absolute}"
   }
   ```
   - 错误消息不够详细

## 修复方案

### 修复1：改进 readFile 错误处理

**文件**: `src/core/tools/fs_toolset.cj`

**修复内容**:
```cangjie
catch (ex: Exception) {
    // 🆕 改进：提供更友好的 UTF-8 错误消息
    let errorMsg = ex.message
    let friendlyMsg = if (errorMsg.contains("utf8") || errorMsg.contains("UTF-8") || 
                          errorMsg.contains("Invalid utf8") || errorMsg.contains("utf8 byte sequence")) {
        "Failed to read file: Invalid UTF-8 byte sequence. " +
        "The file contains binary data or uses an unsupported encoding. " +
        "This file cannot be read as text. Please check the file encoding or use a different file."
    } else {
        "Failed to read file: ${errorMsg}"
    }
    LogUtils.error("[FSToolset] ${friendlyMsg}")
    PrintUtils.printToolFailure("Read File", failureInfo: friendlyMsg)
    return friendlyMsg
}
```

**效果**:
- ✅ 检测 UTF-8 相关错误
- ✅ 提供友好的错误消息
- ✅ 说明原因和解决方案

### 修复2：改进缓存错误处理

**文件**: `src/core/tools/fs_toolset.cj`

**修复内容**:
```cangjie
} catch (cacheError: Exception) {
    // 🆕 改进：如果是因为 UTF-8 错误，记录更详细的信息
    let errorMsg = cacheError.message
    if (errorMsg.contains("utf8") || errorMsg.contains("UTF-8") || 
        errorMsg.contains("Invalid utf8") || errorMsg.contains("utf8 byte sequence")) {
        LogUtils.debug("[FSToolset] Cannot cache file with invalid UTF-8 encoding: ${resolution.absolute}. " +
                     "The file contains binary data or uses an unsupported encoding.")
    } else {
        LogUtils.debug("[FSToolset] Failed to cache full file for ${resolution.absolute}: ${errorMsg}")
    }
}
```

**效果**:
- ✅ 区分 UTF-8 错误和其他错误
- ✅ 提供更详细的日志信息
- ✅ 不影响部分读取的结果

### 修复3：改进 editFileContent 错误处理

**文件**: `src/core/tools/fs_toolset.cj`

**修复内容**:
```cangjie
case None => 
    // 🆕 改进：提供更友好的 UTF-8 错误消息
    let errorMsg = "Invalid UTF-8 byte sequence in file: ${resolution.absolute}. " +
                 "The file contains binary data or uses an unsupported encoding. " +
                 "This file cannot be edited as text. Please check the file encoding or use a different file."
    LogUtils.error("[FSToolset] ${errorMsg}")
    PrintUtils.printToolFailure("Edit File", failureInfo: errorMsg)
    return errorMsg
```

**效果**:
- ✅ 提供友好的错误消息
- ✅ 说明文件路径
- ✅ 提供解决方案建议

### 修复4：改进 batchReadFiles 错误处理

**文件**: `src/core/tools/fs_toolset.cj`

**修复内容**:
```cangjie
case None =>
    synchronized(mutex) {
        results[fileIndex] = Some("Error reading ${filePath}: Invalid UTF-8 byte sequence. " +
                                  "The file contains binary data or uses an unsupported encoding. " +
                                  "This file cannot be read as text.")
    }
```

**效果**:
- ✅ 提供更详细的错误消息
- ✅ 说明原因
- ✅ 与其他方法保持一致

## 修复完成

### 修复内容总结

1. ✅ **readFile 方法**: 改进异常处理，提供友好的 UTF-8 错误消息
2. ✅ **缓存错误处理**: 区分 UTF-8 错误和其他错误，提供详细日志
3. ✅ **editFileContent 方法**: 改进错误消息，提供解决方案
4. ✅ **batchReadFiles 方法**: 统一错误消息格式，提供详细说明

### 错误消息对比

**修复前**:
```
Failed to read file: Invalid utf8 byte sequence.
```

**修复后**:
```
Failed to read file: Invalid UTF-8 byte sequence. The file contains binary data or uses an unsupported encoding. This file cannot be read as text. Please check the file encoding or use a different file.
```

### 测试场景

1. **二进制文件**:
   - 尝试读取图片文件（如 `.png`, `.jpg`）
   - 应该返回友好的错误消息

2. **非 UTF-8 编码文件**:
   - 尝试读取 GBK 编码的文件
   - 应该返回友好的错误消息

3. **正常 UTF-8 文件**:
   - 应该正常读取，不受影响

4. **批量读取**:
   - 如果批量读取中包含无效 UTF-8 文件
   - 应该为每个文件返回独立的错误消息

## 后续建议

1. **文件类型检测**:
   - 在读取文件前检测文件类型
   - 对于已知的二进制文件类型，提前返回友好错误

2. **编码检测**:
   - 尝试检测文件编码（如使用 chardet 库）
   - 如果检测到非 UTF-8 编码，提供转换建议

3. **用户配置**:
   - 允许用户配置如何处理无效 UTF-8 文件
   - 选项：跳过、警告、错误

---

## 总结

**问题根源**: 文件包含无效 UTF-8 字节序列，错误处理不够友好

**修复方案**: 
1. 改进所有文件读取方法的错误处理
2. 提供友好的错误消息，说明原因和解决方案
3. 统一错误消息格式

**预期效果**: 用户遇到无效 UTF-8 文件时，能够清楚了解原因并获得解决方案建议

