# agentmem MCP 服务器启动问题分析报告

## 问题描述

`agentmem` MCP 服务器在 CodeLin 启动时未能正常加载，日志中没有任何关于 `agentmem` 的信息。

## 问题分析

### 1. 配置格式问题

**问题根源**：配置文件中包含了 `"type": "stdio"` 字段，但 CodeLin 的 MCP 配置解析逻辑不支持该字段。

**CodeLin 的类型判断逻辑**：
```cangjie
// src/core/mcp/mcp_config_manager.cj:26-31
redef public static func fromJsonValue(initParams: JsonValue): MCPServerConfig {
    if (let Some(_) <- initParams.asObject().get("url")) {
        return MCPServerConfig.Sse(SseMCPInitParams.fromJsonValue(initParams))
    } else {
        return MCPServerConfig.Stdio(StdioMCPInitParams.fromJsonValue(initParams))
    }
}
```

**判断规则**：
- 如果存在 `url` 字段 → SSE 类型
- 如果不存在 `url` 字段 → Stdio 类型

**问题**：
- `StdioMCPInitParams` 使用 `@jsonable` 宏自动生成 JSON 解析代码
- 如果 JSON 中包含未知字段（如 `type`），可能会导致解析失败
- 解析异常被捕获后只记录 "Invalid server config format"，不会显示具体错误信息

### 2. 错误处理机制

**代码位置**：`src/core/mcp/mcp_config_manager.cj:70-83`

```cangjie
public init() {
    if (let Some(jo) <- CliSettingManager.setting.mcpServers) {
        for ((serverName, initParams) in jo.getFields()) {
            try {
                let config = MCPServerConfig.fromJsonValue(initParams)
                this.mcpServers[serverName] = config
            } catch (ex: JsonException) {
                LogUtils.info("Invalid server config format for: ${serverName}")
            } catch (ex: JsonableException) {
                LogUtils.info("Invalid server config format for: ${serverName}")
            }
        }
    }
}
```

**问题**：
- 异常被静默处理，只记录 INFO 级别的日志
- 没有记录具体的错误信息，难以定位问题
- 如果解析失败，服务器不会被添加到 `mcpServers` 中，后续加载时也不会出现

### 3. 对比其他服务器

**成功加载的服务器**（无 `type` 字段）：
- `context7`: 只有 `command`, `args`, `env`
- `sequential-thinking`: 只有 `command`, `args`
- `playwright-mcp`: 只有 `command`, `args`
- `mastra`: 只有 `command`, `args`

**失败的服务器**：
- `shadcn-ui-server`: 文件不存在（"No such file or directory"）
- `agentmem`: 配置解析失败（包含 `type` 字段）

## 解决方案

### 修复配置

**移除 `type` 字段**，因为 CodeLin 通过是否存在 `url` 字段自动判断类型：

**修复前**：
```json
"agentmem": {
  "type": "stdio",
  "command": "/Users/louloulin/Documents/linchong/gitcode/magic/codelin/.codelin/agentmem-mcp-server",
  "args": [],
  "env": {
    "AGENTMEM_API_URL": "http://127.0.0.1:8080",
    "RUST_LOG": "info"
  }
}
```

**修复后**：
```json
"agentmem": {
  "command": "/Users/louloulin/Documents/linchong/gitcode/magic/codelin/.codelin/agentmem-mcp-server",
  "args": [],
  "env": {
    "AGENTMEM_API_URL": "http://127.0.0.1:8080",
    "RUST_LOG": "info"
  }
}
```

## 验证步骤

1. **检查配置文件**：
   ```bash
   cat .codelin/codelin.json | python3 -m json.tool
   ```

2. **重启 CodeLin**：
   ```bash
   cjpm run --name cli
   ```

3. **查看日志**：
   - 应该看到 "Creating Stdio MCP client for agentmem"
   - 应该看到 "Successfully loaded MCP server: agentmem"

4. **验证服务器加载**：
   - 在 CLI 中输入 `/mcp` 命令
   - 应该看到 `agentmem` 在服务器列表中

## 改进建议

### 1. 增强错误日志

**建议**：在捕获异常时记录详细的错误信息：

```cangjie
catch (ex: JsonException) {
    LogUtils.error("Invalid server config format for: ${serverName}: ${ex.message}")
    LogUtils.debug("Config: ${initParams}")
} catch (ex: JsonableException) {
    LogUtils.error("Invalid server config format for: ${serverName}: ${ex.message}")
    LogUtils.debug("Config: ${initParams}")
}
```

### 2. 支持 `type` 字段（可选）

**建议**：如果用户明确指定了 `type` 字段，优先使用该字段判断类型：

```cangjie
redef public static func fromJsonValue(initParams: JsonValue): MCPServerConfig {
    let obj = initParams.asObject()
    
    // 优先检查 type 字段
    if (let Some(typeValue) <- obj.get("type")) {
        match (typeValue.asString().getValue()) {
            case "stdio" => return MCPServerConfig.Stdio(StdioMCPInitParams.fromJsonValue(initParams))
            case "sse" => return MCPServerConfig.Sse(SseMCPInitParams.fromJsonValue(initParams))
            case _ => throw Exception("Unknown MCP server type: ${typeValue}")
        }
    }
    
    // 回退到自动判断
    if (let Some(_) <- obj.get("url")) {
        return MCPServerConfig.Sse(SseMCPInitParams.fromJsonValue(initParams))
    } else {
        return MCPServerConfig.Stdio(StdioMCPInitParams.fromJsonValue(initParams))
    }
}
```

### 3. 配置验证

**建议**：在启动时验证所有 MCP 服务器配置，并报告失败的服务器：

```cangjie
public func validateConfigs(): Array<String> {
    let failed = ArrayList<String>()
    if (let Some(jo) <- CliSettingManager.setting.mcpServers) {
        for ((serverName, initParams) in jo.getFields()) {
            try {
                let _ = MCPServerConfig.fromJsonValue(initParams)
            } catch (ex: Exception) {
                failed.add(serverName)
            }
        }
    }
    return failed.toArray()
}
```

## 总结

**根本原因**：配置文件中包含 `type` 字段，但 CodeLin 的 JSON 解析不支持该字段，导致解析失败。

**解决方案**：移除 `type` 字段，让 CodeLin 自动判断类型（通过是否存在 `url` 字段）。

**状态**：✅ 已修复配置文件

**下一步**：重启 CodeLin 验证 `agentmem` 服务器是否正常加载

