# Custom Agents Guide

CodeLin supports creating custom agents through configuration files, similar to Claude Code's skills system. This allows you to extend CodeLin's capabilities with specialized agents for your specific needs.

## Overview

Custom agents are loaded from TOML configuration files and automatically integrated into the main CodeLin agent. They can be used just like built-in SubAgents, but with your own custom prompts and tool configurations.

## Quick Start

### 1. Create Example Configuration

Run the following command to create an example agent configuration:

```bash
codelin /agents-init
```

This will create an example configuration file at `~/.codelin/agents/example_agent.toml`.

### 2. Edit the Configuration

Open the example file and customize it for your needs:

```toml
[agent]
name = "MyCustomAgent"
description = "A custom agent for specific tasks"
model = "default"  # Use default model
temperature = 0.7

# System prompt for the agent
prompt = """
You are a custom agent specialized in a specific task.

## Your Capabilities
- Capability 1
- Capability 2
- Capability 3

## Your Workflow
1. Step 1
2. Step 2
3. Step 3

Always be helpful and precise in your responses.
"""

# Available tools for the agent
tools = [
    "FSToolset",
    "SearchToolset"
]

# Tags for categorization
tags = ["custom", "example", "specialized"]

# Usage examples
examples = [
    "Example usage 1",
    "Example usage 2"
]
```

### 3. Restart CodeLin

Custom agents are automatically loaded when CodeLin starts. Simply restart CodeLin to use your new custom agent.

## Configuration File Format

### Required Fields

- `name`: Unique agent name (required)
- `prompt`: System prompt for the agent (required)

### Optional Fields

- `description`: Agent description
- `model`: Model to use (default: "default")
- `temperature`: Temperature parameter (default: 0.7)
- `maxTokens`: Maximum tokens (default: 4096)
- `tools`: List of available tools
- `tags`: Tags for categorization
- `examples`: Usage examples

## Available Tools

You can configure the following tools for your custom agent:

- `FSToolset`: File system operations
- `ShellTool`: Shell command execution
- `PlanToolset`: Planning and task management
- `GitToolset`: Git operations
- `SearchToolset`: Code search
- `LSPToolset`: Language Server Protocol tools
- `BackgroundTaskToolset`: Background task management
- `BatchEditToolset`: Batch file editing
- `CodeQualityToolset`: Code quality analysis

## Configuration Locations

CodeLin searches for custom agent configurations in the following locations (in order):

1. **Project-level**: `.codelin/agents/*.toml`
2. **Global**: `~/.codelin/agents/*.toml`

Project-level configurations take precedence over global ones.

## Usage

Once loaded, your custom agent can be invoked by the main CodeLin agent just like any built-in SubAgent. The main agent will automatically use your custom agent when appropriate based on the task context.

## Example Use Cases

### 1. Specialized Code Reviewer

Create a custom agent focused on security code review:

```toml
[agent]
name = "SecurityReviewer"
description = "Specialized agent for security code review"
prompt = """
You are a security-focused code reviewer. Your primary responsibility is to identify security vulnerabilities, insecure coding practices, and potential security risks in code.

Focus on:
- SQL injection vulnerabilities
- XSS vulnerabilities
- Authentication and authorization issues
- Sensitive data exposure
- Insecure dependencies
"""
tools = ["FSToolset", "LSPToolset", "CodeQualityToolset"]
tags = ["security", "review", "specialized"]
```

### 2. Documentation Generator

Create a custom agent for generating documentation:

```toml
[agent]
name = "DocGenerator"
description = "Specialized agent for generating code documentation"
prompt = """
You are a documentation generator. Your task is to create comprehensive, clear, and well-structured documentation for code.

Guidelines:
- Use clear and concise language
- Include code examples
- Document all public APIs
- Follow project documentation standards
"""
tools = ["FSToolset", "LSPToolset"]
tags = ["documentation", "generation"]
```

### 3. Test Generator

Create a custom agent for generating tests:

```toml
[agent]
name = "TestGenerator"
description = "Specialized agent for generating test cases"
prompt = """
You are a test generator. Your task is to create comprehensive test cases for code.

Focus on:
- Unit tests
- Integration tests
- Edge cases
- Error handling
- Test coverage
"""
tools = ["FSToolset", "LSPToolset"]
tags = ["testing", "generation"]
```

## Best Practices

1. **Clear Prompts**: Write clear and specific prompts that define the agent's role and responsibilities.

2. **Appropriate Tools**: Only include tools that are relevant to the agent's task to avoid confusion.

3. **Descriptive Names**: Use descriptive names that clearly indicate the agent's purpose.

4. **Tags**: Use tags to categorize and organize your custom agents.

5. **Examples**: Include usage examples to help understand when to use the agent.

## Limitations

- Custom agents are loaded at startup, so you need to restart CodeLin after creating or modifying configuration files.
- Tool configuration is currently limited to predefined tool sets.
- Custom agents use the same model as the main agent (unless specified otherwise).

## Troubleshooting

### Agent Not Loading

- Check that the configuration file is in the correct location (`.codelin/agents/` or `~/.codelin/agents/`)
- Verify that the TOML file is valid
- Check that required fields (`name` and `prompt`) are present
- Check CodeLin logs for error messages

### Agent Not Being Used

- Ensure the agent's prompt clearly defines when it should be used
- Check that the agent has appropriate tools for the task
- Verify that the agent is loaded (check startup logs)

## Advanced Usage

### Project-Specific Agents

Place agent configurations in `.codelin/agents/` for project-specific agents that are only available in that project.

### Global Agents

Place agent configurations in `~/.codelin/agents/` for global agents that are available in all projects.

### Multiple Agents

You can create multiple custom agents by creating multiple TOML files. All valid configurations will be loaded automatically.

## See Also

- [CodeLin README](../README.md) - Main documentation
- [Agent Architecture](../docs/AGENT_ARCHITECTURE.md) - Understanding CodeLin's agent system
- [Tool Reference](../docs/TOOLS.md) - Available tools reference

