"""MQTT 5.0 over WebSocket 通用客户端实现模块.
Generated by AI.
"""

import asyncio
import logging
from collections.abc import AsyncGenerator
from dataclasses import dataclass, field
from enum import IntEnum
from typing import Any

import httpx
import orjson as json
from httpx_ws import AsyncWebSocketSession, aconnect_ws

logger = logging.getLogger("MQTTClient")


class MqttRedirectError(Exception):
    """MQTT 5.0 Server Moved (0x9D) 异常."""

    def __init__(self, new_address: str) -> None:
        self.new_address = new_address
        super().__init__(f"Server moved to {new_address}")


class PacketType(IntEnum):
    """MQTT 报文类型枚举."""

    CONNECT = 0x10
    CONNACK = 0x20
    PUBLISH = 0x30
    PUBACK = 0x40
    SUBSCRIBE = 0x82
    SUBACK = 0x90
    PINGREQ = 0xC0
    PINGRESP = 0xD0
    DISCONNECT = 0xE0


class PropertyId(IntEnum):
    """MQTT 5.0 属性 ID 枚举."""

    SERVER_REFERENCE = 0x1C
    REASON_STRING = 0x1F
    AUTH_METHOD = 0x15
    USER_PROPERTY = 0x26


@dataclass
class MqttMessage:
    """通用的 MQTT 消息对象."""

    topic: str
    payload: bytes
    qos: int
    properties: dict[str, str] = field(default_factory=dict)

    @property
    def json(self) -> Any | None:
        """尝试将 payload 解析为 JSON 对象."""
        try:
            return json.loads(self.payload)
        except json.JSONDecodeError:
            return None


class MqttCodec:
    """MQTT 5.0 协议编解码工具类."""

    @staticmethod
    def encode_varbyte(val: int) -> bytes:
        """编码 MQTT 变长字节整数."""
        out = bytearray()
        while True:
            byte = val % 128
            val //= 128
            if val > 0:
                byte |= 128
            out.append(byte)
            if val == 0:
                break
        return bytes(out)

    @staticmethod
    def decode_varbyte(data: bytes, offset: int) -> tuple[int, int]:
        """解码 MQTT 变长字节整数."""
        value = 0
        multiplier = 1
        length = 0
        while True:
            if offset + length >= len(data):
                return 0, 0
            byte = data[offset + length]
            value += (byte & 127) * multiplier
            multiplier *= 128
            length += 1
            if (byte & 128) == 0:
                break
        return value, length

    @staticmethod
    def encode_string(s: str) -> bytes:
        """编码 MQTT 字符串 (UTF-8 及其 2 字节长度前缀)."""
        b = s.encode("utf-8")
        return len(b).to_bytes(2, "big") + b

    @staticmethod
    def decode_string(data: bytes, offset: int) -> tuple[str, int]:
        """解码 MQTT 字符串."""
        str_len = int.from_bytes(data[offset : offset + 2], "big")
        return data[offset + 2 : offset + 2 + str_len].decode("utf-8"), 2 + str_len

    @staticmethod
    def encode_props(props: dict[Any, Any]) -> bytes:
        """编码 MQTT 5.0 属性部分."""
        out = bytearray()
        for pid, val in props.items():
            match pid:
                case PropertyId.USER_PROPERTY:
                    for k, v in val:
                        out.append(pid)
                        out.extend(MqttCodec.encode_string(k) + MqttCodec.encode_string(v))
                case PropertyId.AUTH_METHOD:
                    out.append(pid)
                    out.extend(MqttCodec.encode_string(val))
                case _:
                    pass
        return bytes(out)

    @staticmethod
    def decode_user_properties(data: bytes, start: int, end: int) -> dict[str, str]:
        """仅解码 User Properties (用于 PUBLISH 消息)."""
        res = {}
        curr = start
        while curr < end:
            pid = data[curr]
            curr += 1
            match pid:
                case PropertyId.USER_PROPERTY:
                    k, length = MqttCodec.decode_string(data, curr)
                    curr += length
                    v, length = MqttCodec.decode_string(data, curr)
                    curr += length
                    res[k] = v
                case _:
                    logger.warning(f"Skipping unhandled property ID in PUBLISH: {pid}")
                    break
        return res

    @staticmethod
    def decode_connack_properties(data: bytes, offset: int) -> dict[int, Any]:
        """解码 CONNACK 属性, 提取重定向地址."""
        props_len, length = MqttCodec.decode_varbyte(data, offset)
        p_start = offset + length
        p_end = p_start + props_len

        props = {}
        curr = p_start
        while curr < p_end:
            pid = data[curr]
            curr += 1

            match pid:
                case PropertyId.USER_PROPERTY:
                    _, k_len = MqttCodec.decode_string(data, curr)
                    curr += k_len
                    _, v_len = MqttCodec.decode_string(data, curr)
                    curr += v_len

                case PropertyId.SERVER_REFERENCE:
                    val, v_len = MqttCodec.decode_string(data, curr)
                    curr += v_len
                    props[pid] = val

                case PropertyId.REASON_STRING:
                    val, v_len = MqttCodec.decode_string(data, curr)
                    curr += v_len
                    props[pid] = val

                case _:
                    logger.warning(f"Unknown property {hex(pid)} in CONNACK, aborting property decode")
                    break
        return props


class Client:
    """通用、轻量级的 MQTT 5.0 over WebSocket 客户端."""

    def __init__(self, client_id: str, host: str, port: int, path: str = "/mqtt", keep_alive: int = 45) -> None:
        self.client_id = client_id
        self.host = host
        self.port = port
        self.path = path
        self.keep_alive = keep_alive
        self._ws: AsyncWebSocketSession | None = None
        self._ws_ctx: Any | None = None
        self._http_client: httpx.AsyncClient | None = None

    async def __aenter__(self) -> "Client":
        return self

    async def __aexit__(
        self, exc_type: type[BaseException] | None, exc_val: BaseException | None, exc_tb: Any | None
    ) -> None:
        await self.disconnect()

    async def _get_http_client(self) -> httpx.AsyncClient:
        """获取 HTTP 客户端."""
        if self._http_client is None or self._http_client.is_closed:
            self._http_client = httpx.AsyncClient(timeout=30.0)
        return self._http_client

    async def connect(self, properties: dict[Any, Any] | None = None, headers: dict[str, str] | None = None) -> None:
        """建立 WebSocket 连接并发送 MQTT CONNECT 报文."""
        if self._ws:
            await self.disconnect_ws_only()

        url = f"wss://{self.host}:{self.port}{self.path}"
        logger.info(f"Connecting to {url}...")

        client = await self._get_http_client()

        self._ws_ctx = aconnect_ws(url, subprotocols=["mqtt"], headers=headers, client=client)
        ws = await self._ws_ctx.__aenter__()
        self._ws = ws

        # 1. Variable Header
        proto = MqttCodec.encode_string("MQTT") + b"\x05\x02" + self.keep_alive.to_bytes(2, "big")
        props_bytes = MqttCodec.encode_props(properties or {})
        props_len = MqttCodec.encode_varbyte(len(props_bytes))
        var_header = proto + props_len + props_bytes

        # 2. Payload
        payload = MqttCodec.encode_string(self.client_id)

        # 3. Fixed Header
        rem_len = MqttCodec.encode_varbyte(len(var_header) + len(payload))
        packet = bytes([PacketType.CONNECT]) + rem_len + var_header + payload

        await ws.send_bytes(packet)
        resp = await ws.receive_bytes()

        if not resp or resp[0] != PacketType.CONNACK:
            raise ConnectionError(f"Invalid Packet: {resp.hex() if resp else 'None'}")

        _, rl_len = MqttCodec.decode_varbyte(resp, 1)
        reason_idx = 1 + rl_len + 1

        if reason_idx >= len(resp):
            raise ConnectionError("Malformed CONNACK packet")

        reason_code = resp[reason_idx]

        if reason_code == 0x00:
            logger.info("Connected.")
            return

        if reason_code == 0x9D:
            logger.info("Received 0x9D Server Moved.")
            conn_props = MqttCodec.decode_connack_properties(resp, reason_idx + 1)
            new_server = conn_props.get(PropertyId.SERVER_REFERENCE)
            if new_server:
                raise MqttRedirectError(new_server)
            else:
                logger.error("Server moved but no Server Reference provided.")

        raise ConnectionError(f"MQTT Connect Failed. Reason Code: {hex(reason_code)}. Response: {resp.hex()}")

    async def subscribe(self, topic: str, properties: dict[Any, Any] | None = None) -> None:
        """发送 SUBSCRIBE 报文."""
        if not self._ws:
            raise ConnectionError("WebSocket is not connected")

        packet_id = b"\x00\x01"
        props_bytes = MqttCodec.encode_props(properties or {})
        props_len = MqttCodec.encode_varbyte(len(props_bytes))
        var_header = packet_id + props_len + props_bytes
        payload = MqttCodec.encode_string(topic) + b"\x00"
        rem_len = MqttCodec.encode_varbyte(len(var_header) + len(payload))
        packet = bytes([PacketType.SUBSCRIBE]) + rem_len + var_header + payload

        await self._ws.send_bytes(packet)
        await self._ws.receive_bytes()

    async def disconnect_ws_only(self) -> None:
        """仅断开 WebSocket 连接."""
        if self._ws:
            try:
                await self._ws.send_bytes(bytes([PacketType.DISCONNECT, 0x00]))
            except Exception:
                pass

            if self._ws_ctx:
                try:
                    await self._ws_ctx.__aexit__(None, None, None)
                except RuntimeError as e:
                    # [修复] 忽略在不同 Task (如 GC) 中关闭 anyio cancel scope 的错误
                    if "cancel scope" not in str(e):
                        raise
                except Exception:
                    pass
            self._ws = None

    async def disconnect(self) -> None:
        """断开连接并释放所有资源."""
        await self.disconnect_ws_only()
        if self._http_client:
            await self._http_client.aclose()
            self._http_client = None
        logger.info("Disconnected.")

    async def messages(self) -> AsyncGenerator[MqttMessage, None]:
        """获取消息监听迭代器."""
        if not self._ws:
            raise ConnectionError("Not connected")

        while True:
            try:
                msg_bytes = await asyncio.wait_for(self._ws.receive_bytes(), timeout=self.keep_alive - 5)
            except asyncio.TimeoutError:
                await self._ws.send_bytes(bytes([PacketType.PINGREQ, 0x00]))
                continue

            if not msg_bytes:
                break

            pkt_type = msg_bytes[0] & 0xF0
            match pkt_type:
                case PacketType.PINGRESP:
                    continue
                case PacketType.PUBLISH:
                    _, length = MqttCodec.decode_varbyte(msg_bytes, 1)
                    offset = 1 + length
                    topic, length = MqttCodec.decode_string(msg_bytes, offset)
                    offset += length
                    qos = (msg_bytes[0] & 0x06) >> 1
                    if qos > 0:
                        offset += 2
                    props_len, length = MqttCodec.decode_varbyte(msg_bytes, offset)
                    p_start = offset + length
                    p_end = p_start + props_len
                    user_props = MqttCodec.decode_user_properties(msg_bytes, p_start, p_end)
                    payload = msg_bytes[p_end:]
                    yield MqttMessage(topic=topic, payload=payload, qos=qos, properties=user_props)
                case _:
                    pass
