/**
  ******************************************************************************
  * @file    COMP/COMP_Hygrometer/main.c 
  * @author  MCD Application Team
  * @version V1.1.3
  * @date    14-December-2021
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * Copyright (c) 2015 STMicroelectronics.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */


/* Includes ------------------------------------------------------------------*/
#include "main.h"

/** @addtogroup COMP_Hygrometer
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
__IO uint16_t AvrgICReadValue = 0;
uint16_t ICError = 0;
uint32_t DisplayValue = 50;
uint32_t RHmin = 99;
uint32_t RHmax = 0;
float Capacitance;
float TriggerTime;
float Capacitance55RH = 180e-12; /* typical capacitance value at 55% RH */
float RelativeHumidity;

/* Private function prototypes -----------------------------------------------*/
static void TIM3_PWM_Config(void);
static void COMP4_Config(void);
static void TIM4_IC_Config(void);
static void Calibration_Menu(void);
static void Display_Humidity(uint32_t DisplayHumidity);

/* Private functions ---------------------------------------------------------*/

/**
  * @brief  Main program.
  * @param  None
  * @retval None
  */
int main(void)
{
  float capacitanceratio;

  /*!< At this stage the microcontroller clock setting is already configured, 
       this is done through SystemInit() function which is called from startup
       file (startup_stm32f30x.s) before to branch to application main.
       To reconfigure the default setting of SystemInit() function, refer to
       system_stm32f30x.c file
     */ 
  
  /* Configure the SEL button in GPIO mode */
  STM_EVAL_PBInit(BUTTON_SEL, BUTTON_MODE_GPIO);
  
  /* Configure the UP button in GPIO mode */
  STM_EVAL_PBInit(BUTTON_UP, BUTTON_MODE_GPIO);
  
  /* Configure the DOWN button in GPIO mode */
  STM_EVAL_PBInit(BUTTON_DOWN, BUTTON_MODE_GPIO);
  
  /* Configure the KEY button in GPIO mode */
  STM_EVAL_PBInit(BUTTON_KEY, BUTTON_MODE_GPIO);
  
  /* ------------------------  TFT LCD configuration ------------------------ */
  
  /* Initialize the TFT-LCD */
  STM32303C_LCD_Init();  
  
  /* Clear the LCD */ 
  LCD_Clear(LCD_COLOR_BLACK);
  
  /* Set the LCD Back Color */
  LCD_SetBackColor(LCD_COLOR_BLACK);
  
  /* Set the LCD Text Color */
  LCD_SetTextColor(LCD_COLOR_WHITE);
  
  /* Displays Title Message on line 0 */
  LCD_DisplayStringLine(LINE(0), (uint8_t *)" RELATIVE  HUMIDITY ");
  
  /* -------------- Comparator & Timers configuration ----------------------- */

  /* Comparator Configuration */ 
  COMP4_Config();
  
  /* TIM3 channel3 Configuration in PWM Mode */
  TIM3_PWM_Config();

  /* TIM4 Channel2 Configuration in Input Capture Mode */
  TIM4_IC_Config();
  
  /* Enable TIM3 (TIM4 is enabled in the same cycle) */
  TIM_Cmd(TIM3, ENABLE);
  
  /* wait until first AvrgICReadValue is calculated */
  while(AvrgICReadValue == 0);
  
  /* Enter Calibration menu */
  Calibration_Menu();
  
  /* Infinite loop */
  while (1)
  {
    /* Calculate Trigger Time Value */
    TriggerTime = (float) (AvrgICReadValue-ICError)/SystemCoreClock;

#ifdef USE_DAC  
    /* Comp4 inverted input connected to DAC1 :
     * TriggerTime = RES * Capacitance * ln(VDD/(VDD - VREF))
     * @VREF = 2.086V (generated by DAC),  ln(VDD/(VDD - VREF)) is ~ 1
     *  ==>  Capacitance = TriggerTime/RES
     */
    Capacitance = (float) TriggerTime/RES;

#elif USE_VREFINT
    /* Comp4 inverted input connected to VrefInt
     * TriggerTime = RES * Capacitance * ln(VDD/(VDD - VREF))
     * @VREF = VREFINT = 1.22V,  ln(VDD/(VDD - VREF)) is VREFINT_FACTOR ~ 0.46
     *  ==>  Capacitance = TriggerTime/(RES*VREFINT_FACTOR)
     */
    Capacitance = (float) TriggerTime/(RES*VREFINT_FACTOR);

#endif /* USE_DAC */
    
/* Calculate humidity value using reversed polynomial expression */
    capacitanceratio = Capacitance/Capacitance55RH;
    /* RH (%) = -3.4656*10^3 * X^3 + 1.0732*10^4 * X^2 - 1.0457*10^4*X + 3.2459*10^3
       with X = C (read) / C@55%RH = capacitanceratio */
    RelativeHumidity = RP3 * pow(capacitanceratio, 3) + 
                       RP2 * pow(capacitanceratio, 2) + 
                       RP1 * capacitanceratio + 
                       RP0;

    /* Restrict Relative Humidity Value to 0-99 Domain */
    if (RelativeHumidity < 0)
    {
      RelativeHumidity = 0;
    }
    if (RelativeHumidity > 99)
    {
      RelativeHumidity = 99;
    }
    
    /* Display the humidity value */
    Display_Humidity((uint32_t) RelativeHumidity);
  }
}

/**
  * @brief  Configures TIM3: channels in PWM mode
  * @param  None
  * @retval None
  */
static void TIM3_PWM_Config(void)
{
  TIM_OCInitTypeDef       TIM_OCInitStructure;
  TIM_TimeBaseInitTypeDef TIM_TimeBaseStructure;
  GPIO_InitTypeDef        GPIO_InitStructure;
  uint16_t periodvalue = 0;  
  
  /* GPIO clock enable */
  RCC_AHBPeriphClockCmd(HUM_OUT_GPIO_CLK, ENABLE);

  /* TIM3 channel pin configuration: TIM3_CH3 -> PC8 */
  GPIO_InitStructure.GPIO_Mode  = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin   = HUM_OUT_PIN;
  GPIO_Init(HUM_OUT_GPIO, &GPIO_InitStructure);

  /* Map TIM3 CH3 to PC8 */
  GPIO_PinAFConfig(HUM_OUT_GPIO, HUM_OUT_PIN_SOURCE, HUM_OUT_AF);
  
  /* TIM3 clock enable */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM3, ENABLE);

  /* Calculate the period value */
  periodvalue = (uint16_t) ((SystemCoreClock) / FREQ);

  /* Time Base configuration */
  TIM_TimeBaseStructure.TIM_Prescaler = 1;
  TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;
  TIM_TimeBaseStructure.TIM_Period = periodvalue;
  TIM_TimeBaseStructure.TIM_ClockDivision = 0;
  TIM_TimeBaseStructure.TIM_RepetitionCounter = 0;
  TIM_TimeBaseInit(TIM3, &TIM_TimeBaseStructure);

  /* Channel 3 configuration in PWM mode */
  TIM_OCInitStructure.TIM_OCMode = TIM_OCMode_PWM2;
  TIM_OCInitStructure.TIM_OutputState = TIM_OutputState_Enable;
  TIM_OCInitStructure.TIM_Pulse = periodvalue/2;
  TIM_OCInitStructure.TIM_OCPolarity = TIM_OCPolarity_Low;
  TIM_OCInitStructure.TIM_OCIdleState = TIM_OCIdleState_Set;
  TIM_OC3Init(TIM3, &TIM_OCInitStructure);
  
  /* Select the Master Slave Mode to Synchronize TIM3 with TIM4 */
  TIM_SelectMasterSlaveMode(TIM3, TIM_MasterSlaveMode_Enable);  
  /* Master Mode Trigger Output Selection */
  TIM_SelectOutputTrigger(TIM3, TIM_TRGOSource_Enable);    
}

/**
  * @brief  Configures COMP4: PB0 as COMP4 non inverting input
  *                           VREFINT/DAC1 as COMP4 inverting input
  *                           and COMP4 output to TIM4 IC2.
  * @param  None
  * @retval None
  */
static void COMP4_Config(void)
{
  /* Init Structure definition */
  COMP_InitTypeDef        COMP_InitStructure;
  GPIO_InitTypeDef        GPIO_InitStructure;

#ifdef USE_DAC  
  DAC_InitTypeDef  DAC_InitStructure;

  /* DAC clock enable */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_DAC, ENABLE);

  /* DAC configuration */
  DAC_InitStructure.DAC_Trigger = DAC_Trigger_None;
  DAC_InitStructure.DAC_WaveGeneration = DAC_WaveGeneration_None;
  DAC_InitStructure.DAC_LFSRUnmask_TriangleAmplitude = DAC_LFSRUnmask_Bit0;
  DAC_InitStructure.DAC_Buffer_Switch = DAC_BufferSwitch_Enable;
  DAC_Init(DAC1, DAC_Channel_1, &DAC_InitStructure);
  
  /* Enable DAC Channel1 */
  DAC_Cmd(DAC1, DAC_Channel_1, ENABLE);
  
  /* Set DAC Channel1 DHR register: 
  * DAC_OUT1 = (3.3 * 2588) / 4095 ~ 2.086 V = 3.3*(1-exp(-t/R*C)) with t=R*C */
  DAC_SetChannel1Data(DAC1, DAC_Align_12b_R, 2588);
  
#endif /* USE_DAC */
  
  /* GPIOB Peripheral clock enable */
  RCC_AHBPeriphClockCmd(HUM_IN_GPIO_CLK, ENABLE);
  
  /* Configure PB0 in analog mode: PB0 is connected to COMP4 non inverting input */
  GPIO_InitStructure.GPIO_Pin = HUM_IN_PIN;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_Init(HUM_IN_GPIO, &GPIO_InitStructure);
   
  /* COMP Peripheral clock enable */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_SYSCFG, ENABLE);

  /* COMP4 config */
  COMP_InitStructure.COMP_InvertingInput = COMP_INV_INPUT;
  COMP_InitStructure.COMP_NonInvertingInput = COMP_NINV_INPUT;  
  COMP_InitStructure.COMP_Output = COMP_Output_TIM4IC2;
  COMP_InitStructure.COMP_OutputPol = COMP_OutputPol_NonInverted;
  COMP_InitStructure.COMP_BlankingSrce = COMP_BlankingSrce_None;
  COMP_InitStructure.COMP_Hysteresis = COMP_Hysteresis_No;
  COMP_InitStructure.COMP_Mode = COMP_Mode_HighSpeed;
  COMP_Init(COMP_Selection_COMP4, &COMP_InitStructure);
  
  /* Enable COMP4 */
  COMP_Cmd(COMP_Selection_COMP4, ENABLE);
}

/**
  * @brief  Configures TIM4: channel 2 in Input Capture mode
  * @param  None
  * @retval None
  */
static void TIM4_IC_Config(void)
{
  TIM_ICInitTypeDef  TIM_ICInitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;
  TIM_TimeBaseInitTypeDef TIM_TimeBaseStructure;
  uint16_t periodvalue = 0;
  
  /* TIM4 clocks enable */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM4 , ENABLE);
  
  /* Enable the TIM global Interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = TIM4_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority =0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  /* Calculate the period value */
  periodvalue = (uint16_t) ((SystemCoreClock) / FREQ);
  
  /* Time Base configuration */
  TIM_TimeBaseStructure.TIM_Prescaler = 1;
  TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;
  TIM_TimeBaseStructure.TIM_Period = periodvalue;
  TIM_TimeBaseStructure.TIM_ClockDivision = 0;
  TIM_TimeBaseStructure.TIM_RepetitionCounter = 0;
  TIM_TimeBaseInit(TIM4, &TIM_TimeBaseStructure);
  
  /* TIM4 configuration: Input Capture mode ---------------------
     The external signal is connected to TIM4 CH2 pin
     The Rising edge is used as active edge
  ------------------------------------------------------------ */
  TIM_ICInitStructure.TIM_Channel     = TIM_Channel_2;
  TIM_ICInitStructure.TIM_ICPolarity  = TIM_ICPolarity_Rising;
  TIM_ICInitStructure.TIM_ICSelection = TIM_ICSelection_DirectTI;
  TIM_ICInitStructure.TIM_ICPrescaler = TIM_ICPSC_DIV1;
  TIM_ICInitStructure.TIM_ICFilter = 0;
  TIM_ICInit(TIM4, &TIM_ICInitStructure);
  
  /* Enable the CC2 Interrupt Request */
  TIM_ITConfig(TIM4, TIM_IT_CC2, ENABLE);

  /* Slave Mode selection: TIM4 is synchronized to start with TIM3 */
  TIM_SelectSlaveMode(TIM4, TIM_SlaveMode_Trigger);
  /* Select the input trigger source ITR2 (TIM3) */
  TIM_SelectInputTrigger(TIM4, TIM_TS_ITR2);
}

/**
  * @brief  Enter Calibration menu to correct ICError and Capacitance55RH values
  * @param  None
  * @retval None 
  */
static void Calibration_Menu(void)
{
  uint8_t exitmenu = 0;
  uint8_t LCDstr[20] = {0};
  
  /* Set the LCD Text Color */
  LCD_SetTextColor(LCD_COLOR_RED);
  
  /* Display Calibration Screen */
  LCD_DisplayStringLine(LINE(1), (uint8_t *)" Calibration steps: ");
  LCD_DisplayStringLine(LINE(2), (uint8_t *)"1. Set JP19 to REF  ");  
  LCD_DisplayStringLine(LINE(3), (uint8_t *)"2. Press SEL button "); 

    /* ------------- First step calibration using reference capacitance ----- */
    while (exitmenu != 1)
    { 
      if (STM_EVAL_PBGetState(BUTTON_SEL) == SET) 
      {
        /* Get ICError for reference capacitance */
#ifdef USE_DAC
    /* TriggerTime = (AvrgICReadValue - ICError)/SystemCoreClock
     * TriggerTime = RES * REFCAP * ln(VDD/(VDD - VREF))
     * @VREF = 2.086V (generated by DAC),  ln(VDD/(VDD - VREF)) is ~ 1
     *  ==> TriggerTime = RES * REFCAP
     *  Then RES * REFCAP = (AvrgICReadValue - ICError)/SystemCoreClock
     *  ==>  ICError = AvrgICReadValue - REFCAP * RES * SystemCoreClock
     */
        ICError = (uint16_t) (AvrgICReadValue-REFCAP*RES*SystemCoreClock);
#elif USE_VREFINT
    /* TriggerTime = (AvrgICReadValue - ICError)/SystemCoreClock
     * TriggerTime = RES * REFCAP * ln(VDD/(VDD - VREF))
     * @VREF = VREFINT,  ln(VDD/(VDD - VREFINT)) is VREFINT_FACTOR ~ 0.46
     *  ==> TriggerTime = RES * REFCAP * VREFINT_FACTOR
     *  Then RES * REFCAP * VREFINT_FACTOR = (AvrgICReadValue - ICError)/SystemCoreClock
     *  ==>  ICError = AvrgICReadValue - REFCAP * RES * VREFINT_FACTOR * SystemCoreClock
     */
        ICError = (uint16_t) (AvrgICReadValue-REFCAP*RES*SystemCoreClock*VREFINT_FACTOR);
#endif /* USE_DAC */
        /* Set exitmenu to 1 */
        exitmenu = 1;
      }
    }
  
  /* --------------- Second step calibration using reference humidity ------- */
  exitmenu = 0;
  LCD_DisplayStringLine(LINE(1), (uint8_t *)"now set JP19 to HUM ");
  LCD_DisplayStringLine(LINE(2), (uint8_t *)"and select humidity ");
  LCD_DisplayStringLine(LINE(3), (uint8_t *)"value using U/D keys");
  LCD_DisplayStringLine(LINE(8), (uint8_t *)"Sel Button: Apply   ");
  LCD_DisplayStringLine(LINE(9), (uint8_t *)"Key Button: Cancel  ");
  
  /* Wait for SEL button to be released */
  while (STM_EVAL_PBGetState(BUTTON_SEL) == SET);  
    
  while (exitmenu != 1)
  {
    if ((STM_EVAL_PBGetState(BUTTON_UP) == SET) && (DisplayValue<99))
    {
      /* Wait for UP button to be released */
      while (STM_EVAL_PBGetState(BUTTON_UP) == SET);
      DisplayValue++;
    }
    if ((STM_EVAL_PBGetState(BUTTON_DOWN) == SET) && (DisplayValue>0))
    {
      /* Wait for DOWN button to be released */
      while (STM_EVAL_PBGetState(BUTTON_DOWN) == SET);
      DisplayValue--;
    }
    
   /* Display humidity value on LCD Line 4 */ 
   sprintf((char*)LCDstr, "        %d %%       ", DisplayValue);
   LCD_DisplayStringLine(LINE(4), (uint8_t*) LCDstr);

    if (STM_EVAL_PBGetState(BUTTON_SEL) == SET)
    {
      /* Calculate Trigger Time Value */
      TriggerTime = (float) (AvrgICReadValue-ICError)/SystemCoreClock;
      
      /* Calculate Capacitance Value */
  #ifdef USE_DAC
      Capacitance = (float) TriggerTime/RES;
  #elif USE_VREFINT
      Capacitance = (float) TriggerTime/(RES*VREFINT_FACTOR);
  #endif /* USE_DAC */
      
      /* Update Capacitance55RH value: capacitance @ 55% Relative Humidity */
      Capacitance55RH= Capacitance/(P3 * pow(DisplayValue,3) + 
                                    P2 * pow(DisplayValue,2) + 
                                    P1 * DisplayValue + 
                                    P0 );
      exitmenu = 1;
    }
    else if (STM_EVAL_PBGetState(BUTTON_KEY) != SET) 
    {
      /* Humidity Calibration canceled */
      exitmenu = 1; 
    }
  }

  /* Set the LCD Text Color */
  LCD_SetTextColor(LCD_COLOR_WHITE);
  
  /* Clear Calibration Screen */
  LCD_ClearLine(LCD_LINE_1);
  LCD_ClearLine(LCD_LINE_2);
  LCD_ClearLine(LCD_LINE_3);
  LCD_ClearLine(LCD_LINE_8);  
  LCD_ClearLine(LCD_LINE_9);   
}

/**
  * @brief  Calculate and Dispaly the humidity value on LCD
  * @param  DisplayValue : Relative Humidity in percent
  * @retval None
  */
static void Display_Humidity(uint32_t DisplayHumidity)
{
  uint8_t LCDstr[20] = {0};
  
  /* Display humidity value on LCD Line 4 */ 
  sprintf((char*)LCDstr, "        %d %%        ", DisplayHumidity);
  LCD_DisplayStringLine(LINE(4), (uint8_t *) LCDstr);
  
  /* Store the minimal and maximal RH values */
  if (RelativeHumidity < RHmin)
  {
    RHmin = (uint8_t) RelativeHumidity;
  }
  
  if (RelativeHumidity > RHmax)
  {
    RHmax = (uint8_t) RelativeHumidity;
  }
  
  /* Display humidity range on LCD Line 9 */ 
  sprintf((char*)LCDstr, "Min= %d%%    Max= %d%% ", RHmin,RHmax);
  LCD_DisplayStringLine(LINE(9), (uint8_t *) LCDstr);
}

#ifdef  USE_FULL_ASSERT

/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t* file, uint32_t line)
{ 
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif

/**
  * @}
  */


