import fs from "node:fs";
import os from "node:os";
import path from "node:path";

import { afterEach, beforeEach, describe, expect, it, vi } from "vitest";

import type { HealthSummary } from "./health.js";
import { getHealthSnapshot } from "./health.js";
import { setActivePluginRegistry } from "../plugins/runtime.js";
import { createTestRegistry } from "../test-utils/channel-plugins.js";
import { telegramPlugin } from "../../extensions/telegram/src/channel.js";

let testConfig: Record<string, unknown> = {};
let testStore: Record<string, { updatedAt?: number }> = {};

vi.mock("../config/config.js", async (importOriginal) => {
  const actual = await importOriginal<typeof import("../config/config.js")>();
  return {
    ...actual,
    loadConfig: () => testConfig,
  };
});

vi.mock("../config/sessions.js", () => ({
  resolveStorePath: () => "/tmp/sessions.json",
  loadSessionStore: () => testStore,
  readSessionUpdatedAt: vi.fn(() => undefined),
  recordSessionMetaFromInbound: vi.fn().mockResolvedValue(undefined),
  updateLastRoute: vi.fn().mockResolvedValue(undefined),
}));

vi.mock("../web/auth-store.js", () => ({
  webAuthExists: vi.fn(async () => true),
  getWebAuthAgeMs: vi.fn(() => 1234),
  readWebSelfId: vi.fn(() => ({ e164: null, jid: null })),
  logWebSelfId: vi.fn(),
  logoutWeb: vi.fn(),
}));

describe("getHealthSnapshot", () => {
  beforeEach(async () => {
    setActivePluginRegistry(
      createTestRegistry([{ pluginId: "telegram", plugin: telegramPlugin, source: "test" }]),
    );
    const { createPluginRuntime } = await import("../plugins/runtime/index.js");
    const { setTelegramRuntime } = await import("../../extensions/telegram/src/runtime.js");
    setTelegramRuntime(createPluginRuntime());
  });

  afterEach(() => {
    vi.unstubAllGlobals();
    vi.unstubAllEnvs();
  });

  it("skips telegram probe when not configured", async () => {
    testConfig = { session: { store: "/tmp/x" } };
    testStore = {
      global: { updatedAt: Date.now() },
      unknown: { updatedAt: Date.now() },
      main: { updatedAt: 1000 },
      foo: { updatedAt: 2000 },
    };
    vi.stubEnv("TELEGRAM_BOT_TOKEN", "");
    vi.stubEnv("DISCORD_BOT_TOKEN", "");
    const snap = (await getHealthSnapshot({
      timeoutMs: 10,
    })) satisfies HealthSummary;
    expect(snap.ok).toBe(true);
    const telegram = snap.channels.telegram as {
      configured?: boolean;
      probe?: unknown;
    };
    expect(telegram.configured).toBe(false);
    expect(telegram.probe).toBeUndefined();
    expect(snap.sessions.count).toBe(2);
    expect(snap.sessions.recent[0]?.key).toBe("foo");
  });

  it("probes telegram getMe + webhook info when configured", async () => {
    testConfig = { channels: { telegram: { botToken: "t-1" } } };
    testStore = {};
    vi.stubEnv("DISCORD_BOT_TOKEN", "");

    const calls: string[] = [];
    vi.stubGlobal(
      "fetch",
      vi.fn(async (url: string) => {
        calls.push(url);
        if (url.includes("/getMe")) {
          return {
            ok: true,
            status: 200,
            json: async () => ({
              ok: true,
              result: { id: 1, username: "bot" },
            }),
          } as unknown as Response;
        }
        if (url.includes("/getWebhookInfo")) {
          return {
            ok: true,
            status: 200,
            json: async () => ({
              ok: true,
              result: {
                url: "https://example.com/h",
                has_custom_certificate: false,
              },
            }),
          } as unknown as Response;
        }
        return {
          ok: false,
          status: 404,
          json: async () => ({ ok: false, description: "nope" }),
        } as unknown as Response;
      }),
    );

    const snap = await getHealthSnapshot({ timeoutMs: 25 });
    const telegram = snap.channels.telegram as {
      configured?: boolean;
      probe?: {
        ok?: boolean;
        bot?: { username?: string };
        webhook?: { url?: string };
      };
    };
    expect(telegram.configured).toBe(true);
    expect(telegram.probe?.ok).toBe(true);
    expect(telegram.probe?.bot?.username).toBe("bot");
    expect(telegram.probe?.webhook?.url).toMatch(/^https:/);
    expect(calls.some((c) => c.includes("/getMe"))).toBe(true);
    expect(calls.some((c) => c.includes("/getWebhookInfo"))).toBe(true);
  });

  it("treats telegram.tokenFile as configured", async () => {
    const tmpDir = fs.mkdtempSync(path.join(os.tmpdir(), "clawdbot-health-"));
    const tokenFile = path.join(tmpDir, "telegram-token");
    fs.writeFileSync(tokenFile, "t-file\n", "utf-8");
    testConfig = { channels: { telegram: { tokenFile } } };
    testStore = {};
    vi.stubEnv("TELEGRAM_BOT_TOKEN", "");

    const calls: string[] = [];
    vi.stubGlobal(
      "fetch",
      vi.fn(async (url: string) => {
        calls.push(url);
        if (url.includes("/getMe")) {
          return {
            ok: true,
            status: 200,
            json: async () => ({
              ok: true,
              result: { id: 1, username: "bot" },
            }),
          } as unknown as Response;
        }
        if (url.includes("/getWebhookInfo")) {
          return {
            ok: true,
            status: 200,
            json: async () => ({
              ok: true,
              result: {
                url: "https://example.com/h",
                has_custom_certificate: false,
              },
            }),
          } as unknown as Response;
        }
        return {
          ok: false,
          status: 404,
          json: async () => ({ ok: false, description: "nope" }),
        } as unknown as Response;
      }),
    );

    const snap = await getHealthSnapshot({ timeoutMs: 25 });
    const telegram = snap.channels.telegram as {
      configured?: boolean;
      probe?: { ok?: boolean };
    };
    expect(telegram.configured).toBe(true);
    expect(telegram.probe?.ok).toBe(true);
    expect(calls.some((c) => c.includes("bott-file/getMe"))).toBe(true);

    fs.rmSync(tmpDir, { recursive: true, force: true });
  });

  it("returns a structured telegram probe error when getMe fails", async () => {
    testConfig = { channels: { telegram: { botToken: "bad-token" } } };
    testStore = {};
    vi.stubEnv("DISCORD_BOT_TOKEN", "");

    vi.stubGlobal(
      "fetch",
      vi.fn(async (url: string) => {
        if (url.includes("/getMe")) {
          return {
            ok: false,
            status: 401,
            json: async () => ({ ok: false, description: "unauthorized" }),
          } as unknown as Response;
        }
        throw new Error("unexpected");
      }),
    );

    const snap = await getHealthSnapshot({ timeoutMs: 25 });
    const telegram = snap.channels.telegram as {
      configured?: boolean;
      probe?: { ok?: boolean; status?: number; error?: string };
    };
    expect(telegram.configured).toBe(true);
    expect(telegram.probe?.ok).toBe(false);
    expect(telegram.probe?.status).toBe(401);
    expect(telegram.probe?.error).toMatch(/unauthorized/i);
  });

  it("captures unexpected probe exceptions as errors", async () => {
    testConfig = { channels: { telegram: { botToken: "t-err" } } };
    testStore = {};
    vi.stubEnv("DISCORD_BOT_TOKEN", "");

    vi.stubGlobal(
      "fetch",
      vi.fn(async () => {
        throw new Error("network down");
      }),
    );

    const snap = await getHealthSnapshot({ timeoutMs: 25 });
    const telegram = snap.channels.telegram as {
      configured?: boolean;
      probe?: { ok?: boolean; error?: string };
    };
    expect(telegram.configured).toBe(true);
    expect(telegram.probe?.ok).toBe(false);
    expect(telegram.probe?.error).toMatch(/network down/i);
  });

  it("disables heartbeat for agents without heartbeat blocks", async () => {
    testConfig = {
      agents: {
        defaults: {
          heartbeat: {
            every: "30m",
            target: "last",
          },
        },
        list: [
          { id: "main", default: true },
          { id: "ops", heartbeat: { every: "1h", target: "whatsapp" } },
        ],
      },
    };
    testStore = {};

    const snap = await getHealthSnapshot({ timeoutMs: 10, probe: false });
    const byAgent = new Map(snap.agents.map((agent) => [agent.agentId, agent] as const));
    const main = byAgent.get("main");
    const ops = byAgent.get("ops");

    expect(main?.heartbeat.everyMs).toBeNull();
    expect(main?.heartbeat.every).toBe("disabled");
    expect(ops?.heartbeat.everyMs).toBeTruthy();
    expect(ops?.heartbeat.every).toBe("1h");
  });
});
