# GPU Marching Cubes - Developer Notes

## Personal notes
Do not use Scene Inheritance.
Plan out entire project structure as a monolith.
Enforce composition for medium parts and scripting for small parts.
Top level folder as entire game starting point and configuration.
Folder based activation of functionality.(Modules. Enable. Disable.)
Example: hook up new entities, models, objects, systems.
Singular isolated folders for features that hook up to the system. (data, scripts)
Convert as many things as possible to modules. 

## GPU Resource Cleanup (Building System)

**File:** `building_system/building_mesher.gd`  
**Toggle:** `ENABLE_GPU_CLEANUP` (line ~26)

### If you experience crashes during building/mesh generation:

1. Open `building_system/building_mesher.gd`
2. Find `const ENABLE_GPU_CLEANUP: bool = true`
3. Change to `false`
4. Test if crashes stop

### Background:
- The original code had GPU resource freeing disabled due to crashes
- We re-enabled it with correct freeing order (uniform_set first, then textures/sampler)
- Tested working on Godot 4.5 without crashes
- If crashes return, the freeing order may need more research

### Trade-off:
- **Cleanup ON:** No GPU memory leak
- **Cleanup OFF:** Leaks ~32KB per mesh generation (~3MB per 100 blocks placed)

---

## Slope-Based Cliff Rock Texture

**File:** `marching_cubes/terrain.gdshader`  
**Uniform:** `surface_cliff_threshold` (default: 10.0)

### What it does:
- Applies a rocky texture to steep surfaces (cliffs) based on surface normal angle
- Only active **above** `surface_cliff_threshold` Y level (default Y=10)
- Underground areas show raw materials (stone, ore, etc.) for gameplay clarity

### Potential Gameplay Issue:
> ⚠️ **FUTURE CONSIDERATION:** Players mining near the surface may not see the actual terrain materials (stone, ore, gravel) because the rocky cliff texture overlays them. This is purely visual enhancement and may need to:
> - Be reduced in intensity
> - Have the threshold lowered further
> - Be removed entirely if it causes confusion during early mining

### How to adjust:
1. **Change threshold:** Modify `surface_cliff_threshold` in the shader material (lower = less rocky texture near surface)
2. **Reduce intensity:** In the shader, find `cliff_mix` calculation and multiply by a value < 1.0
3. **Disable entirely:** Comment out the cliff rock blending section (lines ~190-198)

### Current behavior:
- Y 8-12: Smooth transition zone
- Y < 8: No rocky texture (shows actual materials)

---

## Road Edge Blending

**File:** `marching_cubes/terrain.gdshader`  
**Uniforms:** `road_edge_blend_enabled`, `road_edge_blend_width`

### Current Status:
> ⚠️ **Working but not perfect.** Road edge blending has some asymmetry due to Marching Cubes mesh boundary variations. The blend zone uses a fixed distance calculation (`procedural_road_width * 0.43`), but the actual road mesh edge varies with terrain height/slope.

### Why it's imperfect:
- Road **geometry** (mesh) is generated by density shader with smooth blending
- Road **material** (mat_id 6) is assigned with stricter bounds + height check
- The visual road edge depends on both, creating inconsistent boundaries
- Shader blend uses fixed distance, cannot adapt to per-pixel mesh variations

### Potential future fix:
- Pass actual mesh boundary info from density to terrain shader (complex)
- Or accept the current "good enough" state

---

## Procedural Building Generator

**Directory:** `building_system/`  
**Files:** `building_generator.gd`

### Current Status:
> ⚠️ **Imperfect Water Detection:** Procedural buildings are sometimes placed over water or in water despite multiple detection methods. The water detection system cannot reliably prevent all water placements.

> ⚠️ **Floating `small_house` Prefab:** The legacy `small_house` prefab (hardcoded in `prefab_spawner.gd` lines 34-62) sometimes spawns floating above terrain. This is an older placement system that needs updating to properly ground buildings.

### Why it's imperfect:
- **Timing issue:** Buildings are queued when terrain chunk generates, but water density data may not be available yet (water chunks not loaded)
- **Terrain shape:** Water appears in depressions - a building may be placed on terrain above water level while its footprint extends over a lower depression filled with water
- **Regional noise mismatch:** The GDScript noise approximation for "wet regions" doesn't perfectly match the GPU shader's noise function

### Current detection methods (all have limitations):
1. **Terrain height vs water_level:** Works when terrain data is available, but can miss nearby depressions
2. **Water density check:** Only works if water chunks are loaded (often not true at queue time)
3. **Wet region noise:** Approximates shader noise but doesn't match exactly

### Potential future fixes:
- Defer all building spawns until water chunks are loaded
- Improve noise matching between GDScript and shader
- Accept imperfection and let players manually remove misplaced buildings

### Implemented: Save Building Spawn State
Building spawn decisions are now persisted via SaveManager:
- `building_generator.gd` has `get_save_data()` / `load_save_data()` methods
- `save_manager.gd` integrates with building_generator
- Debug toggle: `regenerate_buildings_on_load` (default: false)
  - Set to `true` to regenerate buildings fresh for debugging placement logic

### Future: Deterministic Building Carving During Terrain Generation
> 💡 **Potential Optimization:** Use deterministic building placement and carve prefab interiors block-by-block during terrain generation:
> - Determine building positions deterministically (seeded by world coordinates)
> - During chunk terrain generation, detect if a building will spawn there
> - Carve interior space upward column-by-column until reaching top prefab block
> - Clear vegetation/trees in building footprint (prevent trees inside buildings)
> - Then place the building blocks
> - **Benefit:** Buildings won't have terrain inside them; faster than post-placement carving

---

## Vehicle System

**Directory:** `vehicles/`, `addons/srcoder_simplecar/`

### Current Status:
> ⚠️ **Vehicles are hard to deal with.** The current vehicle system works but has known limitations due to lack of experience in vehicle physics tuning.

### Known Issues:
- **Camera behavior:** Getting the follow camera to behave correctly (decoupled from car rotation, smooth lag, etc.) is tricky with Godot's scene hierarchy
- **Physics tuning:** Car handling requires careful balance of mass, grip, suspension stiffness, and damping - easy to make the car feel "floaty" or unstable
- **High-speed stability:** Cars can become difficult to control at high speeds

### What's working:
- Basic driving with WASD controls
- Enter/exit vehicle interaction
- Vehicle save/load integration
- Deferred spawning (waits for terrain)
- Water physics (buoyancy)
- Flip recovery (B key)

### Future improvements needed:
- Better camera system (possibly custom, not child of car)
- More refined physics parameters
- Possibly switch to a different vehicle addon/approach

---

## Zombie Entity System

**Directory:** `entities/`  
**Files:** `zombie_base.gd`, `zombie_base.tscn`

### Spawning:
- **F10:** Spawn default capsule entity
- **F11:** Spawn zombie

### Known quirks:

#### Floating Gap Fix
- **Issue:** Zombies appear to float slightly above terrain
- **Cause:** `safe_margin` property on CharacterBody3D adds a buffer distance
- **Fix:** Set `safe_margin = 0.04` (4cm) instead of default 0.2 (20cm)
- Also adjust model Y position in scene if needed

#### Model Facing Direction
- **Issue:** GLB model faces backwards (negative Z is forward)
- **Fix:** Rotate the model node 180° in the scene file (negate X and Z scale)
- Do NOT use `rotate_y(PI)` in code - cleaner to fix in scene

### Animation System:
Uses **time-slice animation** on a single "Take 001" animation:
- Idle: 0-1s
- Walk/Chase: 1-2s
- Attack: 3.5-4.5s
- Death: 9.5s+

### Deferred Spawning (Dec 2024):
Entities wait for terrain before spawning:
1. Spawn request added to queue
2. Each frame, queue is checked - spawns only when terrain is ready AND within spawn_radius
3. If player moves away, pending spawn is cancelled

### Terrain Collision Issue:
> ⚠️ **NEEDS MORE WORK:** When player moves far away, terrain chunks unload and entities lose collision, causing them to fall through.

**Current workaround:** `despawn_radius` (80m) is set lower than terrain load distance (~155m) so entities are removed before terrain unloads.

**How it works:**
- `_freeze_entity()`: Disables physics, zeros velocity
- `_unfreeze_entity()`: Only re-enables if terrain is loaded at position
- `dormant_entities`: Stores despawned entity data for respawning when player returns

### Future: World Simulation (TODO)
> 💡 Since we now have persistent dormant entities, a logical next step is **world simulation**:
> - Simulate zombie movement/behavior even when player is far away
> - Zombies could wander, die, or migrate while dormant
> - Would require lightweight "tick" simulation for dormant entities
> - Could use cellular automata or simplified AI for distant entities

### Testing Needed: Collision Cleanup
> ⚠️ **Verify that collisions properly unload when chunks/entities despawn:**
> - Do prefab doors (and other building objects) unload their collision?
> - Do zombies truly release their collision shapes when despawned?
> - Does terrain collision memory get freed when chunks unload?
> - Potential memory leak if collisions persist after objects are removed

---

## Terrain & Chunk Loading Customization (Priority)

> ⚠️ **NEEDS HEAVY WORK:** The terrain and chunk loading systems require significant customization:
> - Loading priority and sequencing improvements needed
> - Player spawn-area prioritization
> - Underground/sky chunk loading optimization
> - Integration with procedural spawning systems
> - Performance tuning for smooth exploration

See also: `TERRAIN_LOADING_PRIORITY.md` for related notes.

---

## Combat System Architecture (Strategy)

### Hitscan vs Projectile Analysis
We have decided on a **hybrid approach** for future weapon implementation:

1.  **Hitscan (Simulated Laser):**
	*   **Use case:** Fast firearms (Pistols, Rifles).
	*   **Reason:** Cheaper performance, easier network sync, "snappy" feel.
	*   **Implementation:** Immediate raycast + Visual Tracer Line. No physical object spawned.

2.  **Projectile (Physical Object):**
	*   **Use case:** Bows, Crossbows, Grenades, Slow Plasma.
	*   **Reason:** Requires travel time, gravity arc, or bouncy physics.
	*   **Implementation:** Spawn `Area3D/RigidBody3D` that moves each frame.
	*   **Structure:** Create `combat_system/projectile_manager.gd` to pool and update these objects efficiently.

### Planned Structure
*   `combat_system/` directory.
*   `weapon_base.gd`: Interactable script attached to props.
*   `projectile_manager.gd`: Global handler for physical projectiles.

---

## Interaction System Refinement (TODO)

### Issue: Object vs. Prop Placement
Currently, there is a disconnect between picking up a placed **Object** (Grid-aligned) and dropping it as a **Prop** (Physics/Free).
*   **Grid Placement:** Snaps to integer coordinates, usually center-aligned.
*   **Physics Drop:** Free rotation, physics-simulated.

### Future Goal: "Smart Re-Placement"
When moving an object via pickup (E), we want to preserve its "intention":
**Current limitation:** Objects dropped via "Physics Drop" lose their grid alignment and exact rotation, making it hard to "tidy up" a room manually.
