/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 SHA-512 + Twofish";
static const u64   KERN_TYPE      = 14633; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$sha512$twofish$cbc-plain64$256$72257$8ca21acee1c388ad1953f25c035e9eebc5077e40ccb5754b07ce82a6508508ab$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$8e35818299b0b563fa07577b015dd7fcbab0a6633b0a8287cfd7cc469d5d8d056bee7a2e9669613627b0178c3ee842d2375b80a891aaf4dd56520c5a3e2d30a0b1f983a1df02ad5538476d969fcd9c271de76d9cd28fb4deb83f8016b7974c6a4591763cc3bd4604b5a8fde7fc578903bced6a4c7dabd986fc59fa27ae9d25f0ab25980473d2a2d83eb95eebf5d08cef411b5a5d5ee7d3b4c53d2441acf8f8b5bd0b535698d23f5d9b4674da9e83f6f5c71114b88f207c306444449f789dad43e16333a7408608b653cc0b808a760223d39ca525a6b680ea8e95266db4bbba4dd40dd8ed9fa24fb8fbc63d5ea90e3a2d4f28b1d56372b88a786c754c7929b98a79db58b1437e1c6cdd140292cec3a32aef73f845b3cb0c32676d92f00c267159f20e7a748d02c8d489b35059217629bd1deb3db8eaf8cb6af50c99e37dfb89c719b0c6aebaa3a0439ade05e192c16f6b20eae5d1d92bf59b609acee2f3163e2c7b3c8306c9e9ded6f67a498c842364903f4094cf312e3e26b913e0efba712731129ef9f80f116fffbf7240cb6dc66f67c2562acea697352f809dd947ea759702b9334cfed7c5f1639a9392b1c4fc0a89811ccdc9fca943623b86bd3036390402c0a7f388fcee403d5de14621dd48bef71b6957b0831efb73ba4d13dd030b2bffae64c0f579cd03391c172b8e721b3a8571d39a74561dbfcc751d98689875fee8";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

const char *module_usage_notice (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  return USAGE_NOTICE_LUKS;
}

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$sha512$twofish$";

bool module_unstable_warning (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra, MAYBE_UNUSED const hc_device_param_t *device_param)
{
  if ((device_param->opencl_platform_vendor_id == VENDOR_ID_APPLE) && (device_param->opencl_device_type & CL_DEVICE_TYPE_GPU))
  {
    if (device_param->is_metal == true)
    {
      if (strncmp (device_param->device_name, "Intel", 5) == 0)
      {
        // Intel Iris Graphics, Metal Version 244.303: failed to create 'm14633_comp' pipeline, timeout reached
        return true;
      }
    }
  }

  return false;
}

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  memset (&token, 0, sizeof (hc_token_t));

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 23;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len[2]     = 3;
  token.attr[2]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len[4]     = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len[6]     = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_SHA512;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_TWOFISH;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];
  const u32 cipher_mode_len = token.len[1];

  if ((strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0) && (cipher_mode_len == 16))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  const luks_t *luks = (const luks_t *) esalt_buf;

  // cipher mode

  const char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char *salt_buf = (char *) hcmalloc (LUKS_SALT_HEX_LEN + 1);

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char *af_buf = (char *) hcmalloc (LUKS_AF_MAX_HEX_LEN + 1);

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char *ct_buf = (char *) hcmalloc (LUKS_CT_HEX_LEN + 1);

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  hcfree (salt_buf);
  hcfree (af_buf);
  hcfree (ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_bridge_name              = MODULE_DEFAULT;
  module_ctx->module_bridge_type              = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_usage_notice             = module_usage_notice;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = MODULE_DEFAULT;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = module_unstable_warning;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
