/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 SHA-512 + AES";
static const u64   KERN_TYPE      = 14631; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$sha512$aes$cbc-plain64$256$71794$33221dc8fd2e55f191e19d77fe95f86f91af4c1858d65ccd0cfee1f515c84b97$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$b411774e7902aa939b2d416a004b504b2a258c05de31e6a33617cf05d547425156981063468460052e206080094680e6edf7b9de8a803286acfbfe536ae44ac173c43f845bc8512c5e7398974a6e226862fdf2e5b4c0c3bf20d7390e1da263f2205fe2eaff5274c3437671827e977343e1a08d70bd60bb18ed3005524a8313220ac12bc450f06eb62a6dbec85845aed08b0ff225bdd1a608ab7aa43a454be51b6dd8df011428c5643a733ed7d8b4f873e91b0419fc6d63aa8dafc8ce1562a4bb9c4fe93fe42dad960e106e21a4dee46bef845dc8519a54ebdb14e6d8e6ad9063045f42bf4a820d602edfc2ccb16a303008828095d63f16b227f343bfd455b89ba6498df1b251490937f6912f121aefb5b5d965d99faa403ca917cbdbf3750b8de9eb9bf8abc53727e7cc80264f4aeff95265fc2658fc813b282fde88bd7260f5502e25023a16ad8c07186183df43516e751c291938e90f9a0b71b8205298b350eab896ae334039b170a54fd188d86768c988c830a89e31b699bc23174e7b0c7ac856e1949b64142dee4558e187b890280996c748fde5a1e10a3a242812ea5839d942a0f39ec41648084de19891e423a6ccfe83bd14aac9eac0cde5f2f37ff1ac1737e26d64502b8a28957d800894dc1796450c1ac8f717db7edc8ca8c4887a6b705f46554197a44606d095dfbb74f5a75cbec2487c4b3ab70352dea1a2986248";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

const char *module_usage_notice (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  return USAGE_NOTICE_LUKS;
}

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$sha512$aes$";

bool module_unstable_warning (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra, MAYBE_UNUSED const hc_device_param_t *device_param)
{
  if ((device_param->opencl_platform_vendor_id == VENDOR_ID_APPLE) && (device_param->opencl_device_type & CL_DEVICE_TYPE_GPU))
  {
    if (device_param->is_metal == true)
    {
      if (strncmp (device_param->device_name, "Intel", 5) == 0)
      {
        // Intel Iris Graphics, Metal Version 244.303: failed to create 'm14631_init' pipeline, timeout reached
        return true;
      }
    }
  }

  return false;
}

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  memset (&token, 0, sizeof (hc_token_t));

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 19;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len[2]     = 3;
  token.attr[2]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len[4]     = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len[6]     = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_SHA512;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_AES;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];
  const u32 cipher_mode_len = token.len[1];

  if ((strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0) && (cipher_mode_len == 16))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  const luks_t *luks = (const luks_t *) esalt_buf;

  // cipher mode

  const char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char *salt_buf = (char *) hcmalloc (LUKS_SALT_HEX_LEN + 1);

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char *af_buf = (char *) hcmalloc (LUKS_AF_MAX_HEX_LEN + 1);

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char *ct_buf = (char *) hcmalloc (LUKS_CT_HEX_LEN + 1);

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  hcfree (salt_buf);
  hcfree (af_buf);
  hcfree (ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_bridge_name              = MODULE_DEFAULT;
  module_ctx->module_bridge_type              = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_usage_notice             = module_usage_notice;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = MODULE_DEFAULT;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = module_unstable_warning;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
