// Copyright 2022 CFC4N <cfc4n.cs@gmail.com>. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package event

import (
	"bytes"
	"encoding/binary"
	"fmt"

	pb "github.com/gojue/ecapture/protobuf/gen/v1"
)

type GnutlsDataEvent struct {
	eventType Type
	DataType  int64             `json:"dataType"`
	Timestamp uint64            `json:"timestamp"`
	Pid       uint32            `json:"pid"`
	Tid       uint32            `json:"tid"`
	Data      [MaxDataSize]byte `json:"data"`
	DataLen   int32             `json:"data_len"`
	Comm      [16]byte          `json:"Comm"`
	base      Base
}

func (ge *GnutlsDataEvent) Decode(payload []byte) (err error) {
	buf := bytes.NewBuffer(payload)
	if err = binary.Read(buf, binary.LittleEndian, &ge.DataType); err != nil {
		return
	}
	if err = binary.Read(buf, binary.LittleEndian, &ge.Timestamp); err != nil {
		return
	}
	if err = binary.Read(buf, binary.LittleEndian, &ge.Pid); err != nil {
		return
	}
	if err = binary.Read(buf, binary.LittleEndian, &ge.Tid); err != nil {
		return
	}
	if err = binary.Read(buf, binary.LittleEndian, &ge.Data); err != nil {
		return
	}
	if err = binary.Read(buf, binary.LittleEndian, &ge.DataLen); err != nil {
		return
	}
	if err = binary.Read(buf, binary.LittleEndian, &ge.Comm); err != nil {
		return
	}

	decodedKtime, err := DecodeKtime(int64(ge.Timestamp), true)
	if err == nil {
		ge.Timestamp = uint64(decodedKtime.UnixNano())
	}

	return nil
}

func (ge *GnutlsDataEvent) StringHex() string {
	var perfix, packetType string
	switch AttachType(ge.DataType) {
	case ProbeEntry:
		packetType = fmt.Sprintf("%sReceived%s", COLORGREEN, COLORRESET)
		perfix = COLORGREEN
	case ProbeRet:
		packetType = fmt.Sprintf("%sSend%s", COLORPURPLE, COLORRESET)
		perfix = fmt.Sprintf("%s\t", COLORPURPLE)
	default:
		perfix = fmt.Sprintf("UNKNOWN_%d", ge.DataType)
	}

	b := dumpByteSlice(ge.Data[:ge.DataLen], perfix)
	b.WriteString(COLORRESET)
	s := fmt.Sprintf("PID:%d, Comm:%s, Type:%s, TID:%d, DataLen:%d bytes, Payload:\n%s", ge.Pid, ge.Comm, packetType, ge.Tid, ge.DataLen, b.String())
	return s
}

func (ge *GnutlsDataEvent) String() string {
	var perfix, packetType string
	switch AttachType(ge.DataType) {
	case ProbeEntry:
		packetType = fmt.Sprintf("%sReceived%s", COLORGREEN, COLORRESET)
		perfix = COLORGREEN
	case ProbeRet:
		packetType = fmt.Sprintf("%sSend%s", COLORPURPLE, COLORRESET)
		perfix = COLORPURPLE
	default:
		packetType = fmt.Sprintf("%sUNKNOWN_%d%s", COLORRED, ge.DataType, COLORRESET)
	}
	s := fmt.Sprintf(" PID:%d, Comm:%s, TID:%d, TYPE:%s, DataLen:%d bytes, Payload:\n%s%s%s", ge.Pid, ge.Comm, ge.Tid, packetType, ge.DataLen, perfix, string(ge.Data[:ge.DataLen]), COLORRESET)
	return s
}

func (ge *GnutlsDataEvent) Clone() IEventStruct {
	event := new(GnutlsDataEvent)
	event.eventType = TypeEventProcessor
	return event
}

func (ge *GnutlsDataEvent) Base() Base {
	ge.base = Base{
		Timestamp: int64(ge.Timestamp),
		UUID:      ge.GetUUID(),
		SrcIP:     "127.0.0.1", // Gnutls events do not have SrcIP
		SrcPort:   0,           // Gnutls events do not have SrcPort
		DstIP:     "127.0.0.1", // Gnutls events do not have DstIP
		DstPort:   0,           // Gnutls events do not have DstPort
		PID:       int64(ge.Pid),
		PName:     string(ge.Comm[:]),
		Type:      uint32(ge.DataType),
	}
	return ge.base
}

func (ge *GnutlsDataEvent) ToProtobufEvent() *pb.Event {
	return &pb.Event{
		Timestamp: int64(ge.Timestamp),
		Uuid:      ge.GetUUID(),
		SrcIp:     "127.0.0.1", // Gnutls events do not have SrcIP
		SrcPort:   0,           // Gnutls events do not have SrcPort
		DstIp:     "127.0.0.1", // Gnutls events do not have DstIP
		DstPort:   0,           // Gnutls events do not have DstPort
		Pid:       int64(ge.Pid),
		Pname:     commStr(ge.Comm[:]),
		Type:      uint32(ge.DataType),
		Length:    uint32(ge.DataLen),
		Payload:   ge.Data[:ge.DataLen],
	}
}

func (ge *GnutlsDataEvent) EventType() Type {
	return ge.eventType
}

func (ge *GnutlsDataEvent) GetUUID() string {
	//return fmt.Sprintf("%d_%d_%s", ge.Pid, ge.Tid, ge.Comm)
	return fmt.Sprintf("%d_%d_%s_%d", ge.Pid, ge.Tid, ge.Comm, ge.DataType)
}

func (ge *GnutlsDataEvent) Payload() []byte {
	return ge.Data[:ge.DataLen]
}

func (ge *GnutlsDataEvent) PayloadLen() int {
	return int(ge.DataLen)
}
