# -*- coding: utf-8 -*-
"""
屏幕画面防检测处理器
实时捕获屏幕画面，添加防检测处理后显示到窗口
直播伴侣可以录制这个窗口

使用方法:
1. 运行此脚本
2. 会打开一个全屏窗口，显示处理后的屏幕画面
3. 直播伴侣选择录制这个窗口
"""

import cv2
import numpy as np
import time
import threading
import sys
from typing import Optional

# 尝试导入屏幕捕获库
try:
    import mss
    MSS_AVAILABLE = True
except ImportError:
    MSS_AVAILABLE = False
    print("请安装 mss: pip install mss")


class ScreenAntiDetect:
    """屏幕画面防检测处理器"""
    
    def __init__(self, monitor_id: int = 1):
        """
        初始化
        
        Args:
            monitor_id: 显示器ID (1=主显示器, 2=第二显示器...)
        """
        self.monitor_id = monitor_id
        self.running = False
        self.frame_count = 0
        self.change_interval = 30  # 每30帧变化参数
        
        # 防检测参数
        self.noise_strength = 3      # 噪点强度
        self.shift_range = 2         # 位移范围
        self.hue_range = 3           # 色调范围
        self.brightness_range = 5    # 亮度范围
        self.contrast_range = 0.02   # 对比度范围
        self.saturation_range = 0.03 # 饱和度范围
        
        # 当前随机参数
        self._update_random_params()
        
        # 屏幕捕获
        self.sct = None
        if MSS_AVAILABLE:
            self.sct = mss.mss()
    
    def _update_random_params(self):
        """更新随机参数"""
        import random
        self.current_shift_x = random.randint(-self.shift_range, self.shift_range)
        self.current_shift_y = random.randint(-self.shift_range, self.shift_range)
        self.current_hue = random.randint(-self.hue_range, self.hue_range)
        self.current_brightness = random.randint(-self.brightness_range, self.brightness_range)
        self.current_contrast = 1.0 + random.uniform(-self.contrast_range, self.contrast_range)
        self.current_saturation = 1.0 + random.uniform(-self.saturation_range, self.saturation_range)
    
    def set_strength(self, level: str = "medium"):
        """设置处理强度"""
        if level == "low":
            self.noise_strength = 1
            self.shift_range = 1
            self.hue_range = 1
            self.brightness_range = 2
        elif level == "medium":
            self.noise_strength = 3
            self.shift_range = 2
            self.hue_range = 3
            self.brightness_range = 5
        elif level == "high":
            self.noise_strength = 5
            self.shift_range = 3
            self.hue_range = 5
            self.brightness_range = 10
        self._update_random_params()
    
    def process_frame(self, frame: np.ndarray) -> np.ndarray:
        """处理单帧画面"""
        self.frame_count += 1
        
        # 周期性更新参数
        if self.frame_count % self.change_interval == 0:
            self._update_random_params()
        
        result = frame.copy()
        
        # 1. 添加噪点
        if self.noise_strength > 0:
            noise = np.random.randint(
                -self.noise_strength,
                self.noise_strength + 1,
                result.shape,
                dtype=np.int16
            )
            result = np.clip(result.astype(np.int16) + noise, 0, 255).astype(np.uint8)
        
        # 2. 随机位移
        if self.current_shift_x != 0 or self.current_shift_y != 0:
            M = np.float32([[1, 0, self.current_shift_x], [0, 1, self.current_shift_y]])
            result = cv2.warpAffine(result, M, (result.shape[1], result.shape[0]))
        
        # 3. 色彩调整
        hsv = cv2.cvtColor(result, cv2.COLOR_BGR2HSV).astype(np.float32)
        hsv[:, :, 0] = np.clip(hsv[:, :, 0] + self.current_hue, 0, 179)
        hsv[:, :, 1] = np.clip(hsv[:, :, 1] * self.current_saturation, 0, 255)
        hsv[:, :, 2] = np.clip(hsv[:, :, 2] + self.current_brightness, 0, 255)
        result = cv2.cvtColor(hsv.astype(np.uint8), cv2.COLOR_HSV2BGR)
        
        # 4. 对比度
        result = cv2.convertScaleAbs(result, alpha=self.current_contrast, beta=0)
        
        return result
    
    def capture_screen(self) -> Optional[np.ndarray]:
        """捕获屏幕"""
        if not self.sct:
            return None
        
        try:
            monitor = self.sct.monitors[self.monitor_id]
            screenshot = self.sct.grab(monitor)
            frame = np.array(screenshot)
            # BGRA -> BGR
            frame = cv2.cvtColor(frame, cv2.COLOR_BGRA2BGR)
            return frame
        except Exception as e:
            print(f"屏幕捕获失败: {e}")
            return None
    
    def run(self, window_name: str = "防检测画面 - 直播伴侣录制此窗口"):
        """
        运行防检测显示窗口
        
        按 Q 退出
        按 1/2/3 切换强度 (低/中/高)
        按 F 全屏/窗口切换
        """
        if not MSS_AVAILABLE:
            print("错误: mss库未安装")
            print("请运行: pip install mss")
            return
        
        self.running = True
        fullscreen = True
        
        # 创建窗口
        cv2.namedWindow(window_name, cv2.WINDOW_NORMAL)
        if fullscreen:
            cv2.setWindowProperty(window_name, cv2.WND_PROP_FULLSCREEN, cv2.WINDOW_FULLSCREEN)
        
        print("=" * 50)
        print("屏幕防检测处理器已启动")
        print("=" * 50)
        print("操作说明:")
        print("  Q - 退出")
        print("  1 - 低强度 (几乎不可见)")
        print("  2 - 中强度 (推荐)")
        print("  3 - 高强度 (明显变化)")
        print("  F - 全屏/窗口切换")
        print("=" * 50)
        print("请让直播伴侣录制此窗口")
        print("=" * 50)
        
        fps_time = time.time()
        fps_count = 0
        current_fps = 0
        
        while self.running:
            # 捕获屏幕
            frame = self.capture_screen()
            if frame is None:
                time.sleep(0.1)
                continue
            
            # 处理画面
            processed = self.process_frame(frame)
            
            # 添加状态信息 (左上角小字)
            info_text = f"FPS:{current_fps} | Frame:{self.frame_count} | Shift:({self.current_shift_x},{self.current_shift_y}) | Hue:{self.current_hue}"
            cv2.putText(processed, info_text, (10, 20), 
                       cv2.FONT_HERSHEY_SIMPLEX, 0.4, (0, 255, 0), 1)
            
            # 显示
            cv2.imshow(window_name, processed)
            
            # FPS计算
            fps_count += 1
            if time.time() - fps_time >= 1.0:
                current_fps = fps_count
                fps_count = 0
                fps_time = time.time()
            
            # 按键处理
            key = cv2.waitKey(1) & 0xFF
            if key == ord('q') or key == ord('Q'):
                break
            elif key == ord('1'):
                self.set_strength("low")
                print("[防检测] 强度: 低")
            elif key == ord('2'):
                self.set_strength("medium")
                print("[防检测] 强度: 中")
            elif key == ord('3'):
                self.set_strength("high")
                print("[防检测] 强度: 高")
            elif key == ord('f') or key == ord('F'):
                fullscreen = not fullscreen
                if fullscreen:
                    cv2.setWindowProperty(window_name, cv2.WND_PROP_FULLSCREEN, cv2.WINDOW_FULLSCREEN)
                else:
                    cv2.setWindowProperty(window_name, cv2.WND_PROP_FULLSCREEN, cv2.WINDOW_NORMAL)
        
        self.running = False
        cv2.destroyAllWindows()
        print("防检测处理器已停止")
    
    def stop(self):
        """停止"""
        self.running = False


def main():
    """主函数"""
    print("正在启动屏幕防检测处理器...")
    
    # 检查依赖
    if not MSS_AVAILABLE:
        print("错误: 缺少依赖库")
        print("请运行: pip install mss opencv-python")
        input("按回车退出...")
        return
    
    # 创建处理器
    processor = ScreenAntiDetect(monitor_id=1)
    processor.set_strength("medium")
    
    # 运行
    try:
        processor.run()
    except KeyboardInterrupt:
        processor.stop()
    except Exception as e:
        print(f"错误: {e}")
        import traceback
        traceback.print_exc()
        input("按回车退出...")


if __name__ == "__main__":
    main()
