# -*- coding: utf-8 -*-
"""
OBS 防检测滤镜脚本

使用方法:
1. 安装 obs-websocket 插件
2. 安装 pyvirtualcam: pip install pyvirtualcam
3. 运行此脚本，会创建一个虚拟摄像头
4. 在OBS中选择这个虚拟摄像头作为来源
5. 虚拟摄像头会自动对画面进行防检测处理

或者使用 OBS Python 脚本功能直接加载
"""

import cv2
import numpy as np
import time
import threading
from typing import Optional

try:
    import pyvirtualcam
    PYVIRTUALCAM_AVAILABLE = True
except ImportError:
    PYVIRTUALCAM_AVAILABLE = False
    print("警告: pyvirtualcam 未安装，请运行: pip install pyvirtualcam")

from video_anti_detection import VideoAntiDetection


class OBSAntiDetectionFilter:
    """OBS防检测滤镜"""
    
    def __init__(self, width: int = 1280, height: int = 720, fps: int = 30):
        self.width = width
        self.height = height
        self.fps = fps
        self.processor = VideoAntiDetection()
        self.running = False
        self.thread: Optional[threading.Thread] = None
        
        # 输入源（可以是摄像头、屏幕捕获等）
        self.capture: Optional[cv2.VideoCapture] = None
        self.input_source = "camera"  # "camera", "screen", "video"
    
    def set_input_camera(self, camera_id: int = 0):
        """设置摄像头输入"""
        self.input_source = "camera"
        self.capture = cv2.VideoCapture(camera_id)
        self.capture.set(cv2.CAP_PROP_FRAME_WIDTH, self.width)
        self.capture.set(cv2.CAP_PROP_FRAME_HEIGHT, self.height)
        self.capture.set(cv2.CAP_PROP_FPS, self.fps)
    
    def set_input_video(self, video_path: str):
        """设置视频文件输入"""
        self.input_source = "video"
        self.capture = cv2.VideoCapture(video_path)
    
    def set_input_screen(self, monitor: int = 0):
        """设置屏幕捕获输入"""
        self.input_source = "screen"
        try:
            import mss
            self.mss = mss.mss()
            self.monitor = self.mss.monitors[monitor + 1]  # 0 是全部屏幕
        except ImportError:
            print("警告: mss 未安装，请运行: pip install mss")
    
    def get_frame(self) -> Optional[np.ndarray]:
        """获取一帧"""
        if self.input_source == "screen" and hasattr(self, 'mss'):
            screenshot = self.mss.grab(self.monitor)
            frame = np.array(screenshot)
            frame = cv2.cvtColor(frame, cv2.COLOR_BGRA2BGR)
            frame = cv2.resize(frame, (self.width, self.height))
            return frame
        elif self.capture and self.capture.isOpened():
            ret, frame = self.capture.read()
            if ret:
                frame = cv2.resize(frame, (self.width, self.height))
                return frame
            elif self.input_source == "video":
                # 视频循环播放
                self.capture.set(cv2.CAP_PROP_POS_FRAMES, 0)
                ret, frame = self.capture.read()
                if ret:
                    frame = cv2.resize(frame, (self.width, self.height))
                    return frame
        return None
    
    def start_virtual_camera(self):
        """启动虚拟摄像头"""
        if not PYVIRTUALCAM_AVAILABLE:
            print("错误: pyvirtualcam 未安装")
            return
        
        self.running = True
        self.thread = threading.Thread(target=self._run_virtual_camera)
        self.thread.daemon = True
        self.thread.start()
        print(f"虚拟摄像头已启动: {self.width}x{self.height}@{self.fps}fps")
    
    def _run_virtual_camera(self):
        """虚拟摄像头主循环"""
        with pyvirtualcam.Camera(width=self.width, height=self.height, fps=self.fps) as cam:
            print(f"虚拟摄像头名称: {cam.device}")
            
            while self.running:
                frame = self.get_frame()
                if frame is None:
                    # 无输入时显示黑屏
                    frame = np.zeros((self.height, self.width, 3), dtype=np.uint8)
                
                # 应用防检测处理
                processed = self.processor.process_frame(frame)
                
                # 转换为RGB（pyvirtualcam需要RGB格式）
                rgb_frame = cv2.cvtColor(processed, cv2.COLOR_BGR2RGB)
                
                # 发送到虚拟摄像头
                cam.send(rgb_frame)
                cam.sleep_until_next_frame()
    
    def stop(self):
        """停止"""
        self.running = False
        if self.thread:
            self.thread.join(timeout=2)
        if self.capture:
            self.capture.release()
        print("虚拟摄像头已停止")
    
    def set_strength(self, level: str):
        """设置防检测强度"""
        self.processor.set_strength(level)
        print(f"防检测强度已设置为: {level}")


class RealtimePreview:
    """实时预览窗口"""
    
    def __init__(self):
        self.processor = VideoAntiDetection()
        self.running = False
    
    def start_preview(self, source: str = "camera", camera_id: int = 0, video_path: str = ""):
        """启动预览
        
        Args:
            source: "camera" 或 "video"
            camera_id: 摄像头ID
            video_path: 视频文件路径
        """
        if source == "camera":
            cap = cv2.VideoCapture(camera_id)
        else:
            cap = cv2.VideoCapture(video_path)
        
        if not cap.isOpened():
            print("无法打开视频源")
            return
        
        self.running = True
        print("按 Q 退出预览")
        print("按 1/2/3 切换强度 (低/中/高)")
        
        while self.running:
            ret, frame = cap.read()
            if not ret:
                if source == "video":
                    cap.set(cv2.CAP_PROP_POS_FRAMES, 0)
                    continue
                break
            
            # 处理帧
            processed = self.processor.process_frame(frame)
            
            # 并排显示
            h, w = frame.shape[:2]
            combined = np.hstack([
                cv2.resize(frame, (w//2, h//2)),
                cv2.resize(processed, (w//2, h//2))
            ])
            
            # 添加文字说明
            cv2.putText(combined, "Original", (10, 30), cv2.FONT_HERSHEY_SIMPLEX, 1, (0, 255, 0), 2)
            cv2.putText(combined, "Anti-Detection", (w//2 + 10, 30), cv2.FONT_HERSHEY_SIMPLEX, 1, (0, 255, 0), 2)
            
            cv2.imshow("Anti-Detection Preview", combined)
            
            key = cv2.waitKey(1) & 0xFF
            if key == ord('q'):
                break
            elif key == ord('1'):
                self.processor.set_strength("low")
                print("强度: 低")
            elif key == ord('2'):
                self.processor.set_strength("medium")
                print("强度: 中")
            elif key == ord('3'):
                self.processor.set_strength("high")
                print("强度: 高")
        
        cap.release()
        cv2.destroyAllWindows()
        self.running = False


if __name__ == "__main__":
    print("=" * 50)
    print("OBS 防检测滤镜")
    print("=" * 50)
    print("\n选择模式:")
    print("1. 预览模式（查看效果）")
    print("2. 虚拟摄像头模式（用于OBS）")
    
    choice = input("\n请选择 (1/2): ").strip()
    
    if choice == "1":
        preview = RealtimePreview()
        print("\n选择输入源:")
        print("1. 摄像头")
        print("2. 视频文件")
        src = input("请选择 (1/2): ").strip()
        
        if src == "1":
            preview.start_preview(source="camera", camera_id=0)
        else:
            path = input("请输入视频文件路径: ").strip()
            preview.start_preview(source="video", video_path=path)
    
    elif choice == "2":
        if not PYVIRTUALCAM_AVAILABLE:
            print("请先安装 pyvirtualcam: pip install pyvirtualcam")
        else:
            filter_obj = OBSAntiDetectionFilter(width=1280, height=720, fps=30)
            
            print("\n选择输入源:")
            print("1. 摄像头")
            print("2. 视频文件")
            print("3. 屏幕捕获")
            src = input("请选择 (1/2/3): ").strip()
            
            if src == "1":
                filter_obj.set_input_camera(0)
            elif src == "2":
                path = input("请输入视频文件路径: ").strip()
                filter_obj.set_input_video(path)
            else:
                filter_obj.set_input_screen(0)
            
            print("\n选择防检测强度:")
            print("1. 低（几乎不可见）")
            print("2. 中（推荐）")
            print("3. 高（明显变化）")
            level = input("请选择 (1/2/3): ").strip()
            levels = {"1": "low", "2": "medium", "3": "high"}
            filter_obj.set_strength(levels.get(level, "medium"))
            
            filter_obj.start_virtual_camera()
            
            print("\n虚拟摄像头运行中...")
            print("在OBS中选择虚拟摄像头作为视频源")
            print("按 Enter 停止...")
            input()
            
            filter_obj.stop()
