# -*- coding: utf-8 -*-
"""
人脸贴纸滤镜处理器
基于OpenCV和Haar Cascade实现的实时人脸贴纸功能

功能:
1. 人脸检测 - 使用Haar Cascade
2. 贴纸叠加 - 支持帽子、眼镜、胡子等贴纸
3. 自动缩放 - 根据人脸大小自动调整贴纸大小
"""

import cv2
import numpy as np
import os
from typing import Optional, Dict, List, Tuple

try:
    from .face_landmarker import FaceLandmarker
    MEDIAPIPE_AVAILABLE = True
except ImportError:
    try:
        from src.utils.face_landmarker import FaceLandmarker
        MEDIAPIPE_AVAILABLE = True
    except ImportError:
        MEDIAPIPE_AVAILABLE = False
        print("Warning: FaceLandmarker not available")


class FaceStickerFilter:
    """人脸贴纸滤镜处理器"""
    
    def __init__(self, assets_dir: str = None):
        # 贴纸资源目录
        if assets_dir is None:
            # 默认为当前文件所在目录下的 assets/stickers
            current_dir = os.path.dirname(os.path.abspath(__file__))
            self.assets_dir = os.path.join(current_dir, "assets", "stickers")
        else:
            self.assets_dir = assets_dir
            
        # 确保目录存在
        if not os.path.exists(self.assets_dir):
            try:
                os.makedirs(self.assets_dir)
            except:
                pass
                
        # 初始化人脸检测 (MediaPipe)
        self.landmarker = None
        if MEDIAPIPE_AVAILABLE:
            self.landmarker = FaceLandmarker()
        
        # 贴纸缓存
        self.stickers = {}
        self.current_sticker = None
        self.enabled = True
        
        # 预加载贴纸（如果有）
        self._load_stickers()
            
    def _load_stickers(self):
        """加载贴纸资源"""
        if not os.path.exists(self.assets_dir):
            return
            
        for filename in os.listdir(self.assets_dir):
            if filename.lower().endswith(('.png', '.jpg', '.jpeg')):
                name = os.path.splitext(filename)[0]
                path = os.path.join(self.assets_dir, filename)
                # 读取图片（保留Alpha通道）
                img = cv2.imread(path, cv2.IMREAD_UNCHANGED)
                if img is not None:
                    self.stickers[name] = img
                    
    def set_sticker(self, sticker_name: str):
        """设置当前贴纸"""
        if sticker_name in self.stickers:
            self.current_sticker = sticker_name
        elif sticker_name is None:
            self.current_sticker = None
            
    def process(self, frame: np.ndarray) -> np.ndarray:
        """处理一帧图像"""
        if not self.enabled or self.current_sticker is None or self.landmarker is None:
            return frame
            
        # 使用 MediaPipe 检测人脸
        # 注意：这里假设输入 frame 已经是 RGB 格式（由 CameraWindow 保证）
        landmarks_data = self.landmarker.process(frame)
        
        if not landmarks_data:
            return frame
            
        result = frame.copy()
        sticker_img = self.stickers[self.current_sticker]
        
        # 获取边界框
        x, y, w, h = landmarks_data.bbox
        
        # 判断贴纸类型并应用
        if "hat" in self.current_sticker.lower():
            self._apply_hat(result, sticker_img, x, y, w, h)
        elif "glass" in self.current_sticker.lower():
            self._apply_glasses(result, sticker_img, x, y, w, h)
        elif "mustache" in self.current_sticker.lower():
            self._apply_mustache(result, sticker_img, x, y, w, h)
        else:
            # 默认放在头顶
            self._apply_hat(result, sticker_img, x, y, w, h)
                
        return result
        
    def _overlay_image(self, bg_img, fg_img, x, y):
        """叠加带有Alpha通道的图片"""
        h_bg, w_bg = bg_img.shape[:2]
        h_fg, w_fg = fg_img.shape[:2]
        
        if x >= w_bg or y >= h_bg:
            return
            
        # 计算重叠区域
        x1, y1 = max(x, 0), max(y, 0)
        x2, y2 = min(x + w_fg, w_bg), min(y + h_fg, h_bg)
        
        # 对应的fg区域
        fg_x1 = max(0, -x)
        fg_y1 = max(0, -y)
        fg_x2 = fg_x1 + (x2 - x1)
        fg_y2 = fg_y1 + (y2 - y1)
        
        if fg_x2 <= fg_x1 or fg_y2 <= fg_y1:
            return
            
        # 获取alpha通道
        if fg_img.shape[2] == 4:
            alpha = fg_img[fg_y1:fg_y2, fg_x1:fg_x2, 3] / 255.0
            color = fg_img[fg_y1:fg_y2, fg_x1:fg_x2, :3]
        else:
            alpha = np.ones((h_fg, w_fg))
            color = fg_img
            
        # 叠加
        for c in range(3):
            bg_img[y1:y2, x1:x2, c] = (alpha * color[:, :, c] + 
                                     (1 - alpha) * bg_img[y1:y2, x1:x2, c])
                                     
    def _apply_hat(self, frame, sticker, x, y, w, h):
        """应用帽子贴纸"""
        # 调整大小：宽度略大于人脸
        scale = 1.2
        new_w = int(w * scale)
        ratio = new_w / sticker.shape[1]
        new_h = int(sticker.shape[0] * ratio)
        
        resized_sticker = cv2.resize(sticker, (new_w, new_h))
        
        # 位置：在人脸上方，稍微重叠一点
        pos_x = x - int((new_w - w) / 2)
        pos_y = y - int(new_h * 0.8)
        
        self._overlay_image(frame, resized_sticker, pos_x, pos_y)
        
    def _apply_glasses(self, frame, sticker, x, y, w, h):
        """应用眼镜贴纸"""
        scale = 1.0
        new_w = int(w * scale)
        ratio = new_w / sticker.shape[1]
        new_h = int(sticker.shape[0] * ratio)
        
        resized_sticker = cv2.resize(sticker, (new_w, new_h))
        
        pos_x = x
        pos_y = y + int(h * 0.35)  # 眼睛位置大概在1/3处
        
        self._overlay_image(frame, resized_sticker, pos_x, pos_y)
        
    def _apply_mustache(self, frame, sticker, x, y, w, h):
        """应用胡子贴纸"""
        scale = 0.5
        new_w = int(w * scale)
        ratio = new_w / sticker.shape[1]
        new_h = int(sticker.shape[0] * ratio)
        
        resized_sticker = cv2.resize(sticker, (new_w, new_h))
        
        pos_x = x + int((w - new_w) / 2)
        pos_y = y + int(h * 0.65)  # 嘴巴上方
        
        self._overlay_image(frame, resized_sticker, pos_x, pos_y)
        
    def toggle(self) -> bool:
        self.enabled = not self.enabled
        return self.enabled

# 测试代码
if __name__ == "__main__":
    import sys
    
    # 创建滤镜
    sticker_filter = FaceStickerFilter()
    
    # 模拟创建一个贴纸（红色的帽子）
    # 在实际使用中应该提供真实的PNG文件
    dummy_hat = np.zeros((100, 150, 4), dtype=np.uint8)
    # 绘制一个三角形作为帽子
    pts = np.array([[75, 0], [0, 100], [150, 100]], np.int32)
    cv2.fillPoly(dummy_hat, [pts], (0, 0, 255, 255)) # 红色，不透明
    
    sticker_filter.stickers['hat_test'] = dummy_hat
    sticker_filter.set_sticker('hat_test')
    
    cap = cv2.VideoCapture(0)
    print("贴纸滤镜测试")
    print("按 'q' 退出, 's' 切换贴纸状态")
    
    while True:
        ret, frame = cap.read()
        if not ret:
            break
            
        result = sticker_filter.process(frame)
        cv2.imshow('Sticker Filter', result)
        
        key = cv2.waitKey(1) & 0xFF
        if key == ord('q'):
            break
        elif key == ord('s'):
            sticker_filter.toggle()
            print(f"贴纸: {'开启' if sticker_filter.enabled else '关闭'}")
            
    cap.release()
    cv2.destroyAllWindows()
