# -*- coding: utf-8 -*-
"""
美颜滤镜处理器
基于OpenCV实现的实时美颜功能

功能:
1. 磨皮 - 双边滤波去除皮肤瑕疵
2. 美白 - 中间调提亮，提升肤色
3. 亮度调节 - 整体亮度调整
4. 饱和度调节 - 色彩饱和度调整
5. 锐化 - 提升清晰度
"""

import cv2
import numpy as np
from typing import Optional, Tuple


class BeautyFilter:
    """美颜滤镜处理器"""
    
    def __init__(self):
        # 美颜参数（0-100）
        self.smoothing = 50      # 磨皮程度
        self.whitening = 30      # 美白程度
        self.brightness = 0      # 亮度 (-50 ~ 50)
        self.saturation = 0      # 饱和度 (-50 ~ 50)
        self.sharpness = 0       # 锐化 (0 ~ 100)
        
        # 是否启用
        self.enabled = True
        
        # 皮肤检测掩码缓存
        self._skin_mask_cache = None
        self._last_frame_shape = None
        
        # 美白查找表缓存
        self._whitening_lut = None
        self._last_whitening_value = None
    
    def set_params(self, smoothing: int = None, whitening: int = None, 
                   brightness: int = None, saturation: int = None, 
                   sharpness: int = None):
        """设置美颜参数"""
        if smoothing is not None:
            self.smoothing = max(0, min(100, smoothing))
        if whitening is not None:
            self.whitening = max(0, min(100, whitening))
        if brightness is not None:
            self.brightness = max(-50, min(50, brightness))
        if saturation is not None:
            self.saturation = max(-50, min(50, saturation))
        if sharpness is not None:
            self.sharpness = max(0, min(100, sharpness))
    
    def process(self, frame: np.ndarray) -> np.ndarray:
        """处理一帧图像"""
        if not self.enabled or frame is None:
            return frame
        
        result = frame.copy()
        
        # 1. 磨皮
        if self.smoothing > 0:
            result = self._apply_smoothing(result)
        
        # 2. 美白
        if self.whitening > 0:
            result = self._apply_whitening(result)
        
        # 3. 亮度
        if self.brightness != 0:
            result = self._apply_brightness(result)
        
        # 4. 饱和度
        if self.saturation != 0:
            result = self._apply_saturation(result)
        
        # 5. 锐化
        if self.sharpness > 0:
            result = self._apply_sharpness(result)
        
        return result
    
    def _detect_skin(self, frame: np.ndarray) -> np.ndarray:
        """检测皮肤区域（使用YCrCb色彩空间）"""
        # 检查缓存
        if (self._skin_mask_cache is not None and 
            self._last_frame_shape == frame.shape):
            return self._skin_mask_cache
        
        # 转换到YCrCb色彩空间
        ycrcb = cv2.cvtColor(frame, cv2.COLOR_BGR2YCrCb)
        
        # 皮肤颜色范围
        lower = np.array([0, 133, 77], dtype=np.uint8)
        upper = np.array([255, 173, 127], dtype=np.uint8)
        
        # 创建皮肤掩码
        mask = cv2.inRange(ycrcb, lower, upper)
        
        # 形态学操作平滑掩码
        kernel = cv2.getStructuringElement(cv2.MORPH_ELLIPSE, (5, 5))
        mask = cv2.morphologyEx(mask, cv2.MORPH_CLOSE, kernel)
        mask = cv2.morphologyEx(mask, cv2.MORPH_OPEN, kernel)
        
        # 高斯模糊使边缘更平滑
        mask = cv2.GaussianBlur(mask, (5, 5), 0)
        
        # 缓存结果
        self._skin_mask_cache = mask
        self._last_frame_shape = frame.shape
        
        return mask
    
    def _apply_smoothing(self, frame: np.ndarray) -> np.ndarray:
        """磨皮 - 双边滤波"""
        # 根据磨皮程度计算滤波参数
        # smoothing: 0-100 -> d: 0-15, sigmaColor: 0-75, sigmaSpace: 0-75
        d = int(self.smoothing * 0.15) + 1
        sigma = int(self.smoothing * 0.75) + 10
        
        # 检测皮肤区域
        skin_mask = self._detect_skin(frame)
        
        # 对整个图像应用双边滤波
        smoothed = cv2.bilateralFilter(frame, d, sigma, sigma)
        
        # 使用高保真磨皮算法（保留边缘细节）
        # 计算细节层
        diff = cv2.subtract(smoothed, frame)
        diff = cv2.add(diff, 128)
        
        # 高斯模糊细节层
        blur_size = max(3, (self.smoothing // 20) * 2 + 1)
        diff_blur = cv2.GaussianBlur(diff, (blur_size, blur_size), 0)
        
        # 融合
        result = cv2.add(frame, cv2.subtract(diff_blur, 128) * 2)
        result = np.clip(result, 0, 255).astype(np.uint8)
        
        # 只对皮肤区域应用效果
        skin_mask_3ch = cv2.merge([skin_mask, skin_mask, skin_mask])
        skin_mask_norm = skin_mask_3ch.astype(np.float32) / 255.0
        
        result = (result * skin_mask_norm + frame * (1 - skin_mask_norm)).astype(np.uint8)
        
        return result
    
    def _apply_whitening(self, frame: np.ndarray) -> np.ndarray:
        """美白 - 中间调提亮"""
        # 检查缓存
        if self._whitening_lut is None or self._last_whitening_value != self.whitening:
            self._build_whitening_lut()
            self._last_whitening_value = self.whitening
        
        # 检测皮肤区域
        skin_mask = self._detect_skin(frame)
        
        # 应用查找表
        whitened = cv2.LUT(frame, self._whitening_lut)
        
        # 只对皮肤区域应用效果
        skin_mask_3ch = cv2.merge([skin_mask, skin_mask, skin_mask])
        skin_mask_norm = skin_mask_3ch.astype(np.float32) / 255.0
        
        result = (whitened * skin_mask_norm + frame * (1 - skin_mask_norm)).astype(np.uint8)
        
        return result
    
    def _build_whitening_lut(self):
        """构建美白查找表"""
        # 中间调曲线函数
        value = self.whitening * 0.5  # 缩放到合适范围
        
        midtones_add = np.zeros(256)
        for i in range(256):
            midtones_add[i] = 0.667 * (1 - ((i - 127.0) / 127) ** 2)
        
        lut = np.zeros(256, dtype=np.uint8)
        for i in range(256):
            val = i + int(value * midtones_add[i])
            lut[i] = max(0, min(255, val))
        
        # 扩展为3通道
        self._whitening_lut = np.dstack([lut, lut, lut])
    
    def _apply_brightness(self, frame: np.ndarray) -> np.ndarray:
        """亮度调节"""
        # brightness: -50 ~ 50 -> -50 ~ 50
        beta = self.brightness
        result = cv2.convertScaleAbs(frame, alpha=1.0, beta=beta)
        return result
    
    def _apply_saturation(self, frame: np.ndarray) -> np.ndarray:
        """饱和度调节"""
        # saturation: -50 ~ 50 -> 0.5 ~ 1.5
        factor = 1.0 + self.saturation / 100.0
        
        # 转换到HSV
        hsv = cv2.cvtColor(frame, cv2.COLOR_BGR2HSV).astype(np.float32)
        hsv[:, :, 1] = np.clip(hsv[:, :, 1] * factor, 0, 255)
        hsv = hsv.astype(np.uint8)
        
        result = cv2.cvtColor(hsv, cv2.COLOR_HSV2BGR)
        return result
    
    def _apply_sharpness(self, frame: np.ndarray) -> np.ndarray:
        """锐化"""
        # sharpness: 0-100 -> sigma: 0-2, amount: 0-2
        if self.sharpness <= 0:
            return frame
        
        amount = self.sharpness / 50.0  # 0-2
        
        # Unsharp Mask锐化
        blurred = cv2.GaussianBlur(frame, (0, 0), 3)
        sharpened = cv2.addWeighted(frame, 1.0 + amount, blurred, -amount, 0)
        
        return sharpened
    
    def toggle(self) -> bool:
        """切换开关"""
        self.enabled = not self.enabled
        return self.enabled
    
    def reset(self):
        """重置参数"""
        self.smoothing = 50
        self.whitening = 30
        self.brightness = 0
        self.saturation = 0
        self.sharpness = 0


class QuickBeautyPresets:
    """快速美颜预设"""
    
    NATURAL = {
        'smoothing': 30,
        'whitening': 20,
        'brightness': 5,
        'saturation': 5,
        'sharpness': 10
    }
    
    SOFT = {
        'smoothing': 50,
        'whitening': 30,
        'brightness': 10,
        'saturation': 0,
        'sharpness': 5
    }
    
    BRIGHT = {
        'smoothing': 40,
        'whitening': 50,
        'brightness': 15,
        'saturation': 10,
        'sharpness': 15
    }
    
    DRAMATIC = {
        'smoothing': 70,
        'whitening': 60,
        'brightness': 20,
        'saturation': 15,
        'sharpness': 20
    }
    
    @classmethod
    def apply(cls, filter: BeautyFilter, preset_name: str):
        """应用预设"""
        presets = {
            'natural': cls.NATURAL,
            'soft': cls.SOFT,
            'bright': cls.BRIGHT,
            'dramatic': cls.DRAMATIC
        }
        
        if preset_name.lower() in presets:
            filter.set_params(**presets[preset_name.lower()])


# 测试代码
if __name__ == "__main__":
    # 创建美颜滤镜
    beauty = BeautyFilter()
    
    # 应用预设
    QuickBeautyPresets.apply(beauty, 'natural')
    
    # 打开摄像头测试
    cap = cv2.VideoCapture(0)
    
    print("美颜滤镜测试")
    print("按 'q' 退出, 'b' 切换美颜, '1-4' 切换预设")
    
    while True:
        ret, frame = cap.read()
        if not ret:
            break
        
        # 应用美颜
        result = beauty.process(frame)
        
        # 显示
        cv2.imshow('Beauty Filter', result)
        
        key = cv2.waitKey(1) & 0xFF
        if key == ord('q'):
            break
        elif key == ord('b'):
            beauty.toggle()
            print(f"美颜: {'开启' if beauty.enabled else '关闭'}")
        elif key == ord('1'):
            QuickBeautyPresets.apply(beauty, 'natural')
            print("预设: 自然")
        elif key == ord('2'):
            QuickBeautyPresets.apply(beauty, 'soft')
            print("预设: 柔和")
        elif key == ord('3'):
            QuickBeautyPresets.apply(beauty, 'bright')
            print("预设: 明亮")
        elif key == ord('4'):
            QuickBeautyPresets.apply(beauty, 'dramatic')
            print("预设: 戏剧")
    
    cap.release()
    cv2.destroyAllWindows()
