# -*- coding: utf-8 -*-
"""
推流服务整合
一键开播、自动获取推流码和直播间ID
"""

import os
import time
import json
import threading
import subprocess
from typing import Optional, Callable, Dict, Any, List
from dataclasses import dataclass, field
from enum import Enum
from pathlib import Path

from src.utils.logger import logger

from .live_companion import LiveCompanionManager, LiveCompanionInfo
from .packet_capture import PacketCaptureService, StreamInfo


class StreamServiceState(Enum):
    """服务状态"""
    IDLE = "idle"                       # 空闲
    STARTING = "starting"               # 启动中
    WAITING_COMPANION = "waiting_companion"  # 等待直播伴侣
    CAPTURING = "capturing"             # 抓包中
    STREAM_CAPTURED = "stream_captured" # 已捕获推流码
    CONNECTED = "connected"             # 已连接直播间
    ERROR = "error"                     # 错误


@dataclass
class StreamServiceConfig:
    """服务配置"""
    auto_start_companion: bool = True       # 自动启动直播伴侣
    auto_capture: bool = True               # 自动开始抓包
    auto_connect: bool = True               # 自动连接直播间
    capture_timeout: int = 300              # 抓包超时(秒)
    network_interface: Optional[str] = None # 指定网卡
    target_platform: str = "douyin"  # 目标平台
    companion_path: Optional[str] = None    # 直播伴侣路径
    obs_auto_config: bool = False           # 自动配置OBS
    obs_path: Optional[str] = None          # OBS路径


@dataclass 
class StreamServiceStatus:
    """服务状态信息"""
    state: StreamServiceState
    message: str = ""
    companion_info: Optional[LiveCompanionInfo] = None
    stream_info: Optional[StreamInfo] = None
    room_id: str = ""
    web_rid: str = ""
    error: str = ""
    progress: int = 0  # 0-100


class StreamService:
    """
    推流服务
    
    整合直播伴侣管理、网络抓包、自动连接等功能
    实现一键开播的完整流程
    """
    
    def __init__(self, config: Optional[StreamServiceConfig] = None):
        self.config = config or StreamServiceConfig()
        
        # 子服务
        self._companion = LiveCompanionManager()
        self._capture = PacketCaptureService()
        
        # 状态
        self._state = StreamServiceState.IDLE
        self._message = ""
        self._error = ""
        self._room_id = ""
        self._web_rid = ""
        self._stream_info: Optional[StreamInfo] = None
        
        # 线程和事件
        self._service_thread: Optional[threading.Thread] = None
        self._stop_event = threading.Event()
        self._lock = threading.Lock()
        
        # 回调
        self._callbacks: Dict[str, List[Callable]] = {
            'state_changed': [],
            'stream_captured': [],
            'room_id_captured': [],
            'error': [],
            'progress': [],
        }
        
        # 设置直播伴侣路径
        if self.config.companion_path:
            self._companion.exe_path = self.config.companion_path
    
    @property
    def state(self) -> StreamServiceState:
        return self._state
    
    @property
    def room_id(self) -> str:
        return self._room_id
    
    @property
    def web_rid(self) -> str:
        return self._web_rid
    
    @property
    def stream_info(self) -> Optional[StreamInfo]:
        return self._stream_info
    
    def on(self, event: str, callback: Callable):
        """注册事件回调"""
        if event in self._callbacks:
            self._callbacks[event].append(callback)
    
    def off(self, event: str, callback: Callable):
        """移除事件回调"""
        if event in self._callbacks and callback in self._callbacks[event]:
            self._callbacks[event].remove(callback)
    
    def _emit(self, event: str, *args, **kwargs):
        """触发事件"""
        for callback in self._callbacks.get(event, []):
            try:
                callback(*args, **kwargs)
            except Exception as e:
                logger.error(f"回调执行异常: {e}")
    
    def _set_state(self, state: StreamServiceState, message: str = ""):
        """设置状态"""
        with self._lock:
            self._state = state
            self._message = message
        
        logger.info(f"服务状态: {state.value} - {message}")
        self._emit('state_changed', self.get_status())
    
    def _set_error(self, error: str):
        """设置错误"""
        with self._lock:
            self._error = error
            self._state = StreamServiceState.ERROR
        
        logger.error(f"服务错误: {error}")
        self._emit('error', error)
    
    def get_status(self) -> StreamServiceStatus:
        """获取当前状态"""
        with self._lock:
            return StreamServiceStatus(
                state=self._state,
                message=self._message,
                companion_info=self._companion.get_info(),
                stream_info=self._stream_info,
                room_id=self._room_id,
                web_rid=self._web_rid,
                error=self._error,
                progress=self._calculate_progress()
            )
    
    def _calculate_progress(self) -> int:
        """计算进度"""
        progress_map = {
            StreamServiceState.IDLE: 0,
            StreamServiceState.STARTING: 10,
            StreamServiceState.WAITING_COMPANION: 30,
            StreamServiceState.CAPTURING: 50,
            StreamServiceState.STREAM_CAPTURED: 80,
            StreamServiceState.CONNECTED: 100,
            StreamServiceState.ERROR: 0,
        }
        return progress_map.get(self._state, 0)
    
    # ========== 网络接口 ==========
    
    @staticmethod
    def get_network_interfaces() -> List[str]:
        """获取可用网络接口"""
        capture = PacketCaptureService()
        return capture.get_all_interfaces()
    
    @staticmethod
    def is_npcap_installed() -> bool:
        """检查Npcap是否安装"""
        return PacketCaptureService.is_npcap_installed()
    
    # ========== 直播伴侣 ==========
    
    def get_companion_path(self) -> Optional[str]:
        """获取直播伴侣路径"""
        return self._companion.exe_path
    
    def set_companion_path(self, path: str):
        """设置直播伴侣路径"""
        self._companion.exe_path = path
        self.config.companion_path = path
    
    def is_companion_running(self) -> bool:
        """检查直播伴侣是否运行"""
        return self._companion.is_running()
    
    def start_companion(self) -> bool:
        """启动直播伴侣"""
        return self._companion.start()
    
    def stop_companion(self) -> bool:
        """停止直播伴侣"""
        return self._companion.stop()
    
    # ========== 一键开播 ==========
    
    def start_one_click(self) -> bool:
        """
        一键开播
        
        流程:
        1. 检查环境（Npcap等）
        2. 启动直播伴侣（如果配置）
        3. 开始网络抓包
        4. 等待捕获推流码
        5. 提取room_id
        6. 触发自动连接回调
        """
        if self._state not in [StreamServiceState.IDLE, StreamServiceState.ERROR]:
            logger.warning(f"当前状态不允许启动: {self._state}")
            return False
        
        self._stop_event.clear()
        self._error = ""
        self._room_id = ""
        self._web_rid = ""
        self._stream_info = None
        
        self._service_thread = threading.Thread(target=self._one_click_flow, daemon=True)
        self._service_thread.start()
        
        return True
    
    def stop(self):
        """停止服务"""
        self._stop_event.set()
        
        # 停止抓包
        self._capture.stop_capture()
        
        # 停止监控
        self._companion.stop_monitor()
        
        self._set_state(StreamServiceState.IDLE, "服务已停止")
    
    def _one_click_flow(self):
        """一键开播流程"""
        try:
            # Step 1: 检查环境
            self._set_state(StreamServiceState.STARTING, "检查环境...")
            self._emit('progress', 5)
            
            if not self._check_environment():
                return
            
            # Step 2: 启动直播伴侣
            if self.config.auto_start_companion:
                self._set_state(StreamServiceState.WAITING_COMPANION, "启动直播伴侣...")
                self._emit('progress', 20)
                
                if not self._companion.is_running():
                    if not self._companion.start():
                        self._set_error("启动直播伴侣失败")
                        return
                
                # 等待直播伴侣完全启动
                time.sleep(2)
            
            # Step 3: 开始抓包
            self._set_state(StreamServiceState.CAPTURING, "监听网络流量...")
            self._emit('progress', 40)
            
            if not self._start_capture():
                return
            
            # Step 4: 等待捕获
            self._wait_for_capture()
            
        except Exception as e:
            self._set_error(f"一键开播异常: {e}")
            logger.exception("一键开播异常")
    
    def _check_environment(self) -> bool:
        """检查运行环境"""
        # 检查Npcap
        if not self.is_npcap_installed():
            self._set_error("未检测到Npcap，请先安装Npcap驱动")
            return False
        
        # 检查直播伴侣路径
        if self.config.auto_start_companion:
            if not self._companion.exe_path:
                self._set_error("未找到直播伴侣安装路径，请手动设置")
                return False
        
        return True
    
    def _start_capture(self) -> bool:
        """启动抓包 - 监听所有网卡"""
        success = self._capture.start_capture_all_interfaces(
            on_stream_captured=self._on_stream_captured,
            on_log=lambda msg: logger.info(msg),
            on_error=self._on_capture_error
        )
        
        if not success:
            self._set_error("启动网络抓包失败")
            return False
        
        # 同时启动进程监控
        self._companion.start_monitor(
            on_streaming_start=self._on_streaming_start,
            on_streaming_stop=self._on_streaming_stop
        )
        
        return True
    
    def _wait_for_capture(self):
        """等待捕获推流码"""
        start_time = time.time()
        
        while not self._stop_event.is_set():
            # 检查超时
            elapsed = time.time() - start_time
            if elapsed > self.config.capture_timeout:
                self._set_error(f"捕获超时({self.config.capture_timeout}秒)")
                self._capture.stop_capture()
                return
            
            # 检查是否已捕获
            if self._stream_info and self._stream_info.is_valid:
                # 已捕获到推流码
                if self._room_id:
                    # 已获取room_id，流程完成
                    self._set_state(StreamServiceState.CONNECTED, "已获取直播间信息")
                    self._emit('progress', 100)
                    break
                else:
                    # 等待room_id
                    self._set_state(StreamServiceState.STREAM_CAPTURED, "已捕获推流码，等待直播间ID...")
                    self._emit('progress', 80)
            
            # 更新进度
            progress = min(70, 40 + int(elapsed / self.config.capture_timeout * 30))
            self._emit('progress', progress)
            
            time.sleep(0.5)
        
        # 停止抓包
        self._capture.stop_capture()
        self._companion.stop_monitor()
    
    def _on_stream_captured(self, stream_info: StreamInfo):
        """推流码捕获回调"""
        with self._lock:
            self._stream_info = stream_info
        
        logger.info(f"捕获到推流码: {stream_info.platform}")
        logger.info(f"  服务器: {stream_info.server_url}")
        logger.info(f"  密钥: {stream_info.stream_key[:50]}...")
        self._emit('stream_captured', stream_info)
    
    def _on_room_id_captured(self, room_id: str, web_rid: str):
        """room_id捕获回调"""
        with self._lock:
            if not self._room_id:  # 只记录第一次
                self._room_id = room_id
            if not self._web_rid and web_rid:
                self._web_rid = web_rid
        
        logger.info(f"捕获到直播间ID: room_id={room_id}, web_rid={web_rid}")
        self._emit('room_id_captured', room_id, web_rid)
    
    def _on_capture_error(self, error: str):
        """抓包错误回调"""
        logger.error(f"抓包错误: {error}")
        # 不直接设置错误状态，可能只是某些包解析失败
    
    def _on_streaming_start(self):
        """开始推流回调"""
        logger.info("检测到开始推流")
        self._set_state(StreamServiceState.CAPTURING, "检测到推流，正在捕获...")
    
    def _on_streaming_stop(self):
        """停止推流回调"""
        logger.info("检测到停止推流")
    
    # ========== OBS集成 ==========
    
    def configure_obs(self, stream_info: Optional[StreamInfo] = None) -> bool:
        """
        配置OBS推流
        
        Args:
            stream_info: 推流信息，None则使用最新捕获的
        """
        info = stream_info or self._stream_info
        if not info or not info.is_valid:
            logger.error("无有效的推流信息")
            return False
        
        try:
            # 查找OBS配置文件
            obs_config_path = self._find_obs_config()
            if not obs_config_path:
                logger.error("未找到OBS配置文件")
                return False
            
            # 读取配置
            with open(obs_config_path, 'r', encoding='utf-8') as f:
                config = json.load(f)
            
            # 更新推流配置
            config['settings'] = config.get('settings', {})
            config['settings']['server'] = info.server_url
            config['settings']['key'] = info.stream_key
            
            # 写回配置
            with open(obs_config_path, 'w', encoding='utf-8') as f:
                json.dump(config, f, indent=4, ensure_ascii=False)
            
            logger.info(f"OBS配置已更新: {obs_config_path}")
            return True
            
        except Exception as e:
            logger.error(f"配置OBS失败: {e}")
            return False
    
    def _find_obs_config(self) -> Optional[str]:
        """查找OBS配置文件"""
        # 常见OBS配置路径
        appdata = os.environ.get('APPDATA', '')
        obs_paths = [
            os.path.join(appdata, 'obs-studio', 'basic', 'profiles'),
        ]
        
        for obs_path in obs_paths:
            if os.path.exists(obs_path):
                # 查找service.json
                for root, dirs, files in os.walk(obs_path):
                    if 'service.json' in files:
                        return os.path.join(root, 'service.json')
        
        return None
    
    def start_obs(self) -> bool:
        """启动OBS"""
        obs_path = self.config.obs_path or self._find_obs_exe()
        if not obs_path:
            logger.error("未找到OBS安装路径")
            return False
        
        try:
            subprocess.Popen(obs_path, shell=False)
            logger.info("OBS已启动")
            return True
        except Exception as e:
            logger.error(f"启动OBS失败: {e}")
            return False
    
    def _find_obs_exe(self) -> Optional[str]:
        """查找OBS可执行文件"""
        common_paths = [
            r"C:\Program Files\obs-studio\bin\64bit\obs64.exe",
            r"C:\Program Files (x86)\obs-studio\bin\64bit\obs64.exe",
            r"D:\Program Files\obs-studio\bin\64bit\obs64.exe",
        ]
        
        for path in common_paths:
            if os.path.exists(path):
                return path
        
        return None
    
    # ========== 手动控制 ==========
    
    def manual_capture_start(self, interface: Optional[str] = None) -> bool:
        """手动开始抓包 - 监听所有网卡"""
        if self._capture.is_capturing:
            return False
        
        self._set_state(StreamServiceState.CAPTURING, "手动抓包中...")
        
        return self._capture.start_capture_all_interfaces(
            on_stream_captured=self._on_stream_captured,
            on_log=lambda msg: logger.info(msg),
            on_error=self._on_capture_error
        )
    
    def manual_capture_stop(self):
        """手动停止抓包"""
        self._capture.stop_capture()
        self._set_state(StreamServiceState.IDLE, "抓包已停止")
    
    def manual_set_room_id(self, room_id: str, web_rid: str = ""):
        """手动设置room_id"""
        with self._lock:
            self._room_id = room_id
            self._web_rid = web_rid or room_id
        
        self._emit('room_id_captured', room_id, web_rid or room_id)
    
    def get_captured_streams(self) -> List[StreamInfo]:
        """获取所有捕获的推流信息"""
        return self._capture.get_captured_streams()
    
    # ========== 配置保存/加载 ==========
    
    def save_config(self, path: str):
        """保存配置"""
        config_dict = {
            'auto_start_companion': self.config.auto_start_companion,
            'auto_capture': self.config.auto_capture,
            'auto_connect': self.config.auto_connect,
            'capture_timeout': self.config.capture_timeout,
            'network_interface': self.config.network_interface,
            'target_platform': self.config.target_platform,
            'companion_path': self.config.companion_path,
            'obs_auto_config': self.config.obs_auto_config,
            'obs_path': self.config.obs_path,
        }
        
        with open(path, 'w', encoding='utf-8') as f:
            json.dump(config_dict, f, indent=2, ensure_ascii=False)
    
    def load_config(self, path: str):
        """加载配置"""
        if not os.path.exists(path):
            return
        
        try:
            with open(path, 'r', encoding='utf-8') as f:
                config_dict = json.load(f)
            
            self.config.auto_start_companion = config_dict.get('auto_start_companion', True)
            self.config.auto_capture = config_dict.get('auto_capture', True)
            self.config.auto_connect = config_dict.get('auto_connect', True)
            self.config.capture_timeout = config_dict.get('capture_timeout', 300)
            self.config.network_interface = config_dict.get('network_interface')
            self.config.target_platform = config_dict.get('target_platform', 'douyin')
            self.config.companion_path = config_dict.get('companion_path')
            self.config.obs_auto_config = config_dict.get('obs_auto_config', False)
            self.config.obs_path = config_dict.get('obs_path')
            
            if self.config.companion_path:
                self._companion.exe_path = self.config.companion_path
                
        except Exception as e:
            logger.error(f"加载配置失败: {e}")
