# -*- coding: utf-8 -*-
"""
OBS管理器
实现自动配置OBS推流参数、启动/停止推流
支持 obs-websocket 5.x API (OBS 28+内置)
"""

import os
import json
import time
import subprocess
from pathlib import Path
from typing import Optional, Callable, Tuple
from dataclasses import dataclass
import configparser

# obs-websocket 5.x 支持
def _ensure_obsws_installed():
    """确保 obsws-python 库已安装"""
    try:
        import obsws_python
        return True
    except ImportError:
        print("[OBS] 正在安装 obsws-python 库...")
        try:
            import subprocess
            import sys
            subprocess.check_call([sys.executable, "-m", "pip", "install", "obsws-python", "-q"])
            print("[OBS] obsws-python 安装成功")
            return True
        except Exception as e:
            print(f"[OBS] obsws-python 安装失败: {e}")
            return False

# 尝试安装并导入
_ensure_obsws_installed()

try:
    import obsws_python as obsws
    HAS_OBSWS = True
except ImportError:
    HAS_OBSWS = False
    print("[OBS] 警告: obsws-python 未安装，WebSocket功能不可用")


@dataclass
class ObsConfig:
    """OBS配置信息"""
    profile_path: str
    profile_name: str
    service_file: str  # service.json
    basic_ini: str     # basic.ini


@dataclass
class ObsStreamStatus:
    """OBS推流状态"""
    is_streaming: bool = False
    is_connected: bool = False
    output_bytes: int = 0
    output_duration: float = 0.0
    error: str = ""


class ObsManager:
    """OBS管理器 - 自动配置和控制OBS推流
    
    支持两种模式：
    1. WebSocket模式 (推荐): 通过obs-websocket实时控制，无需重启OBS
    2. 文件模式 (兼容): 修改配置文件，需要重启OBS生效
    """
    
    DEFAULT_WS_HOST = "localhost"
    DEFAULT_WS_PORT = 4455
    DEFAULT_WS_PASSWORD = ""  # OBS 28+默认无密码
    
    def __init__(self, logger: Optional[Callable[[str], None]] = None,
                 ws_host: str = None, ws_port: int = None, ws_password: str = None):
        self._logger = logger or print
        self._config: Optional[ObsConfig] = None
        self._obs_path: Optional[str] = None
        self._ws_client: Optional[object] = None
        self._event_client: Optional[object] = None
        self._stream_status_callback: Optional[Callable[[str, dict], None]] = None
        self._is_monitoring = False
        
        # WebSocket配置
        self._ws_host = ws_host or self.DEFAULT_WS_HOST
        self._ws_port = ws_port or self.DEFAULT_WS_PORT
        self._ws_password = ws_password or self.DEFAULT_WS_PASSWORD
        
        self._find_obs_config()
        self._find_obs_exe()
    
    def _log(self, msg: str):
        """日志输出"""
        if self._logger:
            self._logger(msg)
    
    def _find_obs_config(self):
        """查找OBS配置文件路径"""
        try:
            # OBS配置通常在 %APPDATA%\obs-studio
            appdata = os.environ.get('APPDATA', '')
            obs_base = Path(appdata) / 'obs-studio'
            
            if not obs_base.exists():
                self._log(f"未找到OBS配置目录: {obs_base}")
                return
            
            # 读取global.ini获取当前profile
            global_ini = obs_base / 'global.ini'
            profile_name = 'Untitled'
            
            if global_ini.exists():
                try:
                    # 尝试用 configparser 解析
                    config = configparser.ConfigParser()
                    # 读取文件内容，移除可能的BOM
                    content = global_ini.read_text(encoding='utf-8-sig')
                    config.read_string(content)
                    if config.has_option('Basic', 'Profile'):
                        profile_name = config.get('Basic', 'Profile')
                except Exception as parse_err:
                    self._log(f"解析global.ini失败: {parse_err}，使用默认profile")
            
            # 构建profile路径
            profile_path = obs_base / 'basic' / 'profiles' / profile_name
            
            if not profile_path.exists():
                # 尝试使用第一个找到的profile
                profiles_dir = obs_base / 'basic' / 'profiles'
                if profiles_dir.exists():
                    profiles = [p for p in profiles_dir.iterdir() if p.is_dir()]
                    if profiles:
                        profile_path = profiles[0]
                        profile_name = profile_path.name
                        self._log(f"使用找到的第一个profile: {profile_name}")
            
            if profile_path.exists():
                self._config = ObsConfig(
                    profile_path=str(profile_path),
                    profile_name=profile_name,
                    service_file=str(profile_path / 'service.json'),
                    basic_ini=str(profile_path / 'basic.ini')
                )
                self._log(f"找到OBS配置: {profile_name}")
            else:
                # 即使没找到profile，也创建一个默认配置
                profiles_dir = obs_base / 'basic' / 'profiles'
                default_profile = profiles_dir / 'Untitled'
                default_profile.mkdir(parents=True, exist_ok=True)
                self._config = ObsConfig(
                    profile_path=str(default_profile),
                    profile_name='Untitled',
                    service_file=str(default_profile / 'service.json'),
                    basic_ini=str(default_profile / 'basic.ini')
                )
                self._log(f"创建默认OBS配置: Untitled")
                
        except Exception as e:
            self._log(f"查找OBS配置失败: {e}")
    
    def _find_obs_exe(self):
        """
        快速自动扫描OBS安装路径
        
        扫描顺序（按优先级）：
        1. Windows注册表 - 最可靠
        2. 常见安装路径
        3. Steam安装路径
        4. 环境变量PATH
        """
        # 方法1: 从注册表查找（最可靠）
        obs_path = self._find_obs_from_registry()
        if obs_path:
            self._obs_path = obs_path
            return
        
        # 方法2: 常见安装路径
        obs_path = self._find_obs_from_common_paths()
        if obs_path:
            self._obs_path = obs_path
            return
        
        # 方法3: Steam安装路径
        obs_path = self._find_obs_from_steam()
        if obs_path:
            self._obs_path = obs_path
            return
        
        # 方法4: 环境变量PATH
        obs_path = self._find_obs_from_path()
        if obs_path:
            self._obs_path = obs_path
    
    def _find_obs_from_registry(self) -> Optional[str]:
        """从Windows注册表查找OBS安装路径"""
        try:
            import winreg
            
            # 尝试多个注册表位置
            registry_keys = [
                # OBS Studio官方安装
                (winreg.HKEY_LOCAL_MACHINE, r"SOFTWARE\OBS Studio", ""),
                (winreg.HKEY_LOCAL_MACHINE, r"SOFTWARE\WOW6432Node\OBS Studio", ""),
                # 卸载信息
                (winreg.HKEY_LOCAL_MACHINE, r"SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\OBS Studio", "InstallLocation"),
                (winreg.HKEY_LOCAL_MACHINE, r"SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall\OBS Studio", "InstallLocation"),
                # 当前用户安装
                (winreg.HKEY_CURRENT_USER, r"SOFTWARE\OBS Studio", ""),
            ]
            
            for hkey, subkey, value_name in registry_keys:
                try:
                    key = winreg.OpenKey(hkey, subkey, 0, 
                                        winreg.KEY_READ | winreg.KEY_WOW64_64KEY)
                    install_path, _ = winreg.QueryValueEx(key, value_name)
                    winreg.CloseKey(key)
                    
                    if install_path:
                        # 检查64位和32位可执行文件
                        for exe in ['bin/64bit/obs64.exe', 'bin/32bit/obs32.exe']:
                            full_path = Path(install_path) / exe
                            if full_path.exists():
                                return str(full_path)
                except (FileNotFoundError, OSError):
                    continue
                    
        except Exception:
            pass
        
        return None
    
    def _find_obs_from_common_paths(self) -> Optional[str]:
        """从常见安装路径查找OBS"""
        # 获取系统盘符
        system_drive = os.environ.get('SystemDrive', 'C:')
        
        common_paths = [
            # 标准Program Files路径
            r"C:\Program Files\obs-studio",
            r"C:\Program Files (x86)\obs-studio",
            # 其他盘符
            r"D:\Program Files\obs-studio",
            r"D:\obs-studio",
            r"E:\Program Files\obs-studio",
            # 用户可能的自定义路径
            os.path.expanduser(r"~\obs-studio"),
            os.path.expanduser(r"~\AppData\Local\obs-studio"),
        ]
        
        # 如果系统盘不是C盘，添加系统盘路径
        if system_drive != 'C:':
            common_paths.insert(0, f"{system_drive}\\Program Files\\obs-studio")
        
        for base_path in common_paths:
            for exe in ['bin/64bit/obs64.exe', 'bin/32bit/obs32.exe']:
                full_path = Path(base_path) / exe
                if full_path.exists():
                    return str(full_path)
        
        return None
    
    def _find_obs_from_steam(self) -> Optional[str]:
        """从Steam安装路径查找OBS"""
        try:
            import winreg
            
            # 获取Steam安装路径
            key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, 
                                r"SOFTWARE\WOW6432Node\Valve\Steam")
            steam_path, _ = winreg.QueryValueEx(key, "InstallPath")
            winreg.CloseKey(key)
            
            # Steam版OBS路径
            obs_steam = Path(steam_path) / "steamapps" / "common" / "OBS Studio"
            for exe in ['bin/64bit/obs64.exe', 'bin/32bit/obs32.exe']:
                full_path = obs_steam / exe
                if full_path.exists():
                    return str(full_path)
                    
        except Exception:
            pass
        
        return None
    
    def _find_obs_from_path(self) -> Optional[str]:
        """从环境变量PATH查找OBS"""
        try:
            import shutil
            # 查找obs64或obs32
            for exe_name in ['obs64.exe', 'obs32.exe', 'obs.exe']:
                obs_path = shutil.which(exe_name)
                if obs_path:
                    return obs_path
        except Exception:
            pass
        
        return None
    
    @property
    def is_configured(self) -> bool:
        """OBS是否已配置"""
        return self._config is not None
    
    @property
    def is_installed(self) -> bool:
        """OBS是否已安装"""
        return self._obs_path is not None
    
    def is_running(self) -> bool:
        """检查OBS是否正在运行"""
        try:
            import psutil
            for proc in psutil.process_iter(['name']):
                name = proc.info['name'].lower()
                if name in ['obs64.exe', 'obs32.exe', 'obs.exe']:
                    return True
        except:
            pass
        return False
    
    def ensure_obs_websocket_enabled(self) -> bool:
        """确保OBS WebSocket服务已启用（无密码）"""
        try:
            appdata = os.environ.get('APPDATA', '')
            ws_config_dir = Path(appdata) / 'obs-studio' / 'plugin_config' / 'obs-websocket'
            ws_config_file = ws_config_dir / 'config.json'
            
            # 确保目录存在
            ws_config_dir.mkdir(parents=True, exist_ok=True)
            
            # WebSocket 配置：启用、无密码、端口4455
            ws_config = {
                "alerts_enabled": False,
                "auth_required": False,
                "first_load": False,
                "server_enabled": True,
                "server_password": "",
                "server_port": 4455
            }
            
            # 检查是否需要更新
            need_update = True
            if ws_config_file.exists():
                try:
                    with open(ws_config_file, 'r', encoding='utf-8') as f:
                        existing = json.load(f)
                    if existing.get('server_enabled') and not existing.get('auth_required'):
                        need_update = False
                        self._log("OBS WebSocket已启用")
                except:
                    pass
            
            if need_update:
                with open(ws_config_file, 'w', encoding='utf-8') as f:
                    json.dump(ws_config, f, indent=2, ensure_ascii=False)
                self._log("已启用OBS WebSocket (端口4455, 无密码)")
                return True  # 配置已更新，需要重启OBS
            
            return False  # 配置无需更新
            
        except Exception as e:
            self._log(f"配置WebSocket失败: {e}")
            return False
    
    def kill_obs(self) -> bool:
        """强制关闭OBS进程"""
        try:
            import psutil
            killed = False
            for proc in psutil.process_iter(['name', 'pid']):
                name = proc.info['name'].lower()
                if name in ['obs64.exe', 'obs32.exe', 'obs.exe']:
                    try:
                        proc.terminate()
                        proc.wait(timeout=5)
                        killed = True
                        self._log(f"已关闭OBS进程 (PID: {proc.info['pid']})")
                    except:
                        try:
                            proc.kill()
                            killed = True
                        except:
                            pass
            return killed
        except Exception as e:
            self._log(f"关闭OBS失败: {e}")
            return False
    
    def set_stream_settings(self, server_url: str, stream_key: str) -> bool:
        """
        设置OBS推流参数 - 写入所有profile确保生效
        
        Args:
            server_url: RTMP服务器地址
            stream_key: 推流密钥
            
        Returns:
            是否成功
        """
        try:
            appdata = os.environ.get('APPDATA', '')
            profiles_dir = Path(appdata) / 'obs-studio' / 'basic' / 'profiles'
            
            if not profiles_dir.exists():
                self._log("OBS配置目录不存在")
                return False
            
            service_data = {
                "settings": {
                    "server": server_url,
                    "key": stream_key,
                    "use_auth": False
                },
                "type": "rtmp_custom"
            }
            
            updated_count = 0
            # 遍历所有profile并更新
            for profile_path in profiles_dir.iterdir():
                if profile_path.is_dir():
                    service_file = profile_path / 'service.json'
                    try:
                        with open(service_file, 'w', encoding='utf-8') as f:
                            json.dump(service_data, f, indent=4, ensure_ascii=False)
                        self._log(f"已更新profile [{profile_path.name}] 的推流配置")
                        updated_count += 1
                    except Exception as e:
                        self._log(f"更新profile [{profile_path.name}] 失败: {e}")
            
            if updated_count > 0:
                self._log(f"服务器: {server_url}")
                self._log(f"密钥: {stream_key[:30]}...")
                return True
            else:
                self._log("未找到任何profile")
                return False
            
        except Exception as e:
            self._log(f"设置OBS推流参数失败: {e}")
            return False
    
    def full_auto_stream(self, server_url: str, stream_key: str, 
                         on_progress: Optional[Callable[[str], None]] = None) -> Tuple[bool, str]:
        """
        完全自动化推流 - 一键完成所有配置和启动
        
        流程:
        1. 检查OBS是否安装
        2. 确保WebSocket已启用
        3. 写入推流配置
        4. 如果OBS运行中且配置有变更，重启OBS
        5. 启动OBS（如果未运行）
        6. 等待OBS完全启动
        7. 连接WebSocket
        8. 设置推流参数（实时）
        9. 开始推流
        
        Returns:
            (success, message)
        """
        def progress(msg: str):
            self._log(msg)
            if on_progress:
                on_progress(msg)
        
        # 步骤1: 检查OBS安装
        if not self.is_installed:
            return False, "未检测到OBS安装，请先安装OBS"
        
        progress("检查OBS配置...")
        
        # 步骤2: 确保WebSocket已启用
        ws_config_changed = self.ensure_obs_websocket_enabled()
        
        # 步骤3: 写入推流配置到文件
        progress("写入推流配置...")
        self.set_stream_settings(server_url, stream_key)
        
        # 步骤4: 处理OBS运行状态
        obs_was_running = self.is_running()
        if obs_was_running:
            if ws_config_changed:
                progress("WebSocket配置已更新，重启OBS...")
                self.kill_obs()
                time.sleep(2)
                obs_was_running = False
            else:
                progress("OBS已在运行")
        
        # 步骤5: 启动OBS
        if not obs_was_running:
            progress("启动OBS...")
            self.start_obs()
            
            # 步骤6: 等待OBS完全启动
            for i in range(20):
                time.sleep(1)
                progress(f"等待OBS启动... {i+1}s")
                if self.is_running():
                    break
            
            if not self.is_running():
                return False, "OBS启动超时"
            
            # 额外等待OBS完全初始化
            progress("等待OBS初始化...")
            time.sleep(3)
        
        # 步骤7: 连接WebSocket
        progress("连接OBS WebSocket...")
        ws_connected = False
        for retry in range(10):
            if self.connect_websocket():
                ws_connected = True
                break
            progress(f"连接WebSocket... 重试 {retry+1}/10")
            time.sleep(1)
        
        if not ws_connected:
            return False, "无法连接OBS WebSocket，请确保OBS已完全启动"
        
        # 步骤8: 通过WebSocket设置推流参数（实时生效）
        progress("配置推流参数...")
        try:
            self._ws_client.set_stream_service_settings(
                ss_type="rtmp_custom",
                ss_settings={
                    "server": server_url,
                    "key": stream_key
                }
            )
        except Exception as e:
            self._log(f"WebSocket设置参数失败: {e}")
        
        # 步骤9: 开始推流
        progress("开始推流...")
        try:
            self._ws_client.start_stream()
            time.sleep(2)
            
            # 验证推流状态
            try:
                resp = self._ws_client.get_stream_status()
                if resp.output_active:
                    progress("推流已启动！")
                    return True, "OBS推流已成功启动"
            except:
                pass
            
            return True, "推流指令已发送，请检查OBS确认状态"
            
        except Exception as e:
            return False, f"启动推流失败: {e}"
        finally:
            self.disconnect_websocket()
    
    def start_obs(self) -> bool:
        """启动OBS"""
        if not self._obs_path:
            self._log("未找到OBS安装路径")
            return False
        
        if self.is_running():
            self._log("OBS已在运行")
            return True
        
        try:
            # 设置环境变量避免 locale 错误
            env = os.environ.copy()
            env['LC_ALL'] = 'en-US'
            env['LANG'] = 'en-US'
            # 清除可能导致问题的环境变量
            env.pop('LC_CTYPE', None)
            env.pop('LC_MESSAGES', None)
            
            obs_dir = os.path.dirname(self._obs_path)
            
            # 使用 --disable-updater 避免启动检查
            subprocess.Popen(
                [self._obs_path, '--disable-updater'],
                creationflags=subprocess.DETACHED_PROCESS | subprocess.CREATE_NO_WINDOW,
                env=env,
                cwd=obs_dir,
                stdin=subprocess.DEVNULL,
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL
            )
            self._log("OBS已启动")
            return True
        except Exception as e:
            self._log(f"启动OBS失败: {e}")
            return False
    
    def start_streaming(self) -> bool:
        """
        开始OBS推流
        优先使用WebSocket API，失败则提示手动操作
        """
        if not self.is_running():
            self._log("OBS未运行，无法开始推流")
            return False
        
        # 优先尝试WebSocket控制
        if self.connect_websocket():
            try:
                self._ws_client.start_stream()
                self._log("已通过WebSocket开始推流")
                return True
            except Exception as e:
                self._log(f"WebSocket启动推流失败: {e}")
            finally:
                self.disconnect_websocket()
        
        self._log("请在OBS中手动点击开始推流")
        return False
    
    def stop_streaming(self) -> bool:
        """停止OBS推流"""
        if not self.is_running():
            return True
        
        if self.connect_websocket():
            try:
                self._ws_client.stop_stream()
                self._log("已通过WebSocket停止推流")
                return True
            except Exception as e:
                self._log(f"WebSocket停止推流失败: {e}")
            finally:
                self.disconnect_websocket()
        
        return False
    
    # ========== WebSocket API 方法 (OBS 28+) ==========
    
    def connect_websocket(self) -> bool:
        """连接OBS WebSocket服务"""
        if not HAS_OBSWS:
            self._log("未安装obsws-python库，请运行: pip install obsws-python")
            return False
        
        if not self.is_running():
            self._log("OBS未运行，无法连接WebSocket")
            return False
        
        try:
            self._ws_client = obsws.ReqClient(
                host=self._ws_host,
                port=self._ws_port,
                password=self._ws_password,
                timeout=5
            )
            self._log("已连接OBS WebSocket")
            return True
        except Exception as e:
            self._log(f"连接OBS WebSocket失败: {e}")
            self._log("请确保OBS已启用WebSocket服务 (工具 -> WebSocket服务器设置)")
            self._ws_client = None
            return False
    
    def disconnect_websocket(self):
        """断开WebSocket连接"""
        if self._ws_client:
            try:
                self._ws_client.base_client.ws.close()
            except:
                pass
            self._ws_client = None
    
    def set_stream_settings_ws(self, server_url: str, stream_key: str, keep_connected: bool = False) -> bool:
        """
        通过WebSocket API设置推流参数 (运行时生效，无需重启OBS)
        
        Args:
            server_url: RTMP服务器地址
            stream_key: 推流密钥
            keep_connected: 是否保持连接（用于后续操作如开始推流）
        """
        if not self._ws_client and not self.connect_websocket():
            return False
        
        try:
            # 设置推流服务类型为自定义RTMP
            self._ws_client.set_stream_service_settings(
                ss_type="rtmp_custom",
                ss_settings={
                    "server": server_url,
                    "key": stream_key
                }
            )
            self._log("已通过WebSocket设置推流参数 (立即生效)")
            self._log(f"  服务器: {server_url}")
            self._log(f"  密钥: {stream_key[:30]}...")
            return True
            
        except Exception as e:
            self._log(f"WebSocket设置推流参数失败: {e}")
            return False
        finally:
            if not keep_connected:
                self.disconnect_websocket()
    
    def get_stream_status(self) -> ObsStreamStatus:
        """获取当前推流状态"""
        status = ObsStreamStatus()
        
        if not self.connect_websocket():
            status.error = "无法连接WebSocket"
            return status
        
        try:
            resp = self._ws_client.get_stream_status()
            status.is_streaming = resp.output_active
            status.is_connected = resp.output_reconnecting == False if hasattr(resp, 'output_reconnecting') else True
            status.output_bytes = getattr(resp, 'output_bytes', 0)
            status.output_duration = getattr(resp, 'output_duration', 0.0)
            return status
            
        except Exception as e:
            status.error = str(e)
            return status
        finally:
            self.disconnect_websocket()
    
    def configure_and_start_streaming(self, server_url: str, stream_key: str, 
                                       wait_for_streaming: bool = True,
                                       timeout: int = 15,
                                       on_progress: Callable[[str], None] = None) -> Tuple[bool, str]:
        """
        一键配置并启动推流 (完整自动化流程)
        
        流程节点:
        1. 验证参数
        2. 检查OBS安装状态
        3. 检查/启动OBS
        4. 等待OBS完全启动
        5. 先写入配置文件(保底)
        6. 通过WebSocket设置参数
        7. 通过WebSocket启动推流
        8. 验证推流成功
        
        Args:
            server_url: RTMP服务器地址
            stream_key: 推流密钥
            wait_for_streaming: 是否等待确认推流成功
            timeout: 等待超时时间(秒)
            on_progress: 进度回调函数
            
        Returns:
            (成功标志, 状态消息)
        """
        def progress(msg: str):
            self._log(msg)
            if on_progress:
                on_progress(msg)
        
        # ===== 节点1: 验证参数 =====
        if not server_url or not stream_key:
            return False, "推流参数不能为空"
        
        if not server_url.startswith(('rtmp://', 'rtmps://')):
            return False, f"服务器地址格式错误，应以rtmp://开头"
        
        # ===== 节点2: 检查OBS安装 =====
        if not self.is_installed:
            return False, "OBS_NOT_INSTALLED"
        
        # ===== 节点3: 先写入配置文件(保底，确保OBS启动时能加载正确配置) =====
        progress("写入推流配置文件...")
        self.set_stream_settings(server_url, stream_key)
        
        # ===== 节点4: 检查/启动OBS =====
        obs_was_running = self.is_running()
        
        if not obs_was_running:
            progress("正在启动OBS...")
            if not self.start_obs():
                return False, "启动OBS失败，请手动打开OBS"
            
            # ===== 节点5: 等待OBS完全启动 =====
            progress("等待OBS启动...")
            if not self._wait_for_obs_ready(timeout=10):
                return False, "等待OBS启动超时"
        
        # ===== 节点6: 通过WebSocket设置参数 =====
        progress("配置推流参数...")
        ws_connected = False
        
        # 尝试连接WebSocket (最多重试3次)
        for attempt in range(3):
            if self.connect_websocket():
                ws_connected = True
                break
            time.sleep(1)
        
        if ws_connected:
            try:
                # 设置推流服务参数
                self._ws_client.set_stream_service_settings(
                    stream_service_type="rtmp_custom",
                    stream_service_settings={
                        "server": server_url,
                        "key": stream_key
                    }
                )
                progress("推流参数已配置")
                
                # ===== 节点7: 启动推流 =====
                progress("正在启动推流...")
                self._ws_client.start_stream()
                progress("推流指令已发送")
                
            except Exception as e:
                self._log(f"WebSocket操作失败: {e}")
                self.disconnect_websocket()
                # 如果OBS是我们启动的，尝试用--startstreaming重启
                if not obs_was_running:
                    progress("尝试重启OBS并自动推流...")
                    self._kill_obs()
                    time.sleep(1)
                    self.start_obs_with_streaming()
                    time.sleep(3)
                else:
                    return False, f"WebSocket控制失败: {e}"
            finally:
                self.disconnect_websocket()
        else:
            # WebSocket连接失败，如果OBS是新启动的，用--startstreaming重启
            progress("WebSocket连接失败，尝试备用方案...")
            if not obs_was_running:
                self._kill_obs()
                time.sleep(1)
                self.start_obs_with_streaming()
                time.sleep(3)
            else:
                return False, "无法连接OBS WebSocket，请在OBS中启用: 工具 → WebSocket服务器设置"
        
        # ===== 节点8: 验证推流成功 =====
        if wait_for_streaming:
            progress("验证推流状态...")
            return self._wait_for_streaming_active(timeout)
        
        return True, "推流指令已发送"
    
    def _wait_for_obs_ready(self, timeout: int = 10) -> bool:
        """等待OBS进程启动完成"""
        start_time = time.time()
        while time.time() - start_time < timeout:
            if self.is_running():
                # OBS进程存在，再等待一下让WebSocket服务就绪
                time.sleep(2)
                return True
            time.sleep(0.5)
        return False
    
    def _kill_obs(self):
        """关闭OBS进程"""
        try:
            import psutil
            for proc in psutil.process_iter(['name']):
                name = proc.info['name'].lower()
                if name in ['obs64.exe', 'obs32.exe', 'obs.exe']:
                    proc.terminate()
        except:
            pass
    
    def _wait_for_streaming_active(self, timeout: int = 10) -> Tuple[bool, str]:
        """等待确认推流已成功启动"""
        start_time = time.time()
        
        while time.time() - start_time < timeout:
            status = self.get_stream_status()
            
            if status.error:
                time.sleep(1)
                continue
            
            if status.is_streaming:
                self._log(f"推流已成功启动")
                return True, "推流已成功启动"
            
            time.sleep(0.5)
        
        return False, f"等待推流启动超时({timeout}秒)"
    
    def start_obs_with_streaming(self) -> bool:
        """启动OBS并自动开始推流"""
        if not self._obs_path:
            self._log("未找到OBS安装路径")
            return False
        
        try:
            # 设置环境变量避免 locale 错误
            env = os.environ.copy()
            env['LC_ALL'] = 'en-US'
            env['LANG'] = 'en-US'
            env.pop('LC_CTYPE', None)
            env.pop('LC_MESSAGES', None)
            
            obs_dir = os.path.dirname(self._obs_path)
            
            # 使用--startstreaming参数启动OBS会自动开始推流
            subprocess.Popen(
                [self._obs_path, '--startstreaming', '--disable-updater'],
                creationflags=subprocess.DETACHED_PROCESS | subprocess.CREATE_NO_WINDOW,
                env=env,
                cwd=obs_dir,
                stdin=subprocess.DEVNULL,
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL
            )
            self._log("OBS已启动并将自动开始推流")
            return True
        except Exception as e:
            self._log(f"启动OBS推流失败: {e}")
            return False
    
    @staticmethod
    def kill_live_companion() -> bool:
        """关闭直播伴侣进程"""
        try:
            import psutil
            killed = False
            for proc in psutil.process_iter(['name', 'pid']):
                name = proc.info['name'].lower()
                # 修复逻辑运算符优先级: 需要加括号
                # 匹配: 直播伴侣 或 (douyin + live)
                if '直播伴侣' in name or ('douyin' in name and 'live' in name):
                    try:
                        proc.terminate()
                        killed = True
                    except:
                        pass
            return killed
        except Exception as e:
            print(f"关闭直播伴侣失败: {e}")
            return False
    
    def start_stream_monitor(self, callback: Callable[[str, dict], None]) -> bool:
        """
        启动推流状态监听
        
        Args:
            callback: 状态回调函数，参数为 (event_type, data)
                event_type: 'started' | 'stopped' | 'reconnecting' | 'reconnected' | 'error'
                data: 事件详情 {'output_active': bool, 'output_bytes': int, ...}
        
        Returns:
            是否成功启动监听
        """
        if not HAS_OBSWS:
            self._log("obsws-python 未安装，无法监听状态")
            return False
        
        if self._is_monitoring:
            self._log("已在监听中")
            return True
        
        self._stream_status_callback = callback
        
        try:
            # 创建事件客户端
            self._event_client = obsws.EventClient(
                host=self._ws_host,
                port=self._ws_port,
                password=self._ws_password if self._ws_password else None
            )
            
            # 注册事件回调
            self._event_client.callback.register(self._on_stream_state_changed)
            
            self._is_monitoring = True
            self._log("推流状态监听已启动")
            return True
            
        except Exception as e:
            self._log(f"启动推流监听失败: {e}")
            return False
    
    def _on_stream_state_changed(self, data):
        """处理推流状态变化事件"""
        try:
            event_type = data.__class__.__name__
            
            # 解析不同事件
            if event_type == 'StreamStateChanged':
                output_active = getattr(data, 'output_active', False)
                output_state = getattr(data, 'output_state', '')
                
                if output_state == 'OBS_WEBSOCKET_OUTPUT_STARTED':
                    self._notify_callback('started', {
                        'output_active': True,
                        'state': output_state
                    })
                elif output_state == 'OBS_WEBSOCKET_OUTPUT_STOPPED':
                    self._notify_callback('stopped', {
                        'output_active': False,
                        'state': output_state
                    })
                elif output_state == 'OBS_WEBSOCKET_OUTPUT_RECONNECTING':
                    self._notify_callback('reconnecting', {
                        'output_active': output_active,
                        'state': output_state
                    })
                elif output_state == 'OBS_WEBSOCKET_OUTPUT_RECONNECTED':
                    self._notify_callback('reconnected', {
                        'output_active': True,
                        'state': output_state
                    })
                else:
                    self._notify_callback('state_change', {
                        'output_active': output_active,
                        'state': output_state
                    })
                    
        except Exception as e:
            self._log(f"处理推流事件失败: {e}")
    
    def _notify_callback(self, event_type: str, data: dict):
        """通知回调"""
        if self._stream_status_callback:
            try:
                self._stream_status_callback(event_type, data)
            except Exception as e:
                self._log(f"回调执行失败: {e}")
    
    def stop_stream_monitor(self):
        """停止推流状态监听"""
        if self._event_client:
            try:
                self._event_client.base_client.ws.close()
            except:
                pass
            self._event_client = None
        
        self._is_monitoring = False
        self._stream_status_callback = None
        self._log("推流状态监听已停止")
    
    def poll_stream_status(self) -> Optional[dict]:
        """
        轮询获取当前推流状态（简单方式，不需要事件订阅）
        
        Returns:
            状态字典 或 None
        """
        try:
            if not self.connect_websocket():
                return None
            
            resp = self._ws_client.get_stream_status()
            status = {
                'is_streaming': resp.output_active,
                'output_bytes': getattr(resp, 'output_bytes', 0),
                'output_duration': getattr(resp, 'output_duration', 0),
                'output_congestion': getattr(resp, 'output_congestion', 0),
                'output_skipped_frames': getattr(resp, 'output_skipped_frames', 0),
                'output_total_frames': getattr(resp, 'output_total_frames', 0),
            }
            return status
        except Exception as e:
            self._log(f"获取推流状态失败: {e}")
            return None
        finally:
            self.disconnect_websocket()


# 测试
if __name__ == '__main__':
    manager = ObsManager()
    print(f"OBS已安装: {manager.is_installed}")
    print(f"OBS已配置: {manager.is_configured}")
    print(f"OBS运行中: {manager.is_running()}")
