# -*- coding: utf-8 -*-
"""
直播伴侣管理器
负责查找、启动、监控抖音直播伴侣进程
"""

import os
import re
import json
import time
import winreg
import subprocess
import threading
from pathlib import Path
from typing import Optional, Callable, Dict, Any
from dataclasses import dataclass

import psutil

from src.utils.logger import logger


@dataclass
class LiveCompanionInfo:
    """直播伴侣信息"""
    exe_path: str
    version: str = ""
    process_id: int = 0
    is_running: bool = False
    is_streaming: bool = False


class LiveCompanionManager:
    """
    直播伴侣管理器
    
    功能:
    - 自动查找直播伴侣安装路径
    - 启动/停止直播伴侣
    - 监控直播伴侣进程状态
    - 检测开播状态
    """
    
    APP_NAME = "直播伴侣"
    PROCESS_NAMES = ["直播伴侣.exe", "webcast_mate.exe", "LiveCompanion.exe"]
    # 常见安装目录
    COMMON_INSTALL_DIRS = [
        r"C:\Program Files\webcast_mate",
        r"C:\Program Files (x86)\webcast_mate",
        r"D:\Program Files\webcast_mate", 
        r"D:\Program Files (x86)\webcast_mate",
        r"E:\Program Files\webcast_mate",
        r"C:\webcast_mate",
        r"D:\webcast_mate",
    ]
    
    def __init__(self):
        self._exe_path: Optional[str] = None
        self._process: Optional[psutil.Process] = None
        self._monitor_thread: Optional[threading.Thread] = None
        self._stop_monitor = threading.Event()
        
        # 回调函数
        self._on_process_start: Optional[Callable[[int], None]] = None
        self._on_process_exit: Optional[Callable[[], None]] = None
        self._on_streaming_start: Optional[Callable[[], None]] = None
        self._on_streaming_stop: Optional[Callable[[], None]] = None
        
        # 状态
        self._is_streaming = False
        
    @property
    def exe_path(self) -> Optional[str]:
        """获取直播伴侣exe路径"""
        if self._exe_path and os.path.exists(self._exe_path):
            return self._exe_path
        
        # 尝试自动查找
        self._exe_path = self._find_exe_path()
        return self._exe_path
    
    @exe_path.setter
    def exe_path(self, value: str):
        """设置直播伴侣exe路径"""
        self._exe_path = value
        
    def _find_exe_path(self) -> Optional[str]:
        """
        查找直播伴侣安装路径 - 快速检测
        优先使用快速方法，避免卡死
        """
        logger.info("[直播伴侣检测] 开始...")
        
        # 方法1: 从运行的进程获取 (最可靠最快)
        logger.debug("[直播伴侣检测] 检查运行进程...")
        exe_path = self._find_from_running_process()
        if exe_path:
            logger.info(f"[直播伴侣检测] ✓ 从进程找到: {exe_path}")
            return exe_path
        
        # 方法2: 常见安装路径 (快速)
        logger.debug("[直播伴侣检测] 检查常见路径...")
        exe_path = self._find_from_common_paths()
        if exe_path:
            logger.info(f"[直播伴侣检测] ✓ 从常见路径找到: {exe_path}")
            return exe_path
        
        # 方法3: 从注册表查找
        logger.debug("[直播伴侣检测] 检查注册表...")
        exe_path = self._find_from_registry_fast()
        if exe_path:
            logger.info(f"[直播伴侣检测] ✓ 从注册表找到: {exe_path}")
            return exe_path
        
        logger.warning("[直播伴侣检测] ✗ 未找到")
        return None
    
    def _find_from_registry_fast(self) -> Optional[str]:
        """快速注册表查找 - 只查关键位置"""
        try:
            # 只查HKCU，速度更快
            key = winreg.OpenKey(winreg.HKEY_CURRENT_USER, 
                               r"SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall", 0,
                               winreg.KEY_READ)
            subkey_count = winreg.QueryInfoKey(key)[0]
            
            for i in range(min(subkey_count, 50)):  # 最多遍卆50个
                try:
                    subkey_name = winreg.EnumKey(key, i)
                    if 'webcast' in subkey_name.lower() or '直播伴侣' in subkey_name:
                        subkey = winreg.OpenKey(key, subkey_name)
                        try:
                            install_dir = winreg.QueryValueEx(subkey, "InstallLocation")[0]
                            if install_dir:
                                exe_path = os.path.join(install_dir, f"{self.APP_NAME}.exe")
                                if os.path.exists(exe_path):
                                    winreg.CloseKey(subkey)
                                    winreg.CloseKey(key)
                                    return self._resolve_launcher(exe_path)
                        except:
                            pass
                        winreg.CloseKey(subkey)
                except:
                    continue
            winreg.CloseKey(key)
        except:
            pass
        return None
    
    def _find_from_registry(self) -> Optional[str]:
        """从注册表查找 - 已废弃，使用_find_from_registry_fast"""
        return self._find_from_registry_fast()
    
    def _find_from_start_menu(self) -> Optional[str]:
        """从开始菜单快捷方式查找"""
        try:
            start_menu_paths = [
                os.path.join(os.environ.get('ProgramData', ''), 
                           r"Microsoft\Windows\Start Menu\Programs"),
                os.path.join(os.environ.get('APPDATA', ''), 
                           r"Microsoft\Windows\Start Menu\Programs")
            ]
            
            for start_menu in start_menu_paths:
                if not os.path.exists(start_menu):
                    continue
                    
                for root, dirs, files in os.walk(start_menu):
                    for file in files:
                        if file == f"{self.APP_NAME}.lnk":
                            lnk_path = os.path.join(root, file)
                            target = self._get_shortcut_target(lnk_path)
                            if target and os.path.exists(target):
                                logger.info(f"从开始菜单找到直播伴侣: {target}")
                                return self._resolve_launcher(target)
        except Exception as e:
            logger.debug(f"开始菜单查找失败: {e}")
        
        return None
    
    def _find_from_common_paths(self) -> Optional[str]:
        """从常见安装路径查找 - 扫描所有磁盘"""
        # 基础路径 - 包含用户实际安装位置
        base_paths = [
            os.path.expanduser(r"~\AppData\Local\webcast_mate"),
            os.path.expanduser(r"~\AppData\Local\Programs\webcast_mate"),
        ]
        
        # 添加所有磁盘的常见路径
        for drive in "CDEFGH":
            drive_path = f"{drive}:\\"
            if os.path.exists(drive_path):
                base_paths.extend([
                    f"{drive}:\\Program Files\\webcast_mate",
                    f"{drive}:\\Program Files (x86)\\webcast_mate",
                    f"{drive}:\\webcast_mate",
                    f"{drive}:\\直播伴侣",
                    # 用户自定义路径
                    f"{drive}:\\soft\\zhibobanlv\\webcast_mate",
                    f"{drive}:\\soft\\直播伴侣",
                    f"{drive}:\\Software\\webcast_mate",
                ])
        
        for path in base_paths:
            if not os.path.isdir(path):
                continue
            
            logger.debug(f"[直播伴侣检测] 检查目录: {path}")
            
            # 直接查找 exe
            exe_path = os.path.join(path, f"{self.APP_NAME}.exe")
            if os.path.exists(exe_path):
                return self._resolve_launcher(exe_path)
            
            # 查找 Launcher
            launcher_path = os.path.join(path, f"{self.APP_NAME} Launcher.exe")
            if os.path.exists(launcher_path):
                return self._resolve_launcher(launcher_path)
            
            # 遍历子目录查找 (如 11.1.3.xxx)
            try:
                for subdir in os.listdir(path):
                    subpath = os.path.join(path, subdir)
                    if os.path.isdir(subpath):
                        exe_path = os.path.join(subpath, f"{self.APP_NAME}.exe")
                        if os.path.exists(exe_path):
                            logger.debug(f"[直播伴侣检测] 在子目录找到: {exe_path}")
                            return exe_path
            except PermissionError:
                pass
        
        return None
    
    def _find_from_running_process(self) -> Optional[str]:
        """从正在运行的进程获取路径"""
        try:
            for proc in psutil.process_iter(['pid', 'name', 'exe']):
                try:
                    name = proc.info.get('name', '')
                    if name in self.PROCESS_NAMES:
                        exe_path = proc.info.get('exe')
                        if exe_path and os.path.exists(exe_path):
                            logger.info(f"从运行进程找到直播伴侣: {exe_path}")
                            return exe_path
                except (psutil.NoSuchProcess, psutil.AccessDenied):
                    continue
        except Exception as e:
            logger.debug(f"进程查找失败: {e}")
        return None
    
    def _resolve_launcher(self, exe_path: str) -> str:
        """
        解析版本选择器
        如果是Launcher则找到真正的exe
        """
        filename = os.path.basename(exe_path)
        
        if "Launcher" in filename:
            install_dir = os.path.dirname(exe_path)
            config_path = os.path.join(install_dir, "launcher_config.json")
            
            if os.path.exists(config_path):
                try:
                    with open(config_path, 'r', encoding='utf-8') as f:
                        config = json.load(f)
                    cur_path = config.get('cur_path', '')
                    if cur_path:
                        real_exe = os.path.join(install_dir, cur_path, f"{self.APP_NAME}.exe")
                        if os.path.exists(real_exe):
                            logger.info(f"解析Launcher得到真实路径: {real_exe}")
                            return real_exe
                except Exception as e:
                    logger.warning(f"解析Launcher配置失败: {e}")
        
        return exe_path
    
    def _get_shortcut_target(self, lnk_path: str) -> Optional[str]:
        """获取快捷方式目标路径"""
        try:
            # 使用PowerShell获取快捷方式目标
            cmd = f'''powershell -command "$WshShell = New-Object -ComObject WScript.Shell; $Shortcut = $WshShell.CreateShortcut('{lnk_path}'); $Shortcut.TargetPath"'''
            result = subprocess.run(cmd, capture_output=True, text=True, shell=True)
            if result.returncode == 0:
                return result.stdout.strip()
        except Exception as e:
            logger.debug(f"获取快捷方式目标失败: {e}")
        return None
    
    def is_running(self) -> bool:
        """检查直播伴侣是否正在运行"""
        return self.find_process() is not None
    
    def find_process(self) -> Optional[psutil.Process]:
        """查找直播伴侣进程"""
        for proc in psutil.process_iter(['pid', 'name', 'exe']):
            try:
                if proc.info['name'] in self.PROCESS_NAMES:
                    return proc
            except (psutil.NoSuchProcess, psutil.AccessDenied):
                pass
        return None
    
    def start(self) -> bool:
        """启动直播伴侣"""
        if self.is_running():
            logger.info("直播伴侣已在运行")
            return True
        
        exe_path = self.exe_path
        if not exe_path:
            logger.error("未找到直播伴侣安装路径")
            return False
        
        try:
            logger.info(f"启动直播伴侣: {exe_path}")
            subprocess.Popen(
                exe_path,
                cwd=os.path.dirname(exe_path),
                shell=False
            )
            
            # 等待进程启动
            for _ in range(30):
                time.sleep(0.5)
                if self.is_running():
                    logger.info("直播伴侣启动成功")
                    return True
            
            logger.warning("直播伴侣启动超时")
            return False
            
        except Exception as e:
            logger.error(f"启动直播伴侣失败: {e}")
            return False
    
    def stop(self) -> bool:
        """停止直播伴侣"""
        proc = self.find_process()
        if not proc:
            return True
        
        try:
            proc.terminate()
            proc.wait(timeout=5)
            logger.info("直播伴侣已停止")
            return True
        except psutil.TimeoutExpired:
            proc.kill()
            logger.warning("强制终止直播伴侣")
            return True
        except Exception as e:
            logger.error(f"停止直播伴侣失败: {e}")
            return False
    
    def start_monitor(self,
                      on_start: Optional[Callable[[int], None]] = None,
                      on_exit: Optional[Callable[[], None]] = None,
                      on_streaming_start: Optional[Callable[[], None]] = None,
                      on_streaming_stop: Optional[Callable[[], None]] = None):
        """
        启动进程监控
        
        Args:
            on_start: 进程启动回调，参数为PID
            on_exit: 进程退出回调
            on_streaming_start: 开始推流回调
            on_streaming_stop: 停止推流回调
        """
        self._on_process_start = on_start
        self._on_process_exit = on_exit
        self._on_streaming_start = on_streaming_start
        self._on_streaming_stop = on_streaming_stop
        
        self._stop_monitor.clear()
        self._monitor_thread = threading.Thread(target=self._monitor_loop, daemon=True)
        self._monitor_thread.start()
        logger.info("直播伴侣进程监控已启动")
    
    def stop_monitor(self):
        """停止进程监控"""
        self._stop_monitor.set()
        if self._monitor_thread:
            self._monitor_thread.join(timeout=2)
        logger.info("直播伴侣进程监控已停止")
    
    def _monitor_loop(self):
        """监控循环"""
        was_running = False
        last_pid = 0
        
        while not self._stop_monitor.is_set():
            try:
                proc = self.find_process()
                is_running = proc is not None
                
                # 检测进程启动
                if is_running and not was_running:
                    last_pid = proc.pid
                    logger.info(f"检测到直播伴侣启动, PID: {last_pid}")
                    if self._on_process_start:
                        self._on_process_start(last_pid)
                
                # 检测进程退出
                elif not is_running and was_running:
                    logger.info("检测到直播伴侣退出")
                    self._is_streaming = False
                    if self._on_process_exit:
                        self._on_process_exit()
                
                # 检测推流状态变化（通过网络连接）
                if is_running:
                    is_streaming = self._check_streaming_status(proc)
                    
                    if is_streaming and not self._is_streaming:
                        logger.info("检测到开始推流")
                        if self._on_streaming_start:
                            self._on_streaming_start()
                    elif not is_streaming and self._is_streaming:
                        logger.info("检测到停止推流")
                        if self._on_streaming_stop:
                            self._on_streaming_stop()
                    
                    self._is_streaming = is_streaming
                
                was_running = is_running
                
            except Exception as e:
                logger.debug(f"监控循环异常: {e}")
            
            self._stop_monitor.wait(timeout=1)
    
    def _check_streaming_status(self, proc: psutil.Process) -> bool:
        """
        检查是否正在推流
        通过检查进程的网络连接判断
        """
        try:
            connections = proc.net_connections()
            for conn in connections:
                # RTMP端口通常是1935
                if conn.raddr and conn.raddr.port in [1935, 443, 19350]:
                    if conn.status == 'ESTABLISHED':
                        return True
        except (psutil.NoSuchProcess, psutil.AccessDenied):
            pass
        return False
    
    def get_info(self) -> LiveCompanionInfo:
        """获取直播伴侣信息"""
        proc = self.find_process()
        
        return LiveCompanionInfo(
            exe_path=self.exe_path or "",
            version=self._get_version(),
            process_id=proc.pid if proc else 0,
            is_running=proc is not None,
            is_streaming=self._is_streaming
        )
    
    def _get_version(self) -> str:
        """获取版本号"""
        if not self.exe_path:
            return ""
        
        try:
            # 从package.json读取版本
            install_dir = os.path.dirname(self.exe_path)
            resources_dir = os.path.join(install_dir, "resources", "app")
            package_json = os.path.join(resources_dir, "package.json")
            
            if os.path.exists(package_json):
                with open(package_json, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                return data.get('version', '')
        except Exception:
            pass
        
        return ""
