# -*- coding: utf-8 -*-
"""
自动开播模块
实现自动操作直播伴侣开播 - 使用OCR定位按钮
"""

import os
import time
import subprocess
import threading
import ctypes
from typing import Optional, Callable, Tuple
from pathlib import Path

# 设置 DPI 感知 - 必须在导入 pyautogui 之前
try:
    ctypes.windll.shcore.SetProcessDpiAwareness(2)  # PROCESS_PER_MONITOR_DPI_AWARE
except Exception:
    try:
        ctypes.windll.user32.SetProcessDPIAware()
    except Exception:
        pass

try:
    import pyautogui
    PYAUTOGUI_AVAILABLE = True
except ImportError:
    PYAUTOGUI_AVAILABLE = False

try:
    import win32gui
    import win32con
    WIN32_AVAILABLE = True
except ImportError:
    WIN32_AVAILABLE = False

try:
    import psutil
    PSUTIL_AVAILABLE = True
except ImportError:
    PSUTIL_AVAILABLE = False

try:
    from rapidocr_onnxruntime import RapidOCR
    OCR_AVAILABLE = True
except ImportError:
    OCR_AVAILABLE = False


class AutoBroadcast:
    """自动开播控制器"""
    
    # 直播伴侣窗口标题关键词
    # 直播伴侣标题格式: "直播伴侣" (可能有多个窗口)
    WINDOW_TITLES = ["直播伴侣"]
    # 排除我们自己的应用和浏览器中的直播间
    EXCLUDE_TITLES = ["梧桐", "DouyinLiveAssistant", "运营伴侣", "Chrome", "Edge", "Firefox", "抖音直播间"]
    
    # 开播按钮的可能文字
    START_BUTTON_TEXTS = ["开始直播", "开播", "开始"]
    
    def __init__(self, companion_path: str = None, logger: Callable = None):
        self.companion_path = companion_path
        self._logger = logger or print
        self._companion_process = None
        self._window_handle = None
    
    def log(self, msg: str):
        """日志输出"""
        print(f"[AutoBroadcast] {msg}")
        if self._logger:
            self._logger(msg)
    
    def is_companion_running(self) -> bool:
        """检查直播伴侣是否运行"""
        if not PSUTIL_AVAILABLE:
            return False
        
        for proc in psutil.process_iter(['name']):
            try:
                name = proc.info.get('name', '').lower()
                if '直播伴侣' in name or 'webcast' in name:
                    return True
            except:
                pass
        return False
    
    def start_companion(self) -> bool:
        """启动直播伴侣"""
        if not self.companion_path or not os.path.exists(self.companion_path):
            self.log("直播伴侣路径未设置或不存在")
            return False
        
        if self.is_companion_running():
            self.log("直播伴侣已在运行")
            return True
        
        try:
            work_dir = os.path.dirname(self.companion_path)
            self._companion_process = subprocess.Popen(
                [self.companion_path],
                cwd=work_dir,
                creationflags=subprocess.DETACHED_PROCESS | subprocess.CREATE_NEW_PROCESS_GROUP
            )
            self.log(f"直播伴侣已启动: {self.companion_path}")
            return True
        except Exception as e:
            self.log(f"启动直播伴侣失败: {e}")
            return False
    
    def find_companion_window(self, timeout: int = 30) -> Optional[int]:
        """查找直播伴侣窗口句柄 - 通过进程名查找"""
        if not WIN32_AVAILABLE or not PSUTIL_AVAILABLE:
            self.log("win32gui或psutil不可用")
            return None
        
        start_time = time.time()
        
        while time.time() - start_time < timeout:
            # 方法1: 通过进程找窗口（最可靠）
            companion_pids = self._get_companion_pids()
            if companion_pids:
                hwnd = self._find_window_by_pids(companion_pids)
                if hwnd:
                    return hwnd
            
            time.sleep(1)
        
        self.log("未找到直播伴侣窗口")
        return None
    
    def _get_companion_pids(self) -> list:
        """获取直播伴侣进程ID列表"""
        pids = []
        for proc in psutil.process_iter(['pid', 'name']):
            try:
                name = proc.info.get('name', '').lower()
                if '直播伴侣' in name or 'webcast' in name or 'livemate' in name:
                    pids.append(proc.info['pid'])
            except:
                pass
        return pids
    
    def _find_window_by_pids(self, pids: list) -> Optional[int]:
        """根据进程ID查找主窗口"""
        import ctypes
        from ctypes import wintypes
        
        user32 = ctypes.windll.user32
        
        def get_window_thread_process_id(hwnd):
            pid = wintypes.DWORD()
            user32.GetWindowThreadProcessId(hwnd, ctypes.byref(pid))
            return pid.value
        
        results = []
        
        def enum_callback(hwnd, _):
            # 不检查 IsWindowVisible，因为后台窗口也需要找到
            pid = get_window_thread_process_id(hwnd)
            if pid in pids:
                # 检查是否有标题（主窗口通常有标题）
                title = win32gui.GetWindowText(hwnd)
                if not title:
                    return True
                
                rect = win32gui.GetWindowRect(hwnd)
                left, top, right, bottom = rect
                width = right - left
                height = bottom - top
                
                # 只选择大小合理的窗口（排除隐藏窗口 left=-32000）
                if width > 400 and height > 300 and left > -1000:
                    results.append((hwnd, width, height, title, pid))
            return True
        
        try:
            win32gui.EnumWindows(enum_callback, None)
            if results:
                # 选择最大的窗口
                results.sort(key=lambda x: x[1] * x[2], reverse=True)
                self._window_handle = results[0][0]
                title = results[0][3]
                width, height = results[0][1], results[0][2]
                self.log(f"找到直播伴侣窗口: {title} ({width}x{height}) PID={results[0][4]}")
                return self._window_handle
        except Exception as e:
            self.log(f"查找窗口异常: {e}")
        
        return None
    
    def activate_window(self, hwnd: int) -> bool:
        """激活窗口到前台"""
        if not WIN32_AVAILABLE:
            return False
        
        try:
            # 如果窗口最小化，先恢复
            if win32gui.IsIconic(hwnd):
                win32gui.ShowWindow(hwnd, win32con.SW_RESTORE)
            
            # 激活窗口
            win32gui.SetForegroundWindow(hwnd)
            time.sleep(0.5)
            return True
        except Exception as e:
            self.log(f"激活窗口失败: {e}")
            return False
    
    def click_start_button(self) -> bool:
        """点击开始直播按钮 - 必须OCR识别到"开始直播"才点击"""
        if not PYAUTOGUI_AVAILABLE:
            self.log("pyautogui不可用")
            return False
        
        if not self._window_handle:
            self.log("窗口句柄不存在")
            return False
        
        if not OCR_AVAILABLE:
            self.log("OCR模块不可用，无法自动开播")
            return False
        
        try:
            # 激活窗口
            self.activate_window(self._window_handle)
            time.sleep(1)
            
            # 必须使用OCR定位按钮，只有识别到"开始直播"才点击
            result = self._click_by_ocr()
            if result:
                return True
            
            # OCR未识别到"开始直播"，不进行点击
            self.log("未识别到【开始直播】按钮，不执行点击")
            return False
            
        except Exception as e:
            self.log(f"点击按钮失败: {e}")
            return False
    
    def _click_by_ocr(self) -> bool:
        """通过OCR识别"开始直播"文字位置并点击"""
        try:
            self.log("使用OCR识别按钮位置...")
            
            # 先将直播伴侣窗口置于最前
            self.activate_window(self._window_handle)
            time.sleep(0.5)
            
            # 获取窗口位置
            rect = win32gui.GetWindowRect(self._window_handle)
            left, top, right, bottom = rect
            width = right - left
            height = bottom - top
            
            self.log(f"窗口区域: ({left}, {top}) - ({right}, {bottom})")
            
            # 截图 - 只截取直播伴侣窗口区域
            screenshot = pyautogui.screenshot(region=(left, top, width, height))
            
            # 保存调试图
            debug_path = Path(__file__).parent / 'ocr_screenshot.png'
            screenshot.save(str(debug_path))
            self.log(f"截图保存: {debug_path}")
            
            # OCR识别
            ocr = RapidOCR()
            result, _ = ocr(str(debug_path))
            
            if not result:
                self.log("OCR未识别到任何文字")
                return False
            
            self.log(f"OCR识别到 {len(result)} 个文本区域")
            
            # 查找"开始直播"按钮
            candidates = []
            
            for item in result:
                box, text, confidence = item
                text_clean = text.strip()
                center_x = sum(p[0] for p in box) / 4
                center_y = sum(p[1] for p in box) / 4
                
                # 精确匹配"开始直播"
                if text_clean == "开始直播":
                    self.log(f"  找到精确匹配: '{text_clean}' @ ({center_x:.0f}, {center_y:.0f})")
                    candidates.append((box, text_clean, center_x, center_y, confidence * 2))
                # 或者包含"开始直播"且长度短
                elif "开始直播" in text_clean and len(text_clean) <= 8:
                    self.log(f"  找到模糊匹配: '{text_clean}' @ ({center_x:.0f}, {center_y:.0f})")
                    candidates.append((box, text_clean, center_x, center_y, confidence))
            
            # 选择位于最底部的按钮（真正的按钮在底部工具栏，中间的是录屏画面）
            best_match = None
            best_y = 0  # 选择Y坐标最大的（最靠近底部）
            
            for box, text, cx, cy, score in candidates:
                self.log(f"  候选: '{text}' @ ({cx:.0f}, {cy:.0f}) 高度比例={cy/height:.2f}")
                # 只选择在底部15%区域的按钮（真正的开始直播按钮在最底部）
                if cy > height * 0.85 and cy > best_y:
                    best_y = cy
                    best_match = (box, text, cx, cy)
            
            # 如果底部没找到，放宽到底部30%
            if not best_match:
                for box, text, cx, cy, score in candidates:
                    if cy > height * 0.7 and cy > best_y:
                        best_y = cy
                        best_match = (box, text, cx, cy)
            
            if best_match:
                box, text, center_x, center_y = best_match
                click_x = int(left + center_x)
                click_y = int(top + center_y)
                
                self.log(f"选中按钮 '{text}' 屏幕坐标: ({click_x}, {click_y})")
                
                # 激活窗口
                self.activate_window(self._window_handle)
                time.sleep(0.5)
                
                # 使用 ctypes 直接调用 Windows API
                user32 = ctypes.windll.user32
                
                # 移动鼠标并点击
                user32.SetCursorPos(click_x, click_y)
                time.sleep(0.3)
                
                # 发送鼠标点击
                MOUSEEVENTF_LEFTDOWN = 0x0002
                MOUSEEVENTF_LEFTUP = 0x0004
                MOUSEEVENTF_ABSOLUTE = 0x8000
                MOUSEEVENTF_MOVE = 0x0001
                
                user32.mouse_event(MOUSEEVENTF_LEFTDOWN, 0, 0, 0, 0)
                time.sleep(0.05)
                user32.mouse_event(MOUSEEVENTF_LEFTUP, 0, 0, 0, 0)
                
                self.log("已点击开始直播按钮 (ctypes)")
                
                # 等待并再点一次
                time.sleep(0.5)
                user32.mouse_event(MOUSEEVENTF_LEFTDOWN, 0, 0, 0, 0)
                time.sleep(0.05)
                user32.mouse_event(MOUSEEVENTF_LEFTUP, 0, 0, 0, 0)
                
                return True
            
            # 如果没有候选，打印所有识别到的文字帮助调试
            self.log("未找到开始直播按钮，打印所有识别结果:")
            for item in result:
                box, text, confidence = item
                if confidence > 0.5:  # 只打印置信度高的
                    center_x = sum(p[0] for p in box) / 4
                    center_y = sum(p[1] for p in box) / 4
                    self.log(f"  '{text}' @ ({center_x:.0f}, {center_y:.0f})")
            
            return False
            
        except Exception as e:
            self.log(f"OCR识别失败: {e}")
            import traceback
            traceback.print_exc()
            return False
    
    def _click_by_color(self) -> bool:
        """通过颜色识别查找绿色开播按钮"""
        try:
            # 获取窗口位置
            rect = win32gui.GetWindowRect(self._window_handle)
            left, top, right, bottom = rect
            width = right - left
            height = bottom - top
            
            # 截图
            screenshot = pyautogui.screenshot(region=(left, top, width, height))
            
            # 在底部区域搜索绿色按钮
            search_top = int(height * 0.8)  # 只搜索底部20%
            
            green_pixels = []
            for y in range(search_top, height):
                for x in range(width):
                    r, g, b = screenshot.getpixel((x, y))
                    # 抖音绿色按钮 RGB(254, 44, 85) 实际是红色，让我用更宽的匹配
                    # 或者是 RGB(0, 200, 100) 绿色
                    if (g > 150 and g > r and g > b) or (r > 200 and g < 100 and b < 150):
                        green_pixels.append((x, y))
            
            if len(green_pixels) > 50:
                avg_x = sum(p[0] for p in green_pixels) // len(green_pixels)
                avg_y = sum(p[1] for p in green_pixels) // len(green_pixels)
                
                click_x = left + avg_x
                click_y = top + avg_y
                
                self.log(f"颜色识别找到按钮: ({click_x}, {click_y})")
                pyautogui.click(click_x, click_y)
                return True
                    
        except Exception as e:
            self.log(f"颜色识别失败: {e}")
        return False
    
    def auto_start_broadcast(self, wait_window_timeout: int = 30, 
                             click_delay: int = 5) -> bool:
        """
        全自动开播流程
        
        Args:
            wait_window_timeout: 等待窗口出现的超时时间(秒)
            click_delay: 窗口出现后等待多久再点击(秒)
        
        Returns:
            是否成功触发开播
        """
        self.log("========== 开始自动开播流程 ==========")
        
        # 1. 启动直播伴侣
        self.log("步骤1: 启动直播伴侣...")
        if not self.start_companion():
            return False
        
        # 2. 等待窗口出现
        self.log("步骤2: 等待直播伴侣窗口...")
        hwnd = self.find_companion_window(timeout=wait_window_timeout)
        if not hwnd:
            return False
        
        # 3. 等待窗口完全加载
        self.log(f"步骤3: 等待{click_delay}秒让窗口完全加载...")
        time.sleep(click_delay)
        
        # 4. 点击开始直播按钮
        self.log("步骤4: 点击开始直播按钮...")
        if not self.click_start_button():
            return False
        
        self.log("========== 自动开播流程完成 ==========")
        return True


def check_dependencies() -> dict:
    """检查依赖"""
    return {
        'pyautogui': PYAUTOGUI_AVAILABLE,
        'win32gui': WIN32_AVAILABLE,
        'psutil': PSUTIL_AVAILABLE,
        'ocr': OCR_AVAILABLE,
    }


# 测试
if __name__ == '__main__':
    print("检查依赖:", check_dependencies())
    
    # 测试自动开播
    auto = AutoBroadcast(
        companion_path=r"D:\soft\zhibobanlv\webcast_mate\11.1.3.263047209\直播伴侣.exe"
    )
    auto.auto_start_broadcast()
