# -*- coding: utf-8 -*-
"""
回复处理器
处理各种事件并生成语音回复
"""

import random
import time
import asyncio
from typing import Dict, List, Optional, Any
from dataclasses import dataclass, field

from src.utils.events import (
    EventType, Event, User, Gift,
    ChatEvent, GiftEvent, EntranceEvent, FollowEvent
)
from src.utils.logger import logger
from src.tts import BaseTTS
from src.audio import AudioPlayer


@dataclass
class ReplyConfig:
    """回复配置"""
    enabled: bool = True
    templates: List[str] = field(default_factory=list)
    cooldown: float = 3.0
    min_value: int = 0
    first_only: bool = False
    read_chat: bool = False


class ReplyHandler:
    """回复处理器"""
    
    def __init__(
        self,
        tts_engine: BaseTTS,
        audio_player: AudioPlayer,
        config: Dict[str, Any] = None
    ):
        self.tts = tts_engine
        self.player = audio_player
        self._load_config(config or {})
        self._cooldowns: Dict[str, float] = {}
        self._entered_users: set = set()
        self._processing = False
        self._queue: asyncio.Queue = asyncio.Queue()
    
    def _load_config(self, config: Dict[str, Any]) -> None:
        """加载配置"""
        reply_config = config.get("reply", {})
        
        entrance_cfg = reply_config.get("entrance", {})
        self.entrance_config = ReplyConfig(
            enabled=entrance_cfg.get("enabled", True),
            templates=entrance_cfg.get("templates", ["欢迎 {username} 来到直播间！"]),
            cooldown=entrance_cfg.get("cooldown", 3),
            first_only=entrance_cfg.get("first_only", False)
        )
        
        gift_cfg = reply_config.get("gift", {})
        self.gift_config = ReplyConfig(
            enabled=gift_cfg.get("enabled", True),
            templates=gift_cfg.get("templates", ["感谢 {username} 送出的 {gift_name}！"]),
            cooldown=gift_cfg.get("cooldown", 1),
            min_value=gift_cfg.get("min_value", 0)
        )
        
        follow_cfg = reply_config.get("follow", {})
        self.follow_config = ReplyConfig(
            enabled=follow_cfg.get("enabled", True),
            templates=follow_cfg.get("templates", ["感谢 {username} 的关注！"]),
            cooldown=follow_cfg.get("cooldown", 2)
        )
        
        chat_cfg = reply_config.get("chat", {})
        self.chat_enabled = chat_cfg.get("enabled", False)
        self.chat_keywords: Dict[str, str] = chat_cfg.get("keywords", {})
        self.read_chat_enabled = chat_cfg.get("read_chat", False)

        ai_cfg = config.get("ai", {})
        self.ai_enabled = ai_cfg.get("enabled", False)
        if self.ai_enabled:
            try:
                from src.ai.llm_client import LLMClient
                self.llm = LLMClient(
                    api_key=ai_cfg.get("api_key", ""),
                    base_url=ai_cfg.get("base_url", "https://api.deepseek.com"),
                    model=ai_cfg.get("model", "deepseek-chat"),
                    system_prompt=ai_cfg.get("system_prompt", "")
                )
                self.ai_reply_rate = ai_cfg.get("reply_rate", 1.0)
                logger.info("AI模块已加载")
            except Exception as e:
                logger.error(f"AI模块加载失败: {e}")
                self.ai_enabled = False
                self.llm = None
        else:
            self.llm = None
        
        self.warm_enabled = ai_cfg.get("warm_enabled", False)
        self.warm_interval = ai_cfg.get("warm_interval", 60)
        self.warm_category = ai_cfg.get("warm_category", "random")
        self.reply_mode = ai_cfg.get("reply_mode", "tts")
        
        self._last_activity_time = time.time()
        
        self._warm_prompts_db = {
            "joke": [
                "讲个笑话活跃一下气氛吧",
                "说个段子逗大家开心",
                "来个幽默的梗",
                "讲个搞笑的短故事",
                "分享一件好笑的事情"
            ],
            "philosophy": [
                "说一句富有哲理的话",
                "分享一句人生感悟",
                "讲一个有深度的短句",
                "关于生活的一句感叹",
                "激励人心的一句话"
            ],
            "warm": [
                "说一句温暖治愈的话",
                "鼓励一下直播间的朋友们",
                "发一句温馨的晚安语录(如果晚上的话)",
                "感谢大家的陪伴",
                "说一句让人感到温暖的话"
            ],
            "trivia": [
                "分享一个冷知识",
                "说一个鲜为人知的小知识",
                "科普一个有趣的事实",
                "关于动物的一个冷知识",
                "关于生活的一个小常识"
            ],
            "random": []
        }
    
    async def handle_event(self, event: Event) -> None:
        """处理事件"""
        self._last_activity_time = time.time()
        
        handlers = {
            EventType.ENTRANCE: self._handle_entrance,
            EventType.GIFT: self._handle_gift,
            EventType.FOLLOW: self._handle_follow,
            EventType.CHAT: self._handle_chat,
        }
        
        handler = handlers.get(event.type)
        if handler:
            await handler(event)
    
    async def check_warm_up(self) -> bool:
        """检查是否需要暖场"""
        # 调试日志
        logger.debug(f"暖场检查: warm_enabled={self.warm_enabled}, ai_enabled={self.ai_enabled}, llm={self.llm is not None}")
        
        if not self.warm_enabled or not self.ai_enabled or not self.llm:
            return False
        
        elapsed = time.time() - self._last_activity_time
        logger.debug(f"暖场检查: elapsed={elapsed:.1f}s, interval={self.warm_interval}s")
        
        if elapsed < self.warm_interval:
            return False
        
        try:
            if self.warm_category == "random" or self.warm_category not in self._warm_prompts_db:
                categories = [k for k in self._warm_prompts_db.keys() if k != "random"]
                category = random.choice(categories)
                prompts = self._warm_prompts_db[category]
            else:
                prompts = self._warm_prompts_db[self.warm_category]
            
            prompt = random.choice(prompts)
            logger.info(f"🎤 暖场触发 [{self.warm_category}]: {prompt}")
            
            reply = await asyncio.to_thread(self.llm.chat, prompt)
            if reply:
                await self._do_reply(reply)
                self._last_activity_time = time.time()
                return True
        except Exception as e:
            logger.error(f"暖场异常: {e}")
        
        return False
    
    def update_warm_config(self, enabled: bool, interval: int, category: str = "random"):
        """更新暖场配置"""
        self.warm_enabled = enabled
        self.warm_interval = interval
        self.warm_category = category
        logger.info(f"暖场配置更新: enabled={enabled}, interval={interval}s, category={category}")

    def update_reply_mode(self, mode: str):
        """更新回复模式"""
        self.reply_mode = mode
        logger.info(f"回复模式更新: {mode}")
    
    async def _handle_entrance(self, event: EntranceEvent) -> None:
        """处理入场事件"""
        if not self.entrance_config.enabled:
            return
        
        user = event.user
        if not user:
            return
        
        if self.entrance_config.first_only:
            if user.id in self._entered_users:
                return
            self._entered_users.add(user.id)
        
        if not self._check_cooldown("entrance", self.entrance_config.cooldown):
            return
        
        template = random.choice(self.entrance_config.templates)
        text = template.format(username=user.nickname)
        await self._do_reply(text)
    
    async def _handle_gift(self, event: GiftEvent) -> None:
        """处理礼物事件"""
        if not self.gift_config.enabled:
            return
        
        user = event.user
        gift = event.gift
        if not user or not gift:
            return
        
        if self.gift_config.min_value > 0:
            total_value = gift.price * gift.count
            if total_value < self.gift_config.min_value:
                return
        
        if not self._check_cooldown("gift", self.gift_config.cooldown):
            return
        
        template = random.choice(self.gift_config.templates)
        text = template.format(
            username=user.nickname,
            gift_name=gift.name,
            count=gift.count,
            price=gift.price
        )
        await self._do_reply(text)
    
    async def _handle_follow(self, event: FollowEvent) -> None:
        """处理关注事件"""
        if not self.follow_config.enabled:
            return
        
        user = event.user
        if not user:
            return
        
        if not self._check_cooldown("follow", self.follow_config.cooldown):
            return
        
        template = random.choice(self.follow_config.templates)
        text = template.format(username=user.nickname)
        await self._do_reply(text)
    
    async def _handle_chat(self, event: ChatEvent) -> None:
        """处理弹幕事件"""
        if not self.chat_enabled and not self.ai_enabled and not self.read_chat_enabled:
            return
        
        content = event.content
        user = event.user
        if not content:
            return
        
        # 过滤自己发送的弹幕，避免AI回复自己导致套娃
        if hasattr(self, '_self_sent_texts') and content in self._self_sent_texts:
            logger.debug(f"跳过自己发送的弹幕: {content}")
            self._self_sent_texts.discard(content)  # 使用后移除
            return
        
        # 朗读弹幕
        if self.read_chat_enabled:
            if self._check_cooldown("read_chat", 2.0):
                text = f"{user.nickname}说: {content}"
                await self._speak(text)
        
        # 关键词匹配
        if self.chat_enabled:
            for keyword, reply in self.chat_keywords.items():
                if keyword in content:
                    cooldown_key = f"chat_{keyword}"
                    if not self._check_cooldown(cooldown_key, 5):
                        return
                    await self._do_reply(reply)
                    return

        # AI 智能回复
        if self.ai_enabled and self.llm:
            if random.random() <= self.ai_reply_rate:
                if not self._check_cooldown("ai_chat", 3):
                    logger.debug(f"AI回复冷却中，跳过: {content}")
                    return
                
                try:
                    logger.info(f"🤖 AI正在思考: {content}")
                    reply = await asyncio.to_thread(self.llm.chat, content)
                    if reply:
                        logger.info(f"🤖 AI回复: {reply}")
                        await self._do_reply(reply)
                except Exception as e:
                    logger.error(f"AI回复异常: {e}")
    
    def _check_cooldown(self, key: str, cooldown: float) -> bool:
        """检查冷却时间"""
        now = time.time()
        last_time = self._cooldowns.get(key, 0)
        
        if now - last_time < cooldown:
            return False
        
        self._cooldowns[key] = now
        return True
    
    async def _do_reply(self, text: str) -> None:
        """统一回复处理"""
        if self.reply_mode in ["tts", "both"]:
            await self._speak(text)
        
        if self.reply_mode == "danmu":
            await self._speak(text)
    
    async def _speak(self, text: str) -> None:
        """语音播报"""
        try:
            text = self._filter_text(text)
            
            if not text.strip():
                return
            
            audio_path = await self.tts.synthesize(text)
            
            if audio_path:
                self.player.play(audio_path)
                logger.info(f"🔊 语音播报: {text}")
            else:
                logger.warning(f"语音合成失败: {text}")
                
        except Exception as e:
            logger.error(f"语音播报错误: {e}")
    
    def _filter_text(self, text: str) -> str:
        """过滤文本"""
        import re
        text = re.sub(r'\*+', '', text)
        text = re.sub(r'\s+', ' ', text)
        return text.strip()
    
    def clear_entered_users(self) -> None:
        """清除已入场用户记录"""
        self._entered_users.clear()
        logger.info("已清除入场用户记录")
    
    def update_config(self, config: Dict[str, Any]) -> None:
        """实时更新配置（包括模板）"""
        self._load_config(config)
        logger.info("回复配置已更新")
