# -*- coding: utf-8 -*-
"""
直连模式弹幕客户端
直接连接抖音WebSocket获取弹幕
"""

import gzip
import hashlib
import random
import re
import string
import asyncio
import threading
from typing import Optional, Dict, Any
from pathlib import Path

import requests
import websocket
from py_mini_racer import MiniRacer

from .base import BaseDanmakuClient
from src.utils.events import (
    EventType, User, Gift, Event,
    ChatEvent, GiftEvent, EntranceEvent, FollowEvent, LikeEvent, RoomStatsEvent
)
from src.utils.logger import logger

# 添加protobuf路径
import sys
PROTOBUF_PATH = Path(__file__).parent.parent.parent / "protobuf"
sys.path.insert(0, str(PROTOBUF_PATH))

from douyin import (
    PushFrame, Response, ChatMessage, GiftMessage, LikeMessage,
    MemberMessage, SocialMessage, RoomUserSeqMessage, FansclubMessage,
    ControlMessage, EmojiChatMessage, RoomStatsMessage, RoomMessage,
    RoomRankMessage
)


def generate_ms_token(length: int = 182) -> str:
    """生成msToken"""
    base_str = string.ascii_letters + string.digits + '-_'
    return ''.join(random.choice(base_str) for _ in range(length))


def generate_signature(wss: str, script_file: str = 'sign.js') -> str:
    """生成签名"""
    import urllib.parse
    
    params = ("live_id,aid,version_code,webcast_sdk_version,"
              "room_id,sub_room_id,sub_channel_id,did_rule,"
              "user_unique_id,device_platform,device_type,ac,"
              "identity").split(',')
    
    wss_params = urllib.parse.urlparse(wss).query.split('&')
    wss_maps = {i.split('=')[0]: i.split("=")[-1] for i in wss_params}
    tpl_params = [f"{i}={wss_maps.get(i, '')}" for i in params]
    param = ','.join(tpl_params)
    
    md5 = hashlib.md5()
    md5.update(param.encode())
    md5_param = md5.hexdigest()
    
    # 查找sign.js文件
    script_paths = [
        Path(__file__).parent.parent.parent / script_file,
        Path(script_file),
    ]
    
    script_path = None
    for p in script_paths:
        if p.exists():
            script_path = p
            break
    
    if not script_path:
        logger.warning(f"签名脚本 {script_file} 未找到，使用空签名")
        return ""
    
    with open(script_path, 'r', encoding='utf-8') as f:
        script = f.read()
    
    ctx = MiniRacer()
    ctx.eval(script)
    
    try:
        signature = ctx.call("get_sign", md5_param)
        return signature
    except Exception as e:
        logger.error(f"生成签名失败: {e}")
        return ""


class DirectDanmakuClient(BaseDanmakuClient):
    """直连模式弹幕客户端"""
    
    def __init__(self, room_id: str):
        super().__init__(room_id)
        self._ttwid: Optional[str] = None
        self._room_id_real: Optional[str] = None
        self._ws: Optional[websocket.WebSocketApp] = None
        self._heartbeat_thread: Optional[threading.Thread] = None
        self._loop: Optional[asyncio.AbstractEventLoop] = None
        
        self.user_agent = (
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 "
            "(KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36"
        )
        self.live_url = "https://live.douyin.com/"
        self.session = requests.Session()
    
    @property
    def ttwid(self) -> str:
        """获取ttwid"""
        if self._ttwid:
            return self._ttwid
        
        headers = {"User-Agent": self.user_agent}
        try:
            response = self.session.get(self.live_url, headers=headers)
            response.raise_for_status()
            self._ttwid = response.cookies.get('ttwid')
            logger.debug(f"获取ttwid成功: {self._ttwid[:20]}...")
            return self._ttwid
        except Exception as e:
            logger.error(f"获取ttwid失败: {e}")
            return ""
    
    @property
    def real_room_id(self) -> str:
        """获取真实房间ID"""
        if self._room_id_real:
            return self._room_id_real
        
        url = self.live_url + self.room_id
        headers = {
            "User-Agent": self.user_agent,
            "cookie": f"ttwid={self.ttwid}&msToken={generate_ms_token()}; __ac_nonce=0123407cc00a9e438deb4",
        }
        
        try:
            response = self.session.get(url, headers=headers)
            response.raise_for_status()
            match = re.search(r'roomId\\":\\"(\d+)\\"', response.text)
            if match:
                self._room_id_real = match.group(1)
                logger.info(f"获取真实房间ID: {self._room_id_real}")
                return self._room_id_real
            else:
                logger.error("无法解析房间ID")
                return self.room_id
        except Exception as e:
            logger.error(f"获取房间ID失败: {e}")
            return self.room_id
    
    async def connect(self) -> bool:
        """连接到直播间"""
        try:
            print(f"[DEBUG] DirectClient.connect() 开始")
            # 预先获取必要信息
            print(f"[DEBUG] 获取 ttwid...")
            _ = self.ttwid
            print(f"[DEBUG] ttwid = {self._ttwid}")
            
            print(f"[DEBUG] 获取 real_room_id...")
            _ = self.real_room_id
            print(f"[DEBUG] real_room_id = {self._room_id_real}")
            
            if not self._ttwid or not self._room_id_real:
                logger.error("无法获取连接所需信息")
                print(f"[DEBUG] 连接失败：ttwid={self._ttwid}, room_id={self._room_id_real}")
                return False
            
            self.is_connected = True
            print(f"[DEBUG] DirectClient.connect() 成功")
            return True
        except Exception as e:
            logger.error(f"连接准备失败: {e}")
            print(f"[DEBUG] DirectClient.connect() 异常: {e}")
            import traceback
            traceback.print_exc()
            return False
    
    async def disconnect(self) -> None:
        """断开连接"""
        self._running = False
        self.is_connected = False
        if self._ws:
            self._ws.close()
            self._ws = None
    
    async def start(self) -> None:
        """开始监听弹幕"""
        print(f"[DEBUG] DirectClient.start() 开始")
        
        if not await self.connect():
            logger.error("连接失败，无法启动监听")
            print(f"[DEBUG] connect() 返回 False，无法启动监听")
            return
        
        print(f"[DEBUG] connect() 成功，开始构建 WebSocket URL")
        self._running = True
        self._loop = asyncio.get_event_loop()
        
        # 构建 WebSocket URL
        try:
            print(f"[DEBUG] 构建 WSS URL...")
            wss = self._build_wss_url()
            print(f"[DEBUG] WSS URL: {wss[:100]}...")
            
            print(f"[DEBUG] 生成签名...")
            signature = generate_signature(wss)
            print(f"[DEBUG] 签名: {signature[:50]}...")
            wss += f"&signature={signature}"
        except Exception as e:
            print(f"[DEBUG] 构建 WSS URL 失败: {e}")
            import traceback
            traceback.print_exc()
            return
        
        headers = {
            "cookie": f"ttwid={self.ttwid}",
            "user-agent": self.user_agent,
        }
        
        logger.info(f"正在连接直播间 {self.room_id}...")
        print(f"[DEBUG] 准备启动 WebSocket, room_id={self.room_id}")
        
        self._ws = websocket.WebSocketApp(
            wss,
            header=headers,
            on_open=self._on_open,
            on_message=self._on_message,
            on_error=self._on_error,
            on_close=self._on_close
        )
        
        # 在线程中运行WebSocket
        ws_thread = threading.Thread(target=self._ws.run_forever)
        ws_thread.daemon = True
        ws_thread.start()
        
        # 保持运行
        while self._running:
            await asyncio.sleep(1)
    
    def _build_wss_url(self) -> str:
        """构建WebSocket URL"""
        return (
            f"wss://webcast100-ws-web-lq.douyin.com/webcast/im/push/v2/?"
            f"app_name=douyin_web&version_code=180800&webcast_sdk_version=1.0.14-beta.0"
            f"&update_version_code=1.0.14-beta.0&compress=gzip&device_platform=web"
            f"&cookie_enabled=true&screen_width=1536&screen_height=864"
            f"&browser_language=zh-CN&browser_platform=Win32&browser_name=Mozilla"
            f"&browser_version=5.0%20(Windows%20NT%2010.0;%20Win64;%20x64)%20AppleWebKit/537.36"
            f"&browser_online=true&tz_name=Asia/Shanghai"
            f"&cursor=d-1_u-1_fh-7392091211001140287_t-1721106114633_r-1"
            f"&internal_ext=internal_src:dim|wss_push_room_id:{self.real_room_id}|"
            f"wss_push_did:7319483754668557238|first_req_ms:1721106114541|fetch_time:1721106114633|"
            f"seq:1|wss_info:0-1721106114633-0-0|wrds_v:7392094459690748497"
            f"&host=https://live.douyin.com&aid=6383&live_id=1&did_rule=3"
            f"&endpoint=live_pc&support_wrds=1&user_unique_id=7319483754668557238"
            f"&im_path=/webcast/im/fetch/&identity=audience"
            f"&need_persist_msg_count=15&insert_task_id=&live_reason="
            f"&room_id={self.real_room_id}&heartbeatDuration=0"
        )
    
    def _on_open(self, ws):
        """WebSocket连接打开"""
        logger.info(f"✅ 已连接到直播间 {self.room_id}")
        self.is_connected = True
        
        # 启动心跳
        self._heartbeat_thread = threading.Thread(target=self._send_heartbeat)
        self._heartbeat_thread.daemon = True
        self._heartbeat_thread.start()
        
        # 触发连接事件
        if self._loop:
            asyncio.run_coroutine_threadsafe(
                self.emit(Event(type=EventType.CONNECTED)),
                self._loop
            )
    
    def _on_message(self, ws, message):
        """接收消息"""
        try:
            package = PushFrame().parse(message)
            response = Response().parse(gzip.decompress(package.payload))
            
            # 发送ACK
            if response.need_ack:
                ack = PushFrame(
                    log_id=package.log_id,
                    payload_type='ack',
                    payload=response.internal_ext.encode('utf-8')
                ).SerializeToString()
                ws.send(ack, websocket.ABNF.OPCODE_BINARY)
            
            # 处理消息
            for msg in response.messages_list:
                self._handle_message(msg)
                
        except Exception as e:
            logger.error(f"处理消息错误: {e}")
    
    def _on_error(self, ws, error):
        """WebSocket错误"""
        logger.error(f"WebSocket错误: {error}")
        if self._loop:
            asyncio.run_coroutine_threadsafe(
                self.emit(Event(type=EventType.ERROR, content=str(error))),
                self._loop
            )
    
    def _on_close(self, ws, *args):
        """WebSocket关闭"""
        logger.warning("WebSocket连接已关闭")
        self.is_connected = False
        if self._loop:
            asyncio.run_coroutine_threadsafe(
                self.emit(Event(type=EventType.DISCONNECTED)),
                self._loop
            )
    
    def _send_heartbeat(self):
        """发送心跳包"""
        import time
        while self._running and self._ws:
            try:
                heartbeat = PushFrame(payload_type='hb').SerializeToString()
                self._ws.send(heartbeat, websocket.ABNF.OPCODE_PING)
                logger.debug("💓 发送心跳")
            except Exception as e:
                logger.error(f"心跳发送失败: {e}")
                break
            time.sleep(10)
    
    def _handle_message(self, msg):
        """处理单条消息"""
        method = msg.method
        payload = msg.payload
        
        try:
            handlers = {
                'WebcastChatMessage': self._parse_chat,
                'WebcastGiftMessage': self._parse_gift,
                'WebcastLikeMessage': self._parse_like,
                'WebcastMemberMessage': self._parse_member,
                'WebcastSocialMessage': self._parse_social,
                'WebcastRoomUserSeqMessage': self._parse_room_stats,
                'WebcastControlMessage': self._parse_control,
            }
            
            handler = handlers.get(method)
            if handler:
                handler(payload)
        except Exception as e:
            logger.debug(f"处理消息 {method} 失败: {e}")
    
    def _parse_chat(self, payload):
        """解析聊天消息"""
        message = ChatMessage().parse(payload)
        user = User(
            id=str(message.user.id),
            nickname=message.user.nick_name,
            gender=message.user.gender
        )
        event = ChatEvent(user=user, content=message.content)
        logger.info(f"💬 [{user.nickname}]: {message.content}")
        
        if self._loop:
            asyncio.run_coroutine_threadsafe(self.emit(event), self._loop)
    
    def _parse_gift(self, payload):
        """解析礼物消息"""
        message = GiftMessage().parse(payload)
        user = User(
            id=str(message.user.id),
            nickname=message.user.nick_name,
            gender=message.user.gender
        )
        gift = Gift(
            id=str(message.gift.id),
            name=message.gift.name,
            count=message.combo_count or 1,
            price=message.gift.diamond_count if hasattr(message.gift, 'diamond_count') else 0
        )
        event = GiftEvent(user=user, gift=gift)
        logger.info(f"🎁 {user.nickname} 送出 {gift.name} x{gift.count}")
        
        if self._loop:
            asyncio.run_coroutine_threadsafe(self.emit(event), self._loop)
    
    def _parse_like(self, payload):
        """解析点赞消息"""
        message = LikeMessage().parse(payload)
        user = User(
            id=str(message.user.id),
            nickname=message.user.nick_name
        )
        event = LikeEvent(user=user, count=message.count, total=message.total)
        logger.debug(f"👍 {user.nickname} 点赞 x{message.count}")
        
        if self._loop:
            asyncio.run_coroutine_threadsafe(self.emit(event), self._loop)
    
    def _parse_member(self, payload):
        """解析进入直播间消息"""
        message = MemberMessage().parse(payload)
        user = User(
            id=str(message.user.id),
            nickname=message.user.nick_name,
            gender=message.user.gender
        )
        event = EntranceEvent(user=user, content="进入直播间")
        logger.info(f"🚪 {user.nickname} 进入直播间")
        
        if self._loop:
            asyncio.run_coroutine_threadsafe(self.emit(event), self._loop)
    
    def _parse_social(self, payload):
        """解析关注消息"""
        message = SocialMessage().parse(payload)
        user = User(
            id=str(message.user.id),
            nickname=message.user.nick_name
        )
        event = FollowEvent(user=user, content="关注了主播")
        logger.info(f"➕ {user.nickname} 关注了主播")
        
        if self._loop:
            asyncio.run_coroutine_threadsafe(self.emit(event), self._loop)
    
    def _parse_room_stats(self, payload):
        """解析直播间统计"""
        message = RoomUserSeqMessage().parse(payload)
        event = RoomStatsEvent(
            online_count=message.total,
            total_count=message.total_pv_for_anchor
        )
        logger.debug(f"📊 在线: {message.total}, 累计: {message.total_pv_for_anchor}")
        
        if self._loop:
            asyncio.run_coroutine_threadsafe(self.emit(event), self._loop)
    
    def _parse_control(self, payload):
        """解析控制消息"""
        message = ControlMessage().parse(payload)
        if message.status == 3:
            logger.warning("📴 直播已结束")
            event = Event(type=EventType.LIVE_END)
            if self._loop:
                asyncio.run_coroutine_threadsafe(self.emit(event), self._loop)
            self._running = False
