# -*- coding: utf-8 -*-
"""
插件管理器
支持动态加载/卸载可选功能模块（如美颜、贴纸、大眼瘦脸等）

设计原则：
1. 核心功能保持轻量
2. 重型依赖（如 MediaPipe）作为可选插件
3. 插件按需下载和加载
"""

import os
import sys
import json
import importlib
import importlib.util
from typing import Dict, Optional, Any, List, Callable
from dataclasses import dataclass, field
from enum import Enum
import logging

logger = logging.getLogger(__name__)


class PluginStatus(Enum):
    """插件状态"""
    NOT_INSTALLED = "not_installed"  # 未安装
    INSTALLED = "installed"          # 已安装但未加载
    LOADED = "loaded"                # 已加载
    ERROR = "error"                  # 加载失败


@dataclass
class PluginInfo:
    """插件信息"""
    id: str                          # 插件ID
    name: str                        # 显示名称
    version: str                     # 版本号
    description: str                 # 描述
    entry_point: str                 # 入口模块路径
    dependencies: List[str] = field(default_factory=list)  # Python依赖
    size_mb: float = 0               # 大小(MB)
    status: PluginStatus = PluginStatus.NOT_INSTALLED
    instance: Any = None             # 插件实例


class PluginManager:
    """插件管理器"""
    
    _instance = None
    
    # 可用插件定义
    AVAILABLE_PLUGINS = {
        "beauty_filter": PluginInfo(
            id="beauty_filter",
            name="美颜滤镜",
            version="1.0.0",
            description="实时美颜效果：磨皮、美白、亮度、饱和度、锐化",
            entry_point="src.utils.beauty_filter",
            dependencies=["opencv-python", "numpy"],
            size_mb=0.1
        ),
        "face_sticker": PluginInfo(
            id="face_sticker",
            name="人脸贴纸",
            version="1.0.0",
            description="实时人脸贴纸：帽子、眼镜、胡子等",
            entry_point="src.utils.face_sticker",
            dependencies=["opencv-python", "numpy", "mediapipe"],
            size_mb=99.0  # MediaPipe ~99MB
        ),
        "face_deformation": PluginInfo(
            id="face_deformation",
            name="大眼瘦脸",
            version="1.0.0",
            description="实时人脸变形：大眼、瘦脸效果",
            entry_point="src.utils.face_deformation",
            dependencies=["opencv-python", "numpy", "mediapipe"],
            size_mb=99.0  # 共享 MediaPipe
        ),
    }
    
    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(PluginManager, cls).__new__(cls)
            cls._instance._initialized = False
        return cls._instance
    
    def __init__(self):
        if self._initialized:
            return
            
        self._initialized = True
        self._plugins: Dict[str, PluginInfo] = {}
        self._plugin_dir = self._get_plugin_dir()
        self._config_file = os.path.join(self._plugin_dir, "plugins.json")
        
        # 确保插件目录存在
        os.makedirs(self._plugin_dir, exist_ok=True)
        
        # 加载插件配置
        self._load_config()
        
        # 事件回调
        self._on_plugin_loaded: List[Callable] = []
        self._on_plugin_unloaded: List[Callable] = []
        
    def _get_plugin_dir(self) -> str:
        """获取插件目录"""
        # 开发环境
        if hasattr(sys, '_MEIPASS'):
            # PyInstaller 打包后
            base_dir = os.path.dirname(sys.executable)
        else:
            # 开发环境
            base_dir = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
        
        return os.path.join(base_dir, "plugins")
    
    def _load_config(self):
        """加载插件配置"""
        # 初始化可用插件
        for plugin_id, info in self.AVAILABLE_PLUGINS.items():
            self._plugins[plugin_id] = PluginInfo(
                id=info.id,
                name=info.name,
                version=info.version,
                description=info.description,
                entry_point=info.entry_point,
                dependencies=info.dependencies.copy(),
                size_mb=info.size_mb,
                status=PluginStatus.NOT_INSTALLED
            )
        
        # 加载保存的状态
        if os.path.exists(self._config_file):
            try:
                with open(self._config_file, 'r', encoding='utf-8') as f:
                    config = json.load(f)
                    for plugin_id, status in config.get("installed", {}).items():
                        if plugin_id in self._plugins:
                            self._plugins[plugin_id].status = PluginStatus.INSTALLED
            except Exception as e:
                logger.error(f"加载插件配置失败: {e}")
    
    def _save_config(self):
        """保存插件配置"""
        config = {
            "installed": {
                pid: "installed" 
                for pid, info in self._plugins.items() 
                if info.status in [PluginStatus.INSTALLED, PluginStatus.LOADED]
            }
        }
        try:
            with open(self._config_file, 'w', encoding='utf-8') as f:
                json.dump(config, f, indent=2)
        except Exception as e:
            logger.error(f"保存插件配置失败: {e}")
    
    def get_all_plugins(self) -> Dict[str, PluginInfo]:
        """获取所有插件信息"""
        return self._plugins.copy()
    
    def get_plugin(self, plugin_id: str) -> Optional[PluginInfo]:
        """获取指定插件信息"""
        return self._plugins.get(plugin_id)
    
    def is_loaded(self, plugin_id: str) -> bool:
        """检查插件是否已加载"""
        plugin = self._plugins.get(plugin_id)
        return plugin and plugin.status == PluginStatus.LOADED
    
    def check_dependencies(self, plugin_id: str) -> List[str]:
        """检查插件依赖，返回缺失的依赖列表"""
        plugin = self._plugins.get(plugin_id)
        if not plugin:
            return []
        
        missing = []
        for dep in plugin.dependencies:
            # 转换包名（如 opencv-python -> cv2）
            import_name = {
                "opencv-python": "cv2",
                "mediapipe": "mediapipe",
                "numpy": "numpy",
            }.get(dep, dep)
            
            try:
                importlib.import_module(import_name)
            except ImportError:
                missing.append(dep)
        
        return missing
    
    def load_plugin(self, plugin_id: str) -> bool:
        """加载插件"""
        plugin = self._plugins.get(plugin_id)
        if not plugin:
            logger.error(f"插件不存在: {plugin_id}")
            return False
        
        if plugin.status == PluginStatus.LOADED:
            return True
        
        # 检查依赖
        missing = self.check_dependencies(plugin_id)
        if missing:
            logger.error(f"插件 {plugin_id} 缺少依赖: {missing}")
            plugin.status = PluginStatus.ERROR
            return False
        
        try:
            # 动态导入模块
            module = importlib.import_module(plugin.entry_point)
            
            # 获取主类（约定：模块中与文件名同名的类）
            class_name = {
                "beauty_filter": "BeautyFilter",
                "face_sticker": "FaceStickerFilter",
                "face_deformation": "FaceDeformation",
            }.get(plugin_id)
            
            if class_name and hasattr(module, class_name):
                plugin.instance = getattr(module, class_name)()
            else:
                plugin.instance = module
            
            plugin.status = PluginStatus.LOADED
            self._save_config()
            
            # 触发回调
            for callback in self._on_plugin_loaded:
                try:
                    callback(plugin_id, plugin.instance)
                except Exception as e:
                    logger.error(f"插件加载回调失败: {e}")
            
            logger.info(f"插件 {plugin.name} 加载成功")
            return True
            
        except Exception as e:
            logger.error(f"加载插件 {plugin_id} 失败: {e}")
            plugin.status = PluginStatus.ERROR
            return False
    
    def unload_plugin(self, plugin_id: str) -> bool:
        """卸载插件"""
        plugin = self._plugins.get(plugin_id)
        if not plugin:
            return False
        
        if plugin.status != PluginStatus.LOADED:
            return True
        
        try:
            # 触发回调
            for callback in self._on_plugin_unloaded:
                try:
                    callback(plugin_id)
                except Exception as e:
                    logger.error(f"插件卸载回调失败: {e}")
            
            # 清理实例
            if plugin.instance:
                if hasattr(plugin.instance, 'cleanup'):
                    plugin.instance.cleanup()
                plugin.instance = None
            
            plugin.status = PluginStatus.INSTALLED
            logger.info(f"插件 {plugin.name} 已卸载")
            return True
            
        except Exception as e:
            logger.error(f"卸载插件 {plugin_id} 失败: {e}")
            return False
    
    def get_instance(self, plugin_id: str) -> Optional[Any]:
        """获取插件实例"""
        plugin = self._plugins.get(plugin_id)
        if plugin and plugin.status == PluginStatus.LOADED:
            return plugin.instance
        return None
    
    def on_plugin_loaded(self, callback: Callable):
        """注册插件加载回调"""
        self._on_plugin_loaded.append(callback)
    
    def on_plugin_unloaded(self, callback: Callable):
        """注册插件卸载回调"""
        self._on_plugin_unloaded.append(callback)


# 便捷函数
def get_plugin_manager() -> PluginManager:
    """获取插件管理器实例"""
    return PluginManager()
